"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var netModule = require("net");
var metrics_1 = require("../services/metrics");
var Debug = require("debug");
var meter_1 = require("../utils/metrics/meter");
var shimmer = require("shimmer");
var serviceManager_1 = require("../serviceManager");
var NetworkTrafficConfig = (function () {
    function NetworkTrafficConfig() {
    }
    return NetworkTrafficConfig;
}());
exports.NetworkTrafficConfig = NetworkTrafficConfig;
var defaultConfig = {
    upload: false,
    download: false
};
var allEnabled = {
    upload: true,
    download: true
};
var NetworkMetric = (function () {
    function NetworkMetric() {
        this.logger = Debug('axm:features:metrics:network');
    }
    NetworkMetric.prototype.init = function (config) {
        if (config === false)
            return;
        if (config === true) {
            config = allEnabled;
        }
        if (config === undefined) {
            config = defaultConfig;
        }
        this.metricService = serviceManager_1.ServiceManager.get('metrics');
        if (this.metricService === undefined) {
            return this.logger("Failed to load metric service");
        }
        if (config.download === true) {
            this.catchDownload();
        }
        if (config.upload === true) {
            this.catchUpload();
        }
        this.logger('init');
    };
    NetworkMetric.prototype.destroy = function () {
        if (this.timer !== undefined) {
            clearTimeout(this.timer);
        }
        if (this.socketProto !== undefined && this.socketProto !== null) {
            shimmer.unwrap(this.socketProto, 'read');
            shimmer.unwrap(this.socketProto, 'write');
        }
        this.logger('destroy');
    };
    NetworkMetric.prototype.catchDownload = function () {
        var _this = this;
        if (this.metricService === undefined)
            return this.logger("Failed to load metric service");
        var downloadMeter = new meter_1.default({});
        this.metricService.registerMetric({
            name: 'Network In',
            id: 'internal/network/in',
            historic: true,
            type: metrics_1.MetricType.meter,
            implementation: downloadMeter,
            unit: 'MBytes/sec',
            handler: function () {
                return this.implementation.val() / 1024 / 1024;
            }
        });
        setTimeout(function () {
            var property = netModule.Socket.prototype.read;
            var isWrapped = property && property.__wrapped === true;
            if (isWrapped) {
                return _this.logger("Already patched socket read, canceling");
            }
            shimmer.wrap(netModule.Socket.prototype, 'read', function (original) {
                return function () {
                    this.on('data', function (data) {
                        if (typeof data.length === 'number') {
                            downloadMeter.mark(data.length);
                        }
                    });
                    return original.apply(this, arguments);
                };
            });
        }, 500);
    };
    NetworkMetric.prototype.catchUpload = function () {
        var _this = this;
        if (this.metricService === undefined)
            return this.logger("Failed to load metric service");
        var uploadMeter = new meter_1.default();
        this.metricService.registerMetric({
            name: 'Network Out',
            id: 'internal/network/out',
            type: metrics_1.MetricType.meter,
            historic: true,
            implementation: uploadMeter,
            unit: 'MBytes/sec',
            handler: function () {
                return this.implementation.val() / 1024 / 1024;
            }
        });
        setTimeout(function () {
            var property = netModule.Socket.prototype.write;
            var isWrapped = property && property.__wrapped === true;
            if (isWrapped) {
                return _this.logger("Already patched socket write, canceling");
            }
            shimmer.wrap(netModule.Socket.prototype, 'write', function (original) {
                return function (data) {
                    if (typeof data.length === 'number') {
                        uploadMeter.mark(data.length);
                    }
                    return original.apply(this, arguments);
                };
            });
        }, 500);
    };
    return NetworkMetric;
}());
exports.default = NetworkMetric;
//# sourceMappingURL=data:application/json;base64,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