'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
var shimmer = require("shimmer");
var debug_1 = require("debug");
var configuration_1 = require("../configuration");
var serviceManager_1 = require("../serviceManager");
var histogram_1 = require("../utils/metrics/histogram");
var Module = require("module");
var metrics_1 = require("../services/metrics");
var HttpMetricsConfig = (function () {
    function HttpMetricsConfig() {
    }
    return HttpMetricsConfig;
}());
exports.HttpMetricsConfig = HttpMetricsConfig;
var HttpMetrics = (function () {
    function HttpMetrics() {
        this.defaultConf = {
            http: true
        };
        this.metrics = new Map();
        this.logger = debug_1.default('axm:features:metrics:http');
        this.modules = {};
    }
    HttpMetrics.prototype.init = function (config) {
        if (config === undefined)
            return;
        if (config === false)
            return;
        if (typeof config !== 'object') {
            config = this.defaultConf;
        }
        this.logger('init');
        configuration_1.default.configureModule({
            latency: true
        });
        this.metricService = serviceManager_1.ServiceManager.get('metrics');
        if (this.metricService === undefined)
            return this.logger("Failed to load metric service");
        this.logger('hooking to require');
        this.hookRequire();
    };
    HttpMetrics.prototype.registerHttpMetric = function () {
        if (this.metricService === undefined)
            return this.logger("Failed to load metric service");
        var histogram = new histogram_1.default();
        var p50 = {
            name: "HTTP Mean Latency",
            id: 'internal/http/builtin/latency/p50',
            type: metrics_1.MetricType.histogram,
            historic: true,
            implementation: histogram,
            unit: 'ms',
            handler: function () {
                var percentiles = histogram.percentiles([0.5]);
                return percentiles[0.5];
            }
        };
        var p95 = {
            name: "HTTP P95 Latency",
            id: 'internal/http/builtin/latency/p95',
            type: metrics_1.MetricType.histogram,
            historic: true,
            implementation: histogram,
            handler: function () {
                var percentiles = histogram.percentiles([0.95]);
                return percentiles[0.95];
            },
            unit: 'ms'
        };
        var meter = {
            name: 'HTTP',
            historic: true,
            id: 'internal/http/builtin/reqs',
            unit: 'req/min'
        };
        this.metricService.registerMetric(p50);
        this.metricService.registerMetric(p95);
        this.metrics.set('http.latency', histogram);
        this.metrics.set('http.meter', this.metricService.meter(meter));
    };
    HttpMetrics.prototype.registerHttpsMetric = function () {
        if (this.metricService === undefined)
            return this.logger("Failed to load metric service");
        var histogram = new histogram_1.default();
        var p50 = {
            name: "HTTPS Mean Latency",
            id: 'internal/https/builtin/latency/p50',
            type: metrics_1.MetricType.histogram,
            historic: true,
            implementation: histogram,
            unit: 'ms',
            handler: function () {
                var percentiles = histogram.percentiles([0.5]);
                return percentiles[0.5];
            }
        };
        var p95 = {
            name: "HTTPS P95 Latency",
            id: 'internal/https/builtin/latency/p95',
            type: metrics_1.MetricType.histogram,
            historic: true,
            implementation: histogram,
            handler: function () {
                var percentiles = histogram.percentiles([0.95]);
                return percentiles[0.95];
            },
            unit: 'ms'
        };
        var meter = {
            name: 'HTTPS',
            historic: true,
            id: 'internal/https/builtin/reqs',
            unit: 'req/min'
        };
        this.metricService.registerMetric(p50);
        this.metricService.registerMetric(p95);
        this.metrics.set('https.latency', histogram);
        this.metrics.set('https.meter', this.metricService.meter(meter));
    };
    HttpMetrics.prototype.destroy = function () {
        if (this.modules.http !== undefined) {
            this.logger('unwraping http module');
            shimmer.unwrap(this.modules.http, 'emit');
        }
        if (this.modules.https !== undefined) {
            this.logger('unwraping https module');
            shimmer.unwrap(this.modules.https, 'emit');
        }
        if (Module['_load'] && Module['_load'].__io_apm === true) {
            shimmer.unwrap(Module, '_load');
            Module['_load'].__io_apm = undefined;
        }
        this.logger('destroy');
    };
    HttpMetrics.prototype.hookHttp = function (nodule, name) {
        if (nodule.Server === undefined || nodule.Server.prototype === undefined)
            return;
        if (this.modules[name] !== undefined)
            return this.logger("Module " + name + " already hooked");
        this.logger("Hooking to " + name + " module");
        this.modules[name] = nodule.Server.prototype;
        var self = this;
        shimmer.wrap(nodule.Server.prototype, 'emit', function (original) {
            return function (event, req, res) {
                if (event !== 'request')
                    return original.apply(this, arguments);
                var meter = self.metrics.get(name + ".meter");
                if (meter !== undefined) {
                    meter.mark();
                }
                var latency = self.metrics.get(name + ".latency");
                if (latency === undefined)
                    return original.apply(this, arguments);
                if (res === undefined || res === null)
                    return original.apply(this, arguments);
                var startTime = Date.now();
                res.once('finish', function (_) {
                    latency.update(Date.now() - startTime);
                });
                return original.apply(this, arguments);
            };
        });
    };
    HttpMetrics.prototype.hookRequire = function () {
        var self = this;
        var isAlreadyWrapped = typeof Module['_load'] === 'function' && Module['_load'].__io_apm === true;
        if (isAlreadyWrapped)
            return this.logger("Module already wrapped, aborting");
        shimmer.wrap(Module, '_load', function (original) {
            return function (file) {
                var returned = original.apply(original, arguments);
                if (file === 'http' && self.modules['http'] === undefined) {
                    self.registerHttpMetric();
                    self.hookHttp(returned, file);
                }
                else if (file === 'https' && self.modules['https'] === undefined) {
                    self.registerHttpsMetric();
                    self.hookHttp(returned, file);
                }
                return returned;
            };
        });
        Module['_load'].__io_apm = true;
    };
    return HttpMetrics;
}());
exports.default = HttpMetrics;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaHR0cE1ldHJpY3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvbWV0cmljcy9odHRwTWV0cmljcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxZQUFZLENBQUE7O0FBRVosaUNBQWtDO0FBQ2xDLCtCQUF5QjtBQUN6QixrREFBNEM7QUFFNUMsb0RBQWtEO0FBRWxELHdEQUFrRDtBQUlsRCwrQkFBZ0M7QUFDaEMsK0NBSzRCO0FBRTVCO0lBQUE7SUFFQSxDQUFDO0lBQUQsd0JBQUM7QUFBRCxDQUFDLEFBRkQsSUFFQztBQUZZLDhDQUFpQjtBQUk5QjtJQUFBO1FBRVUsZ0JBQVcsR0FBc0I7WUFDdkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFBO1FBQ08sWUFBTyxHQUFxQixJQUFJLEdBQUcsRUFBZSxDQUFBO1FBQ2xELFdBQU0sR0FBUSxlQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQTtRQUVoRCxZQUFPLEdBQVEsRUFBRSxDQUFBO0lBeUszQixDQUFDO0lBdktDLDBCQUFJLEdBQUosVUFBTSxNQUFvQztRQUN4QyxJQUFJLE1BQU0sS0FBSyxTQUFTO1lBQUUsT0FBTTtRQUNoQyxJQUFJLE1BQU0sS0FBSyxLQUFLO1lBQUUsT0FBTTtRQUM1QixJQUFJLE9BQU8sTUFBTSxLQUFLLFFBQVEsRUFBRTtZQUM5QixNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQTtTQUMxQjtRQUNELElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUE7UUFDbkIsdUJBQWEsQ0FBQyxlQUFlLENBQUM7WUFDNUIsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDLENBQUE7UUFDRixJQUFJLENBQUMsYUFBYSxHQUFHLCtCQUFjLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFBO1FBQ2xELElBQUksSUFBSSxDQUFDLGFBQWEsS0FBSyxTQUFTO1lBQUUsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLCtCQUErQixDQUFDLENBQUE7UUFFekYsSUFBSSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFBO1FBQ2pDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQTtJQUNwQixDQUFDO0lBRU8sd0NBQWtCLEdBQTFCO1FBQ0UsSUFBSSxJQUFJLENBQUMsYUFBYSxLQUFLLFNBQVM7WUFBRSxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsK0JBQStCLENBQUMsQ0FBQTtRQUN6RixJQUFNLFNBQVMsR0FBRyxJQUFJLG1CQUFTLEVBQUUsQ0FBQTtRQUNqQyxJQUFNLEdBQUcsR0FBbUI7WUFDMUIsSUFBSSxFQUFFLG1CQUFtQjtZQUN6QixFQUFFLEVBQUUsbUNBQW1DO1lBQ3ZDLElBQUksRUFBRSxvQkFBVSxDQUFDLFNBQVM7WUFDMUIsUUFBUSxFQUFFLElBQUk7WUFDZCxjQUFjLEVBQUUsU0FBUztZQUN6QixJQUFJLEVBQUUsSUFBSTtZQUNWLE9BQU8sRUFBRTtnQkFDUCxJQUFNLFdBQVcsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUUsR0FBRyxDQUFFLENBQUMsQ0FBQTtnQkFDbEQsT0FBTyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUE7WUFDekIsQ0FBQztTQUNGLENBQUE7UUFDRCxJQUFNLEdBQUcsR0FBbUI7WUFDMUIsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixFQUFFLEVBQUUsbUNBQW1DO1lBQ3ZDLElBQUksRUFBRSxvQkFBVSxDQUFDLFNBQVM7WUFDMUIsUUFBUSxFQUFFLElBQUk7WUFDZCxjQUFjLEVBQUUsU0FBUztZQUN6QixPQUFPLEVBQUU7Z0JBQ1AsSUFBTSxXQUFXLEdBQUcsU0FBUyxDQUFDLFdBQVcsQ0FBQyxDQUFFLElBQUksQ0FBRSxDQUFDLENBQUE7Z0JBQ25ELE9BQU8sV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFBO1lBQzFCLENBQUM7WUFDRCxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUE7UUFDRCxJQUFNLEtBQUssR0FBVztZQUNwQixJQUFJLEVBQUUsTUFBTTtZQUNaLFFBQVEsRUFBRSxJQUFJO1lBQ2QsRUFBRSxFQUFFLDRCQUE0QjtZQUNoQyxJQUFJLEVBQUUsU0FBUztTQUNoQixDQUFBO1FBQ0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUE7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUE7UUFDdEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFBO1FBQzNDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFBO0lBQ2pFLENBQUM7SUFFTyx5Q0FBbUIsR0FBM0I7UUFDRSxJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssU0FBUztZQUFFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQywrQkFBK0IsQ0FBQyxDQUFBO1FBQ3pGLElBQU0sU0FBUyxHQUFHLElBQUksbUJBQVMsRUFBRSxDQUFBO1FBQ2pDLElBQU0sR0FBRyxHQUFtQjtZQUMxQixJQUFJLEVBQUUsb0JBQW9CO1lBQzFCLEVBQUUsRUFBRSxvQ0FBb0M7WUFDeEMsSUFBSSxFQUFFLG9CQUFVLENBQUMsU0FBUztZQUMxQixRQUFRLEVBQUUsSUFBSTtZQUNkLGNBQWMsRUFBRSxTQUFTO1lBQ3pCLElBQUksRUFBRSxJQUFJO1lBQ1YsT0FBTyxFQUFFO2dCQUNQLElBQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBRSxHQUFHLENBQUUsQ0FBQyxDQUFBO2dCQUNsRCxPQUFPLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQTtZQUN6QixDQUFDO1NBQ0YsQ0FBQTtRQUNELElBQU0sR0FBRyxHQUFtQjtZQUMxQixJQUFJLEVBQUUsbUJBQW1CO1lBQ3pCLEVBQUUsRUFBRSxvQ0FBb0M7WUFDeEMsSUFBSSxFQUFFLG9CQUFVLENBQUMsU0FBUztZQUMxQixRQUFRLEVBQUUsSUFBSTtZQUNkLGNBQWMsRUFBRSxTQUFTO1lBQ3pCLE9BQU8sRUFBRTtnQkFDUCxJQUFNLFdBQVcsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUUsSUFBSSxDQUFFLENBQUMsQ0FBQTtnQkFDbkQsT0FBTyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUE7WUFDMUIsQ0FBQztZQUNELElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQTtRQUNELElBQU0sS0FBSyxHQUFXO1lBQ3BCLElBQUksRUFBRSxPQUFPO1lBQ2IsUUFBUSxFQUFFLElBQUk7WUFDZCxFQUFFLEVBQUUsNkJBQTZCO1lBQ2pDLElBQUksRUFBRSxTQUFTO1NBQ2hCLENBQUE7UUFDRCxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQTtRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQTtRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLEVBQUUsU0FBUyxDQUFDLENBQUE7UUFDNUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUE7SUFDbEUsQ0FBQztJQUVELDZCQUFPLEdBQVA7UUFDRSxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxLQUFLLFNBQVMsRUFBRTtZQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLHVCQUF1QixDQUFDLENBQUE7WUFDcEMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQTtTQUMxQztRQUNELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3BDLElBQUksQ0FBQyxNQUFNLENBQUMsd0JBQXdCLENBQUMsQ0FBQTtZQUNyQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFBO1NBQzNDO1FBQ0QsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVEsS0FBSyxJQUFJLEVBQUU7WUFDeEQsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUE7WUFDL0IsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVEsR0FBRyxTQUFTLENBQUE7U0FDckM7UUFDRCxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFBO0lBQ3hCLENBQUM7SUFLTyw4QkFBUSxHQUFoQixVQUFrQixNQUFXLEVBQUUsSUFBWTtRQUN6QyxJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssU0FBUyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxLQUFLLFNBQVM7WUFBRSxPQUFNO1FBQ2hGLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxTQUFTO1lBQUUsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVUsSUFBSSxvQkFBaUIsQ0FBQyxDQUFBO1FBQ3pGLElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWMsSUFBSSxZQUFTLENBQUMsQ0FBQTtRQUN4QyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFBO1FBQzVDLElBQU0sSUFBSSxHQUFHLElBQUksQ0FBQTtRQUVqQixPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxVQUFDLFFBQWtCO1lBQy9ELE9BQU8sVUFBVSxLQUFhLEVBQUUsR0FBUSxFQUFFLEdBQVE7Z0JBRWhELElBQUksS0FBSyxLQUFLLFNBQVM7b0JBQUUsT0FBTyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQTtnQkFDL0QsSUFBTSxLQUFLLEdBQXNCLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFJLElBQUksV0FBUSxDQUFDLENBQUE7Z0JBQ2xFLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtvQkFDdkIsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFBO2lCQUNiO2dCQUNELElBQU0sT0FBTyxHQUEwQixJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBSSxJQUFJLGFBQVUsQ0FBQyxDQUFBO2dCQUMxRSxJQUFJLE9BQU8sS0FBSyxTQUFTO29CQUFFLE9BQU8sUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUE7Z0JBQ2pFLElBQUksR0FBRyxLQUFLLFNBQVMsSUFBSSxHQUFHLEtBQUssSUFBSTtvQkFBRSxPQUFPLFFBQVEsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFBO2dCQUM3RSxJQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUE7Z0JBRTVCLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFVBQUEsQ0FBQztvQkFDbEIsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUE7Z0JBQ3hDLENBQUMsQ0FBQyxDQUFBO2dCQUNGLE9BQU8sUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUE7WUFDeEMsQ0FBQyxDQUFBO1FBQ0gsQ0FBQyxDQUFDLENBQUE7SUFDSixDQUFDO0lBRU8saUNBQVcsR0FBbkI7UUFDRSxJQUFNLElBQUksR0FBRyxJQUFJLENBQUE7UUFFakIsSUFBTSxnQkFBZ0IsR0FBRyxPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxVQUFVLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVEsS0FBSyxJQUFJLENBQUE7UUFDbkcsSUFBSSxnQkFBZ0I7WUFBRSxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsa0NBQWtDLENBQUMsQ0FBQTtRQUU1RSxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLEVBQUUsVUFBVSxRQUFrQjtZQUN4RCxPQUFPLFVBQVUsSUFBSTtnQkFFbkIsSUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsU0FBUyxDQUFDLENBQUE7Z0JBRXBELElBQUksSUFBSSxLQUFLLE1BQU0sSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLFNBQVMsRUFBRTtvQkFDekQsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUE7b0JBQ3pCLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFBO2lCQUM5QjtxQkFBTSxJQUFJLElBQUksS0FBSyxPQUFPLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxTQUFTLEVBQUU7b0JBQ2xFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFBO29CQUMxQixJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQTtpQkFDOUI7Z0JBRUQsT0FBTyxRQUFRLENBQUE7WUFDakIsQ0FBQyxDQUFBO1FBQ0gsQ0FBQyxDQUFDLENBQUE7UUFFRixNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQTtJQUNqQyxDQUFDO0lBQ0gsa0JBQUM7QUFBRCxDQUFDLEFBakxELElBaUxDIn0=