#ifndef _EFL_UI_STACK_EO_H_
#define _EFL_UI_STACK_EO_H_

#ifndef _EFL_UI_STACK_EO_CLASS_TYPE
#define _EFL_UI_STACK_EO_CLASS_TYPE

typedef Eo Efl_Ui_Stack;

#endif

#ifndef _EFL_UI_STACK_EO_TYPES
#define _EFL_UI_STACK_EO_TYPES

/** Information of loaded event.
 *
 * @ingroup Efl_Ui
 */
typedef struct _Efl_Ui_Stack_Event_Loaded
{
  Efl_Canvas_Object *content; /**< Loaded content. */
} Efl_Ui_Stack_Event_Loaded;

/** Information of unloaded event.
 *
 * @ingroup Efl_Ui
 */
typedef struct _Efl_Ui_Stack_Event_Unloaded
{
  Efl_Canvas_Object *content; /**< Unloaded content. */
} Efl_Ui_Stack_Event_Unloaded;

/** Information of activated event.
 *
 * @ingroup Efl_Ui
 */
typedef struct _Efl_Ui_Stack_Event_Activated
{
  Efl_Canvas_Object *content; /**< Activated content. */
} Efl_Ui_Stack_Event_Activated;

/** Information of deactivated event.
 *
 * @ingroup Efl_Ui
 */
typedef struct _Efl_Ui_Stack_Event_Deactivated
{
  Efl_Canvas_Object *content; /**< Deactivated content. */
} Efl_Ui_Stack_Event_Deactivated;


#endif
/**
 * @brief Stack widget.
 *
 * Stack widget arranges objects in stack structure by pushing and poping them.
 *
 * @ingroup Efl_Ui_Stack
 */
#define EFL_UI_STACK_CLASS efl_ui_stack_class_get()

EWAPI const Efl_Class *efl_ui_stack_class_get(void);

/**
 * @brief Pushes a new object to the top of the stack and shows it.
 *
 * @param[in] obj The object.
 * @param[in] content The pushed object which becomes the top content of the
 * stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_push(Eo *obj, Efl_Canvas_Object *content);

/**
 * @brief Pops the top content from the stack and deletes it.
 *
 * @param[in] obj The object.
 *
 * @return The top content which is removed from the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI Efl_Canvas_Object *efl_ui_stack_pop(Eo *obj);

/**
 * @brief Inserts an object before the given base content in the stack.
 *
 * @param[in] obj The object.
 * @param[in] base_content @c content is inserted before this @c base_content.
 * @param[in] content The inserted object in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_insert_before(Eo *obj, Efl_Canvas_Object *base_content, Efl_Canvas_Object *content);

/**
 * @brief Inserts an object after the given base content in the stack.
 *
 * @param[in] obj The object.
 * @param[in] base_content @c content is inserted after this @c base_content.
 * @param[in] content The inserted object in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_insert_after(Eo *obj, Efl_Canvas_Object *base_content, Efl_Canvas_Object *content);

/**
 * @brief Inserts an object at the given place in the stack.
 *
 * @param[in] obj The object.
 * @param[in] index The index of the inserted object in the stack. @c index
 * begins from bottom to top of the stack. @c index of the bottom content is 0.
 * @param[in] content The inserted object in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_insert_at(Eo *obj, int index, Efl_Canvas_Object *content);

/**
 * @brief Removes the given content in the stack.
 *
 * @param[in] obj The object.
 * @param[in] content The removed content from the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_remove(Eo *obj, Efl_Canvas_Object *content);

/**
 * @brief Removes a content matched to the given index in the stack.
 *
 * @param[in] obj The object.
 * @param[in] index The index of the removed object in the stack. @c index
 * begins from bottom to top of the stack. @c index of the bottom content is 0.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI void efl_ui_stack_remove_at(Eo *obj, int index);

/**
 * @brief Gets the index of the given content in the stack. The index begins
 * from bottom to top of the stack. The index of the bottom content is 0.
 *
 * @param[in] obj The object.
 * @param[in] content The content matched to the index to be returned in the
 * stack.
 *
 * @return The index of @c content in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI int efl_ui_stack_index_get(Eo *obj, Efl_Canvas_Object *content);

/**
 * @brief Gets the content matched to the given index in the stack.
 *
 * @param[in] obj The object.
 * @param[in] index The index of the content to be returned in the stack.
 *
 * @return The content matched to @c index in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI Efl_Canvas_Object *efl_ui_stack_content_get(Eo *obj, int index);

/**
 * @brief Gets the top content in the stack.
 *
 * @param[in] obj The object.
 *
 * @return The top content in the stack.
 *
 * @ingroup Efl_Ui_Stack
 */
EOAPI Efl_Canvas_Object *efl_ui_stack_top(Eo *obj);

EWAPI extern const Efl_Event_Description _EFL_UI_STACK_EVENT_LOADED;

/** Called when content is loaded right before transition.
 * @return Efl_Ui_Stack_Event_Loaded
 *
 * @ingroup Efl_Ui_Stack
 */
#define EFL_UI_STACK_EVENT_LOADED (&(_EFL_UI_STACK_EVENT_LOADED))

EWAPI extern const Efl_Event_Description _EFL_UI_STACK_EVENT_UNLOADED;

/** Called when content is unloaded right after being deactivated.
 * @return Efl_Ui_Stack_Event_Unloaded
 *
 * @ingroup Efl_Ui_Stack
 */
#define EFL_UI_STACK_EVENT_UNLOADED (&(_EFL_UI_STACK_EVENT_UNLOADED))

EWAPI extern const Efl_Event_Description _EFL_UI_STACK_EVENT_ACTIVATED;

/** Called when content is activated right after transition.
 * @return Efl_Ui_Stack_Event_Activated
 *
 * @ingroup Efl_Ui_Stack
 */
#define EFL_UI_STACK_EVENT_ACTIVATED (&(_EFL_UI_STACK_EVENT_ACTIVATED))

EWAPI extern const Efl_Event_Description _EFL_UI_STACK_EVENT_DEACTIVATED;

/** Called when content is deactivated right after transition.
 * @return Efl_Ui_Stack_Event_Deactivated
 *
 * @ingroup Efl_Ui_Stack
 */
#define EFL_UI_STACK_EVENT_DEACTIVATED (&(_EFL_UI_STACK_EVENT_DEACTIVATED))

#endif
