'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _createClass2 = require('babel-runtime/helpers/createClass');

var _createClass3 = _interopRequireDefault(_createClass2);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _utils = require('../utils');

var _utils2 = _interopRequireDefault(_utils);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Internal module. Management of outgoing room key requests.
 *
 * See https://docs.google.com/document/d/1m4gQkcnJkxNuBmb5NoFCIadIY-DyqqNAS3lloE73BlQ
 * for draft documentation on what we're supposed to be implementing here.
 *
 * @module
 */

// delay between deciding we want some keys, and sending out the request, to
// allow for (a) it turning up anyway, (b) grouping requests together
/*
Copyright 2017 Vector Creations Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

var SEND_KEY_REQUESTS_DELAY_MS = 500;

/** possible states for a room key request
 *
 * The state machine looks like:
 *
 *     |         (cancellation sent)
 *     | .-------------------------------------------------.
 *     | |                                                 |
 *     V V       (cancellation requested)                  |
 *   UNSENT  -----------------------------+                |
 *     |                                  |                |
 *     |                                  |                |
 *     | (send successful)                |  CANCELLATION_PENDING_AND_WILL_RESEND
 *     V                                  |                Λ
 *    SENT                                |                |
 *     |--------------------------------  |  --------------'
 *     |                                  |  (cancellation requested with intent
 *     |                                  |   to resend the original request)
 *     |                                  |
 *     | (cancellation requested)         |
 *     V                                  |
 * CANCELLATION_PENDING                   |
 *     |                                  |
 *     | (cancellation sent)              |
 *     V                                  |
 * (deleted)  <---------------------------+
 *
 * @enum {number}
 */
var ROOM_KEY_REQUEST_STATES = {
    /** request not yet sent */
    UNSENT: 0,

    /** request sent, awaiting reply */
    SENT: 1,

    /** reply received, cancellation not yet sent */
    CANCELLATION_PENDING: 2,

    /**
     * Cancellation not yet sent and will transition to UNSENT instead of
     * being deleted once the cancellation has been sent.
     */
    CANCELLATION_PENDING_AND_WILL_RESEND: 3
};

var OutgoingRoomKeyRequestManager = function () {
    function OutgoingRoomKeyRequestManager(baseApis, deviceId, cryptoStore) {
        (0, _classCallCheck3.default)(this, OutgoingRoomKeyRequestManager);

        this._baseApis = baseApis;
        this._deviceId = deviceId;
        this._cryptoStore = cryptoStore;

        // handle for the delayed call to _sendOutgoingRoomKeyRequests. Non-null
        // if the callback has been set, or if it is still running.
        this._sendOutgoingRoomKeyRequestsTimer = null;

        // sanity check to ensure that we don't end up with two concurrent runs
        // of _sendOutgoingRoomKeyRequests
        this._sendOutgoingRoomKeyRequestsRunning = false;

        this._clientRunning = false;
    }

    /**
     * Called when the client is started. Sets background processes running.
     */


    (0, _createClass3.default)(OutgoingRoomKeyRequestManager, [{
        key: 'start',
        value: function start() {
            this._clientRunning = true;

            // set the timer going, to handle any requests which didn't get sent
            // on the previous run of the client.
            this._startTimer();
        }

        /**
         * Called when the client is stopped. Stops any running background processes.
         */

    }, {
        key: 'stop',
        value: function stop() {
            console.log('stopping OutgoingRoomKeyRequestManager');
            // stop the timer on the next run
            this._clientRunning = false;
        }

        /**
         * Send off a room key request, if we haven't already done so.
         *
         * The `requestBody` is compared (with a deep-equality check) against
         * previous queued or sent requests and if it matches, no change is made.
         * Otherwise, a request is added to the pending list, and a job is started
         * in the background to send it.
         *
         * @param {module:crypto~RoomKeyRequestBody} requestBody
         * @param {Array<{userId: string, deviceId: string}>} recipients
         *
         * @returns {Promise} resolves when the request has been added to the
         *    pending list (or we have established that a similar request already
         *    exists)
         */

    }, {
        key: 'sendRoomKeyRequest',
        value: function sendRoomKeyRequest(requestBody, recipients) {
            var _this = this;

            return this._cryptoStore.getOrAddOutgoingRoomKeyRequest({
                requestBody: requestBody,
                recipients: recipients,
                requestId: this._baseApis.makeTxnId(),
                state: ROOM_KEY_REQUEST_STATES.UNSENT
            }).then(function (req) {
                if (req.state === ROOM_KEY_REQUEST_STATES.UNSENT) {
                    _this._startTimer();
                }
            });
        }

        /**
         * Cancel room key requests, if any match the given requestBody
         *
         * @param {module:crypto~RoomKeyRequestBody} requestBody
         * @param {boolean} andResend if true, transition to UNSENT instead of
         *                            deleting after sending cancellation.
         *
         * @returns {Promise} resolves when the request has been updated in our
         *    pending list.
         */

    }, {
        key: 'cancelRoomKeyRequest',
        value: function cancelRoomKeyRequest(requestBody) {
            var _this2 = this;

            var andResend = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

            return this._cryptoStore.getOutgoingRoomKeyRequest(requestBody).then(function (req) {
                if (!req) {
                    // no request was made for this key
                    return;
                }
                switch (req.state) {
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING:
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND:
                        // nothing to do here
                        return;

                    case ROOM_KEY_REQUEST_STATES.UNSENT:
                        // just delete it

                        // FIXME: ghahah we may have attempted to send it, and
                        // not yet got a successful response. So the server
                        // may have seen it, so we still need to send a cancellation
                        // in that case :/

                        console.log('deleting unnecessary room key request for ' + stringifyRequestBody(requestBody));
                        return _this2._cryptoStore.deleteOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.UNSENT);

                    case ROOM_KEY_REQUEST_STATES.SENT:
                        {
                            // If `andResend` is set, transition to UNSENT once the
                            // cancellation has successfully been sent.
                            var state = andResend ? ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND : ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING;
                            // send a cancellation.
                            return _this2._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.SENT, {
                                state: state,
                                cancellationTxnId: _this2._baseApis.makeTxnId()
                            }).then(function (updatedReq) {
                                if (!updatedReq) {
                                    // updateOutgoingRoomKeyRequest couldn't find the
                                    // request in state ROOM_KEY_REQUEST_STATES.SENT,
                                    // so we must have raced with another tab to mark
                                    // the request cancelled. There is no point in
                                    // sending another cancellation since the other tab
                                    // will do it.
                                    console.log('Tried to cancel room key request for ' + stringifyRequestBody(requestBody) + ' but it was already cancelled in another tab');
                                    return;
                                }

                                // We don't want to wait for the timer, so we send it
                                // immediately. (We might actually end up racing with the timer,
                                // but that's ok: even if we make the request twice, we'll do it
                                // with the same transaction_id, so only one message will get
                                // sent).
                                //
                                // (We also don't want to wait for the response from the server
                                // here, as it will slow down processing of received keys if we
                                // do.)
                                _this2._sendOutgoingRoomKeyRequestCancellation(updatedReq, andResend).catch(function (e) {
                                    console.error("Error sending room key request cancellation;" + " will retry later.", e);
                                    _this2._startTimer();
                                }).then(function () {
                                    if (!andResend) return;
                                    // The request has transitioned from
                                    // CANCELLATION_PENDING_AND_WILL_RESEND to UNSENT. We
                                    // still need to resend the request which is now UNSENT, so
                                    // start the timer if it isn't already started.
                                    _this2._startTimer();
                                });
                            });
                        }
                    default:
                        throw new Error('unhandled state: ' + req.state);
                }
            });
        }

        // start the background timer to send queued requests, if the timer isn't
        // already running

    }, {
        key: '_startTimer',
        value: function _startTimer() {
            var _this3 = this;

            if (this._sendOutgoingRoomKeyRequestsTimer) {
                return;
            }

            var startSendingOutgoingRoomKeyRequests = function startSendingOutgoingRoomKeyRequests() {
                if (_this3._sendOutgoingRoomKeyRequestsRunning) {
                    throw new Error("RoomKeyRequestSend already in progress!");
                }
                _this3._sendOutgoingRoomKeyRequestsRunning = true;

                _this3._sendOutgoingRoomKeyRequests().finally(function () {
                    _this3._sendOutgoingRoomKeyRequestsRunning = false;
                }).catch(function (e) {
                    // this should only happen if there is an indexeddb error,
                    // in which case we're a bit stuffed anyway.
                    console.warn('error in OutgoingRoomKeyRequestManager: ' + e);
                }).done();
            };

            this._sendOutgoingRoomKeyRequestsTimer = global.setTimeout(startSendingOutgoingRoomKeyRequests, SEND_KEY_REQUESTS_DELAY_MS);
        }

        // look for and send any queued requests. Runs itself recursively until
        // there are no more requests, or there is an error (in which case, the
        // timer will be restarted before the promise resolves).

    }, {
        key: '_sendOutgoingRoomKeyRequests',
        value: function _sendOutgoingRoomKeyRequests() {
            var _this4 = this;

            if (!this._clientRunning) {
                this._sendOutgoingRoomKeyRequestsTimer = null;
                return _bluebird2.default.resolve();
            }

            console.log("Looking for queued outgoing room key requests");

            return this._cryptoStore.getOutgoingRoomKeyRequestByState([ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND, ROOM_KEY_REQUEST_STATES.UNSENT]).then(function (req) {
                if (!req) {
                    console.log("No more outgoing room key requests");
                    _this4._sendOutgoingRoomKeyRequestsTimer = null;
                    return;
                }

                var prom = void 0;
                switch (req.state) {
                    case ROOM_KEY_REQUEST_STATES.UNSENT:
                        prom = _this4._sendOutgoingRoomKeyRequest(req);
                        break;
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING:
                        prom = _this4._sendOutgoingRoomKeyRequestCancellation(req);
                        break;
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND:
                        prom = _this4._sendOutgoingRoomKeyRequestCancellation(req, true);
                        break;
                }

                return prom.then(function () {
                    // go around the loop again
                    return _this4._sendOutgoingRoomKeyRequests();
                }).catch(function (e) {
                    console.error("Error sending room key request; will retry later.", e);
                    _this4._sendOutgoingRoomKeyRequestsTimer = null;
                    _this4._startTimer();
                }).done();
            });
        }

        // given a RoomKeyRequest, send it and update the request record

    }, {
        key: '_sendOutgoingRoomKeyRequest',
        value: function _sendOutgoingRoomKeyRequest(req) {
            var _this5 = this;

            console.log('Requesting keys for ' + stringifyRequestBody(req.requestBody) + (' from ' + stringifyRecipientList(req.recipients)) + ('(id ' + req.requestId + ')'));

            var requestMessage = {
                action: "request",
                requesting_device_id: this._deviceId,
                request_id: req.requestId,
                body: req.requestBody
            };

            return this._sendMessageToDevices(requestMessage, req.recipients, req.requestId).then(function () {
                return _this5._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.UNSENT, { state: ROOM_KEY_REQUEST_STATES.SENT });
            });
        }

        // Given a RoomKeyRequest, cancel it and delete the request record unless
        // andResend is set, in which case transition to UNSENT.

    }, {
        key: '_sendOutgoingRoomKeyRequestCancellation',
        value: function _sendOutgoingRoomKeyRequestCancellation(req, andResend) {
            var _this6 = this;

            console.log('Sending cancellation for key request for ' + (stringifyRequestBody(req.requestBody) + ' to ') + (stringifyRecipientList(req.recipients) + ' ') + ('(cancellation id ' + req.cancellationTxnId + ')'));

            var requestMessage = {
                action: "request_cancellation",
                requesting_device_id: this._deviceId,
                request_id: req.requestId
            };

            return this._sendMessageToDevices(requestMessage, req.recipients, req.cancellationTxnId).then(function () {
                if (andResend) {
                    // We want to resend, so transition to UNSENT
                    return _this6._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND, { state: ROOM_KEY_REQUEST_STATES.UNSENT });
                }
                return _this6._cryptoStore.deleteOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING);
            });
        }

        // send a RoomKeyRequest to a list of recipients

    }, {
        key: '_sendMessageToDevices',
        value: function _sendMessageToDevices(message, recipients, txnId) {
            var contentMap = {};
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = (0, _getIterator3.default)(recipients), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var recip = _step.value;

                    if (!contentMap[recip.userId]) {
                        contentMap[recip.userId] = {};
                    }
                    contentMap[recip.userId][recip.deviceId] = message;
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            return this._baseApis.sendToDevice('m.room_key_request', contentMap, txnId);
        }
    }]);
    return OutgoingRoomKeyRequestManager;
}();

exports.default = OutgoingRoomKeyRequestManager;


function stringifyRequestBody(requestBody) {
    // we assume that the request is for megolm keys, which are identified by
    // room id and session id
    return requestBody.room_id + " / " + requestBody.session_id;
}

function stringifyRecipientList(recipients) {
    return '[' + _utils2.default.map(recipients, function (r) {
        return r.userId + ':' + r.deviceId;
    }).join(",") + ']';
}
//# sourceMappingURL=OutgoingRoomKeyRequestManager.js.map