'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var configuration_1 = require("./configuration");
var debug_1 = require("debug");
var serviceManager_1 = require("./serviceManager");
var transport_1 = require("./services/transport");
var featureManager_1 = require("./featureManager");
var actions_1 = require("./services/actions");
var metrics_1 = require("./services/metrics");
var constants_1 = require("./constants");
var runtimeStats_1 = require("./services/runtimeStats");
var entrypoint_1 = require("./features/entrypoint");
var IOConfig = (function () {
    function IOConfig() {
        this.catchExceptions = true;
        this.profiling = true;
        this.tracing = false;
        this.standalone = false;
    }
    return IOConfig;
}());
exports.IOConfig = IOConfig;
exports.defaultConfig = {
    catchExceptions: true,
    profiling: true,
    metrics: {
        v8: true,
        network: false,
        eventLoop: true,
        runtime: true,
        http: true
    },
    standalone: false,
    tracing: false
};
var PMX = (function () {
    function PMX() {
        this.featureManager = new featureManager_1.FeatureManager();
        this.transport = null;
        this.actionService = null;
        this.metricService = null;
        this.runtimeStatsService = null;
        this.logger = debug_1.default('axm:main');
        this.initialized = false;
        this.Entrypoint = entrypoint_1.Entrypoint;
    }
    PMX.prototype.init = function (config) {
        var callsite = (new Error().stack || '').split('\n')[2];
        if (callsite && callsite.length > 0) {
            this.logger("init from " + callsite);
        }
        if (this.initialized === true) {
            this.logger("Calling init but was already the case, destroying and recreating");
            this.destroy();
        }
        if (config === undefined) {
            config = exports.defaultConfig;
        }
        this.transport = transport_1.createTransport(config.standalone === true ? 'websocket' : 'ipc', config.apmOptions);
        serviceManager_1.ServiceManager.set('transport', this.transport);
        if (constants_1.canUseInspector()) {
            var Inspector = require('./services/inspector');
            var inspectorService = new Inspector();
            inspectorService.init();
            serviceManager_1.ServiceManager.set('inspector', inspectorService);
        }
        this.actionService = new actions_1.ActionService();
        this.actionService.init();
        serviceManager_1.ServiceManager.set('actions', this.actionService);
        this.metricService = new metrics_1.MetricService();
        this.metricService.init();
        serviceManager_1.ServiceManager.set('metrics', this.metricService);
        this.runtimeStatsService = new runtimeStats_1.RuntimeStatsService();
        this.runtimeStatsService.init();
        if (this.runtimeStatsService.isEnabled()) {
            serviceManager_1.ServiceManager.set('runtimeStats', this.runtimeStatsService);
        }
        this.featureManager.init(config);
        configuration_1.default.init(config);
        this.initialConfig = config;
        this.initialized = true;
        return this;
    };
    PMX.prototype.destroy = function () {
        this.logger('destroy');
        this.featureManager.destroy();
        if (this.actionService !== null) {
            this.actionService.destroy();
        }
        if (this.transport !== null) {
            this.transport.destroy();
        }
        if (this.metricService !== null) {
            this.metricService.destroy();
        }
        if (this.runtimeStatsService !== null) {
            this.runtimeStatsService.destroy();
        }
        var inspectorService = serviceManager_1.ServiceManager.get('inspector');
        if (inspectorService !== undefined) {
            inspectorService.destroy();
        }
    };
    PMX.prototype.getConfig = function () {
        return this.initialConfig;
    };
    PMX.prototype.notifyError = function (error, context) {
        var notify = this.featureManager.get('notify');
        return notify.notifyError(error, context);
    };
    PMX.prototype.metrics = function (metric) {
        var e_1, _a;
        var res = [];
        if (metric === undefined || metric === null) {
            console.error("Received empty metric to create");
            console.trace();
            return [];
        }
        var metrics = !Array.isArray(metric) ? [metric] : metric;
        try {
            for (var metrics_2 = tslib_1.__values(metrics), metrics_2_1 = metrics_2.next(); !metrics_2_1.done; metrics_2_1 = metrics_2.next()) {
                var metric_1 = metrics_2_1.value;
                if (typeof metric_1.name !== 'string') {
                    console.error("Trying to create a metrics without a name", metric_1);
                    console.trace();
                    res.push({});
                    continue;
                }
                if (metric_1.type === undefined) {
                    metric_1.type = metrics_1.MetricType.gauge;
                }
                switch (metric_1.type) {
                    case metrics_1.MetricType.counter: {
                        res.push(this.counter(metric_1));
                        continue;
                    }
                    case metrics_1.MetricType.gauge: {
                        res.push(this.gauge(metric_1));
                        continue;
                    }
                    case metrics_1.MetricType.histogram: {
                        res.push(this.histogram(metric_1));
                        continue;
                    }
                    case metrics_1.MetricType.meter: {
                        res.push(this.meter(metric_1));
                        continue;
                    }
                    case metrics_1.MetricType.metric: {
                        res.push(this.gauge(metric_1));
                        continue;
                    }
                    default: {
                        console.error("Invalid metric type " + metric_1.type + " for metric " + metric_1.name);
                        console.trace();
                        res.push({});
                        continue;
                    }
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (metrics_2_1 && !metrics_2_1.done && (_a = metrics_2.return)) _a.call(metrics_2);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return res;
    };
    PMX.prototype.histogram = function (config) {
        if (typeof config === 'string') {
            config = {
                name: config,
                measurement: metrics_1.MetricMeasurements.mean
            };
        }
        if (this.metricService === null) {
            return console.trace("Tried to register a metric without initializing @pm2/io");
        }
        return this.metricService.histogram(config);
    };
    PMX.prototype.metric = function (config) {
        if (typeof config === 'string') {
            config = {
                name: config
            };
        }
        if (this.metricService === null) {
            return console.trace("Tried to register a metric without initializing @pm2/io");
        }
        return this.metricService.metric(config);
    };
    PMX.prototype.gauge = function (config) {
        if (typeof config === 'string') {
            config = {
                name: config
            };
        }
        if (this.metricService === null) {
            return console.trace("Tried to register a metric without initializing @pm2/io");
        }
        return this.metricService.metric(config);
    };
    PMX.prototype.counter = function (config) {
        if (typeof config === 'string') {
            config = {
                name: config
            };
        }
        if (this.metricService === null) {
            return console.trace("Tried to register a metric without initializing @pm2/io");
        }
        return this.metricService.counter(config);
    };
    PMX.prototype.meter = function (config) {
        if (typeof config === 'string') {
            config = {
                name: config
            };
        }
        if (this.metricService === null) {
            return console.trace("Tried to register a metric without initializing @pm2/io");
        }
        return this.metricService.meter(config);
    };
    PMX.prototype.action = function (name, opts, fn) {
        if (typeof name === 'object') {
            var tmp = name;
            name = tmp.name;
            opts = tmp.options;
            fn = tmp.action;
        }
        if (this.actionService === null) {
            return console.trace("Tried to register a action without initializing @pm2/io");
        }
        return this.actionService.registerAction(name, opts, fn);
    };
    PMX.prototype.onExit = function (callback) {
        if (typeof callback === 'function') {
            var onExit = require('signal-exit');
            return onExit(callback);
        }
    };
    PMX.prototype.emit = function (name, data) {
        var events = this.featureManager.get('events');
        return events.emit(name, data);
    };
    PMX.prototype.initModule = function (opts, cb) {
        if (!opts)
            opts = {};
        if (opts.reference) {
            opts.name = opts.reference;
            delete opts.reference;
        }
        opts = Object.assign({
            widget: {}
        }, opts);
        opts.widget = Object.assign({
            type: 'generic',
            logo: 'https://app.keymetrics.io/img/logo/keymetrics-300.png',
            theme: ['#111111', '#1B2228', '#807C7C', '#807C7C']
        }, opts.widget);
        opts.isModule = true;
        opts = configuration_1.default.init(opts);
        return typeof cb === 'function' ? cb(null, opts) : opts;
    };
    PMX.prototype.expressErrorHandler = function () {
        var notify = this.featureManager.get('notify');
        return notify.expressErrorHandler();
    };
    PMX.prototype.koaErrorHandler = function () {
        var notify = this.featureManager.get('notify');
        return notify.koaErrorHandler();
    };
    return PMX;
}());
exports.default = PMX;
//# sourceMappingURL=data:application/json;base64,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