package Lire::Config::DlfAnalyserSpec;

use strict;

use base qw/Lire::Config::PluginSpec/;

use Lire::PluginManager;
use Lire::Config::OptionSpec;
use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::Config::DlfAnalyserSpec - PluginSpec to select an analyser.

=head1 DESCRIPTION

This Lire::Config::TypeSpec is a PluginSpec which can be used to
select DlfAnalyser that should be used to create the Dlf records of an
extended or derived schema.

=head2 new( 'name' => $dst_stream_name, ['src_schema' => $from_schema])

The name of the type specification is the name of the schema for which
the analyser is selected. If the 'src_stream' parameter is used, only
analysers which uses that value as their src_schema() will be
selected.

The constructor will fill the PluginSpec automatically with the
available options and it will be set the first analyser as default.

=cut

sub new {
    my $self = shift->SUPER::new( @_ );

    my %args = @_;

    $self->add( new Lire::Config::OptionSpec( 'name' => 'none',
                                              'summary' => N__( 'No analyser' ),
                                              'description' => '<para>' . join( "" ,N__( 'No analyser will be used.' )) . '</para>' ) );

    my $analysers = Lire::PluginManager->analysers_by_dst( $self->name() );
    my $default = 'none';
    foreach my $analyser ( sort { $a->name() cmp $b->name() } @$analysers  ) {
        next if ( defined $args{'src_schema'}
                  && $analyser->src_schema() ne $args{'src_schema'} );
        $default = $analyser->name() if $default eq 'none';
        $self->add( new Lire::Config::OptionSpec( 'name' => $analyser->name(),
                                                  'summary' => $analyser->title(),
                                                  'description' => $analyser->description(),
                                                ) );
    }
    $self->default( $self->instance( 'value' => $default ) );
    $self->{'_src_schema'} = $args{'src_schema'};

    return $self;
}

=pod

=head2 src_schema()

Returns the src_schema() that should be used by the selected
DlfAnalyser. If no particular schema is required, undef will be returned.

=cut

sub src_schema {
    return $_[0]{'_src_schema'};
}

sub summary {
    return __x( 'Analyser to use for {schema} schema',
                'schema' => $_[0]->name() );
}

sub description {
    return '<para>' . __x( q{Select the analyser to use to create {schema} DLF records. If you select 'none', no records will be created. Otherwise, the selected analyser will be run after each ImportJob that created records in the Dlf stream used by the analyser.},
                           'schema' => $_[0]->name() ) . '</para>';
}

1;

__END__

=pod

=head2 SEE ALSO

  Lire::Config::PluginSpec(3pm), Lire::Config::DlfStreamSpec(3pm)

=head1 VERSION

$Id: DlfAnalyserSpec.pm,v 1.2 2006/07/23 13:16:30 vanbaal Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
