"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Debug = require("debug");
const configuration_1 = require("../configuration");
const path_1 = require("path");
const propagation_b3_1 = require("@opencensus/propagation-b3");
const exporter_1 = require("../census/exporter");
const tracer_1 = require("../census/tracer");
const httpMethodToIgnore = [
    'options',
    'head'
];
const defaultTracingConfig = {
    enabled: false,
    outbound: false,
    samplingRate: 0,
    ignoreIncomingPaths: [],
    ignoreOutgoingUrls: [],
    detailedDatabasesCalls: false
};
const enabledTracingConfig = {
    enabled: true,
    outbound: false,
    samplingRate: 0.5,
    ignoreIncomingPaths: [
        (url, request) => {
            const method = (request.method || 'GET').toLowerCase();
            return httpMethodToIgnore.indexOf(method) > -1;
        },
        /(.*).js/,
        /(.*).css/,
        /(.*).png/,
        /(.*).ico/,
        /(.*).svg/,
        /webpack/
    ],
    ignoreOutgoingUrls: [],
    detailedDatabasesCalls: false
};
class TracingFeature {
    constructor() {
        this.logger = Debug('axm:tracing');
    }
    init(config) {
        this.logger('init tracing');
        if (config.tracing === undefined) {
            config.tracing = defaultTracingConfig;
        }
        else if (config.tracing === true) {
            config.tracing = enabledTracingConfig;
        }
        else if (config.tracing === false) {
            config.tracing = defaultTracingConfig;
        }
        if (config.tracing.enabled === false) {
            return this.logger('tracing disabled');
        }
        this.options = Object.assign(enabledTracingConfig, config.tracing);
        if (typeof config.apmOptions === 'object' && typeof config.apmOptions.appName === 'string') {
            this.options.serviceName = config.apmOptions.appName;
        }
        else if (typeof process.env.name === 'string') {
            this.options.serviceName = process.env.name;
        }
        if (config.tracing.ignoreOutgoingUrls === undefined) {
            config.tracing.ignoreOutgoingUrls = enabledTracingConfig.ignoreOutgoingUrls;
        }
        if (config.tracing.ignoreIncomingPaths === undefined) {
            config.tracing.ignoreIncomingPaths = enabledTracingConfig.ignoreIncomingPaths;
        }
        this.exporter = new exporter_1.CustomCensusExporter(this.options);
        if (this.tracer && this.tracer.active) {
            throw new Error(`Tracing was already enabled`);
        }
        this.logger('start census tracer');
        const tracer = tracer_1.Tracing.instance;
        this.tracer = tracer.start({
            exporter: this.exporter,
            plugins: {
                'http': {
                    module: path_1.resolve(__dirname, '../census/plugins/http'),
                    config: config.tracing
                },
                'http2': path_1.resolve(__dirname, '../census/plugins/http2'),
                'https': path_1.resolve(__dirname, '../census/plugins/https'),
                'mongodb-core': {
                    module: path_1.resolve(__dirname, '../census/plugins/mongodb'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'mysql': {
                    module: path_1.resolve(__dirname, '../census/plugins/mysql'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'mysql2': {
                    module: path_1.resolve(__dirname, '../census/plugins/mysql2'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'redis': {
                    module: path_1.resolve(__dirname, '../census/plugins/redis'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'ioredis': {
                    module: path_1.resolve(__dirname, '../census/plugins/ioredis'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'pg': {
                    module: path_1.resolve(__dirname, '../census/plugins/pg'),
                    config: { detailedCommands: config.tracing.detailedDatabasesCalls }
                },
                'vue-server-renderer': {
                    module: path_1.resolve(__dirname, '../census/plugins/vue'),
                    config: {}
                }
            },
            propagation: new propagation_b3_1.B3Format(),
            samplingRate: this.options.samplingRate || 0.5,
            logLevel: this.isDebugEnabled() ? 4 : 1
        });
        configuration_1.default.configureModule({
            census_tracing: true
        });
    }
    isDebugEnabled() {
        return typeof process.env.DEBUG === 'string' &&
            (process.env.DEBUG.indexOf('axm:*') >= 0 || process.env.DEBUG.indexOf('axm:tracing') >= 0);
    }
    getTracer() {
        return this.tracer ? this.tracer.tracer : undefined;
    }
    destroy() {
        if (!this.tracer)
            return;
        this.logger('stop census tracer');
        configuration_1.default.configureModule({
            census_tracing: false
        });
        this.tracer.stop();
    }
}
exports.TracingFeature = TracingFeature;
//# sourceMappingURL=data:application/json;base64,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