#ifndef _EFL_UI_WIN_EO_LEGACY_H_
#define _EFL_UI_WIN_EO_LEGACY_H_

#ifndef _EFL_UI_WIN_EO_CLASS_TYPE
#define _EFL_UI_WIN_EO_CLASS_TYPE

typedef Eo Efl_Ui_Win;

#endif

#ifndef _EFL_UI_WIN_EO_TYPES
#define _EFL_UI_WIN_EO_TYPES

/**
 * @brief Defines the types of window that can be created
 *
 * These are hints set on a window so that a running Window Manager knows how
 * the window should be handled and/or what kind of decorations it should have.
 *
 * Currently, only the X11 backed engines use them.
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_UNKNOWN = -1 /* +1 */, /**< Default, unknown, type */
  EFL_UI_WIN_BASIC, /**< A normal window. Indicates a normal, top-level window.
                     * Almost every window will be created with this type. */
  EFL_UI_WIN_DIALOG_BASIC, /**< Used for simple dialog windows. */
  EFL_UI_WIN_DESKTOP, /**< For special desktop windows, like a background window
                       * holding desktop icons. */
  EFL_UI_WIN_DOCK, /**< The window is used as a dock or panel. Usually would be
                    * kept on top of any other window by the Window Manager. */
  EFL_UI_WIN_TOOLBAR, /**< The window is used to hold a floating toolbar, or
                       * similar. */
  EFL_UI_WIN_MENU, /**< Similar to @.toolbar. */
  EFL_UI_WIN_UTILITY, /**< A persistent utility window, like a toolbox or
                       * palette. */
  EFL_UI_WIN_SPLASH, /**< Splash window for a starting up application. */
  EFL_UI_WIN_DROPDOWN_MENU, /**< The window is a dropdown menu, as when an
                             *  entry in a menubar is clicked. This hint exists
                             * for completion only, as the EFL way of
                             * implementing a menu would not normally use a
                             * separate window for its contents. */
  EFL_UI_WIN_POPUP_MENU, /**< Like @.dropdown_menu, but for the menu triggered
                          * by right-clicking an object. */
  EFL_UI_WIN_TOOLTIP, /**< The window is a tooltip. A short piece of explanatory
                       * text that typically appear after the mouse cursor
                       * hovers over an object for a while. Typically not very
                       * commonly used in the EFL. */
  EFL_UI_WIN_NOTIFICATION, /**< A notification window, like a warning about
                            * battery life or a new E-Mail received. */
  EFL_UI_WIN_COMBO, /**< A window holding the contents of a combo box. Not
                     * usually used in the EFL. */
  EFL_UI_WIN_NAVIFRAME_BASIC = 17 /**< Used for naviframe style replacement with
                                   * a back button instead of a close button.
                                   *
                                   * @since 1.19 */
} Efl_Ui_Win_Type;

/**
 * @brief The different layouts that can be requested for the virtual keyboard.
 *
 * When the application window is being managed by Illume it may request any of
 * the following layouts for the virtual keyboard.
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_KEYBOARD_UNKNOWN = 0, /**< Unknown keyboard state */
  EFL_UI_WIN_KEYBOARD_OFF, /**< Request to deactivate the keyboard */
  EFL_UI_WIN_KEYBOARD_ON, /**< Enable keyboard with default layout */
  EFL_UI_WIN_KEYBOARD_ALPHA, /**< Alpha (a-z) keyboard layout */
  EFL_UI_WIN_KEYBOARD_NUMERIC, /**< Numeric keyboard layout */
  EFL_UI_WIN_KEYBOARD_PIN, /**< PIN keyboard layout */
  EFL_UI_WIN_KEYBOARD_PHONE_NUMBER, /**< Phone keyboard layout */
  EFL_UI_WIN_KEYBOARD_HEX, /**< Hexadecimal numeric keyboard layout */
  EFL_UI_WIN_KEYBOARD_TERMINAL, /**< Full (QWERTY) keyboard layout */
  EFL_UI_WIN_KEYBOARD_PASSWORD, /**< Password keyboard layout */
  EFL_UI_WIN_KEYBOARD_IP, /**< IP keyboard layout */
  EFL_UI_WIN_KEYBOARD_HOST, /**< Host keyboard layout */
  EFL_UI_WIN_KEYBOARD_FILE, /**< File keyboard layout */
  EFL_UI_WIN_KEYBOARD_URL, /**< URL keyboard layout */
  EFL_UI_WIN_KEYBOARD_KEYPAD, /**< Keypad layout */
  EFL_UI_WIN_KEYBOARD_J2ME /**< J2ME keyboard layout */
} Efl_Ui_Win_Keyboard_Mode;

/** Defines the type indicator that can be shown
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_INDICATOR_OFF = 0, /**< Request to deactivate the indicator */
  EFL_UI_WIN_INDICATOR_BG_OPAQUE, /**< The indicator icon is opaque, as is the
                                   * indicator background. The content of
                                   * window is located at the end of the
                                   * indicator. The area of indicator and
                                   * window content are not overlapped */
  EFL_UI_WIN_INDICATOR_BG_TRANSPARENT, /**< The icon of indicator is opaque, but
                                        * the background is transparent. The
                                        * content of window is located under
                                        * the indicator in Z-order. The area of
                                        * indicator and window content are
                                        * overlapped */
  EFL_UI_WIN_INDICATOR_HIDDEN /**< The indicator is hidden so user can see only
                               * the content of window such as in video mode.
                               * If user flicks the upper side of window, the
                               * indicator is shown temporarily. */
} Efl_Ui_Win_Indicator_Mode;

/** Define the keygrab modes of window. A window may send commands to the
 * Keyrouter according this mode, and perform different actions.
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_KEYGRAB_UNKNOWN = 0, /**< Unknown keygrab mode */
  EFL_UI_WIN_KEYGRAB_SHARED = 256 /* 1 >> 8 */, /**< Get the grabbed-key
                                                 * together with the other
                                                 * client windows */
  EFL_UI_WIN_KEYGRAB_TOPMOST = 512 /* 1 >> 9 */, /**< Get the grabbed-key only
                                                  * when window is top of the
                                                  * stack */
  EFL_UI_WIN_KEYGRAB_EXCLUSIVE = 1024 /* 1 >> 10 */, /**< Get the grabbed-key
                                                      * exclusively regardless
                                                      * of window's position */
  EFL_UI_WIN_KEYGRAB_OVERRIDE_EXCLUSIVE = 2048 /* 1 >> 11 */ /**< Get the grabbed-key exclusively
                                                              * regardless of window's position.
                                                              * This is overrided by grabs from the
                                                              * other client window */
} Efl_Ui_Win_Keygrab_Mode;

/** Defines the mode of a modal window
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_MODAL_NONE = 0, /**< The window is not modal window. */
  EFL_UI_WIN_MODAL_MODAL /**< The window is modal window. */
} Efl_Ui_Win_Modal_Mode;

/** Defines the mode of a urgent window.
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_URGENT_NONE = 0, /**< The window is not a urgent window. */
  EFL_UI_WIN_URGENT_URGENT /**< The window is a urgent window. */
} Efl_Ui_Win_Urgent_Mode;

/**
 * @brief Define the move or resize mode of window.
 *
 * The user can request the display server to start moving or resizing the
 * window by combining these modes. However only limited combinations are
 * allowed.
 *
 * Currently, only the following 9 combinations are permitted.  More
 * combinations may be added in future: 1. move, 2. top, 3. bottom, 4. left, 5.
 * right, 6. top | left, 7. top | right, 8. bottom | left, 9. bottom | right.
 *
 * @since 1.19
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_WIN_MOVE_RESIZE_MOVE = 1, /**< Start moving window */
  EFL_UI_WIN_MOVE_RESIZE_TOP = 2 /* 1 >> 1 */, /**< Start resizing window to the
                                                * top */
  EFL_UI_WIN_MOVE_RESIZE_BOTTOM = 4 /* 1 >> 2 */, /**< Start resizing window to
                                                   * the bottom */
  EFL_UI_WIN_MOVE_RESIZE_LEFT = 8 /* 1 >> 3 */, /**< Start resizing window to
                                                 * the left */
  EFL_UI_WIN_MOVE_RESIZE_RIGHT = 16 /* 1 >> 4 */ /**< Start resizing window to
                                                  * the right */
} Efl_Ui_Win_Move_Resize_Mode;


#endif



/**
 * @brief Sets the keyboard mode of the window.
 *
 * @param[in] obj The object.
 * @param[in] mode The mode, one of #Efl_Ui_Win_Keyboard_Mode.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_keyboard_mode_set(Efl_Ui_Win *obj, Efl_Ui_Win_Keyboard_Mode mode);

/**
 * @brief Get the keyboard mode of the window.
 *
 * @param[in] obj The object.
 *
 * @return The mode, one of #Efl_Ui_Win_Keyboard_Mode.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Efl_Ui_Win_Keyboard_Mode elm_win_keyboard_mode_get(const Efl_Ui_Win *obj);





/**
 * @brief Constrain the maximum width and height of a window to the width and
 * height of the screen.
 *
 * When @c constrain is @c true, @c obj will never resize larger than the
 * screen.
 *
 * @param[in] obj The object.
 * @param[in] constrain @c true to restrict the window's maximum size.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_screen_constrain_set(Efl_Ui_Win *obj, Eina_Bool constrain);

/**
 * @brief Get the constraints on the maximum width and height of a window
 * relative to the width and height of the screen.
 *
 * When this function returns @c true, @c obj will never resize larger than the
 * screen.
 *
 * @param[in] obj The object.
 *
 * @return @c true to restrict the window's maximum size.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_screen_constrain_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the window to be skipped by keyboard focus.
 *
 * This sets the window to be skipped by normal keyboard input. This means a
 * window manager will be asked not to focus this window as well as omit it
 * from things like the taskbar, pager, "alt-tab" list etc. etc.
 *
 * Call this and enable it on a window BEFORE you show it for the first time,
 * otherwise it may have no effect.
 *
 * Use this for windows that have only output information or might only be
 * interacted with by the mouse or touchscreen, never for typing. This may have
 * side-effects like making the window non-accessible in some cases unless the
 * window is specially handled. Use this with care.
 *
 * @param[in] obj The object.
 * @param[in] skip The skip flag state ($true if it is to be skipped).
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_prop_focus_skip_set(Efl_Ui_Win *obj, Eina_Bool skip);

/**
 * @brief Window's autohide state.
 *
 * When closing the window in any way outside of the program control, like
 * pressing the X button in the titlebar or using a command from the Window
 * Manager, a "delete,request" signal is emitted to indicate that this event
 * occurred and the developer can take any action, which may include, or not,
 * destroying the window object.
 *
 * When this property is set to @c true, the window will be automatically
 * hidden when this event occurs, after the signal is emitted. If this property
 * is @c false nothing will happen, beyond the event emission.
 *
 * C applications can use this option along with the quit policy
 * @c ELM_POLICY_QUIT_LAST_WINDOW_HIDDEN which allows exiting EFL's main loop
 * when all the windows are hidden.
 *
 * @note @c autodel and @c autohide are not mutually exclusive. The window will
 * be deleted if both are set to @c true.
 *
 * @param[in] obj The object.
 * @param[in] autohide If @c true, the window will automatically hide itself
 * when closed.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_autohide_set(Efl_Ui_Win *obj, Eina_Bool autohide);

/**
 * @brief Window's autohide state.
 *
 * When closing the window in any way outside of the program control, like
 * pressing the X button in the titlebar or using a command from the Window
 * Manager, a "delete,request" signal is emitted to indicate that this event
 * occurred and the developer can take any action, which may include, or not,
 * destroying the window object.
 *
 * When this property is set to @c true, the window will be automatically
 * hidden when this event occurs, after the signal is emitted. If this property
 * is @c false nothing will happen, beyond the event emission.
 *
 * C applications can use this option along with the quit policy
 * @c ELM_POLICY_QUIT_LAST_WINDOW_HIDDEN which allows exiting EFL's main loop
 * when all the windows are hidden.
 *
 * @note @c autodel and @c autohide are not mutually exclusive. The window will
 * be deleted if both are set to @c true.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window will automatically hide itself when closed.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_autohide_get(const Efl_Ui_Win *obj);

/**
 * @brief Set a window object's icon.
 *
 * This sets an image to be used as the icon for the given window, in the
 * window manager decoration part. The exact pixel dimensions of the object
 * (not object size) will be used and the image pixels will be used as-is when
 * this function is called. If the image object has been updated, then call
 * this function again to source the image pixels and place them in the
 * window's icon. Note that only objects of type @ref Efl_Canvas_Image or
 * @ref Efl_Ui_Image are allowed.
 *
 * @param[in] obj The object.
 * @param[in] icon The image object to use for an icon.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_icon_object_set(Efl_Ui_Win *obj, Efl_Canvas_Object *icon);

/**
 * @brief Get the icon object used for the window.
 *
 * The object returns is the one marked by @ref elm_win_icon_object_set as the
 * object to use for the window icon.
 *
 * @param[in] obj The object.
 *
 * @return The Evas image object to use for an icon.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI const Efl_Canvas_Object *elm_win_icon_object_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the iconified state of a window.
 *
 * @param[in] obj The object.
 * @param[in] iconified If @c true, the window is iconified.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_iconified_set(Efl_Ui_Win *obj, Eina_Bool iconified);

/**
 * @brief Get the iconified state of a window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window is iconified.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_iconified_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the maximized state of a window.
 *
 * @param[in] obj The object.
 * @param[in] maximized If @c true, the window is maximized.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_maximized_set(Efl_Ui_Win *obj, Eina_Bool maximized);

/**
 * @brief Get the maximized state of a window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window is maximized.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_maximized_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the fullscreen state of a window.
 *
 * @param[in] obj The object.
 * @param[in] fullscreen If @c true, the window is fullscreen.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_fullscreen_set(Efl_Ui_Win *obj, Eina_Bool fullscreen);

/**
 * @brief Get the fullscreen state of a window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window is fullscreen.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_fullscreen_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the sticky state of the window.
 *
 * Hints the Window Manager that the window in @c obj should be left fixed at
 * its position even when the virtual desktop it's on moves or changes.
 *
 * @param[in] obj The object.
 * @param[in] sticky If @c true, the window's sticky state is enabled.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_sticky_set(Efl_Ui_Win *obj, Eina_Bool sticky);

/**
 * @brief Get the sticky state of the window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window's sticky state is enabled.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_sticky_get(const Efl_Ui_Win *obj);





/**
 * @brief Set the noblank property of a window.
 *
 * The "noblank" property is a way to request the display on which the window
 * is shown does not blank, go to screensaver or otherwise hide or obscure the
 * window. It is intended for uses such as media playback on a television where
 * a user may not want to be interrupted by an idle screen. The noblank
 * property may have no effect if the window is iconified/minimized or hidden.
 *
 * @param[in] obj The object.
 * @param[in] noblank If @c true, the window is set to noblank.
 *
 * @since 1.11
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_noblank_set(Efl_Ui_Win *obj, Eina_Bool noblank);

/**
 * @brief Get the noblank property of a window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window is set to noblank.
 *
 * @since 1.11
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_noblank_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the borderless state of a window.
 *
 * This function requests the Window Manager not to draw any decoration around
 * the window.
 *
 * @param[in] obj The object.
 * @param[in] borderless If @c true, the window is borderless.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_borderless_set(Efl_Ui_Win *obj, Eina_Bool borderless);

/**
 * @brief Get the borderless state of a window.
 *
 * @param[in] obj The object.
 *
 * @return If @c true, the window is borderless.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_borderless_get(const Efl_Ui_Win *obj);

/**
 * @brief The role of the window.
 *
 * It is a hint of how the Window Manager should handle it. Unlike
 * @ref elm_win_type_get and @ref elm_win_name_get this can be changed at
 * runtime.
 *
 * Set the role of the window.
 *
 * @param[in] obj The object.
 * @param[in] role The role to set.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_role_set(Efl_Ui_Win *obj, const char *role);

/**
 * @brief The role of the window.
 *
 * It is a hint of how the Window Manager should handle it. Unlike
 * @ref elm_win_type_get and @ref elm_win_name_get this can be changed at
 * runtime.
 *
 * Get the role of the window.
 *
 * The returned string is an internal one and should not be freed or modified.
 * It will also be invalid if a new role is set or if the window is destroyed.
 *
 * @param[in] obj The object.
 *
 * @return The role to set.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI const char *elm_win_role_get(const Efl_Ui_Win *obj);


/**
 * @brief The window name.
 *
 * The meaning of name depends on the underlying windowing system.
 *
 * The window name is a construction property that can only be set at creation
 * time, before finalize. In C this means inside @c efl_add().
 *
 * @note Once set, it cannot be modified afterwards.
 *
 * No description supplied.
 *
 * @param[in] obj The object.
 *
 * @return Window name
 *
 * @since 1.18
 *
 * @ingroup Efl_Ui_Win
 */
EAPI const char *elm_win_name_get(const Efl_Ui_Win *obj);


/**
 * @brief The type of the window.
 *
 * It is a hint of how the Window Manager should handle it.
 *
 * The window type is a construction property that can only be set at creation
 * time, before finalize. In C this means inside @c efl_add().
 *
 * @note Once set, it cannot be modified afterward.
 *
 * If the object is not window object, returns @c unknown.
 *
 * @param[in] obj The object.
 *
 * @return Window type
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Efl_Ui_Win_Type elm_win_type_get(const Efl_Ui_Win *obj);


/**
 * @brief The hardware acceleration preference for this window.
 *
 * This is a constructor function and can only be called before
 * @ref Efl.Object.finalize.
 *
 * This property overrides the global EFL configuration option
 * "accel_preference" for this single window, and accepts the same syntax.
 *
 * The @c accel string is a freeform C string that indicates what kind of
 * acceleration is preferred. Here "acceleration" generally refers to rendering
 * and the hardware with which the unit application renders GUIs. This may or
 * may not be honored but a best attempt will be made. Known strings are as
 * follows:
 *
 * "gl", "opengl" - try use OpenGL. "3d" - try to use a 3d acceleration unit.
 * "hw", "hardware", "accel" - try any acceleration unit (best possible) "none"
 * - use no acceleration but software instead (since 1.16)
 *
 * Since 1.14, it is also possible to specify some GL properties for the GL
 * window surface. This allows applications to use GLView with depth, stencil
 * and MSAA buffers with direct rendering. The new accel preference string
 * format is thus "{HW Accel}[:depth{value}[:stencil{value}[:msaa{str}$]$]$]".
 *
 * Accepted values for depth are for instance "depth", "depth16", "depth24".
 * Accepted values for stencil are "stencil", "stencil1", "stencil8". For MSAA,
 * only predefined strings are accepted: "msaa", "msaa_low", "msaa_mid" and
 * "msaa_high". The selected configuration is not guaranteed and is only valid
 * in case of GL acceleration. Only the base acceleration string will be saved
 * (e.g. "gl" or "hw").
 *
 * Full examples include:
 *
 * "gl", - try to use OpenGL "hw:depth:stencil", - use HW acceleration with
 * default depth and stencil buffers "opengl:depth24:stencil8:msaa_mid" - use
 * OpenGL with 24-bit depth, 8-bit stencil and a medium number of MSAA samples
 * in the backbuffer.
 *
 * Note that this option may be overriden by environment variables or the
 * configuration option "accel_preference_override".
 *
 * This will return the value of "accel_preference" when the window was
 * created.
 *
 * @param[in] obj The object.
 *
 * @return Acceleration
 *
 * @since 1.18
 *
 * @ingroup Efl_Ui_Win
 */
EAPI const char *elm_win_accel_preference_get(const Efl_Ui_Win *obj);

/**
 * @brief Set the alpha channel state of a window.
 *
 * If @c alpha is true, the alpha channel of the canvas will be enabled
 * possibly making parts of the window completely or partially transparent.
 * This is also subject to the underlying system supporting it, for example a
 * system using a compositing manager.
 *
 * @note Alpha window can be enabled automatically by window theme style's
 * property. If "alpha" data.item is "1" or "true" in window style(eg.
 * elm/win/base/default), the window is switched to alpha automatically without
 * the explicit api call.
 *
 * @param[in] obj The object.
 * @param[in] alpha @c true if the window alpha channel is enabled, @c false
 * otherwise.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_alpha_set(Efl_Ui_Win *obj, Eina_Bool alpha);

/**
 * @brief Get the alpha channel state of a window.
 *
 * @param[in] obj The object.
 *
 * @return @c true if the window alpha channel is enabled, @c false otherwise.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_alpha_get(const Efl_Ui_Win *obj);

/**
 * @brief Activate a window object.
 *
 * This function sends a request to the Window Manager to activate the window
 * pointed by @c obj. If honored by the WM, the window will receive the
 * keyboard focus.
 *
 * @note This is just a request that a Window Manager may ignore, so calling
 * this function does not ensure in any way that the window will be the active
 * one afterwards.
 * @param[in] obj The object.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_activate(Efl_Ui_Win *obj);

/**
 * @brief Center a window on the screen.
 *
 * This function centers window @c obj horizontally and/or vertically based on
 * the values of @c h and @c v.
 *
 * @note This is just a request that a Window Manager may ignore, so calling
 * this function does not ensure in any way that the window will be centered
 * afterwards.
 *
 * @param[in] obj The object.
 * @param[in] h If @c true, center horizontally. If @c false, do not change
 * horizontal location.
 * @param[in] v If @c true, center vertically. If @c false, do not change
 * vertical location.
 *
 * @ingroup Efl_Ui_Win
 */
EAPI void elm_win_center(Efl_Ui_Win *obj, Eina_Bool h, Eina_Bool v);



/**
 * @brief Start moving or resizing the window.
 *
 * The user can request the display server to start moving or resizing the
 * window by combining modes from @ref Efl_Ui_Win_Move_Resize_Mode. This API
 * can only be called if none of the following conditions is true:
 *
 * 1. Called in the absence of a pointer down event, 2. Called more than once
 * before a pointer up event, 3. Called when the window is already being
 * resized or moved, 4. Called with an unsupported combination of modes.
 *
 * Right usage: 1. Pointer (mouse or touch) down event, 2.
 * @ref elm_win_move_resize_start called only once with a supported mode, 3.
 * Pointer (mouse or touch) up event.
 *
 * If a pointer up event occurs after calling the function, it automatically
 * ends the window move and resize operation.
 *
 * Currently, only the following 9 combinations are allowed, and possibly more
 * combinations may be added in the future: 1. @ref EFL_UI_WIN_MOVE_RESIZE_MOVE
 * 2. @ref EFL_UI_WIN_MOVE_RESIZE_TOP 3. @ref EFL_UI_WIN_MOVE_RESIZE_BOTTOM 4.
 * @ref EFL_UI_WIN_MOVE_RESIZE_LEFT 5. @ref EFL_UI_WIN_MOVE_RESIZE_RIGHT 6.
 * @ref EFL_UI_WIN_MOVE_RESIZE_TOP | @ref EFL_UI_WIN_MOVE_RESIZE_LEFT 7.
 * @ref EFL_UI_WIN_MOVE_RESIZE_TOP | @ref EFL_UI_WIN_MOVE_RESIZE_RIGHT 8.
 * @ref EFL_UI_WIN_MOVE_RESIZE_BOTTOM | @ref EFL_UI_WIN_MOVE_RESIZE_LEFT 9.
 * @ref EFL_UI_WIN_MOVE_RESIZE_BOTTOM | @ref EFL_UI_WIN_MOVE_RESIZE_RIGHT
 *
 * In particular move and resize cannot happen simultaneously.
 *
 * @note the result of this API can only guarantee that the request has been
 * forwarded to the server, but there is no guarantee that the request can be
 * processed by the display server.
 *
 * @param[in] obj The object.
 * @param[in] mode The requested move or resize mode.
 *
 * @return @c true if the request was successfully sent to the display server,
 * @c false in case of error.
 *
 * @since 1.19
 *
 * @ingroup Efl_Ui_Win
 */
EAPI Eina_Bool elm_win_move_resize_start(Efl_Ui_Win *obj, Efl_Ui_Win_Move_Resize_Mode mode);

#endif
