/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.support.replication;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.Assertions;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.UnavailableShardsException;
import org.elasticsearch.action.support.ActiveShardCount;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.action.support.replication.ReplicationRequest;
import org.elasticsearch.action.support.replication.ReplicationResponse;
import org.elasticsearch.cluster.action.shard.ShardStateAction;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.index.shard.ReplicationGroup;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.transport.TransportException;

public class ReplicationOperation<Request extends ReplicationRequest<Request>, ReplicaRequest extends ReplicationRequest<ReplicaRequest>, PrimaryResultT extends PrimaryResult<ReplicaRequest>> {
    private final Logger logger;
    private final Request request;
    private final String opType;
    private final AtomicInteger totalShards = new AtomicInteger();
    private final AtomicInteger pendingActions = new AtomicInteger();
    private final AtomicInteger successfulShards = new AtomicInteger();
    private final Primary<Request, ReplicaRequest, PrimaryResultT> primary;
    private final Replicas<ReplicaRequest> replicasProxy;
    private final AtomicBoolean finished = new AtomicBoolean();
    protected final ActionListener<PrimaryResultT> resultListener;
    private volatile PrimaryResultT primaryResult = null;
    private final List<ReplicationResponse.ShardInfo.Failure> shardReplicaFailures = Collections.synchronizedList(new ArrayList());

    public ReplicationOperation(Request request, Primary<Request, ReplicaRequest, PrimaryResultT> primary, ActionListener<PrimaryResultT> listener, Replicas<ReplicaRequest> replicas, Logger logger, String opType) {
        this.replicasProxy = replicas;
        this.primary = primary;
        this.resultListener = listener;
        this.logger = logger;
        this.request = request;
        this.opType = opType;
    }

    public void execute() throws Exception {
        String activeShardCountFailure = this.checkActiveShardCount();
        ShardRouting primaryRouting = this.primary.routingEntry();
        ShardId primaryId = primaryRouting.shardId();
        if (activeShardCountFailure != null) {
            this.finishAsFailed(new UnavailableShardsException(primaryId, "{} Timeout: [{}], request: [{}]", activeShardCountFailure, ((ReplicationRequest)this.request).timeout(), this.request));
            return;
        }
        this.totalShards.incrementAndGet();
        this.pendingActions.incrementAndGet();
        this.primaryResult = this.primary.perform(this.request);
        this.primary.updateLocalCheckpointForShard(primaryRouting.allocationId().getId(), this.primary.localCheckpoint());
        Object replicaRequest = this.primaryResult.replicaRequest();
        if (replicaRequest != null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("[{}] op [{}] completed on primary for request [{}]", (Object)primaryId, (Object)this.opType, this.request);
            }
            long globalCheckpoint = this.primary.globalCheckpoint();
            long maxSeqNoOfUpdatesOrDeletes = this.primary.maxSeqNoOfUpdatesOrDeletes();
            assert (maxSeqNoOfUpdatesOrDeletes != -2L) : "seqno_of_updates still uninitialized";
            ReplicationGroup replicationGroup = this.primary.getReplicationGroup();
            this.markUnavailableShardsAsStale(replicaRequest, replicationGroup);
            this.performOnReplicas(replicaRequest, globalCheckpoint, maxSeqNoOfUpdatesOrDeletes, replicationGroup);
        }
        this.successfulShards.incrementAndGet();
        this.decPendingAndFinishIfNeeded();
    }

    private void markUnavailableShardsAsStale(ReplicaRequest replicaRequest, ReplicationGroup replicationGroup) {
        for (String allocationId : replicationGroup.getUnavailableInSyncShards()) {
            this.pendingActions.incrementAndGet();
            this.replicasProxy.markShardCopyAsStaleIfNeeded(((ReplicationRequest)replicaRequest).shardId(), allocationId, ActionListener.wrap(r -> this.decPendingAndFinishIfNeeded(), this::onNoLongerPrimary));
        }
    }

    private void performOnReplicas(ReplicaRequest replicaRequest, long globalCheckpoint, long maxSeqNoOfUpdatesOrDeletes, ReplicationGroup replicationGroup) {
        this.totalShards.addAndGet(replicationGroup.getSkippedShards().size());
        ShardRouting primaryRouting = this.primary.routingEntry();
        for (ShardRouting shard : replicationGroup.getReplicationTargets()) {
            if (shard.isSameAllocation(primaryRouting)) continue;
            this.performOnReplica(shard, replicaRequest, globalCheckpoint, maxSeqNoOfUpdatesOrDeletes);
        }
    }

    private void performOnReplica(final ShardRouting shard, final ReplicaRequest replicaRequest, long globalCheckpoint, long maxSeqNoOfUpdatesOrDeletes) {
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("[{}] sending op [{}] to replica {} for request [{}]", (Object)shard.shardId(), (Object)this.opType, (Object)shard, replicaRequest);
        }
        this.totalShards.incrementAndGet();
        this.pendingActions.incrementAndGet();
        this.replicasProxy.performOn(shard, replicaRequest, globalCheckpoint, maxSeqNoOfUpdatesOrDeletes, new ActionListener<ReplicaResponse>(){

            @Override
            public void onResponse(ReplicaResponse response) {
                ReplicationOperation.this.successfulShards.incrementAndGet();
                try {
                    ReplicationOperation.this.primary.updateLocalCheckpointForShard(shard.allocationId().getId(), response.localCheckpoint());
                    ReplicationOperation.this.primary.updateGlobalCheckpointForShard(shard.allocationId().getId(), response.globalCheckpoint());
                }
                catch (AlreadyClosedException alreadyClosedException) {
                }
                catch (Exception e) {
                    String message = String.format(Locale.ROOT, "primary failed updating local checkpoint for replica %s", shard);
                    ReplicationOperation.this.primary.failShard(message, e);
                }
                ReplicationOperation.this.decPendingAndFinishIfNeeded();
            }

            @Override
            public void onFailure(Exception replicaException) {
                ReplicationOperation.this.logger.trace(() -> new ParameterizedMessage("[{}] failure while performing [{}] on replica {}, request [{}]", new Object[]{shard.shardId(), ReplicationOperation.this.opType, shard, replicaRequest}), (Throwable)replicaException);
                if (!TransportActions.isShardNotAvailableException(replicaException)) {
                    RestStatus restStatus = ExceptionsHelper.status(replicaException);
                    ReplicationOperation.this.shardReplicaFailures.add(new ReplicationResponse.ShardInfo.Failure(shard.shardId(), shard.currentNodeId(), replicaException, restStatus, false));
                }
                String message = String.format(Locale.ROOT, "failed to perform %s on replica %s", ReplicationOperation.this.opType, shard);
                ReplicationOperation.this.replicasProxy.failShardIfNeeded(shard, message, replicaException, ActionListener.wrap(r -> ReplicationOperation.this.decPendingAndFinishIfNeeded(), x$0 -> ReplicationOperation.this.onNoLongerPrimary(x$0)));
            }

            public String toString() {
                return "[" + replicaRequest + "][" + shard + "]";
            }
        });
    }

    private void onNoLongerPrimary(Exception failure) {
        String message;
        boolean nodeIsClosing;
        Throwable cause = ExceptionsHelper.unwrapCause(failure);
        boolean bl = nodeIsClosing = cause instanceof NodeClosedException || cause instanceof TransportException && ("TransportService is closed stopped can't send request".equals(cause.getMessage()) || "transport stopped, action: internal:cluster/shard/failure".equals(cause.getMessage()));
        if (nodeIsClosing) {
            message = String.format(Locale.ROOT, "node with primary [%s] is shutting down while failing replica shard", this.primary.routingEntry());
        } else {
            if (Assertions.ENABLED && !(failure instanceof ShardStateAction.NoLongerPrimaryShardException)) {
                throw new AssertionError("unexpected failure", failure);
            }
            message = String.format(Locale.ROOT, "primary shard [%s] was demoted while failing replica shard", this.primary.routingEntry());
            this.primary.failShard(message, failure);
        }
        this.finishAsFailed(new RetryOnPrimaryException(this.primary.routingEntry().shardId(), message, failure));
    }

    protected String checkActiveShardCount() {
        ShardId shardId = this.primary.routingEntry().shardId();
        ActiveShardCount waitForActiveShards = ((ReplicationRequest)this.request).waitForActiveShards();
        if (waitForActiveShards == ActiveShardCount.NONE) {
            return null;
        }
        IndexShardRoutingTable shardRoutingTable = this.primary.getReplicationGroup().getRoutingTable();
        if (waitForActiveShards.enoughShardsActive(shardRoutingTable)) {
            return null;
        }
        String resolvedShards = waitForActiveShards == ActiveShardCount.ALL ? Integer.toString(shardRoutingTable.shards().size()) : waitForActiveShards.toString();
        this.logger.trace("[{}] not enough active copies to meet shard count of [{}] (have {}, needed {}), scheduling a retry. op [{}], request [{}]", (Object)shardId, (Object)waitForActiveShards, (Object)shardRoutingTable.activeShards().size(), (Object)resolvedShards, (Object)this.opType, this.request);
        return "Not enough active copies to meet shard count of [" + waitForActiveShards + "] (have " + shardRoutingTable.activeShards().size() + ", needed " + resolvedShards + ").";
    }

    private void decPendingAndFinishIfNeeded() {
        assert (this.pendingActions.get() > 0) : "pending action count goes below 0 for request [" + this.request + "]";
        if (this.pendingActions.decrementAndGet() == 0) {
            this.finish();
        }
    }

    private void finish() {
        if (this.finished.compareAndSet(false, true)) {
            ReplicationResponse.ShardInfo.Failure[] failuresArray;
            if (this.shardReplicaFailures.isEmpty()) {
                failuresArray = ReplicationResponse.EMPTY;
            } else {
                failuresArray = new ReplicationResponse.ShardInfo.Failure[this.shardReplicaFailures.size()];
                this.shardReplicaFailures.toArray(failuresArray);
            }
            this.primaryResult.setShardInfo(new ReplicationResponse.ShardInfo(this.totalShards.get(), this.successfulShards.get(), failuresArray));
            this.resultListener.onResponse(this.primaryResult);
        }
    }

    private void finishAsFailed(Exception exception) {
        if (this.finished.compareAndSet(false, true)) {
            this.resultListener.onFailure(exception);
        }
    }

    public static interface PrimaryResult<RequestT extends ReplicationRequest<RequestT>> {
        @Nullable
        public RequestT replicaRequest();

        public void setShardInfo(ReplicationResponse.ShardInfo var1);
    }

    public static class RetryOnPrimaryException
    extends ElasticsearchException {
        public RetryOnPrimaryException(ShardId shardId, String msg) {
            this(shardId, msg, null);
        }

        public RetryOnPrimaryException(ShardId shardId, String msg, Throwable cause) {
            super(msg, cause, new Object[0]);
            this.setShard(shardId);
        }

        public RetryOnPrimaryException(StreamInput in) throws IOException {
            super(in);
        }
    }

    public static interface ReplicaResponse {
        public long localCheckpoint();

        public long globalCheckpoint();
    }

    public static interface Replicas<RequestT extends ReplicationRequest<RequestT>> {
        public void performOn(ShardRouting var1, RequestT var2, long var3, long var5, ActionListener<ReplicaResponse> var7);

        public void failShardIfNeeded(ShardRouting var1, String var2, Exception var3, ActionListener<Void> var4);

        public void markShardCopyAsStaleIfNeeded(ShardId var1, String var2, ActionListener<Void> var3);
    }

    public static interface Primary<RequestT extends ReplicationRequest<RequestT>, ReplicaRequestT extends ReplicationRequest<ReplicaRequestT>, PrimaryResultT extends PrimaryResult<ReplicaRequestT>> {
        public ShardRouting routingEntry();

        public void failShard(String var1, Exception var2);

        public PrimaryResultT perform(RequestT var1) throws Exception;

        public void updateLocalCheckpointForShard(String var1, long var2);

        public void updateGlobalCheckpointForShard(String var1, long var2);

        public long localCheckpoint();

        public long globalCheckpoint();

        public long maxSeqNoOfUpdatesOrDeletes();

        public ReplicationGroup getReplicationGroup();
    }
}

