"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const notify_1 = require("./features/notify");
const profiling_1 = require("./features/profiling");
const eventLoopInspector_1 = require("./features/eventLoopInspector");
const events_1 = require("./features/events");
const metrics_1 = require("./features/metrics");
const Debug = require("debug");
const tracing_1 = require("./features/tracing");
function getObjectAtPath(context, path) {
    if (path.indexOf('.') === -1 && path.indexOf('[') === -1) {
        return context[path];
    }
    let crumbs = path.split(/\.|\[|\]/g);
    let i = -1;
    let len = crumbs.length;
    let result;
    while (++i < len) {
        if (i === 0)
            result = context;
        if (!crumbs[i])
            continue;
        if (result === undefined)
            break;
        result = result[crumbs[i]];
    }
    return result;
}
exports.getObjectAtPath = getObjectAtPath;
class AvailableFeature {
}
const availablesFeatures = [
    {
        name: 'notify',
        optionsPath: '.',
        module: notify_1.NotifyFeature
    },
    {
        name: 'profiler',
        optionsPath: 'profiling',
        module: profiling_1.ProfilingFeature
    },
    {
        name: 'eventLoopInspector',
        optionsPath: 'actions.eventLoopDump',
        module: eventLoopInspector_1.EventLoopInspectorFeature
    },
    {
        name: 'events',
        module: events_1.EventsFeature
    },
    {
        name: 'metrics',
        optionsPath: 'metrics',
        module: metrics_1.MetricsFeature
    },
    {
        name: 'tracing',
        optionsPath: '.',
        module: tracing_1.TracingFeature
    }
];
class FeatureManager {
    constructor() {
        this.logger = Debug('axm:features');
    }
    init(options) {
        for (let availableFeature of availablesFeatures) {
            this.logger(`Creating feature ${availableFeature.name}`);
            const feature = new availableFeature.module();
            let config = undefined;
            if (typeof availableFeature.optionsPath !== 'string') {
                config = {};
            }
            else if (availableFeature.optionsPath === '.') {
                config = options;
            }
            else {
                config = getObjectAtPath(options, availableFeature.optionsPath);
            }
            this.logger(`Init feature ${availableFeature.name}`);
            feature.init(config);
            availableFeature.instance = feature;
        }
    }
    get(name) {
        const feature = availablesFeatures.find(feature => feature.name === name);
        if (feature === undefined || feature.instance === undefined) {
            throw new Error(`Tried to call feature ${name} which doesn't exist or wasn't initiated`);
        }
        return feature.instance;
    }
    destroy() {
        for (let availableFeature of availablesFeatures) {
            if (availableFeature.instance === undefined)
                continue;
            this.logger(`Destroy feature ${availableFeature.name}`);
            availableFeature.instance.destroy();
        }
    }
}
exports.FeatureManager = FeatureManager;
class FeatureConfig {
}
exports.FeatureConfig = FeatureConfig;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmVhdHVyZU1hbmFnZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZmVhdHVyZU1hbmFnZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSw4Q0FBaUQ7QUFDakQsb0RBQXVEO0FBQ3ZELHNFQUF5RTtBQUN6RSw4Q0FBaUQ7QUFFakQsZ0RBQW1EO0FBQ25ELCtCQUE4QjtBQUM5QixnREFBbUQ7QUFFbkQsU0FBZ0IsZUFBZSxDQUFFLE9BQWUsRUFBRSxJQUFZO0lBQzVELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1FBQ3hELE9BQU8sT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFBO0tBQ3JCO0lBRUQsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQTtJQUNwQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQTtJQUNWLElBQUksR0FBRyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUE7SUFDdkIsSUFBSSxNQUFNLENBQUE7SUFFVixPQUFPLEVBQUUsQ0FBQyxHQUFHLEdBQUcsRUFBRTtRQUNoQixJQUFJLENBQUMsS0FBSyxDQUFDO1lBQUUsTUFBTSxHQUFHLE9BQU8sQ0FBQTtRQUM3QixJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUFFLFNBQVE7UUFDeEIsSUFBSSxNQUFNLEtBQUssU0FBUztZQUFFLE1BQUs7UUFDL0IsTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtLQUMzQjtJQUVELE9BQU8sTUFBTSxDQUFBO0FBQ2YsQ0FBQztBQWxCRCwwQ0FrQkM7QUFFRCxNQUFNLGdCQUFnQjtDQXFCckI7QUFFRCxNQUFNLGtCQUFrQixHQUF1QjtJQUM3QztRQUNFLElBQUksRUFBRSxRQUFRO1FBQ2QsV0FBVyxFQUFFLEdBQUc7UUFDaEIsTUFBTSxFQUFFLHNCQUFhO0tBQ3RCO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsVUFBVTtRQUNoQixXQUFXLEVBQUUsV0FBVztRQUN4QixNQUFNLEVBQUUsNEJBQWdCO0tBQ3pCO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsb0JBQW9CO1FBQzFCLFdBQVcsRUFBRSx1QkFBdUI7UUFDcEMsTUFBTSxFQUFFLDhDQUF5QjtLQUNsQztJQUNEO1FBQ0UsSUFBSSxFQUFFLFFBQVE7UUFDZCxNQUFNLEVBQUUsc0JBQWE7S0FDdEI7SUFDRDtRQUNFLElBQUksRUFBRSxTQUFTO1FBQ2YsV0FBVyxFQUFFLFNBQVM7UUFDdEIsTUFBTSxFQUFFLHdCQUFjO0tBQ3ZCO0lBQ0Q7UUFDRSxJQUFJLEVBQUUsU0FBUztRQUNmLFdBQVcsRUFBRSxHQUFHO1FBQ2hCLE1BQU0sRUFBRSx3QkFBYztLQUN2QjtDQUNGLENBQUE7QUFFRCxNQUFhLGNBQWM7SUFBM0I7UUFFVSxXQUFNLEdBQWEsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFBO0lBNkNsRCxDQUFDO0lBeENDLElBQUksQ0FBRSxPQUFpQjtRQUNyQixLQUFLLElBQUksZ0JBQWdCLElBQUksa0JBQWtCLEVBQUU7WUFDL0MsSUFBSSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQTtZQUN4RCxNQUFNLE9BQU8sR0FBRyxJQUFJLGdCQUFnQixDQUFDLE1BQU0sRUFBRSxDQUFBO1lBQzdDLElBQUksTUFBTSxHQUFRLFNBQVMsQ0FBQTtZQUMzQixJQUFJLE9BQU8sZ0JBQWdCLENBQUMsV0FBVyxLQUFLLFFBQVEsRUFBRTtnQkFDcEQsTUFBTSxHQUFHLEVBQUUsQ0FBQTthQUNaO2lCQUFNLElBQUksZ0JBQWdCLENBQUMsV0FBVyxLQUFLLEdBQUcsRUFBRTtnQkFDL0MsTUFBTSxHQUFHLE9BQU8sQ0FBQTthQUNqQjtpQkFBTTtnQkFDTCxNQUFNLEdBQUcsZUFBZSxDQUFDLE9BQU8sRUFBRSxnQkFBZ0IsQ0FBQyxXQUFXLENBQUMsQ0FBQTthQUNoRTtZQUNELElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLGdCQUFnQixDQUFDLElBQUksRUFBRSxDQUFDLENBQUE7WUFJcEQsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQTtZQUNwQixnQkFBZ0IsQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFBO1NBQ3BDO0lBQ0gsQ0FBQztJQU1ELEdBQUcsQ0FBRSxJQUFZO1FBQ2YsTUFBTSxPQUFPLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksS0FBSyxJQUFJLENBQUMsQ0FBQTtRQUN6RSxJQUFJLE9BQU8sS0FBSyxTQUFTLElBQUksT0FBTyxDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7WUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsSUFBSSwwQ0FBMEMsQ0FBQyxDQUFBO1NBQ3pGO1FBQ0QsT0FBTyxPQUFPLENBQUMsUUFBUSxDQUFBO0lBQ3pCLENBQUM7SUFFRCxPQUFPO1FBQ0wsS0FBSyxJQUFJLGdCQUFnQixJQUFJLGtCQUFrQixFQUFFO1lBQy9DLElBQUksZ0JBQWdCLENBQUMsUUFBUSxLQUFLLFNBQVM7Z0JBQUUsU0FBUTtZQUNyRCxJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFBO1lBQ3ZELGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQTtTQUNwQztJQUNILENBQUM7Q0FDRjtBQS9DRCx3Q0ErQ0M7QUFHRCxNQUFhLGFBQWE7Q0FBSTtBQUE5QixzQ0FBOEIifQ==