"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@opencensus/core");
const httpModule = require("http");
const semver = require("semver");
const shimmer = require("shimmer");
const url = require("url");
const uuid = require("uuid");
class HttpPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
    }
    applyPatch() {
        this.logger.debug('applying patch to %s@%s', this.moduleName, this.version);
        shimmer.wrap(this.moduleExports, 'request', this.getPatchOutgoingRequestFunction());
        if (semver.satisfies(this.version, '>=8.0.0')) {
            shimmer.wrap(this.moduleExports, 'get', () => {
                return function getTrace(options, callback) {
                    const req = httpModule.request(options, callback);
                    req.end();
                    return req;
                };
            });
        }
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.wrap(this.moduleExports.Server.prototype, 'emit', this.getPatchIncomingRequestFunction());
        }
        else {
            this.logger.error('Could not apply patch to %s.emit. Interface is not as expected.', this.moduleName);
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.moduleExports, 'request');
        if (semver.satisfies(this.version, '>=8.0.0')) {
            shimmer.unwrap(this.moduleExports, 'get');
        }
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.unwrap(this.moduleExports.Server.prototype, 'emit');
        }
    }
    isIgnored(url, request, list) {
        if (!list) {
            return false;
        }
        for (const pattern of list) {
            if (this.isSatisfyPattern(url, request, pattern)) {
                return true;
            }
        }
        return false;
    }
    isSatisfyPattern(url, request, pattern) {
        if (typeof pattern === 'string') {
            return pattern === url;
        }
        else if (pattern instanceof RegExp) {
            return pattern.test(url);
        }
        else if (typeof pattern === 'function') {
            return pattern(url, request);
        }
        else {
            throw new TypeError('Pattern is in unsupported datatype');
        }
    }
    getPatchIncomingRequestFunction() {
        return (original) => {
            const plugin = this;
            if (plugin.options === undefined) {
                plugin.options = {
                    ignoreIncomingPaths: [],
                    ignoreOutgoingUrls: []
                };
            }
            return function incomingRequest(event, ...args) {
                if (event !== 'request') {
                    return original.apply(this, arguments);
                }
                const request = args[0];
                const response = args[1];
                const path = url.parse(request.url).pathname;
                plugin.logger.debug('%s plugin incomingRequest', plugin.moduleName);
                if (plugin.isIgnored(path, request, plugin.options.ignoreIncomingPaths)) {
                    return original.apply(this, arguments);
                }
                const propagation = plugin.tracer.propagation;
                const headers = request.headers;
                const getter = {
                    getHeader(name) {
                        return headers[name];
                    }
                };
                const context = propagation ? propagation.extract(getter) : null;
                const traceOptions = {
                    name: path,
                    kind: core_1.SpanKind.SERVER,
                    spanContext: context !== null ? context : undefined
                };
                return plugin.tracer.startRootSpan(traceOptions, rootSpan => {
                    if (!rootSpan)
                        return original.apply(this, arguments);
                    plugin.tracer.wrapEmitter(request);
                    plugin.tracer.wrapEmitter(response);
                    const originalEnd = response.end;
                    response.end = function () {
                        response.end = originalEnd;
                        const returned = response.end.apply(this, arguments);
                        const requestUrl = url.parse(request.url || 'localhost');
                        const host = headers.host || 'localhost';
                        const userAgent = (headers['user-agent'] || headers['User-Agent']);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_HOST, host.replace(/^(.*)(\:[0-9]{1,5})/, '$1'));
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_METHOD, request.method || 'GET');
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_PATH, `${requestUrl.pathname}`);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ROUTE, `${requestUrl.path}`);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT, userAgent);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE, response.statusCode.toString());
                        rootSpan.setStatus(HttpPlugin.convertTraceStatus(response.statusCode));
                        rootSpan.addMessageEvent(core_1.MessageEventType.RECEIVED, uuid.v4().split('-').join(''));
                        rootSpan.end();
                        return returned;
                    };
                    return original.apply(this, arguments);
                });
            };
        };
    }
    getPatchOutgoingRequestFunction() {
        return (original) => {
            const plugin = this;
            const kind = plugin.moduleName === 'https' ? 'HTTPS' : 'HTTP';
            if (plugin.options === undefined) {
                plugin.options = {
                    ignoreIncomingPaths: [],
                    ignoreOutgoingUrls: []
                };
            }
            return function outgoingRequest(options, callback) {
                if (!options) {
                    return original.apply(this, arguments);
                }
                let pathname = '';
                let method = 'GET';
                let origin = '';
                if (typeof (options) === 'string') {
                    const parsedUrl = url.parse(options);
                    options = parsedUrl;
                    pathname = parsedUrl.pathname || '/';
                    origin = `${parsedUrl.protocol || 'http:'}//${parsedUrl.host}`;
                }
                else {
                    if (options.headers &&
                        options.headers['x-opencensus-outgoing-request']) {
                        plugin.logger.debug('header with "x-opencensus-outgoing-request" - do not trace');
                        return original.apply(this, arguments);
                    }
                    try {
                        pathname = options.pathname || '';
                        if (pathname.length === 0 && typeof options.path === 'string') {
                            pathname = url.parse(options.path).pathname || '';
                        }
                        method = options.method || 'GET';
                        origin = `${options.protocol || 'http:'}//${options.host}`;
                    }
                    catch (e) {
                        return original.apply(this, arguments);
                    }
                }
                const request = original.apply(this, arguments);
                if (plugin.isIgnored(origin + pathname, request, plugin.options.ignoreOutgoingUrls)) {
                    return request;
                }
                plugin.tracer.wrapEmitter(request);
                plugin.logger.debug('%s plugin outgoingRequest', plugin.moduleName);
                const traceOptions = {
                    name: `${kind.toLowerCase()}-${(method || 'GET').toLowerCase()}`,
                    kind: core_1.SpanKind.CLIENT
                };
                if (!plugin.tracer.currentRootSpan) {
                    plugin.logger.debug('outgoingRequest starting a root span');
                    return plugin.tracer.startRootSpan(traceOptions, plugin.getMakeRequestTraceFunction(request, options, plugin));
                }
                else {
                    plugin.logger.debug('outgoingRequest starting a child span');
                    const span = plugin.tracer.startChildSpan(traceOptions.name, traceOptions.kind);
                    return (plugin.getMakeRequestTraceFunction(request, options, plugin))(span);
                }
            };
        };
    }
    getMakeRequestTraceFunction(request, options, plugin) {
        return (span) => {
            plugin.logger.debug('makeRequestTrace');
            if (!span) {
                plugin.logger.debug('makeRequestTrace span is null');
                return request;
            }
            const setter = {
                setHeader(name, value) {
                    if (plugin.hasExpectHeader(options) && options.headers) {
                        if (options.__cloned !== true) {
                            options = Object.assign({}, options);
                            options.headers = Object.assign({}, options.headers);
                            options.__cloned = true;
                        }
                        options.headers[name] = value;
                    }
                    else {
                        request.setHeader(name, value);
                    }
                }
            };
            const propagation = plugin.tracer.propagation;
            if (propagation) {
                propagation.inject(setter, span.spanContext);
            }
            request.on('response', (response) => {
                plugin.tracer.wrapEmitter(response);
                plugin.logger.debug('outgoingRequest on response()');
                response.on('end', () => {
                    plugin.logger.debug('outgoingRequest on end()');
                    const method = response.method ? response.method : 'GET';
                    const headers = options.headers;
                    const userAgent = headers ? (headers['user-agent'] || headers['User-Agent']) : null;
                    if (options.host || options.hostname) {
                        const value = options.host || options.hostname;
                        span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_HOST, `${value}`);
                    }
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_METHOD, method);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_PATH, `${options.path}`);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ROUTE, `${options.path}`);
                    if (userAgent) {
                        span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT, userAgent.toString());
                    }
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE, `${response.statusCode}`);
                    span.setStatus(HttpPlugin.convertTraceStatus(response.statusCode || 0));
                    span.addMessageEvent(core_1.MessageEventType.SENT, uuid.v4().split('-').join(''));
                    span.end();
                });
                response.on('error', error => {
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME, error.name);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE, error.message);
                    span.setStatus(core_1.CanonicalCode.UNKNOWN);
                    span.end();
                });
            });
            request.on('error', error => {
                span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME, error.name);
                span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE, error.message);
                span.setStatus(core_1.CanonicalCode.UNKNOWN);
                span.end();
            });
            plugin.logger.debug('makeRequestTrace return request');
            return request;
        };
    }
    static convertTraceStatus(statusCode) {
        if (statusCode < 200 || statusCode > 504) {
            return TraceStatusCodes.UNKNOWN;
        }
        else if (statusCode >= 200 && statusCode < 400) {
            return TraceStatusCodes.OK;
        }
        else {
            switch (statusCode) {
                case (400):
                    return TraceStatusCodes.INVALID_ARGUMENT;
                case (504):
                    return TraceStatusCodes.DEADLINE_EXCEEDED;
                case (404):
                    return TraceStatusCodes.NOT_FOUND;
                case (403):
                    return TraceStatusCodes.PERMISSION_DENIED;
                case (401):
                    return TraceStatusCodes.UNAUTHENTICATED;
                case (429):
                    return TraceStatusCodes.RESOURCE_EXHAUSTED;
                case (501):
                    return TraceStatusCodes.UNIMPLEMENTED;
                case (503):
                    return TraceStatusCodes.UNAVAILABLE;
                default:
                    return TraceStatusCodes.UNKNOWN;
            }
        }
    }
    hasExpectHeader(options) {
        return !!(options.headers &&
            options.headers.Expect);
    }
}
HttpPlugin.ATTRIBUTE_HTTP_HOST = 'http.host';
HttpPlugin.ATTRIBUTE_HTTP_METHOD = 'http.method';
HttpPlugin.ATTRIBUTE_HTTP_PATH = 'http.path';
HttpPlugin.ATTRIBUTE_HTTP_ROUTE = 'http.route';
HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT = 'http.user_agent';
HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE = 'http.status_code';
HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME = 'http.error_name';
HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE = 'http.error_message';
exports.HttpPlugin = HttpPlugin;
var TraceStatusCodes;
(function (TraceStatusCodes) {
    TraceStatusCodes[TraceStatusCodes["UNKNOWN"] = 2] = "UNKNOWN";
    TraceStatusCodes[TraceStatusCodes["OK"] = 0] = "OK";
    TraceStatusCodes[TraceStatusCodes["INVALID_ARGUMENT"] = 3] = "INVALID_ARGUMENT";
    TraceStatusCodes[TraceStatusCodes["DEADLINE_EXCEEDED"] = 4] = "DEADLINE_EXCEEDED";
    TraceStatusCodes[TraceStatusCodes["NOT_FOUND"] = 5] = "NOT_FOUND";
    TraceStatusCodes[TraceStatusCodes["PERMISSION_DENIED"] = 7] = "PERMISSION_DENIED";
    TraceStatusCodes[TraceStatusCodes["UNAUTHENTICATED"] = 16] = "UNAUTHENTICATED";
    TraceStatusCodes[TraceStatusCodes["RESOURCE_EXHAUSTED"] = 8] = "RESOURCE_EXHAUSTED";
    TraceStatusCodes[TraceStatusCodes["UNIMPLEMENTED"] = 12] = "UNIMPLEMENTED";
    TraceStatusCodes[TraceStatusCodes["UNAVAILABLE"] = 14] = "UNAVAILABLE";
})(TraceStatusCodes = exports.TraceStatusCodes || (exports.TraceStatusCodes = {}));
exports.plugin = new HttpPlugin('http');
//# sourceMappingURL=data:application/json;base64,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