#
# Copyright 2014 Ettus Research LLC
# Copyright 2018 Ettus Research, a National Instruments Company
#
# SPDX-License-Identifier: GPL-3.0-or-later
#
########################################################################
#
# Find the header <uhd/config.hpp> and library "libuhd" for the USRP
# Hardware Driver.  Priorty for prefix search is:
# 1) ENV(UHD_DIR)
# 2) pkg-config results, if available;
# 3) CMAKE_INSTALL_PREFIX
# 4) /usr/local/
# 5) /usr/
#
# Version info is handled by UHDConfigVersion.cmake only; not here.
#
########################################################################

# set that this file was found, for use in GNU Radio's FindUHD.cmake.
# Have to use the ENV, since this file might not allow CACHE changes.

set(ENV{UHD_CONFIG_USED} TRUE)

# set default values

set(UHD_FOUND TRUE)
set(UHD_RFNOC_FOUND FALSE)
set(UHD_INCLUDE_HINTS)
set(UHD_LIBDIR_HINTS)
set(UHD_DIR $ENV{UHD_DIR})

if(UHD_DIR)
    list(APPEND UHD_INCLUDE_HINTS ${UHD_DIR}/include)
    list(APPEND UHD_LIBDIR_HINTS ${UHD_DIR}/lib)
endif()

include(FindPkgConfig)
if(PKG_CONFIG_FOUND)
  if(NOT ${CMAKE_VERSION} VERSION_LESS "2.8.0")
    set(UHD_QUIET "QUIET")
  endif()
  PKG_CHECK_MODULES(PC_UHD ${UHD_QUIET} uhd)
  if(PC_UHD_FOUND)
    list(APPEND UHD_INCLUDE_HINTS ${PC_UHD_INCLUDEDIR})
    list(APPEND UHD_LIBDIR_HINTS ${PC_UHD_LIBDIR})
  endif()
endif()

list(APPEND UHD_INCLUDE_HINTS ${CMAKE_INSTALL_PREFIX}/include)
list(APPEND UHD_LIBDIR_HINTS ${CMAKE_INSTALL_PREFIX}/lib)


# Search for static libs if so required
if( UHD_USE_STATIC_LIBS )
    set( _UHD_ORIG_CMAKE_FIND_LIBRARY_SUFFIXES ${CMAKE_FIND_LIBRARY_SUFFIXES})
    if(WIN32)
        set(CMAKE_FIND_LIBRARY_SUFFIXES .lib .a ${CMAKE_FIND_LIBRARY_SUFFIXES})
    else()
        set(CMAKE_FIND_LIBRARY_SUFFIXES .a )
    endif()
    # This is set during build of UHD to match the installed version:
    set(UHD_STATIC_LIB_DEPS "dl;pthread;boost_chrono;boost_date_time;boost_filesystem;boost_program_options;boost_regex;boost_system;boost_serialization;boost_thread;usb-1.0")
endif()

# Verify that <uhd/config.hpp> and libuhd are available, and, if a
# version is provided, that UHD meets the version requirements -- no
# matter what pkg-config might think.

find_path(
    UHD_INCLUDE_DIRS
    NAMES uhd/config.hpp
    HINTS ${UHD_INCLUDE_HINTS}
    PATHS /usr/local/include
          /usr/include
)

find_library(
    UHD_LIBRARIES
    NAMES uhd
    HINTS ${UHD_LIBDIR_HINTS}
    PATHS /usr/local/lib
          /usr/lib
)

# Set up linker flags for static linking:
if(UHD_USE_STATIC_LIBS)
    if(WIN32)
        message(FATAL_ERROR "Static linking not available on Windows")
    else(WIN32)
        # This works for gcc and Clang:
        set(UHD_STATIC_LIB_LINK_FLAG "-Wl,-whole-archive ${UHD_LIBRARIES} -Wl,-no-whole-archive")
    endif(WIN32)
    # Restore the library suffixes, if we changed them:
    set(CMAKE_FIND_LIBRARY_SUFFIXES ${_UHD_ORIG_CMAKE_FIND_LIBRARY_SUFFIXES})
endif(UHD_USE_STATIC_LIBS)

if(UHD_LIBRARIES AND UHD_INCLUDE_DIRS)

  include(FindPackageHandleStandardArgs)
  FIND_PACKAGE_HANDLE_STANDARD_ARGS(UHD DEFAULT_MSG UHD_LIBRARIES UHD_INCLUDE_DIRS)
  mark_as_advanced(UHD_LIBRARIES UHD_INCLUDE_DIRS)

elseif(UHD_FIND_REQUIRED)

  message(FATAL_ERROR "UHD is required, but was not found.")

endif()
