#!/bin/bash
#
# Simple shell script to update db-ip.com database files
#
# Project    : ipv6calc/DBIP
# File       : DBIP-update.sh
# Version    : $Id: 7f92f42a0ed33724534b3e45e02985ac7bc690a0 $
# Copyright  : 2014-2018 by Peter Bieringer <pb (at) bieringer.de>
# License    : GNU GPL version 2

DBIP_DAT_DIR_DEFAULT="/usr/share/DBIP"
[ -z "$DBIP_DAT_DIR" ] && DBIP_DAT_DIR="$DBIP_DAT_DIR_DEFAULT"

DBIP_DAT_URL_BASE="http://download.db-ip.com/free/"
DBIP_DAT_FILES="dbip-country-%Y-%m.csv.gz dbip-city-%Y-%m.csv.gz"
DBIP_GENERATOR_LIST="./DBIP-generate-db.pl /usr/share/ipv6calc/tools/DBIP-generate-db.pl"


help() {
	cat <<END
Usage: $(basename "$0") [-s] [-D <dir>] [-U <url>]
	-s		skip download
	-g		skip generation of database file
	-D <dir>	database directory (optional)
	-U <url>	use URL to download full database

	database directory: $DBIP_DAT_DIR (default: $DBIP_DAT_DIR_DEFAULT)

 it honors externally defined environment value: DBIP_DAT_DIR

 this script (and its helper scripts) will download data from
 DB-IP.com (default: free versions) and generate Berkeley DB files for ipv6calc
 "DBIP" database support

 DBIP_DAT_URL_BASE=$DBIP_DAT_URL_BASE
 DBIP_DAT_FILES=$DBIP_DAT_FILES

 in case -U <url> (take URL from subscription e-mail) is provided, it downloads the full database
END
}

while getopts "D:U:gsh\?" opt; do
	case $opt in
	    s)
		skip_download=1
		;;
	    g)
		skip_generation=1
		;;
	    D)
		DBIP_DAT_DIR="$OPTARG"
		;;
	    U)
		DBIP_URL="$OPTARG"
		DBIP_DAT_FILES="dbip-full-%Y-%m.csv.gz"
		;;
	    *)
		help
		exit 1
		;;
	esac
done


for entry in $DBIP_GENERATOR_LIST; do
	if [ -e "$entry" -a -x "$entry" ]; then
		generate_db="$entry"
		break
	fi
done

if [ -z "$generate_db" ]; then
	echo "ERROR : no DBIP database generator found from list: $DBIP_GENERATOR_LIST"
	exit 1
else
	echo "INFO  : selected DBIP database generator: $generate_db"
fi

if [ ! -t 0 ]; then
	options_generate="-q"
fi

if [ ! -d "$DBIP_DAT_DIR" ]; then
	echo "ERROR : missing directory: $DBIP_DAT_DIR"
	exit 1
fi

if [ ! -w "$DBIP_DAT_DIR" ]; then
	echo "ERROR : missing write permissions on directory: $DBIP_DAT_DIR"
	exit 1
fi

# Download files
download_result=1
if [ "$skip_download" != "1" ]; then
	for file in $DBIP_DAT_FILES; do
		# convert tokens
		year=$(date +%Y)
		month=$(date +%m)

		file=${file//%Y/$year}
		file=${file//%m/$month}

		file_dest="$DBIP_DAT_DIR/`basename "$file"`"

		if [ -z "$DBIP_URL" ]; then
			echo "INFO  : try to download file: $file ($file_dest)"
			wget -q -O "$file_dest" "$DBIP_DAT_URL_BASE$file"
			if [ $? -ne 0 ]; then
				echo "ERROR : download of file not successful: $file ($file_dest)"
				continue
				download_result=0
			fi
		else
			echo "INFO  : try to download file: $DBIP_URL ($file_dest)"
			wget -q -O "$file_dest" "$DBIP_URL"
			if [ $? -ne 0 ]; then
				echo "ERROR : download of file not successful: $DBIP_URL ($file_dest)"
				continue
				download_result=0
			fi
		fi
		echo "INFO  : download of file successful: $DBIP_URL ($file_dest)"
	done
fi

# create db files from downloaded files
if [ "$download_result" = "1" -a "$skip_generation" != "1" ]; then
	for file in $DBIP_DAT_FILES; do
		time_begin=$(date '+%s')
		echo "INFO  : begin $(date '+%Y%m%d-%H%M%S')"

		# convert tokens
		year=$(date +%Y)
		month=$(date +%m)

		file=${file//%Y/$year}
		file=${file//%m/$month}

		file_input="$DBIP_DAT_DIR/`basename "$file"`"

		nice -n 19 $generate_db $options_generate -I "$file_input" -O "$DBIP_DAT_DIR" -A
		result=$?

		echo "INFO  : end $(date '+%Y%m%d-%H%M%S')"

		time_end=$(date '+%s')
		time_delta=$[ $time_end - $time_begin ]
		time_delta_min=$[ ($time_delta + 59) / 60 ]

		echo "INFO  : processing time for $file: $time_delta sec ($time_delta_min min)"

		if [ $result -ne 0 ]; then
			echo "ERROR : processing of file was not successful, keep downloaded file: $file_input"
		else
			echo "INFO  : processing of file was successful, remove downloaded file now: $file_input"
			rm -f "$file_input"
		fi
	done
fi
