// Copyright (c) 2013, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

library barback.transformer.transformer_group;

/// A [TransformerGroup] encapsulates a phased collection of transformers.
///
/// A transformer group is defined like a collection of phases, such as you
/// might pass to [Barback.updateTransformers]. Input assets are transformed by
/// the first phase, whose results are passed to the second phase, and so on.
///
/// However, from the perspective of someone using a [TransformerGroup], it
/// works just like a [Transformer]. It can be included in phases, and will run
/// in parallel to other transformers or groups in those phases. Other
/// transformers and groups will be unable to see any intermediate assets that
/// are generated by one phase of the group and consumed by another. Phases
/// after the one containing the group will be able to see its outputs, though.
class TransformerGroup {
  /// The phases that comprise this group.
  ///
  /// Each element of the inner iterable must be either a [Transformer] or a
  /// [TransformerGroup].
  final Iterable<Iterable> phases;

  TransformerGroup(Iterable<Iterable> phases)
      : this.phases = phases.map((phase) => phase.toList()).toList();

  String toString() => "group of $phases";
}
