// Copyright (c) 2019 Ultimaker B.V.
// Cura is released under the terms of the LGPLv3 or higher.

import QtQuick 2.10
import QtQuick.Controls 2.3

import UM 1.3 as UM
import Cura 1.1 as Cura


//
// ComboBox with Cura styling.
//
ComboBox
{
    id: control

    background: Rectangle
    {
        color:
        {
            if (!enabled)
            {
                return UM.Theme.getColor("setting_control_disabled")
            }

            if (control.hovered || control.activeFocus)
            {
                return UM.Theme.getColor("setting_control_highlight")
            }

            return UM.Theme.getColor("setting_control")
        }

        radius: UM.Theme.getSize("setting_control_radius").width
        border.width: UM.Theme.getSize("default_lining").width
        border.color:
        {
            if (!enabled)
            {
                return UM.Theme.getColor("setting_control_disabled_border")
            }

            if (control.hovered || control.activeFocus)
            {
                return UM.Theme.getColor("setting_control_border_highlight")
            }

            return UM.Theme.getColor("setting_control_border")
        }
    }

    indicator: UM.RecolorImage
    {
        id: downArrow
        x: control.width - width - control.rightPadding
        y: control.topPadding + Math.round((control.availableHeight - height) / 2)

        source: UM.Theme.getIcon("arrow_bottom")
        width: UM.Theme.getSize("standard_arrow").width
        height: UM.Theme.getSize("standard_arrow").height
        sourceSize.width: width + 5 * screenScaleFactor
        sourceSize.height: width + 5 * screenScaleFactor

        color: UM.Theme.getColor("setting_control_button")
    }

    contentItem: Label
    {
        anchors.left: parent.left
        anchors.leftMargin: UM.Theme.getSize("setting_unit_margin").width
        anchors.verticalCenter: parent.verticalCenter
        anchors.right: downArrow.left

        text: control.currentText
        textFormat: Text.PlainText
        renderType: Text.NativeRendering
        font: UM.Theme.getFont("default")
        color: !enabled ? UM.Theme.getColor("setting_control_disabled_text") : UM.Theme.getColor("setting_control_text")
        elide: Text.ElideRight
        verticalAlignment: Text.AlignVCenter
    }

    popup: Popup
    {
        y: control.height - UM.Theme.getSize("default_lining").height
        width: control.width
        implicitHeight: contentItem.implicitHeight + 2 * UM.Theme.getSize("default_lining").width
        padding: UM.Theme.getSize("default_lining").width

        contentItem: ListView
        {
            clip: true
            implicitHeight: contentHeight
            model: control.popup.visible ? control.delegateModel : null
            currentIndex: control.highlightedIndex

            ScrollIndicator.vertical: ScrollIndicator { }
        }

        background: Rectangle
        {
            color: UM.Theme.getColor("setting_control")
            border.color: UM.Theme.getColor("setting_control_border")
        }
    }

    delegate: ItemDelegate
    {
        id: delegateItem
        width: control.width - 2 * UM.Theme.getSize("default_lining").width
        height: control.height
        highlighted: control.highlightedIndex == index
        text:
        // FIXME: Maybe there is a better way to do this. Check model and modelData doc page:
        // https://doc.qt.io/qt-5/qtquick-modelviewsdata-modelview.html
        {
            var _val = undefined
            if (typeof _val === 'undefined')  // try to get textRole from "model".
            {
                _val = model[textRole]
            }
            if (typeof _val === 'undefined')  // try to get textRole from "modelData" if it's still undefined.
            {
                _val = modelData[textRole]
            }
            return (typeof _val !== 'undefined') ? _val : ""
        }

        contentItem: Label
        {
            // FIXME: Somehow the top/bottom anchoring is not correct on Linux and it results in invisible texts.
            anchors.fill: parent
            anchors.leftMargin: UM.Theme.getSize("setting_unit_margin").width
            anchors.rightMargin: UM.Theme.getSize("setting_unit_margin").width

            text: delegateItem.text
            textFormat: Text.PlainText
            renderType: Text.NativeRendering
            color: control.contentItem.color
            font: UM.Theme.getFont("default")
            elide: Text.ElideRight
            verticalAlignment: Text.AlignVCenter
        }

        background: Rectangle
        {
            color: parent.highlighted ? UM.Theme.getColor("setting_control_highlight") : "transparent"
            border.color: parent.highlighted ? UM.Theme.getColor("setting_control_border_highlight") : "transparent"
        }
    }
}
