/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.network;

import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.common.network.NetworkUtils;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;

public final class NetworkService {
    public static final String DEFAULT_NETWORK_HOST = "_local_";
    public static final Setting<Boolean> NETWORK_SERVER = Setting.boolSetting("network.server", true, Setting.Property.NodeScope);
    public static final Setting<List<String>> GLOBAL_NETWORK_HOST_SETTING = Setting.listSetting("network.host", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> GLOBAL_NETWORK_BIND_HOST_SETTING = Setting.listSetting("network.bind_host", GLOBAL_NETWORK_HOST_SETTING, Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> GLOBAL_NETWORK_PUBLISH_HOST_SETTING = Setting.listSetting("network.publish_host", GLOBAL_NETWORK_HOST_SETTING, Function.identity(), Setting.Property.NodeScope);
    public static final Setting<Boolean> TCP_NO_DELAY = Setting.boolSetting("network.tcp.no_delay", true, Setting.Property.NodeScope);
    public static final Setting<Boolean> TCP_KEEP_ALIVE = Setting.boolSetting("network.tcp.keep_alive", true, Setting.Property.NodeScope);
    public static final Setting<Boolean> TCP_REUSE_ADDRESS = Setting.boolSetting("network.tcp.reuse_address", NetworkUtils.defaultReuseAddress(), Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> TCP_SEND_BUFFER_SIZE = Setting.byteSizeSetting("network.tcp.send_buffer_size", new ByteSizeValue(-1L), Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> TCP_RECEIVE_BUFFER_SIZE = Setting.byteSizeSetting("network.tcp.receive_buffer_size", new ByteSizeValue(-1L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> TCP_CONNECT_TIMEOUT = Setting.timeSetting("network.tcp.connect_timeout", new TimeValue(30L, TimeUnit.SECONDS), Setting.Property.NodeScope, Setting.Property.Deprecated);
    private final List<CustomNameResolver> customNameResolvers;

    public NetworkService(List<CustomNameResolver> customNameResolvers) {
        this.customNameResolvers = Objects.requireNonNull(customNameResolvers, "customNameResolvers must be non null");
    }

    public InetAddress[] resolveBindHostAddresses(String[] bindHosts) throws IOException {
        InetAddress[] addresses;
        if (bindHosts == null || bindHosts.length == 0) {
            for (CustomNameResolver customNameResolver : this.customNameResolvers) {
                InetAddress[] addresses2 = customNameResolver.resolveDefault();
                if (addresses2 == null) continue;
                return addresses2;
            }
            bindHosts = new String[]{DEFAULT_NETWORK_HOST};
        }
        for (InetAddress address : addresses = this.resolveInetAddresses(bindHosts)) {
            if (address.isMulticastAddress()) {
                throw new IllegalArgumentException("bind address: {" + NetworkAddress.format(address) + "} is invalid: multicast address");
            }
            if (!address.isAnyLocalAddress() || addresses.length <= 1) continue;
            throw new IllegalArgumentException("bind address: {" + NetworkAddress.format(address) + "} is wildcard, but multiple addresses specified: this makes no sense");
        }
        return addresses;
    }

    public InetAddress resolvePublishHostAddresses(String[] publishHosts) throws IOException {
        InetAddress[] addresses;
        if (publishHosts == null || publishHosts.length == 0) {
            for (CustomNameResolver customNameResolver : this.customNameResolvers) {
                InetAddress[] addresses2 = customNameResolver.resolveDefault();
                if (addresses2 == null) continue;
                return addresses2[0];
            }
            publishHosts = new String[]{DEFAULT_NETWORK_HOST};
        }
        if ((addresses = this.resolveInetAddresses(publishHosts)).length == 1 && addresses[0].isAnyLocalAddress()) {
            HashSet<InetAddress> all = new HashSet<InetAddress>(Arrays.asList(NetworkUtils.getAllAddresses()));
            addresses = all.toArray(new InetAddress[all.size()]);
        }
        for (InetAddress address : addresses) {
            if (address.isMulticastAddress()) {
                throw new IllegalArgumentException("publish address: {" + NetworkAddress.format(address) + "} is invalid: multicast address");
            }
            if (!address.isAnyLocalAddress()) continue;
            throw new IllegalArgumentException("publish address: {" + NetworkAddress.format(address) + "} is wildcard, but multiple addresses specified: this makes no sense");
        }
        if (addresses.length > 1) {
            ArrayList<InetAddress> sorted = new ArrayList<InetAddress>(Arrays.asList(addresses));
            NetworkUtils.sortAddresses(sorted);
            addresses = new InetAddress[]{(InetAddress)sorted.get(0)};
        }
        return addresses[0];
    }

    private InetAddress[] resolveInetAddresses(String[] hosts) throws IOException {
        if (hosts.length == 0) {
            throw new IllegalArgumentException("empty host specification");
        }
        HashSet<InetAddress> set = new HashSet<InetAddress>();
        for (String host : hosts) {
            set.addAll(Arrays.asList(this.resolveInternal(host)));
        }
        return set.toArray(new InetAddress[set.size()]);
    }

    private InetAddress[] resolveInternal(String host) throws IOException {
        if (host.startsWith("#") && host.endsWith("#") || host.startsWith("_") && host.endsWith("_")) {
            host = host.substring(1, host.length() - 1);
            if (this.customNameResolvers != null) {
                for (CustomNameResolver customNameResolver : this.customNameResolvers) {
                    InetAddress[] addresses = customNameResolver.resolveIfPossible(host);
                    if (addresses == null) continue;
                    return addresses;
                }
            }
            switch (host) {
                case "local": {
                    return NetworkUtils.getLoopbackAddresses();
                }
                case "local:ipv4": {
                    return NetworkUtils.filterIPV4(NetworkUtils.getLoopbackAddresses());
                }
                case "local:ipv6": {
                    return NetworkUtils.filterIPV6(NetworkUtils.getLoopbackAddresses());
                }
                case "site": {
                    return NetworkUtils.getSiteLocalAddresses();
                }
                case "site:ipv4": {
                    return NetworkUtils.filterIPV4(NetworkUtils.getSiteLocalAddresses());
                }
                case "site:ipv6": {
                    return NetworkUtils.filterIPV6(NetworkUtils.getSiteLocalAddresses());
                }
                case "global": {
                    return NetworkUtils.getGlobalAddresses();
                }
                case "global:ipv4": {
                    return NetworkUtils.filterIPV4(NetworkUtils.getGlobalAddresses());
                }
                case "global:ipv6": {
                    return NetworkUtils.filterIPV6(NetworkUtils.getGlobalAddresses());
                }
            }
            if (host.endsWith(":ipv4")) {
                host = host.substring(0, host.length() - 5);
                return NetworkUtils.filterIPV4(NetworkUtils.getAddressesForInterface(host));
            }
            if (host.endsWith(":ipv6")) {
                host = host.substring(0, host.length() - 5);
                return NetworkUtils.filterIPV6(NetworkUtils.getAddressesForInterface(host));
            }
            return NetworkUtils.getAddressesForInterface(host);
        }
        return InetAddress.getAllByName(host);
    }

    public static interface CustomNameResolver {
        public InetAddress[] resolveDefault();

        public InetAddress[] resolveIfPossible(String var1) throws IOException;
    }
}

