/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.analysis.util.CharFilterFactory;
import org.apache.lucene.analysis.util.TokenFilterFactory;
import org.apache.lucene.analysis.util.TokenizerFactory;
import org.apache.lucene.codecs.Codec;
import org.apache.lucene.codecs.DocValuesFormat;
import org.apache.lucene.codecs.PostingsFormat;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.admin.cluster.node.info.PluginsAndModules;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.LifecycleComponent;
import org.elasticsearch.common.inject.Module;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.IndexModule;
import org.elasticsearch.plugins.ExtensiblePlugin;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.plugins.PluginInfo;
import org.elasticsearch.plugins.PluginLoaderIndirection;
import org.elasticsearch.threadpool.ExecutorBuilder;

public class PluginsService {
    private static final Logger logger = LogManager.getLogger(PluginsService.class);
    private final Settings settings;
    private final Path configPath;
    private final List<Tuple<PluginInfo, Plugin>> plugins;
    private final PluginsAndModules info;
    public static final Setting<List<String>> MANDATORY_SETTING = Setting.listSetting("plugin.mandatory", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);

    public List<Setting<?>> getPluginSettings() {
        return this.plugins.stream().flatMap(p -> ((Plugin)p.v2()).getSettings().stream()).collect(Collectors.toList());
    }

    public List<String> getPluginSettingsFilter() {
        return this.plugins.stream().flatMap(p -> ((Plugin)p.v2()).getSettingsFilter().stream()).collect(Collectors.toList());
    }

    public PluginsService(Settings settings, Path configPath, Path modulesDirectory, Path pluginsDirectory, Collection<Class<? extends Plugin>> classpathPlugins) {
        this.settings = settings;
        this.configPath = configPath;
        ArrayList<Object> pluginsLoaded = new ArrayList<Object>();
        ArrayList<PluginInfo> pluginsList = new ArrayList<PluginInfo>();
        ArrayList<String> pluginsNames = new ArrayList<String>();
        for (Class<? extends Plugin> pluginClass : classpathPlugins) {
            Plugin plugin = this.loadPlugin(pluginClass, settings, configPath);
            Iterator<Bundle> pluginInfo = new PluginInfo(pluginClass.getName(), "classpath plugin", "NA", Version.CURRENT, "1.8", pluginClass.getName(), Collections.emptyList(), false);
            if (logger.isTraceEnabled()) {
                logger.trace("plugin loaded from classpath [{}]", pluginInfo);
            }
            pluginsLoaded.add(new Tuple(pluginInfo, (Object)plugin));
            pluginsList.add((PluginInfo)((Object)pluginInfo));
            pluginsNames.add(((PluginInfo)((Object)pluginInfo)).getName());
        }
        LinkedHashSet<Bundle> seenBundles = new LinkedHashSet<Bundle>();
        ArrayList<PluginInfo> modulesList = new ArrayList<PluginInfo>();
        if (modulesDirectory != null) {
            try {
                Set<Bundle> modules = PluginsService.getModuleBundles(modulesDirectory);
                for (Bundle bundle : modules) {
                    modulesList.add(bundle.plugin);
                }
                seenBundles.addAll(modules);
            }
            catch (IOException ex) {
                throw new IllegalStateException("Unable to initialize modules", ex);
            }
        }
        if (pluginsDirectory != null) {
            try {
                if (FileSystemUtils.isAccessibleDirectory(pluginsDirectory, logger)) {
                    PluginsService.checkForFailedPluginRemovals(pluginsDirectory);
                    Set<Bundle> plugins = PluginsService.getPluginBundles(pluginsDirectory);
                    for (Bundle bundle : plugins) {
                        pluginsList.add(bundle.plugin);
                        pluginsNames.add(bundle.plugin.getName());
                    }
                    seenBundles.addAll(plugins);
                }
            }
            catch (IOException ex) {
                throw new IllegalStateException("Unable to initialize plugins", ex);
            }
        }
        List<Tuple<PluginInfo, Plugin>> loaded = this.loadBundles(seenBundles);
        pluginsLoaded.addAll(loaded);
        this.info = new PluginsAndModules(pluginsList, modulesList);
        this.plugins = Collections.unmodifiableList(pluginsLoaded);
        List<String> mandatoryPlugins = MANDATORY_SETTING.get(settings);
        if (!mandatoryPlugins.isEmpty()) {
            HashSet<String> missingPlugins = new HashSet<String>();
            for (String mandatoryPlugin : mandatoryPlugins) {
                if (pluginsNames.contains(mandatoryPlugin) || missingPlugins.contains(mandatoryPlugin)) continue;
                missingPlugins.add(mandatoryPlugin);
            }
            if (!missingPlugins.isEmpty()) {
                String message = String.format(Locale.ROOT, "missing mandatory plugins [%s], found plugins [%s]", Strings.collectionToDelimitedString(missingPlugins, ", "), Strings.collectionToDelimitedString(pluginsNames, ", "));
                throw new IllegalStateException(message);
            }
        }
        PluginsService.logPluginInfo(this.info.getModuleInfos(), "module", logger);
        PluginsService.logPluginInfo(this.info.getPluginInfos(), "plugin", logger);
    }

    private static void logPluginInfo(List<PluginInfo> pluginInfos, String type, Logger logger) {
        assert (pluginInfos != null);
        if (pluginInfos.isEmpty()) {
            logger.info("no " + type + "s loaded");
        } else {
            for (String name : pluginInfos.stream().map(PluginInfo::getName).sorted().collect(Collectors.toList())) {
                logger.info("loaded " + type + " [" + name + "]");
            }
        }
    }

    public Settings updatedSettings() {
        HashMap<String, String> foundSettings = new HashMap<String, String>();
        TreeMap<String, String> features = new TreeMap<String, String>();
        Settings.Builder builder = Settings.builder();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            Settings settings = ((Plugin)plugin.v2()).additionalSettings();
            for (String setting : settings.keySet()) {
                String oldPlugin = foundSettings.put(setting, ((PluginInfo)plugin.v1()).getName());
                if (oldPlugin == null) continue;
                throw new IllegalArgumentException("Cannot have additional setting [" + setting + "] in plugin [" + ((PluginInfo)plugin.v1()).getName() + "], already added in plugin [" + oldPlugin + "]");
            }
            builder.put(settings);
            Optional<String> maybeFeature = ((Plugin)plugin.v2()).getFeature();
            if (!maybeFeature.isPresent()) continue;
            String feature = maybeFeature.get();
            if (features.containsKey(feature)) {
                String message = String.format(Locale.ROOT, "duplicate feature [%s] in plugin [%s], already added in [%s]", feature, ((PluginInfo)plugin.v1()).getName(), features.get(feature));
                throw new IllegalArgumentException(message);
            }
            features.put(feature, ((PluginInfo)plugin.v1()).getName());
        }
        for (String feature : features.keySet()) {
            builder.put("transport.features." + feature, true);
        }
        return builder.put(this.settings).build();
    }

    public Collection<Module> createGuiceModules() {
        ArrayList<Module> modules = new ArrayList<Module>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            modules.addAll(((Plugin)plugin.v2()).createGuiceModules());
        }
        return modules;
    }

    public List<ExecutorBuilder<?>> getExecutorBuilders(Settings settings) {
        ArrayList builders = new ArrayList();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            builders.addAll(((Plugin)plugin.v2()).getExecutorBuilders(settings));
        }
        return builders;
    }

    public Collection<Class<? extends LifecycleComponent>> getGuiceServiceClasses() {
        ArrayList<Class<? extends LifecycleComponent>> services = new ArrayList<Class<? extends LifecycleComponent>>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            services.addAll(((Plugin)plugin.v2()).getGuiceServiceClasses());
        }
        return services;
    }

    public void onIndexModule(IndexModule indexModule) {
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            ((Plugin)plugin.v2()).onIndexModule(indexModule);
        }
    }

    public PluginsAndModules info() {
        return this.info;
    }

    public static List<Path> findPluginDirs(Path rootPath) throws IOException {
        ArrayList<Path> plugins = new ArrayList<Path>();
        HashSet<String> seen = new HashSet<String>();
        if (Files.exists(rootPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(rootPath);){
                for (Path plugin : stream) {
                    if (FileSystemUtils.isDesktopServicesStore(plugin) || plugin.getFileName().toString().startsWith(".removing-")) continue;
                    if (!seen.add(plugin.getFileName().toString())) {
                        throw new IllegalStateException("duplicate plugin: " + plugin);
                    }
                    plugins.add(plugin);
                }
            }
        }
        return plugins;
    }

    static void verifyCompatibility(PluginInfo info) {
        if (!info.getElasticsearchVersion().equals(Version.CURRENT)) {
            throw new IllegalArgumentException("Plugin [" + info.getName() + "] was built for Elasticsearch version " + info.getElasticsearchVersion() + " but version " + Version.CURRENT + " is running");
        }
        JarHell.checkJavaVersion((String)info.getName(), (String)info.getJavaVersion());
    }

    static void checkForFailedPluginRemovals(Path pluginsDirectory) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(pluginsDirectory, ".removing-*");){
            Iterator<Path> iterator = stream.iterator();
            if (iterator.hasNext()) {
                Path removing = iterator.next();
                String fileName = removing.getFileName().toString();
                String name = fileName.substring(1 + fileName.indexOf("-"));
                String message = String.format(Locale.ROOT, "found file [%s] from a failed attempt to remove the plugin [%s]; execute [elasticsearch-plugin remove %2$s]", removing, name);
                throw new IllegalStateException(message);
            }
        }
    }

    static Set<Bundle> getModuleBundles(Path modulesDirectory) throws IOException {
        return PluginsService.findBundles(modulesDirectory, "module");
    }

    static Set<Bundle> getPluginBundles(Path pluginsDirectory) throws IOException {
        return PluginsService.findBundles(pluginsDirectory, "plugin");
    }

    private static Set<Bundle> findBundles(Path directory, String type) throws IOException {
        HashSet<Bundle> bundles = new HashSet<Bundle>();
        for (Path plugin : PluginsService.findPluginDirs(directory)) {
            Bundle bundle = PluginsService.readPluginBundle(bundles, plugin, type);
            bundles.add(bundle);
        }
        return bundles;
    }

    private static Bundle readPluginBundle(Set<Bundle> bundles, Path plugin, String type) throws IOException {
        PluginInfo info;
        LogManager.getLogger(PluginsService.class).trace("--- adding [{}] [{}]", (Object)type, (Object)plugin.toAbsolutePath());
        try {
            info = PluginInfo.readFromProperties(plugin);
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not load plugin descriptor for " + type + " directory [" + plugin.getFileName() + "]", e);
        }
        Bundle bundle = new Bundle(info, plugin);
        if (!bundles.add(bundle)) {
            throw new IllegalStateException("duplicate " + type + ": " + info);
        }
        return bundle;
    }

    static List<Bundle> sortBundles(Set<Bundle> bundles) {
        Map<String, Bundle> namedBundles = bundles.stream().collect(Collectors.toMap(b -> b.plugin.getName(), Function.identity()));
        LinkedHashSet<Bundle> sortedBundles = new LinkedHashSet<Bundle>();
        LinkedHashSet<String> dependencyStack = new LinkedHashSet<String>();
        for (Bundle bundle : bundles) {
            PluginsService.addSortedBundle(bundle, namedBundles, sortedBundles, dependencyStack);
        }
        return new ArrayList<Bundle>(sortedBundles);
    }

    private static void addSortedBundle(Bundle bundle, Map<String, Bundle> bundles, LinkedHashSet<Bundle> sortedBundles, LinkedHashSet<String> dependencyStack) {
        String name = bundle.plugin.getName();
        if (dependencyStack.contains(name)) {
            StringBuilder msg = new StringBuilder("Cycle found in plugin dependencies: ");
            dependencyStack.forEach(s -> {
                msg.append((String)s);
                msg.append(" -> ");
            });
            msg.append(name);
            throw new IllegalStateException(msg.toString());
        }
        if (sortedBundles.contains(bundle)) {
            return;
        }
        dependencyStack.add(name);
        for (String dependency : bundle.plugin.getExtendedPlugins()) {
            Bundle depBundle = bundles.get(dependency);
            if (depBundle == null) {
                throw new IllegalArgumentException("Missing plugin [" + dependency + "], dependency of [" + name + "]");
            }
            PluginsService.addSortedBundle(depBundle, bundles, sortedBundles, dependencyStack);
            assert (sortedBundles.contains(depBundle));
        }
        dependencyStack.remove(name);
        sortedBundles.add(bundle);
    }

    private List<Tuple<PluginInfo, Plugin>> loadBundles(Set<Bundle> bundles) {
        ArrayList<Tuple> plugins = new ArrayList<Tuple>();
        HashMap<String, Plugin> loaded = new HashMap<String, Plugin>();
        HashMap<String, Set<URL>> transitiveUrls = new HashMap<String, Set<URL>>();
        List<Bundle> sortedBundles = PluginsService.sortBundles(bundles);
        for (Bundle bundle : sortedBundles) {
            PluginsService.checkBundleJarHell(JarHell.parseClassPath(), bundle, transitiveUrls);
            Plugin plugin = this.loadBundle(bundle, loaded);
            plugins.add(new Tuple((Object)bundle.plugin, (Object)plugin));
        }
        return Collections.unmodifiableList(plugins);
    }

    static void checkBundleJarHell(Set<URL> classpath, Bundle bundle, Map<String, Set<URL>> transitiveUrls) {
        List<String> exts = bundle.plugin.getExtendedPlugins();
        try {
            Logger logger = LogManager.getLogger(JarHell.class);
            HashSet<URL> urls = new HashSet<URL>();
            for (String extendedPlugin : exts) {
                Set<URL> pluginUrls = transitiveUrls.get(extendedPlugin);
                assert (pluginUrls != null) : "transitive urls should have already been set for " + extendedPlugin;
                HashSet<Object> intersection = new HashSet(urls);
                intersection.retainAll(pluginUrls);
                if (!intersection.isEmpty()) {
                    throw new IllegalStateException("jar hell! extended plugins " + exts + " have duplicate codebases with each other: " + intersection);
                }
                intersection = new HashSet<URL>(bundle.urls);
                intersection.retainAll(pluginUrls);
                if (!intersection.isEmpty()) {
                    throw new IllegalStateException("jar hell! duplicate codebases with extended plugin [" + extendedPlugin + "]: " + intersection);
                }
                urls.addAll(pluginUrls);
                JarHell.checkJarHell(urls, arg_0 -> ((Logger)logger).debug(arg_0));
            }
            urls.addAll(bundle.urls);
            JarHell.checkJarHell(urls, arg_0 -> ((Logger)logger).debug(arg_0));
            transitiveUrls.put(bundle.plugin.getName(), urls);
            HashSet<URL> intersection = new HashSet<URL>(classpath);
            intersection.retainAll(bundle.urls);
            if (!intersection.isEmpty()) {
                throw new IllegalStateException("jar hell! duplicate codebases between plugin and core: " + intersection);
            }
            HashSet<URL> union = new HashSet<URL>(classpath);
            union.addAll(bundle.urls);
            JarHell.checkJarHell(union, arg_0 -> ((Logger)logger).debug(arg_0));
        }
        catch (Exception e) {
            throw new IllegalStateException("failed to load plugin " + bundle.plugin.getName() + " due to jar hell", e);
        }
    }

    private Plugin loadBundle(Bundle bundle, Map<String, Plugin> loaded) {
        String name = bundle.plugin.getName();
        PluginsService.verifyCompatibility(bundle.plugin);
        ArrayList<ClassLoader> extendedLoaders = new ArrayList<ClassLoader>();
        for (String extendedPluginName : bundle.plugin.getExtendedPlugins()) {
            Plugin extendedPlugin = loaded.get(extendedPluginName);
            assert (extendedPlugin != null);
            if (!ExtensiblePlugin.class.isInstance(extendedPlugin)) {
                throw new IllegalStateException("Plugin [" + name + "] cannot extend non-extensible plugin [" + extendedPluginName + "]");
            }
            extendedLoaders.add(extendedPlugin.getClass().getClassLoader());
        }
        ClassLoader parentLoader = PluginLoaderIndirection.createLoader(this.getClass().getClassLoader(), extendedLoaders);
        URLClassLoader loader = URLClassLoader.newInstance(bundle.urls.toArray(new URL[0]), parentLoader);
        PluginsService.reloadLuceneSPI(loader);
        for (String extendedPluginName : bundle.plugin.getExtendedPlugins()) {
            ((ExtensiblePlugin)ExtensiblePlugin.class.cast(loaded.get(extendedPluginName))).reloadSPI(loader);
        }
        Class<? extends Plugin> pluginClass = this.loadPluginClass(bundle.plugin.getClassname(), loader);
        Plugin plugin = this.loadPlugin(pluginClass, this.settings, this.configPath);
        loaded.put(name, plugin);
        return plugin;
    }

    static void reloadLuceneSPI(ClassLoader loader) {
        PostingsFormat.reloadPostingsFormats((ClassLoader)loader);
        DocValuesFormat.reloadDocValuesFormats((ClassLoader)loader);
        Codec.reloadCodecs((ClassLoader)loader);
        CharFilterFactory.reloadCharFilters((ClassLoader)loader);
        TokenFilterFactory.reloadTokenFilters((ClassLoader)loader);
        TokenizerFactory.reloadTokenizers((ClassLoader)loader);
    }

    private Class<? extends Plugin> loadPluginClass(String className, ClassLoader loader) {
        try {
            return loader.loadClass(className).asSubclass(Plugin.class);
        }
        catch (ClassNotFoundException e) {
            throw new ElasticsearchException("Could not find plugin class [" + className + "]", (Throwable)e, new Object[0]);
        }
    }

    private Plugin loadPlugin(Class<? extends Plugin> pluginClass, Settings settings, Path configPath) {
        Constructor<?>[] constructors = pluginClass.getConstructors();
        if (constructors.length == 0) {
            throw new IllegalStateException("no public constructor for [" + pluginClass.getName() + "]");
        }
        if (constructors.length > 1) {
            throw new IllegalStateException("no unique public constructor for [" + pluginClass.getName() + "]");
        }
        Constructor<?> constructor = constructors[0];
        if (constructor.getParameterCount() > 2) {
            throw new IllegalStateException(this.signatureMessage(pluginClass));
        }
        Class<?>[] parameterTypes = constructor.getParameterTypes();
        try {
            if (constructor.getParameterCount() == 2 && parameterTypes[0] == Settings.class && parameterTypes[1] == Path.class) {
                return (Plugin)constructor.newInstance(settings, configPath);
            }
            if (constructor.getParameterCount() == 1 && parameterTypes[0] == Settings.class) {
                return (Plugin)constructor.newInstance(settings);
            }
            if (constructor.getParameterCount() == 0) {
                return (Plugin)constructor.newInstance(new Object[0]);
            }
            throw new IllegalStateException(this.signatureMessage(pluginClass));
        }
        catch (ReflectiveOperationException e) {
            throw new IllegalStateException("failed to load plugin class [" + pluginClass.getName() + "]", e);
        }
    }

    private String signatureMessage(Class<? extends Plugin> clazz) {
        return String.format(Locale.ROOT, "no public constructor of correct signature for [%s]; must be [%s], [%s], or [%s]", clazz.getName(), "(org.elasticsearch.common.settings.Settings,java.nio.file.Path)", "(org.elasticsearch.common.settings.Settings)", "()");
    }

    public <T> List<T> filterPlugins(Class<T> type) {
        return this.plugins.stream().filter(x -> type.isAssignableFrom(((Plugin)x.v2()).getClass())).map(p -> p.v2()).collect(Collectors.toList());
    }

    static class Bundle {
        final PluginInfo plugin;
        final Set<URL> urls;

        Bundle(PluginInfo plugin, Path dir) throws IOException {
            this.plugin = Objects.requireNonNull(plugin);
            LinkedHashSet<URL> urls = new LinkedHashSet<URL>();
            try (DirectoryStream<Path> jarStream = Files.newDirectoryStream(dir, "*.jar");){
                for (Path jar : jarStream) {
                    URL url = jar.toRealPath(new LinkOption[0]).toUri().toURL();
                    if (urls.add(url)) continue;
                    throw new IllegalStateException("duplicate codebase: " + url);
                }
            }
            this.urls = Objects.requireNonNull(urls);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bundle bundle = (Bundle)o;
            return Objects.equals(this.plugin, bundle.plugin);
        }

        public int hashCode() {
            return Objects.hash(this.plugin);
        }
    }
}

