/*
Copyright 2016 OpenMarket Ltd
Copyright 2017 Vector Creations Ltd
Copyright 2018-2019 New Vector Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
"use strict";

/**
 * @module crypto
 */

Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.verificationMethods = undefined;

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _set = require('babel-runtime/core-js/set');

var _set2 = _interopRequireDefault(_set);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _toConsumableArray2 = require('babel-runtime/helpers/toConsumableArray');

var _toConsumableArray3 = _interopRequireDefault(_toConsumableArray2);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _assign = require('babel-runtime/core-js/object/assign');

var _assign2 = _interopRequireDefault(_assign);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _regenerator = require('babel-runtime/regenerator');

var _regenerator2 = _interopRequireDefault(_regenerator);

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _map = require('babel-runtime/core-js/map');

var _map2 = _interopRequireDefault(_map);

var _defineProperty2 = require('babel-runtime/helpers/defineProperty');

var _defineProperty3 = _interopRequireDefault(_defineProperty2);

var _defaultVerificationM;

// returns a promise which resolves to the response
var _uploadOneTimeKeys = function () {
    var _ref6 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee6(crypto) {
        var oneTimeKeys, oneTimeJson, promises, keyId, k, res;
        return _regenerator2.default.wrap(function _callee6$(_context6) {
            while (1) {
                switch (_context6.prev = _context6.next) {
                    case 0:
                        _context6.next = 2;
                        return (0, _bluebird.resolve)(crypto._olmDevice.getOneTimeKeys());

                    case 2:
                        oneTimeKeys = _context6.sent;
                        oneTimeJson = {};
                        promises = [];


                        for (keyId in oneTimeKeys.curve25519) {
                            if (oneTimeKeys.curve25519.hasOwnProperty(keyId)) {
                                k = {
                                    key: oneTimeKeys.curve25519[keyId]
                                };

                                oneTimeJson["signed_curve25519:" + keyId] = k;
                                promises.push(crypto._signObject(k));
                            }
                        }

                        _context6.next = 8;
                        return (0, _bluebird.resolve)(_bluebird2.default.all(promises));

                    case 8:
                        _context6.next = 10;
                        return (0, _bluebird.resolve)(crypto._baseApis.uploadKeysRequest({
                            one_time_keys: oneTimeJson
                        }, {
                            // for now, we set the device id explicitly, as we may not be using the
                            // same one as used in login.
                            device_id: crypto._deviceId
                        }));

                    case 10:
                        res = _context6.sent;
                        _context6.next = 13;
                        return (0, _bluebird.resolve)(crypto._olmDevice.markKeysAsPublished());

                    case 13:
                        return _context6.abrupt('return', res);

                    case 14:
                    case 'end':
                        return _context6.stop();
                }
            }
        }, _callee6, this);
    }));

    return function _uploadOneTimeKeys(_x3) {
        return _ref6.apply(this, arguments);
    };
}();

/**
 * Download the keys for a list of users and stores the keys in the session
 * store.
 * @param {Array} userIds The users to fetch.
 * @param {bool} forceDownload Always download the keys even if cached.
 *
 * @return {Promise} A promise which resolves to a map userId->deviceId->{@link
 * module:crypto/deviceinfo|DeviceInfo}.
 */


exports.isCryptoAvailable = isCryptoAvailable;
exports.default = Crypto;

var _events = require('events');

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _randomstring = require('../randomstring');

var _OutgoingRoomKeyRequestManager = require('./OutgoingRoomKeyRequestManager');

var _OutgoingRoomKeyRequestManager2 = _interopRequireDefault(_OutgoingRoomKeyRequestManager);

var _indexeddbCryptoStore = require('./store/indexeddb-crypto-store');

var _indexeddbCryptoStore2 = _interopRequireDefault(_indexeddbCryptoStore);

var _QRCode = require('./verification/QRCode');

var _SAS = require('./verification/SAS');

var _SAS2 = _interopRequireDefault(_SAS);

var _Error = require('./verification/Error');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var anotherjson = require('another-json');

var utils = require("../utils");
var OlmDevice = require("./OlmDevice");
var olmlib = require("./olmlib");
var algorithms = require("./algorithms");
var DeviceInfo = require("./deviceinfo");
var DeviceVerification = DeviceInfo.DeviceVerification;
var DeviceList = require('./DeviceList').default;


var defaultVerificationMethods = (_defaultVerificationM = {}, (0, _defineProperty3.default)(_defaultVerificationM, _QRCode.ScanQRCode.NAME, _QRCode.ScanQRCode), (0, _defineProperty3.default)(_defaultVerificationM, _QRCode.ShowQRCode.NAME, _QRCode.ShowQRCode), (0, _defineProperty3.default)(_defaultVerificationM, _SAS2.default.NAME, _SAS2.default), _defaultVerificationM);

/**
 * verification method names
 */
var verificationMethods = exports.verificationMethods = {
    QR_CODE_SCAN: _QRCode.ScanQRCode.NAME,
    QR_CODE_SHOW: _QRCode.ShowQRCode.NAME,
    SAS: _SAS2.default.NAME
};

function isCryptoAvailable() {
    return Boolean(global.Olm);
}

var MIN_FORCE_SESSION_INTERVAL_MS = 60 * 60 * 1000;
var KEY_BACKUP_KEYS_PER_REQUEST = 200;

/**
 * Cryptography bits
 *
 * This module is internal to the js-sdk; the public API is via MatrixClient.
 *
 * @constructor
 * @alias module:crypto
 *
 * @internal
 *
 * @param {module:base-apis~MatrixBaseApis} baseApis base matrix api interface
 *
 * @param {module:store/session/webstorage~WebStorageSessionStore} sessionStore
 *    Store to be used for end-to-end crypto session data
 *
 * @param {string} userId The user ID for the local user
 *
 * @param {string} deviceId The identifier for this device.
 *
 * @param {Object} clientStore the MatrixClient data store.
 *
 * @param {module:crypto/store/base~CryptoStore} cryptoStore
 *    storage for the crypto layer.
 *
 * @param {RoomList} roomList An initialised RoomList object
 *
 * @param {Array} verificationMethods Array of verification methods to use.
 *    Each element can either be a string from MatrixClient.verificationMethods
 *    or a class that implements a verification method.
 */
function Crypto(baseApis, sessionStore, userId, deviceId, clientStore, cryptoStore, roomList, verificationMethods) {
    this._baseApis = baseApis;
    this._sessionStore = sessionStore;
    this._userId = userId;
    this._deviceId = deviceId;
    this._clientStore = clientStore;
    this._cryptoStore = cryptoStore;
    this._roomList = roomList;
    this._verificationMethods = new _map2.default();
    if (verificationMethods) {
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
            for (var _iterator = (0, _getIterator3.default)(verificationMethods), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                var method = _step.value;

                if (typeof method === "string") {
                    if (defaultVerificationMethods[method]) {
                        this._verificationMethods.set(method, defaultVerificationMethods[method]);
                    }
                } else if (method.NAME) {
                    this._verificationMethods.set(method.NAME, method);
                }
            }
        } catch (err) {
            _didIteratorError = true;
            _iteratorError = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion && _iterator.return) {
                    _iterator.return();
                }
            } finally {
                if (_didIteratorError) {
                    throw _iteratorError;
                }
            }
        }
    }

    // track whether this device's megolm keys are being backed up incrementally
    // to the server or not.
    // XXX: this should probably have a single source of truth from OlmAccount
    this.backupInfo = null; // The info dict from /room_keys/version
    this.backupKey = null; // The encryption key object
    this._checkedForBackup = false; // Have we checked the server for a backup we can use?
    this._sendingBackups = false; // Are we currently sending backups?

    this._olmDevice = new OlmDevice(cryptoStore);
    this._deviceList = new DeviceList(baseApis, cryptoStore, this._olmDevice);

    // the last time we did a check for the number of one-time-keys on the
    // server.
    this._lastOneTimeKeyCheck = null;
    this._oneTimeKeyCheckInProgress = false;

    // EncryptionAlgorithm instance for each room
    this._roomEncryptors = {};

    // map from algorithm to DecryptionAlgorithm instance, for each room
    this._roomDecryptors = {};

    this._supportedAlgorithms = utils.keys(algorithms.DECRYPTION_CLASSES);

    this._deviceKeys = {};

    this._globalBlacklistUnverifiedDevices = false;

    this._outgoingRoomKeyRequestManager = new _OutgoingRoomKeyRequestManager2.default(baseApis, this._deviceId, this._cryptoStore);

    // list of IncomingRoomKeyRequests/IncomingRoomKeyRequestCancellations
    // we received in the current sync.
    this._receivedRoomKeyRequests = [];
    this._receivedRoomKeyRequestCancellations = [];
    // true if we are currently processing received room key requests
    this._processingRoomKeyRequests = false;
    // controls whether device tracking is delayed
    // until calling encryptEvent or trackRoomDevices,
    // or done immediately upon enabling room encryption.
    this._lazyLoadMembers = false;
    // in case _lazyLoadMembers is true,
    // track if an initial tracking of all the room members
    // has happened for a given room. This is delayed
    // to avoid loading room members as long as possible.
    this._roomDeviceTrackingState = {};

    // The timestamp of the last time we forced establishment
    // of a new session for each device, in milliseconds.
    // {
    //     userId: {
    //         deviceId: 1234567890000,
    //     },
    // }
    this._lastNewSessionForced = {};

    this._verificationTransactions = new _map2.default();
}
utils.inherits(Crypto, _events.EventEmitter);

/**
 * Initialise the crypto module so that it is ready for use
 *
 * Returns a promise which resolves once the crypto module is ready for use.
 */
Crypto.prototype.init = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee() {
    var myDevices, deviceInfo;
    return _regenerator2.default.wrap(function _callee$(_context) {
        while (1) {
            switch (_context.prev = _context.next) {
                case 0:
                    _logger2.default.log("Crypto: initialising Olm...");
                    _context.next = 3;
                    return (0, _bluebird.resolve)(global.Olm.init());

                case 3:
                    _logger2.default.log("Crypto: initialising Olm device...");
                    _context.next = 6;
                    return (0, _bluebird.resolve)(this._olmDevice.init());

                case 6:
                    _logger2.default.log("Crypto: loading device list...");
                    _context.next = 9;
                    return (0, _bluebird.resolve)(this._deviceList.load());

                case 9:

                    // build our device keys: these will later be uploaded
                    this._deviceKeys["ed25519:" + this._deviceId] = this._olmDevice.deviceEd25519Key;
                    this._deviceKeys["curve25519:" + this._deviceId] = this._olmDevice.deviceCurve25519Key;

                    _logger2.default.log("Crypto: fetching own devices...");
                    myDevices = this._deviceList.getRawStoredDevicesForUser(this._userId);


                    if (!myDevices) {
                        myDevices = {};
                    }

                    if (!myDevices[this._deviceId]) {
                        // add our own deviceinfo to the cryptoStore
                        _logger2.default.log("Crypto: adding this device to the store...");
                        deviceInfo = {
                            keys: this._deviceKeys,
                            algorithms: this._supportedAlgorithms,
                            verified: DeviceVerification.VERIFIED,
                            known: true
                        };


                        myDevices[this._deviceId] = deviceInfo;
                        this._deviceList.storeDevicesForUser(this._userId, myDevices);
                        this._deviceList.saveIfDirty();
                    }

                    _logger2.default.log("Crypto: checking for key backup...");
                    this._checkAndStartKeyBackup();

                case 17:
                case 'end':
                    return _context.stop();
            }
        }
    }, _callee, this);
}));

/**
 * Check the server for an active key backup and
 * if one is present and has a valid signature from
 * one of the user's verified devices, start backing up
 * to it.
 */
Crypto.prototype._checkAndStartKeyBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee2() {
    var backupInfo, trustInfo;
    return _regenerator2.default.wrap(function _callee2$(_context2) {
        while (1) {
            switch (_context2.prev = _context2.next) {
                case 0:
                    _logger2.default.log("Checking key backup status...");

                    if (!this._baseApis.isGuest()) {
                        _context2.next = 5;
                        break;
                    }

                    _logger2.default.log("Skipping key backup check since user is guest");
                    this._checkedForBackup = true;
                    return _context2.abrupt('return', null);

                case 5:
                    backupInfo = void 0;
                    _context2.prev = 6;
                    _context2.next = 9;
                    return (0, _bluebird.resolve)(this._baseApis.getKeyBackupVersion());

                case 9:
                    backupInfo = _context2.sent;
                    _context2.next = 17;
                    break;

                case 12:
                    _context2.prev = 12;
                    _context2.t0 = _context2['catch'](6);

                    _logger2.default.log("Error checking for active key backup", _context2.t0);
                    if (_context2.t0.httpStatus / 100 === 4) {
                        // well that's told us. we won't try again.
                        this._checkedForBackup = true;
                    }
                    return _context2.abrupt('return', null);

                case 17:
                    this._checkedForBackup = true;

                    _context2.next = 20;
                    return (0, _bluebird.resolve)(this.isKeyBackupTrusted(backupInfo));

                case 20:
                    trustInfo = _context2.sent;


                    if (trustInfo.usable && !this.backupInfo) {
                        _logger2.default.log("Found usable key backup v" + backupInfo.version + ": enabling key backups");
                        this._baseApis.enableKeyBackup(backupInfo);
                    } else if (!trustInfo.usable && this.backupInfo) {
                        _logger2.default.log("No usable key backup: disabling key backup");
                        this._baseApis.disableKeyBackup();
                    } else if (!trustInfo.usable && !this.backupInfo) {
                        _logger2.default.log("No usable key backup: not enabling key backup");
                    } else if (trustInfo.usable && this.backupInfo) {
                        // may not be the same version: if not, we should switch
                        if (backupInfo.version !== this.backupInfo.version) {
                            _logger2.default.log("On backup version " + this.backupInfo.version + " but found " + "version " + backupInfo.version + ": switching.");
                            this._baseApis.disableKeyBackup();
                            this._baseApis.enableKeyBackup(backupInfo);
                        } else {
                            _logger2.default.log("Backup version " + backupInfo.version + " still current");
                        }
                    }

                    return _context2.abrupt('return', { backupInfo: backupInfo, trustInfo: trustInfo });

                case 23:
                case 'end':
                    return _context2.stop();
            }
        }
    }, _callee2, this, [[6, 12]]);
}));

Crypto.prototype.setTrustedBackupPubKey = function () {
    var _ref3 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee3(trustedPubKey) {
        return _regenerator2.default.wrap(function _callee3$(_context3) {
            while (1) {
                switch (_context3.prev = _context3.next) {
                    case 0:
                        // This should be redundant post cross-signing is a thing, so just
                        // plonk it in localStorage for now.
                        this._sessionStore.setLocalTrustedBackupPubKey(trustedPubKey);
                        _context3.next = 3;
                        return (0, _bluebird.resolve)(this.checkKeyBackup());

                    case 3:
                    case 'end':
                        return _context3.stop();
                }
            }
        }, _callee3, this);
    }));

    return function (_x) {
        return _ref3.apply(this, arguments);
    };
}();

/**
 * Forces a re-check of the key backup and enables/disables it
 * as appropriate.
 *
 * @return {Object} Object with backup info (as returned by
 *     getKeyBackupVersion) in backupInfo and
 *     trust information (as returned by isKeyBackupTrusted)
 *     in trustInfo.
 */
Crypto.prototype.checkKeyBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee4() {
    var returnInfo;
    return _regenerator2.default.wrap(function _callee4$(_context4) {
        while (1) {
            switch (_context4.prev = _context4.next) {
                case 0:
                    this._checkedForBackup = false;
                    _context4.next = 3;
                    return (0, _bluebird.resolve)(this._checkAndStartKeyBackup());

                case 3:
                    returnInfo = _context4.sent;
                    return _context4.abrupt('return', returnInfo);

                case 5:
                case 'end':
                    return _context4.stop();
            }
        }
    }, _callee4, this);
}));

/**
 * @param {object} backupInfo key backup info dict from /room_keys/version
 * @return {object} {
 *     usable: [bool], // is the backup trusted, true iff there is a sig that is valid & from a trusted device
 *     sigs: [
 *         valid: [bool || null], // true: valid, false: invalid, null: cannot attempt validation
 *         deviceId: [string],
 *         device: [DeviceInfo || null],
 *     ]
 * }
 */
Crypto.prototype.isKeyBackupTrusted = function () {
    var _ref5 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee5(backupInfo) {
        var ret, trustedPubkey, mySigs, _iteratorNormalCompletion2, _didIteratorError2, _iteratorError2, _iterator2, _step2, keyId, keyIdParts, sigInfo, device;

        return _regenerator2.default.wrap(function _callee5$(_context5) {
            while (1) {
                switch (_context5.prev = _context5.next) {
                    case 0:
                        ret = {
                            usable: false,
                            trusted_locally: false,
                            sigs: []
                        };

                        if (!(!backupInfo || !backupInfo.algorithm || !backupInfo.auth_data || !backupInfo.auth_data.public_key || !backupInfo.auth_data.signatures)) {
                            _context5.next = 4;
                            break;
                        }

                        _logger2.default.info("Key backup is absent or missing required data");
                        return _context5.abrupt('return', ret);

                    case 4:
                        trustedPubkey = this._sessionStore.getLocalTrustedBackupPubKey();


                        if (backupInfo.auth_data.public_key === trustedPubkey) {
                            _logger2.default.info("Backup public key " + trustedPubkey + " is trusted locally");
                            ret.trusted_locally = true;
                        }

                        mySigs = backupInfo.auth_data.signatures[this._userId] || [];
                        _iteratorNormalCompletion2 = true;
                        _didIteratorError2 = false;
                        _iteratorError2 = undefined;
                        _context5.prev = 10;
                        _iterator2 = (0, _getIterator3.default)((0, _keys2.default)(mySigs));

                    case 12:
                        if (_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done) {
                            _context5.next = 40;
                            break;
                        }

                        keyId = _step2.value;
                        keyIdParts = keyId.split(':');

                        if (!(keyIdParts[0] !== 'ed25519')) {
                            _context5.next = 18;
                            break;
                        }

                        _logger2.default.log("Ignoring unknown signature type: " + keyIdParts[0]);
                        return _context5.abrupt('continue', 37);

                    case 18:
                        sigInfo = { deviceId: keyIdParts[1] }; // XXX: is this how we're supposed to get the device ID?

                        device = this._deviceList.getStoredDevice(this._userId, sigInfo.deviceId);

                        if (!device) {
                            _context5.next = 34;
                            break;
                        }

                        sigInfo.device = device;
                        _context5.prev = 22;
                        _context5.next = 25;
                        return (0, _bluebird.resolve)(olmlib.verifySignature(this._olmDevice,
                        // verifySignature modifies the object so we need to copy
                        // if we verify more than one sig
                        (0, _assign2.default)({}, backupInfo.auth_data), this._userId, device.deviceId, device.getFingerprint()));

                    case 25:
                        sigInfo.valid = true;
                        _context5.next = 32;
                        break;

                    case 28:
                        _context5.prev = 28;
                        _context5.t0 = _context5['catch'](22);

                        _logger2.default.info("Bad signature from key ID " + keyId + " userID " + this._userId + " device ID " + device.deviceId + " fingerprint: " + device.getFingerprint(), backupInfo.auth_data, _context5.t0);
                        sigInfo.valid = false;

                    case 32:
                        _context5.next = 36;
                        break;

                    case 34:
                        sigInfo.valid = null; // Can't determine validity because we don't have the signing device
                        _logger2.default.info("Ignoring signature from unknown key " + keyId);

                    case 36:
                        ret.sigs.push(sigInfo);

                    case 37:
                        _iteratorNormalCompletion2 = true;
                        _context5.next = 12;
                        break;

                    case 40:
                        _context5.next = 46;
                        break;

                    case 42:
                        _context5.prev = 42;
                        _context5.t1 = _context5['catch'](10);
                        _didIteratorError2 = true;
                        _iteratorError2 = _context5.t1;

                    case 46:
                        _context5.prev = 46;
                        _context5.prev = 47;

                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                            _iterator2.return();
                        }

                    case 49:
                        _context5.prev = 49;

                        if (!_didIteratorError2) {
                            _context5.next = 52;
                            break;
                        }

                        throw _iteratorError2;

                    case 52:
                        return _context5.finish(49);

                    case 53:
                        return _context5.finish(46);

                    case 54:

                        ret.usable = ret.sigs.some(function (s) {
                            return s.valid && s.device.isVerified();
                        }) || ret.trusted_locally;
                        return _context5.abrupt('return', ret);

                    case 56:
                    case 'end':
                        return _context5.stop();
                }
            }
        }, _callee5, this, [[10, 42, 46, 54], [22, 28], [47,, 49, 53]]);
    }));

    return function (_x2) {
        return _ref5.apply(this, arguments);
    };
}();

/**
 */
Crypto.prototype.enableLazyLoading = function () {
    this._lazyLoadMembers = true;
};

/**
 * Tell the crypto module to register for MatrixClient events which it needs to
 * listen for
 *
 * @param {external:EventEmitter} eventEmitter event source where we can register
 *    for event notifications
 */
Crypto.prototype.registerEventHandlers = function (eventEmitter) {
    var crypto = this;

    eventEmitter.on("RoomMember.membership", function (event, member, oldMembership) {
        try {
            crypto._onRoomMembership(event, member, oldMembership);
        } catch (e) {
            _logger2.default.error("Error handling membership change:", e);
        }
    });

    eventEmitter.on("toDeviceEvent", function (event) {
        crypto._onToDeviceEvent(event);
    });
};

/** Start background processes related to crypto */
Crypto.prototype.start = function () {
    this._outgoingRoomKeyRequestManager.start();
};

/** Stop background processes related to crypto */
Crypto.prototype.stop = function () {
    this._outgoingRoomKeyRequestManager.stop();
    this._deviceList.stop();
};

/**
 * @return {string} The version of Olm.
 */
Crypto.getOlmVersion = function () {
    return OlmDevice.getOlmVersion();
};

/**
 * Get the Ed25519 key for this device
 *
 * @return {string} base64-encoded ed25519 key.
 */
Crypto.prototype.getDeviceEd25519Key = function () {
    return this._olmDevice.deviceEd25519Key;
};

/**
 * Set the global override for whether the client should ever send encrypted
 * messages to unverified devices.  This provides the default for rooms which
 * do not specify a value.
 *
 * @param {boolean} value whether to blacklist all unverified devices by default
 */
Crypto.prototype.setGlobalBlacklistUnverifiedDevices = function (value) {
    this._globalBlacklistUnverifiedDevices = value;
};

/**
 * @return {boolean} whether to blacklist all unverified devices by default
 */
Crypto.prototype.getGlobalBlacklistUnverifiedDevices = function () {
    return this._globalBlacklistUnverifiedDevices;
};

/**
 * Upload the device keys to the homeserver.
 * @return {object} A promise that will resolve when the keys are uploaded.
 */
Crypto.prototype.uploadDeviceKeys = function () {
    var crypto = this;
    var userId = crypto._userId;
    var deviceId = crypto._deviceId;

    var deviceKeys = {
        algorithms: crypto._supportedAlgorithms,
        device_id: deviceId,
        keys: crypto._deviceKeys,
        user_id: userId
    };

    return crypto._signObject(deviceKeys).then(function () {
        crypto._baseApis.uploadKeysRequest({
            device_keys: deviceKeys
        }, {
            // for now, we set the device id explicitly, as we may not be using the
            // same one as used in login.
            device_id: deviceId
        });
    });
};

/**
 * Stores the current one_time_key count which will be handled later (in a call of
 * onSyncCompleted). The count is e.g. coming from a /sync response.
 *
 * @param {Number} currentCount The current count of one_time_keys to be stored
 */
Crypto.prototype.updateOneTimeKeyCount = function (currentCount) {
    if (isFinite(currentCount)) {
        this._oneTimeKeyCount = currentCount;
    } else {
        throw new TypeError("Parameter for updateOneTimeKeyCount has to be a number");
    }
};

// check if it's time to upload one-time keys, and do so if so.
function _maybeUploadOneTimeKeys(crypto) {
    // frequency with which to check & upload one-time keys
    var uploadPeriod = 1000 * 60; // one minute

    // max number of keys to upload at once
    // Creating keys can be an expensive operation so we limit the
    // number we generate in one go to avoid blocking the application
    // for too long.
    var maxKeysPerCycle = 5;

    if (crypto._oneTimeKeyCheckInProgress) {
        return;
    }

    var now = Date.now();
    if (crypto._lastOneTimeKeyCheck !== null && now - crypto._lastOneTimeKeyCheck < uploadPeriod) {
        // we've done a key upload recently.
        return;
    }

    crypto._lastOneTimeKeyCheck = now;

    // We need to keep a pool of one time public keys on the server so that
    // other devices can start conversations with us. But we can only store
    // a finite number of private keys in the olm Account object.
    // To complicate things further then can be a delay between a device
    // claiming a public one time key from the server and it sending us a
    // message. We need to keep the corresponding private key locally until
    // we receive the message.
    // But that message might never arrive leaving us stuck with duff
    // private keys clogging up our local storage.
    // So we need some kind of enginering compromise to balance all of
    // these factors.

    // Check how many keys we can store in the Account object.
    var maxOneTimeKeys = crypto._olmDevice.maxNumberOfOneTimeKeys();
    // Try to keep at most half that number on the server. This leaves the
    // rest of the slots free to hold keys that have been claimed from the
    // server but we haven't recevied a message for.
    // If we run out of slots when generating new keys then olm will
    // discard the oldest private keys first. This will eventually clean
    // out stale private keys that won't receive a message.
    var keyLimit = Math.floor(maxOneTimeKeys / 2);

    function uploadLoop(keyCount) {
        if (keyLimit <= keyCount) {
            // If we don't need to generate any more keys then we are done.
            return _bluebird2.default.resolve();
        }

        var keysThisLoop = Math.min(keyLimit - keyCount, maxKeysPerCycle);

        // Ask olm to generate new one time keys, then upload them to synapse.
        return crypto._olmDevice.generateOneTimeKeys(keysThisLoop).then(function () {
            return _uploadOneTimeKeys(crypto);
        }).then(function (res) {
            if (res.one_time_key_counts && res.one_time_key_counts.signed_curve25519) {
                // if the response contains a more up to date value use this
                // for the next loop
                return uploadLoop(res.one_time_key_counts.signed_curve25519);
            } else {
                throw new Error("response for uploading keys does not contain " + "one_time_key_counts.signed_curve25519");
            }
        });
    }

    crypto._oneTimeKeyCheckInProgress = true;
    _bluebird2.default.resolve().then(function () {
        if (crypto._oneTimeKeyCount !== undefined) {
            // We already have the current one_time_key count from a /sync response.
            // Use this value instead of asking the server for the current key count.
            return _bluebird2.default.resolve(crypto._oneTimeKeyCount);
        }
        // ask the server how many keys we have
        return crypto._baseApis.uploadKeysRequest({}, {
            device_id: crypto._deviceId
        }).then(function (res) {
            return res.one_time_key_counts.signed_curve25519 || 0;
        });
    }).then(function (keyCount) {
        // Start the uploadLoop with the current keyCount. The function checks if
        // we need to upload new keys or not.
        // If there are too many keys on the server then we don't need to
        // create any more keys.
        return uploadLoop(keyCount);
    }).catch(function (e) {
        _logger2.default.error("Error uploading one-time keys", e.stack || e);
    }).finally(function () {
        // reset _oneTimeKeyCount to prevent start uploading based on old data.
        // it will be set again on the next /sync-response
        crypto._oneTimeKeyCount = undefined;
        crypto._oneTimeKeyCheckInProgress = false;
    }).done();
}Crypto.prototype.downloadKeys = function (userIds, forceDownload) {
    return this._deviceList.downloadKeys(userIds, forceDownload);
};

/**
 * Get the stored device keys for a user id
 *
 * @param {string} userId the user to list keys for.
 *
 * @return {module:crypto/deviceinfo[]|null} list of devices, or null if we haven't
 * managed to get a list of devices for this user yet.
 */
Crypto.prototype.getStoredDevicesForUser = function (userId) {
    return this._deviceList.getStoredDevicesForUser(userId);
};

/**
 * Get the stored keys for a single device
 *
 * @param {string} userId
 * @param {string} deviceId
 *
 * @return {module:crypto/deviceinfo?} device, or undefined
 * if we don't know about this device
 */
Crypto.prototype.getStoredDevice = function (userId, deviceId) {
    return this._deviceList.getStoredDevice(userId, deviceId);
};

/**
 * Save the device list, if necessary
 *
 * @param {integer} delay Time in ms before which the save actually happens.
 *     By default, the save is delayed for a short period in order to batch
 *     multiple writes, but this behaviour can be disabled by passing 0.
 *
 * @return {Promise<bool>} true if the data was saved, false if
 *     it was not (eg. because no changes were pending). The promise
 *     will only resolve once the data is saved, so may take some time
 *     to resolve.
 */
Crypto.prototype.saveDeviceList = function (delay) {
    return this._deviceList.saveIfDirty(delay);
};

/**
 * Update the blocked/verified state of the given device
 *
 * @param {string} userId owner of the device
 * @param {string} deviceId unique identifier for the device
 *
 * @param {?boolean} verified whether to mark the device as verified. Null to
 *     leave unchanged.
 *
 * @param {?boolean} blocked whether to mark the device as blocked. Null to
 *      leave unchanged.
 *
 * @param {?boolean} known whether to mark that the user has been made aware of
 *      the existence of this device. Null to leave unchanged
 *
 * @return {Promise<module:crypto/deviceinfo>} updated DeviceInfo
 */
Crypto.prototype.setDeviceVerification = function () {
    var _ref7 = (0, _bluebird.method)(function (userId, deviceId, verified, blocked, known) {
        var devices = this._deviceList.getRawStoredDevicesForUser(userId);
        if (!devices || !devices[deviceId]) {
            throw new Error("Unknown device " + userId + ":" + deviceId);
        }

        var dev = devices[deviceId];
        var verificationStatus = dev.verified;

        if (verified) {
            verificationStatus = DeviceVerification.VERIFIED;
        } else if (verified !== null && verificationStatus == DeviceVerification.VERIFIED) {
            verificationStatus = DeviceVerification.UNVERIFIED;
        }

        if (blocked) {
            verificationStatus = DeviceVerification.BLOCKED;
        } else if (blocked !== null && verificationStatus == DeviceVerification.BLOCKED) {
            verificationStatus = DeviceVerification.UNVERIFIED;
        }

        var knownStatus = dev.known;
        if (known !== null && known !== undefined) {
            knownStatus = known;
        }

        if (dev.verified !== verificationStatus || dev.known !== knownStatus) {
            dev.verified = verificationStatus;
            dev.known = knownStatus;
            this._deviceList.storeDevicesForUser(userId, devices);
            this._deviceList.saveIfDirty();
        }
        return DeviceInfo.fromStorage(dev, deviceId);
    });

    return function (_x4, _x5, _x6, _x7, _x8) {
        return _ref7.apply(this, arguments);
    };
}();

Crypto.prototype.requestVerification = function (userId, methods, devices) {
    var _this = this;

    if (!methods) {
        // .keys() returns an iterator, so we need to explicitly turn it into an array
        methods = [].concat((0, _toConsumableArray3.default)(this._verificationMethods.keys()));
    }
    if (!devices) {
        devices = (0, _keys2.default)(this._deviceList.getRawStoredDevicesForUser(userId));
    }
    if (!this._verificationTransactions.has(userId)) {
        this._verificationTransactions.set(userId, new _map2.default());
    }

    var transactionId = (0, _randomstring.randomString)(32);

    var promise = new _bluebird2.default(function (resolve, reject) {
        _this._verificationTransactions.get(userId).set(transactionId, {
            request: {
                methods: methods,
                devices: devices,
                resolve: resolve,
                reject: reject
            }
        });
    });

    var message = {
        transaction_id: transactionId,
        from_device: this._baseApis.deviceId,
        methods: methods,
        timestamp: Date.now()
    };
    var msgMap = {};
    var _iteratorNormalCompletion3 = true;
    var _didIteratorError3 = false;
    var _iteratorError3 = undefined;

    try {
        for (var _iterator3 = (0, _getIterator3.default)(devices), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
            var deviceId = _step3.value;

            msgMap[deviceId] = message;
        }
    } catch (err) {
        _didIteratorError3 = true;
        _iteratorError3 = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion3 && _iterator3.return) {
                _iterator3.return();
            }
        } finally {
            if (_didIteratorError3) {
                throw _iteratorError3;
            }
        }
    }

    this._baseApis.sendToDevice("m.key.verification.request", (0, _defineProperty3.default)({}, userId, msgMap));

    return promise;
};

Crypto.prototype.beginKeyVerification = function (method, userId, deviceId, transactionId) {
    if (!this._verificationTransactions.has(userId)) {
        this._verificationTransactions.set(userId, new _map2.default());
    }
    transactionId = transactionId || (0, _randomstring.randomString)(32);
    if (method instanceof Array) {
        if (method.length !== 2 || !this._verificationMethods.has(method[0]) || !this._verificationMethods.has(method[1])) {
            throw (0, _Error.newUnknownMethodError)();
        }
        /*
        return new TwoPartVerification(
            this._verificationMethods[method[0]],
            this._verificationMethods[method[1]],
            userId, deviceId, transactionId,
        );
        */
    } else if (this._verificationMethods.has(method)) {
        var verifier = new (this._verificationMethods.get(method))(this._baseApis, userId, deviceId, transactionId);
        if (!this._verificationTransactions.get(userId).has(transactionId)) {
            this._verificationTransactions.get(userId).set(transactionId, {});
        }
        this._verificationTransactions.get(userId).get(transactionId).verifier = verifier;
        return verifier;
    } else {
        throw (0, _Error.newUnknownMethodError)();
    }
};

/**
 * Get information on the active olm sessions with a user
 * <p>
 * Returns a map from device id to an object with keys 'deviceIdKey' (the
 * device's curve25519 identity key) and 'sessions' (an array of objects in the
 * same format as that returned by
 * {@link module:crypto/OlmDevice#getSessionInfoForDevice}).
 * <p>
 * This method is provided for debugging purposes.
 *
 * @param {string} userId id of user to inspect
 *
 * @return {Promise<Object.<string, {deviceIdKey: string, sessions: object[]}>>}
 */
Crypto.prototype.getOlmSessionsForUser = function () {
    var _ref8 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee7(userId) {
        var devices, result, j, device, deviceKey, sessions;
        return _regenerator2.default.wrap(function _callee7$(_context7) {
            while (1) {
                switch (_context7.prev = _context7.next) {
                    case 0:
                        devices = this.getStoredDevicesForUser(userId) || [];
                        result = {};
                        j = 0;

                    case 3:
                        if (!(j < devices.length)) {
                            _context7.next = 13;
                            break;
                        }

                        device = devices[j];
                        deviceKey = device.getIdentityKey();
                        _context7.next = 8;
                        return (0, _bluebird.resolve)(this._olmDevice.getSessionInfoForDevice(deviceKey));

                    case 8:
                        sessions = _context7.sent;


                        result[device.deviceId] = {
                            deviceIdKey: deviceKey,
                            sessions: sessions
                        };

                    case 10:
                        ++j;
                        _context7.next = 3;
                        break;

                    case 13:
                        return _context7.abrupt('return', result);

                    case 14:
                    case 'end':
                        return _context7.stop();
                }
            }
        }, _callee7, this);
    }));

    return function (_x9) {
        return _ref8.apply(this, arguments);
    };
}();

/**
 * Get the device which sent an event
 *
 * @param {module:models/event.MatrixEvent} event event to be checked
 *
 * @return {module:crypto/deviceinfo?}
 */
Crypto.prototype.getEventSenderDeviceInfo = function (event) {
    var senderKey = event.getSenderKey();
    var algorithm = event.getWireContent().algorithm;

    if (!senderKey || !algorithm) {
        return null;
    }

    var forwardingChain = event.getForwardingCurve25519KeyChain();
    if (forwardingChain.length > 0) {
        // we got this event from somewhere else
        // TODO: check if we can trust the forwarders.
        return null;
    }

    // senderKey is the Curve25519 identity key of the device which the event
    // was sent from. In the case of Megolm, it's actually the Curve25519
    // identity key of the device which set up the Megolm session.

    var device = this._deviceList.getDeviceByIdentityKey(algorithm, senderKey);

    if (device === null) {
        // we haven't downloaded the details of this device yet.
        return null;
    }

    // so far so good, but now we need to check that the sender of this event
    // hadn't advertised someone else's Curve25519 key as their own. We do that
    // by checking the Ed25519 claimed by the event (or, in the case of megolm,
    // the event which set up the megolm session), to check that it matches the
    // fingerprint of the purported sending device.
    //
    // (see https://github.com/vector-im/vector-web/issues/2215)

    var claimedKey = event.getClaimedEd25519Key();
    if (!claimedKey) {
        _logger2.default.warn("Event " + event.getId() + " claims no ed25519 key: " + "cannot verify sending device");
        return null;
    }

    if (claimedKey !== device.getFingerprint()) {
        _logger2.default.warn("Event " + event.getId() + " claims ed25519 key " + claimedKey + "but sender device has key " + device.getFingerprint());
        return null;
    }

    return device;
};

/**
 * Forces the current outbound group session to be discarded such
 * that another one will be created next time an event is sent.
 *
 * @param {string} roomId The ID of the room to discard the session for
 *
 * This should not normally be necessary.
 */
Crypto.prototype.forceDiscardSession = function (roomId) {
    var alg = this._roomEncryptors[roomId];
    if (alg === undefined) throw new Error("Room not encrypted");
    if (alg.forceDiscardSession === undefined) {
        throw new Error("Room encryption algorithm doesn't support session discarding");
    }
    alg.forceDiscardSession();
};

/**
 * Configure a room to use encryption (ie, save a flag in the cryptoStore).
 *
 * @param {string} roomId The room ID to enable encryption in.
 *
 * @param {object} config The encryption config for the room.
 *
 * @param {boolean=} inhibitDeviceQuery true to suppress device list query for
 *   users in the room (for now). In case lazy loading is enabled,
 *   the device query is always inhibited as the members are not tracked.
 */
Crypto.prototype.setRoomEncryption = function () {
    var _ref9 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee8(roomId, config, inhibitDeviceQuery) {
        var existingConfig, existingAlg, storeConfigPromise, AlgClass, alg;
        return _regenerator2.default.wrap(function _callee8$(_context8) {
            while (1) {
                switch (_context8.prev = _context8.next) {
                    case 0:
                        // if state is being replayed from storage, we might already have a configuration
                        // for this room as they are persisted as well.
                        // We just need to make sure the algorithm is initialized in this case.
                        // However, if the new config is different,
                        // we should bail out as room encryption can't be changed once set.
                        existingConfig = this._roomList.getRoomEncryption(roomId);

                        if (!existingConfig) {
                            _context8.next = 5;
                            break;
                        }

                        if (!((0, _stringify2.default)(existingConfig) != (0, _stringify2.default)(config))) {
                            _context8.next = 5;
                            break;
                        }

                        _logger2.default.error("Ignoring m.room.encryption event which requests " + "a change of config in " + roomId);
                        return _context8.abrupt('return');

                    case 5:
                        // if we already have encryption in this room, we should ignore this event,
                        // as it would reset the encryption algorithm.
                        // This is at least expected to be called twice, as sync calls onCryptoEvent
                        // for both the timeline and state sections in the /sync response,
                        // the encryption event would appear in both.
                        // If it's called more than twice though,
                        // it signals a bug on client or server.
                        existingAlg = this._roomEncryptors[roomId];

                        if (!existingAlg) {
                            _context8.next = 8;
                            break;
                        }

                        return _context8.abrupt('return');

                    case 8:

                        // _roomList.getRoomEncryption will not race with _roomList.setRoomEncryption
                        // because it first stores in memory. We should await the promise only
                        // after all the in-memory state (_roomEncryptors and _roomList) has been updated
                        // to avoid races when calling this method multiple times. Hence keep a hold of the promise.
                        storeConfigPromise = null;

                        if (!existingConfig) {
                            storeConfigPromise = this._roomList.setRoomEncryption(roomId, config);
                        }

                        AlgClass = algorithms.ENCRYPTION_CLASSES[config.algorithm];

                        if (AlgClass) {
                            _context8.next = 13;
                            break;
                        }

                        throw new Error("Unable to encrypt with " + config.algorithm);

                    case 13:
                        alg = new AlgClass({
                            userId: this._userId,
                            deviceId: this._deviceId,
                            crypto: this,
                            olmDevice: this._olmDevice,
                            baseApis: this._baseApis,
                            roomId: roomId,
                            config: config
                        });

                        this._roomEncryptors[roomId] = alg;

                        if (!storeConfigPromise) {
                            _context8.next = 18;
                            break;
                        }

                        _context8.next = 18;
                        return (0, _bluebird.resolve)(storeConfigPromise);

                    case 18:
                        if (this._lazyLoadMembers) {
                            _context8.next = 25;
                            break;
                        }

                        _logger2.default.log("Enabling encryption in " + roomId + "; " + "starting to track device lists for all users therein");

                        _context8.next = 22;
                        return (0, _bluebird.resolve)(this.trackRoomDevices(roomId));

                    case 22:
                        // TODO: this flag is only not used from MatrixClient::setRoomEncryption
                        // which is never used (inside riot at least)
                        // but didn't want to remove it as it technically would
                        // be a breaking change.
                        if (!this.inhibitDeviceQuery) {
                            this._deviceList.refreshOutdatedDeviceLists();
                        }
                        _context8.next = 26;
                        break;

                    case 25:
                        _logger2.default.log("Enabling encryption in " + roomId);

                    case 26:
                    case 'end':
                        return _context8.stop();
                }
            }
        }, _callee8, this);
    }));

    return function (_x10, _x11, _x12) {
        return _ref9.apply(this, arguments);
    };
}();

/**
 * Make sure we are tracking the device lists for all users in this room.
 *
 * @param {string} roomId The room ID to start tracking devices in.
 * @returns {Promise} when all devices for the room have been fetched and marked to track
 */
Crypto.prototype.trackRoomDevices = function (roomId) {
    var _this2 = this;

    var trackMembers = function () {
        var _ref10 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee9() {
            var room, members;
            return _regenerator2.default.wrap(function _callee9$(_context9) {
                while (1) {
                    switch (_context9.prev = _context9.next) {
                        case 0:
                            if (_this2._roomEncryptors[roomId]) {
                                _context9.next = 2;
                                break;
                            }

                            return _context9.abrupt('return');

                        case 2:
                            room = _this2._clientStore.getRoom(roomId);

                            if (room) {
                                _context9.next = 5;
                                break;
                            }

                            throw new Error('Unable to start tracking devices in unknown room ' + roomId);

                        case 5:
                            _logger2.default.log('Starting to track devices for room ' + roomId + ' ...');
                            _context9.next = 8;
                            return (0, _bluebird.resolve)(room.getEncryptionTargetMembers());

                        case 8:
                            members = _context9.sent;

                            members.forEach(function (m) {
                                _this2._deviceList.startTrackingDeviceList(m.userId);
                            });

                        case 10:
                        case 'end':
                            return _context9.stop();
                    }
                }
            }, _callee9, _this2);
        }));

        return function trackMembers() {
            return _ref10.apply(this, arguments);
        };
    }();

    var promise = this._roomDeviceTrackingState[roomId];
    if (!promise) {
        promise = trackMembers();
        this._roomDeviceTrackingState[roomId] = promise;
    }
    return promise;
};

/**
 * @typedef {Object} module:crypto~OlmSessionResult
 * @property {module:crypto/deviceinfo} device  device info
 * @property {string?} sessionId base64 olm session id; null if no session
 *    could be established
 */

/**
 * Try to make sure we have established olm sessions for all known devices for
 * the given users.
 *
 * @param {string[]} users list of user ids
 *
 * @return {module:client.Promise} resolves once the sessions are complete, to
 *    an Object mapping from userId to deviceId to
 *    {@link module:crypto~OlmSessionResult}
 */
Crypto.prototype.ensureOlmSessionsForUsers = function (users) {
    var devicesByUser = {};

    for (var i = 0; i < users.length; ++i) {
        var userId = users[i];
        devicesByUser[userId] = [];

        var devices = this.getStoredDevicesForUser(userId) || [];
        for (var j = 0; j < devices.length; ++j) {
            var deviceInfo = devices[j];

            var key = deviceInfo.getIdentityKey();
            if (key == this._olmDevice.deviceCurve25519Key) {
                // don't bother setting up session to ourself
                continue;
            }
            if (deviceInfo.verified == DeviceVerification.BLOCKED) {
                // don't bother setting up sessions with blocked users
                continue;
            }

            devicesByUser[userId].push(deviceInfo);
        }
    }

    return olmlib.ensureOlmSessionsForDevices(this._olmDevice, this._baseApis, devicesByUser);
};

/**
 * Get a list containing all of the room keys
 *
 * @return {module:crypto/OlmDevice.MegolmSessionData[]} a list of session export objects
 */
Crypto.prototype.exportRoomKeys = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee10() {
    var _this3 = this;

    var exportedSessions;
    return _regenerator2.default.wrap(function _callee10$(_context10) {
        while (1) {
            switch (_context10.prev = _context10.next) {
                case 0:
                    exportedSessions = [];
                    _context10.next = 3;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                        _this3._cryptoStore.getAllEndToEndInboundGroupSessions(txn, function (s) {
                            if (s === null) return;

                            var sess = _this3._olmDevice.exportInboundGroupSession(s.senderKey, s.sessionId, s.sessionData);
                            delete sess.first_known_index;
                            sess.algorithm = olmlib.MEGOLM_ALGORITHM;
                            exportedSessions.push(sess);
                        });
                    }));

                case 3:
                    return _context10.abrupt('return', exportedSessions);

                case 4:
                case 'end':
                    return _context10.stop();
            }
        }
    }, _callee10, this);
}));

/**
 * Import a list of room keys previously exported by exportRoomKeys
 *
 * @param {Object[]} keys a list of session export objects
 * @return {module:client.Promise} a promise which resolves once the keys have been imported
 */
Crypto.prototype.importRoomKeys = function (keys) {
    var _this4 = this;

    return _bluebird2.default.map(keys, function (key) {
        if (!key.room_id || !key.algorithm) {
            _logger2.default.warn("ignoring room key entry with missing fields", key);
            return null;
        }

        var alg = _this4._getRoomDecryptor(key.room_id, key.algorithm);
        return alg.importRoomKey(key);
    });
};

/**
 * Schedules sending all keys waiting to be sent to the backup, if not already
 * scheduled. Retries if necessary.
 *
 * @param {number} maxDelay Maximum delay to wait in ms. 0 means no delay.
 */
Crypto.prototype.scheduleKeyBackupSend = function () {
    var _ref12 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee11() {
        var maxDelay = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 10000;
        var delay, numFailures, numBackedUp;
        return _regenerator2.default.wrap(function _callee11$(_context11) {
            while (1) {
                switch (_context11.prev = _context11.next) {
                    case 0:
                        if (!this._sendingBackups) {
                            _context11.next = 2;
                            break;
                        }

                        return _context11.abrupt('return');

                    case 2:

                        this._sendingBackups = true;

                        _context11.prev = 3;

                        // wait between 0 and `maxDelay` seconds, to avoid backup
                        // requests from different clients hitting the server all at
                        // the same time when a new key is sent
                        delay = Math.random() * maxDelay;
                        _context11.next = 7;
                        return (0, _bluebird.resolve)(_bluebird2.default.delay(delay));

                    case 7:
                        numFailures = 0; // number of consecutive failures

                    case 8:
                        if (!1) {
                            _context11.next = 35;
                            break;
                        }

                        if (this.backupKey) {
                            _context11.next = 11;
                            break;
                        }

                        return _context11.abrupt('return');

                    case 11:
                        _context11.prev = 11;
                        _context11.next = 14;
                        return (0, _bluebird.resolve)(this._backupPendingKeys(KEY_BACKUP_KEYS_PER_REQUEST));

                    case 14:
                        numBackedUp = _context11.sent;

                        if (!(numBackedUp === 0)) {
                            _context11.next = 17;
                            break;
                        }

                        return _context11.abrupt('return');

                    case 17:
                        numFailures = 0;
                        _context11.next = 30;
                        break;

                    case 20:
                        _context11.prev = 20;
                        _context11.t0 = _context11['catch'](11);

                        numFailures++;
                        _logger2.default.log("Key backup request failed", _context11.t0);

                        if (!_context11.t0.data) {
                            _context11.next = 30;
                            break;
                        }

                        if (!(_context11.t0.data.errcode == 'M_NOT_FOUND' || _context11.t0.data.errcode == 'M_WRONG_ROOM_KEYS_VERSION')) {
                            _context11.next = 30;
                            break;
                        }

                        _context11.next = 28;
                        return (0, _bluebird.resolve)(this.checkKeyBackup());

                    case 28:
                        // Backup version has changed or this backup version
                        // has been deleted
                        this.emit("crypto.keyBackupFailed", _context11.t0.data.errcode);
                        throw _context11.t0;

                    case 30:
                        if (!numFailures) {
                            _context11.next = 33;
                            break;
                        }

                        _context11.next = 33;
                        return (0, _bluebird.resolve)(_bluebird2.default.delay(1000 * Math.pow(2, Math.min(numFailures - 1, 4))));

                    case 33:
                        _context11.next = 8;
                        break;

                    case 35:
                        _context11.prev = 35;

                        this._sendingBackups = false;
                        return _context11.finish(35);

                    case 38:
                    case 'end':
                        return _context11.stop();
                }
            }
        }, _callee11, this, [[3,, 35, 38], [11, 20]]);
    }));

    return function (_x13) {
        return _ref12.apply(this, arguments);
    };
}();

/**
 * Take some e2e keys waiting to be backed up and send them
 * to the backup.
 *
 * @param {integer} limit Maximum number of keys to back up
 * @returns {integer} Number of sessions backed up
 */
Crypto.prototype._backupPendingKeys = function () {
    var _ref13 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee12(limit) {
        var sessions, remaining, data, _iteratorNormalCompletion4, _didIteratorError4, _iteratorError4, _iterator4, _step4, session, roomId, sessionData, firstKnownIndex, encrypted, forwardedCount, device;

        return _regenerator2.default.wrap(function _callee12$(_context12) {
            while (1) {
                switch (_context12.prev = _context12.next) {
                    case 0:
                        _context12.next = 2;
                        return (0, _bluebird.resolve)(this._cryptoStore.getSessionsNeedingBackup(limit));

                    case 2:
                        sessions = _context12.sent;

                        if (sessions.length) {
                            _context12.next = 5;
                            break;
                        }

                        return _context12.abrupt('return', 0);

                    case 5:
                        _context12.next = 7;
                        return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                    case 7:
                        remaining = _context12.sent;

                        this.emit("crypto.keyBackupSessionsRemaining", remaining);

                        data = {};
                        _iteratorNormalCompletion4 = true;
                        _didIteratorError4 = false;
                        _iteratorError4 = undefined;
                        _context12.prev = 13;
                        _iterator4 = (0, _getIterator3.default)(sessions);

                    case 15:
                        if (_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done) {
                            _context12.next = 34;
                            break;
                        }

                        session = _step4.value;
                        roomId = session.sessionData.room_id;

                        if (data[roomId] === undefined) {
                            data[roomId] = { sessions: {} };
                        }

                        _context12.next = 21;
                        return (0, _bluebird.resolve)(this._olmDevice.exportInboundGroupSession(session.senderKey, session.sessionId, session.sessionData));

                    case 21:
                        sessionData = _context12.sent;

                        sessionData.algorithm = olmlib.MEGOLM_ALGORITHM;
                        delete sessionData.session_id;
                        delete sessionData.room_id;
                        firstKnownIndex = sessionData.first_known_index;

                        delete sessionData.first_known_index;
                        encrypted = this.backupKey.encrypt((0, _stringify2.default)(sessionData));
                        forwardedCount = (sessionData.forwarding_curve25519_key_chain || []).length;
                        device = this._deviceList.getDeviceByIdentityKey(olmlib.MEGOLM_ALGORITHM, session.senderKey);


                        data[roomId]['sessions'][session.sessionId] = {
                            first_message_index: firstKnownIndex,
                            forwarded_count: forwardedCount,
                            is_verified: !!(device && device.isVerified()),
                            session_data: encrypted
                        };

                    case 31:
                        _iteratorNormalCompletion4 = true;
                        _context12.next = 15;
                        break;

                    case 34:
                        _context12.next = 40;
                        break;

                    case 36:
                        _context12.prev = 36;
                        _context12.t0 = _context12['catch'](13);
                        _didIteratorError4 = true;
                        _iteratorError4 = _context12.t0;

                    case 40:
                        _context12.prev = 40;
                        _context12.prev = 41;

                        if (!_iteratorNormalCompletion4 && _iterator4.return) {
                            _iterator4.return();
                        }

                    case 43:
                        _context12.prev = 43;

                        if (!_didIteratorError4) {
                            _context12.next = 46;
                            break;
                        }

                        throw _iteratorError4;

                    case 46:
                        return _context12.finish(43);

                    case 47:
                        return _context12.finish(40);

                    case 48:
                        _context12.next = 50;
                        return (0, _bluebird.resolve)(this._baseApis.sendKeyBackup(undefined, undefined, this.backupInfo.version, { rooms: data }));

                    case 50:
                        _context12.next = 52;
                        return (0, _bluebird.resolve)(this._cryptoStore.unmarkSessionsNeedingBackup(sessions));

                    case 52:
                        _context12.next = 54;
                        return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                    case 54:
                        remaining = _context12.sent;

                        this.emit("crypto.keyBackupSessionsRemaining", remaining);

                        return _context12.abrupt('return', sessions.length);

                    case 57:
                    case 'end':
                        return _context12.stop();
                }
            }
        }, _callee12, this, [[13, 36, 40, 48], [41,, 43, 47]]);
    }));

    return function (_x15) {
        return _ref13.apply(this, arguments);
    };
}();

Crypto.prototype.backupGroupSession = function () {
    var _ref14 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee13(roomId, senderKey, forwardingCurve25519KeyChain, sessionId, sessionKey, keysClaimed, exportFormat) {
        return _regenerator2.default.wrap(function _callee13$(_context13) {
            while (1) {
                switch (_context13.prev = _context13.next) {
                    case 0:
                        if (this.backupInfo) {
                            _context13.next = 2;
                            break;
                        }

                        throw new Error("Key backups are not enabled");

                    case 2:
                        _context13.next = 4;
                        return (0, _bluebird.resolve)(this._cryptoStore.markSessionsNeedingBackup([{
                            senderKey: senderKey,
                            sessionId: sessionId
                        }]));

                    case 4:

                        // don't wait for this to complete: it will delay so
                        // happens in the background
                        this.scheduleKeyBackupSend();

                    case 5:
                    case 'end':
                        return _context13.stop();
                }
            }
        }, _callee13, this);
    }));

    return function (_x16, _x17, _x18, _x19, _x20, _x21, _x22) {
        return _ref14.apply(this, arguments);
    };
}();

/**
 * Marks all group sessions as needing to be backed up and schedules them to
 * upload in the background as soon as possible.
 */
Crypto.prototype.scheduleAllGroupSessionsForBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee14() {
    return _regenerator2.default.wrap(function _callee14$(_context14) {
        while (1) {
            switch (_context14.prev = _context14.next) {
                case 0:
                    _context14.next = 2;
                    return (0, _bluebird.resolve)(this.flagAllGroupSessionsForBackup());

                case 2:

                    // Schedule keys to upload in the background as soon as possible.
                    this.scheduleKeyBackupSend(0 /* maxDelay */);

                case 3:
                case 'end':
                    return _context14.stop();
            }
        }
    }, _callee14, this);
}));

/**
 * Marks all group sessions as needing to be backed up without scheduling
 * them to upload in the background.
 * @returns {Promise<int>} Resolves to the number of sessions requiring a backup.
 */
Crypto.prototype.flagAllGroupSessionsForBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee15() {
    var _this5 = this;

    var remaining;
    return _regenerator2.default.wrap(function _callee15$(_context15) {
        while (1) {
            switch (_context15.prev = _context15.next) {
                case 0:
                    _context15.next = 2;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS, _indexeddbCryptoStore2.default.STORE_BACKUP], function (txn) {
                        _this5._cryptoStore.getAllEndToEndInboundGroupSessions(txn, function (session) {
                            if (session !== null) {
                                _this5._cryptoStore.markSessionsNeedingBackup([session], txn);
                            }
                        });
                    }));

                case 2:
                    _context15.next = 4;
                    return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                case 4:
                    remaining = _context15.sent;

                    this.emit("crypto.keyBackupSessionsRemaining", remaining);
                    return _context15.abrupt('return', remaining);

                case 7:
                case 'end':
                    return _context15.stop();
            }
        }
    }, _callee15, this);
}));

/* eslint-disable valid-jsdoc */ //https://github.com/eslint/eslint/issues/7307
/**
 * Encrypt an event according to the configuration of the room.
 *
 * @param {module:models/event.MatrixEvent} event  event to be sent
 *
 * @param {module:models/room} room destination room.
 *
 * @return {module:client.Promise?} Promise which resolves when the event has been
 *     encrypted, or null if nothing was needed
 */
/* eslint-enable valid-jsdoc */
Crypto.prototype.encryptEvent = function () {
    var _ref17 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee16(event, room) {
        var roomId, alg, content, mRelatesTo, encryptedContent;
        return _regenerator2.default.wrap(function _callee16$(_context16) {
            while (1) {
                switch (_context16.prev = _context16.next) {
                    case 0:
                        if (room) {
                            _context16.next = 2;
                            break;
                        }

                        throw new Error("Cannot send encrypted messages in unknown rooms");

                    case 2:
                        roomId = event.getRoomId();
                        alg = this._roomEncryptors[roomId];

                        if (alg) {
                            _context16.next = 6;
                            break;
                        }

                        throw new Error("Room was previously configured to use encryption, but is " + "no longer. Perhaps the homeserver is hiding the " + "configuration event.");

                    case 6:

                        if (!this._roomDeviceTrackingState[roomId]) {
                            this.trackRoomDevices(roomId);
                        }
                        // wait for all the room devices to be loaded
                        _context16.next = 9;
                        return (0, _bluebird.resolve)(this._roomDeviceTrackingState[roomId]);

                    case 9:
                        content = event.getContent();
                        // If event has an m.relates_to then we need
                        // to put this on the wrapping event instead

                        mRelatesTo = content['m.relates_to'];

                        if (mRelatesTo) {
                            // Clone content here so we don't remove `m.relates_to` from the local-echo
                            content = (0, _assign2.default)({}, content);
                            delete content['m.relates_to'];
                        }

                        _context16.next = 14;
                        return (0, _bluebird.resolve)(alg.encryptMessage(room, event.getType(), content));

                    case 14:
                        encryptedContent = _context16.sent;


                        if (mRelatesTo) {
                            encryptedContent['m.relates_to'] = mRelatesTo;
                        }

                        event.makeEncrypted("m.room.encrypted", encryptedContent, this._olmDevice.deviceCurve25519Key, this._olmDevice.deviceEd25519Key);

                    case 17:
                    case 'end':
                        return _context16.stop();
                }
            }
        }, _callee16, this);
    }));

    return function (_x23, _x24) {
        return _ref17.apply(this, arguments);
    };
}();

/**
 * Decrypt a received event
 *
 * @param {MatrixEvent} event
 *
 * @return {Promise<module:crypto~EventDecryptionResult>} resolves once we have
 *  finished decrypting. Rejects with an `algorithms.DecryptionError` if there
 *  is a problem decrypting the event.
 */
Crypto.prototype.decryptEvent = function (event) {
    if (event.isRedacted()) {
        return _bluebird2.default.resolve({
            clearEvent: {
                room_id: event.getRoomId(),
                type: "m.room.message",
                content: {}
            }
        });
    }
    var content = event.getWireContent();
    var alg = this._getRoomDecryptor(event.getRoomId(), content.algorithm);
    return alg.decryptEvent(event);
};

/**
 * Handle the notification from /sync or /keys/changes that device lists have
 * been changed.
 *
 * @param {Object} syncData Object containing sync tokens associated with this sync
 * @param {Object} syncDeviceLists device_lists field from /sync, or response from
 * /keys/changes
 */
Crypto.prototype.handleDeviceListChanges = function () {
    var _ref18 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee17(syncData, syncDeviceLists) {
        return _regenerator2.default.wrap(function _callee17$(_context17) {
            while (1) {
                switch (_context17.prev = _context17.next) {
                    case 0:
                        if (syncData.oldSyncToken) {
                            _context17.next = 2;
                            break;
                        }

                        return _context17.abrupt('return');

                    case 2:
                        _context17.next = 4;
                        return (0, _bluebird.resolve)(this._evalDeviceListChanges(syncDeviceLists));

                    case 4:
                    case 'end':
                        return _context17.stop();
                }
            }
        }, _callee17, this);
    }));

    return function (_x25, _x26) {
        return _ref18.apply(this, arguments);
    };
}();

/**
 * Send a request for some room keys, if we have not already done so
 *
 * @param {module:crypto~RoomKeyRequestBody} requestBody
 * @param {Array<{userId: string, deviceId: string}>} recipients
 * @param {boolean} resend whether to resend the key request if there is
 *    already one
 *
 * @return {Promise} a promise that resolves when the key request is queued
 */
Crypto.prototype.requestRoomKey = function (requestBody, recipients) {
    var resend = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

    return this._outgoingRoomKeyRequestManager.sendRoomKeyRequest(requestBody, recipients, resend).catch(function (e) {
        // this normally means we couldn't talk to the store
        _logger2.default.error('Error requesting key for event', e);
    }).done();
};

/**
 * Cancel any earlier room key request
 *
 * @param {module:crypto~RoomKeyRequestBody} requestBody
 *    parameters to match for cancellation
 */
Crypto.prototype.cancelRoomKeyRequest = function (requestBody) {
    this._outgoingRoomKeyRequestManager.cancelRoomKeyRequest(requestBody).catch(function (e) {
        _logger2.default.warn("Error clearing pending room key requests", e);
    }).done();
};

/**
 * handle an m.room.encryption event
 *
 * @param {module:models/event.MatrixEvent} event encryption event
 */
Crypto.prototype.onCryptoEvent = function () {
    var _ref19 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee18(event) {
        var roomId, content;
        return _regenerator2.default.wrap(function _callee18$(_context18) {
            while (1) {
                switch (_context18.prev = _context18.next) {
                    case 0:
                        roomId = event.getRoomId();
                        content = event.getContent();
                        _context18.prev = 2;
                        _context18.next = 5;
                        return (0, _bluebird.resolve)(this.setRoomEncryption(roomId, content, true));

                    case 5:
                        _context18.next = 10;
                        break;

                    case 7:
                        _context18.prev = 7;
                        _context18.t0 = _context18['catch'](2);

                        _logger2.default.error("Error configuring encryption in room " + roomId + ":", _context18.t0);

                    case 10:
                    case 'end':
                        return _context18.stop();
                }
            }
        }, _callee18, this, [[2, 7]]);
    }));

    return function (_x28) {
        return _ref19.apply(this, arguments);
    };
}();

/**
 * Called before the result of a sync is procesed
 *
 * @param {Object} syncData  the data from the 'MatrixClient.sync' event
 */
Crypto.prototype.onSyncWillProcess = function () {
    var _ref20 = (0, _bluebird.method)(function (syncData) {
        if (!syncData.oldSyncToken) {
            // If there is no old sync token, we start all our tracking from
            // scratch, so mark everything as untracked. onCryptoEvent will
            // be called for all e2e rooms during the processing of the sync,
            // at which point we'll start tracking all the users of that room.
            _logger2.default.log("Initial sync performed - resetting device tracking state");
            this._deviceList.stopTrackingAllDeviceLists();
            this._roomDeviceTrackingState = {};
        }
    });

    return function (_x29) {
        return _ref20.apply(this, arguments);
    };
}();

/**
 * handle the completion of a /sync
 *
 * This is called after the processing of each successful /sync response.
 * It is an opportunity to do a batch process on the information received.
 *
 * @param {Object} syncData  the data from the 'MatrixClient.sync' event
 */
Crypto.prototype.onSyncCompleted = function () {
    var _ref21 = (0, _bluebird.method)(function (syncData) {
        var nextSyncToken = syncData.nextSyncToken;

        this._deviceList.setSyncToken(syncData.nextSyncToken);
        this._deviceList.saveIfDirty();

        // catch up on any new devices we got told about during the sync.
        this._deviceList.lastKnownSyncToken = nextSyncToken;

        // we always track our own device list (for key backups etc)
        this._deviceList.startTrackingDeviceList(this._userId);

        this._deviceList.refreshOutdatedDeviceLists();

        // we don't start uploading one-time keys until we've caught up with
        // to-device messages, to help us avoid throwing away one-time-keys that we
        // are about to receive messages for
        // (https://github.com/vector-im/riot-web/issues/2782).
        if (!syncData.catchingUp) {
            _maybeUploadOneTimeKeys(this);
            this._processReceivedRoomKeyRequests();
        }
    });

    return function (_x30) {
        return _ref21.apply(this, arguments);
    };
}();

/**
 * Trigger the appropriate invalidations and removes for a given
 * device list
 *
 * @param {Object} deviceLists device_lists field from /sync, or response from
 * /keys/changes
 */
Crypto.prototype._evalDeviceListChanges = function () {
    var _ref22 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee19(deviceLists) {
        var _this6 = this;

        var e2eUserIds;
        return _regenerator2.default.wrap(function _callee19$(_context19) {
            while (1) {
                switch (_context19.prev = _context19.next) {
                    case 0:
                        if (deviceLists.changed && Array.isArray(deviceLists.changed)) {
                            deviceLists.changed.forEach(function (u) {
                                _this6._deviceList.invalidateUserDeviceList(u);
                            });
                        }

                        if (!(deviceLists.left && Array.isArray(deviceLists.left) && deviceLists.left.length)) {
                            _context19.next = 8;
                            break;
                        }

                        _context19.t0 = _set2.default;
                        _context19.next = 5;
                        return (0, _bluebird.resolve)(this._getTrackedE2eUsers());

                    case 5:
                        _context19.t1 = _context19.sent;
                        e2eUserIds = new _context19.t0(_context19.t1);


                        deviceLists.left.forEach(function (u) {
                            if (!e2eUserIds.has(u)) {
                                _this6._deviceList.stopTrackingDeviceList(u);
                            }
                        });

                    case 8:
                    case 'end':
                        return _context19.stop();
                }
            }
        }, _callee19, this);
    }));

    return function (_x31) {
        return _ref22.apply(this, arguments);
    };
}();

/**
 * Get a list of all the IDs of users we share an e2e room with
 * for which we are tracking devices already
 *
 * @returns {string[]} List of user IDs
 */
Crypto.prototype._getTrackedE2eUsers = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee20() {
    var e2eUserIds, _iteratorNormalCompletion5, _didIteratorError5, _iteratorError5, _iterator5, _step5, room, members, _iteratorNormalCompletion6, _didIteratorError6, _iteratorError6, _iterator6, _step6, member;

    return _regenerator2.default.wrap(function _callee20$(_context20) {
        while (1) {
            switch (_context20.prev = _context20.next) {
                case 0:
                    e2eUserIds = [];
                    _iteratorNormalCompletion5 = true;
                    _didIteratorError5 = false;
                    _iteratorError5 = undefined;
                    _context20.prev = 4;
                    _iterator5 = (0, _getIterator3.default)(this._getTrackedE2eRooms());

                case 6:
                    if (_iteratorNormalCompletion5 = (_step5 = _iterator5.next()).done) {
                        _context20.next = 33;
                        break;
                    }

                    room = _step5.value;
                    _context20.next = 10;
                    return (0, _bluebird.resolve)(room.getEncryptionTargetMembers());

                case 10:
                    members = _context20.sent;
                    _iteratorNormalCompletion6 = true;
                    _didIteratorError6 = false;
                    _iteratorError6 = undefined;
                    _context20.prev = 14;

                    for (_iterator6 = (0, _getIterator3.default)(members); !(_iteratorNormalCompletion6 = (_step6 = _iterator6.next()).done); _iteratorNormalCompletion6 = true) {
                        member = _step6.value;

                        e2eUserIds.push(member.userId);
                    }
                    _context20.next = 22;
                    break;

                case 18:
                    _context20.prev = 18;
                    _context20.t0 = _context20['catch'](14);
                    _didIteratorError6 = true;
                    _iteratorError6 = _context20.t0;

                case 22:
                    _context20.prev = 22;
                    _context20.prev = 23;

                    if (!_iteratorNormalCompletion6 && _iterator6.return) {
                        _iterator6.return();
                    }

                case 25:
                    _context20.prev = 25;

                    if (!_didIteratorError6) {
                        _context20.next = 28;
                        break;
                    }

                    throw _iteratorError6;

                case 28:
                    return _context20.finish(25);

                case 29:
                    return _context20.finish(22);

                case 30:
                    _iteratorNormalCompletion5 = true;
                    _context20.next = 6;
                    break;

                case 33:
                    _context20.next = 39;
                    break;

                case 35:
                    _context20.prev = 35;
                    _context20.t1 = _context20['catch'](4);
                    _didIteratorError5 = true;
                    _iteratorError5 = _context20.t1;

                case 39:
                    _context20.prev = 39;
                    _context20.prev = 40;

                    if (!_iteratorNormalCompletion5 && _iterator5.return) {
                        _iterator5.return();
                    }

                case 42:
                    _context20.prev = 42;

                    if (!_didIteratorError5) {
                        _context20.next = 45;
                        break;
                    }

                    throw _iteratorError5;

                case 45:
                    return _context20.finish(42);

                case 46:
                    return _context20.finish(39);

                case 47:
                    return _context20.abrupt('return', e2eUserIds);

                case 48:
                case 'end':
                    return _context20.stop();
            }
        }
    }, _callee20, this, [[4, 35, 39, 47], [14, 18, 22, 30], [23,, 25, 29], [40,, 42, 46]]);
}));

/**
 * Get a list of the e2e-enabled rooms we are members of,
 * and for which we are already tracking the devices
 *
 * @returns {module:models.Room[]}
 */
Crypto.prototype._getTrackedE2eRooms = function () {
    var _this7 = this;

    return this._clientStore.getRooms().filter(function (room) {
        // check for rooms with encryption enabled
        var alg = _this7._roomEncryptors[room.roomId];
        if (!alg) {
            return false;
        }
        if (!_this7._roomDeviceTrackingState[room.roomId]) {
            return false;
        }

        // ignore any rooms which we have left
        var myMembership = room.getMyMembership();
        return myMembership === "join" || myMembership === "invite";
    });
};

Crypto.prototype._onToDeviceEvent = function (event) {
    var _this8 = this;

    try {
        if (event.getType() == "m.room_key" || event.getType() == "m.forwarded_room_key") {
            this._onRoomKeyEvent(event);
        } else if (event.getType() == "m.room_key_request") {
            this._onRoomKeyRequestEvent(event);
        } else if (event.getType() === "m.key.verification.request") {
            this._onKeyVerificationRequest(event);
        } else if (event.getType() === "m.key.verification.start") {
            this._onKeyVerificationStart(event);
        } else if (event.getContent().transaction_id) {
            this._onKeyVerificationMessage(event);
        } else if (event.getContent().msgtype === "m.bad.encrypted") {
            this._onToDeviceBadEncrypted(event);
        } else if (event.isBeingDecrypted()) {
            // once the event has been decrypted, try again
            event.once('Event.decrypted', function (ev) {
                _this8._onToDeviceEvent(ev);
            });
        }
    } catch (e) {
        _logger2.default.error("Error handling toDeviceEvent:", e);
    }
};

/**
 * Handle a key event
 *
 * @private
 * @param {module:models/event.MatrixEvent} event key event
 */
Crypto.prototype._onRoomKeyEvent = function (event) {
    var content = event.getContent();

    if (!content.room_id || !content.algorithm) {
        _logger2.default.error("key event is missing fields");
        return;
    }

    if (!this._checkedForBackup) {
        // don't bother awaiting on this - the important thing is that we retry if we
        // haven't managed to check before
        this._checkAndStartKeyBackup();
    }

    var alg = this._getRoomDecryptor(content.room_id, content.algorithm);
    alg.onRoomKeyEvent(event);
};

/**
 * Handle a key verification request event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification request event
 */
Crypto.prototype._onKeyVerificationRequest = function (event) {
    var _this9 = this;

    if (event.isCancelled()) {
        _logger2.default.warn("Ignoring flagged verification request from " + event.getSender());
        return;
    }

    var content = event.getContent();
    if (!("from_device" in content) || typeof content.from_device !== "string" || !("transaction_id" in content) || typeof content.from_device !== "string" || !("methods" in content) || !(content.methods instanceof Array) || !("timestamp" in content) || typeof content.timestamp !== "number") {
        _logger2.default.warn("received invalid verification request from " + event.getSender());
        // ignore event if malformed
        return;
    }

    var now = Date.now();
    if (now < content.timestamp - 5 * 60 * 1000 || now > content.timestamp + 10 * 60 * 1000) {
        // ignore if event is too far in the past or too far in the future
        _logger2.default.log("received verification that is too old or from the future");
        return;
    }

    var sender = event.getSender();
    if (sender === this._userId && content.from_device === this._deviceId) {
        // ignore requests from ourselves, because it doesn't make sense for a
        // device to verify itself
        return;
    }
    if (this._verificationTransactions.has(sender)) {
        if (this._verificationTransactions.get(sender).has(content.transaction_id)) {
            // transaction already exists: cancel it and drop the existing
            // request because someone has gotten confused
            var err = (0, _Error.newUnexpectedMessageError)({
                transaction_id: content.transaction_id
            });
            if (this._verificationTransactions.get(sender).get(content.transaction_id).verifier) {
                this._verificationTransactions.get(sender).get(content.transaction_id).verifier.cancel(err);
            } else {
                this._verificationTransactions.get(sender).get(content.transaction_id).reject(err);
                this.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, err.getContent())));
            }
            this._verificationTransactions.get(sender).delete(content.transaction_id);
            return;
        }
    } else {
        this._verificationTransactions.set(sender, new _map2.default());
    }

    // determine what requested methods we support
    var methods = [];
    var _iteratorNormalCompletion7 = true;
    var _didIteratorError7 = false;
    var _iteratorError7 = undefined;

    try {
        for (var _iterator7 = (0, _getIterator3.default)(content.methods), _step7; !(_iteratorNormalCompletion7 = (_step7 = _iterator7.next()).done); _iteratorNormalCompletion7 = true) {
            var method = _step7.value;

            if (typeof method !== "string") {
                continue;
            }
            if (this._verificationMethods.has(method)) {
                methods.push(method);
            }
        }
    } catch (err) {
        _didIteratorError7 = true;
        _iteratorError7 = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion7 && _iterator7.return) {
                _iterator7.return();
            }
        } finally {
            if (_didIteratorError7) {
                throw _iteratorError7;
            }
        }
    }

    if (methods.length === 0) {
        this._baseApis.emit("crypto.verification.request.unknown", event.getSender(), function () {
            _this9.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, (0, _Error.newUserCancelledError)({
                transaction_id: content.transaction_id
            }).getContent())));
        });
    } else {
        // notify the application of the verification request, so it can
        // decide what to do with it
        var request = {
            event: event,
            methods: methods,
            beginKeyVerification: function beginKeyVerification(method) {
                var verifier = _this9.beginKeyVerification(method, sender, content.from_device, content.transaction_id);
                _this9._verificationTransactions.get(sender).get(content.transaction_id).verifier = verifier;
                return verifier;
            },
            cancel: function cancel() {
                _this9._baseApis.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, (0, _Error.newUserCancelledError)({
                    transaction_id: content.transaction_id
                }).getContent())));
            }
        };
        this._verificationTransactions.get(sender).set(content.transaction_id, {
            request: request
        });
        this._baseApis.emit("crypto.verification.request", request);
    }
};

/**
 * Handle a key verification start event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification start event
 */
Crypto.prototype._onKeyVerificationStart = function (event) {
    var _this10 = this;

    if (event.isCancelled()) {
        _logger2.default.warn("Ignoring flagged verification start from " + event.getSender());
        return;
    }

    var sender = event.getSender();
    var content = event.getContent();
    var transactionId = content.transaction_id;
    var deviceId = content.from_device;
    if (!transactionId || !deviceId) {
        // invalid request, and we don't have enough information to send a
        // cancellation, so just ignore it
        return;
    }

    var handler = this._verificationTransactions.has(sender) && this._verificationTransactions.get(sender).get(transactionId);
    // if the verification start message is invalid, send a cancel message to
    // the other side, and also send a cancellation event
    var cancel = function cancel(err) {
        if (handler.verifier) {
            handler.verifier.cancel(err);
        } else if (handler.request && handler.request.cancel) {
            handler.request.cancel(err);
        }
        _this10.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, deviceId, err.getContent())));
    };
    if (!this._verificationMethods.has(content.method)) {
        cancel((0, _Error.newUnknownMethodError)({
            transaction_id: content.transactionId
        }));
        return;
    } else if (content.next_method) {
        if (!this._verificationMethods.has(content.next_method)) {
            cancel((0, _Error.newUnknownMethodError)({
                transaction_id: content.transactionId
            }));
            return;
        } else {
            /* TODO:
            const verification = new TwoPartVerification(
                this._verificationMethods[content.method],
                this._verificationMethods[content.next_method],
                userId, deviceId,
            );
            this.emit(verification.event_type, verification);
            this.emit(verification.first.event_type, verification);*/
        }
    } else {
        var verifier = new (this._verificationMethods.get(content.method))(this._baseApis, sender, deviceId, content.transaction_id, event, handler && handler.request);
        if (!handler) {
            if (!this._verificationTransactions.has(sender)) {
                this._verificationTransactions.set(sender, new _map2.default());
            }
            handler = this._verificationTransactions.get(sender).set(transactionId, {
                verifier: verifier
            });
        } else {
            if (!handler.verifier) {
                handler.verifier = verifier;
                if (handler.request) {
                    // the verification start was sent as a response to a
                    // verification request

                    if (!handler.request.devices.includes(deviceId)) {
                        // didn't send a request to that device, so it
                        // shouldn't have responded
                        cancel((0, _Error.newUnexpectedMessageError)({
                            transaction_id: content.transactionId
                        }));
                        return;
                    }
                    if (!handler.request.methods.includes(content.method)) {
                        // verification method wasn't one that was requested
                        cancel((0, _Error.newUnknownMethodError)({
                            transaction_id: content.transactionId
                        }));
                        return;
                    }

                    // send cancellation messages to all the other devices that
                    // the request was sent to
                    var message = {
                        transaction_id: transactionId,
                        code: "m.accepted",
                        reason: "Verification request accepted by another device"
                    };
                    var msgMap = {};
                    var _iteratorNormalCompletion8 = true;
                    var _didIteratorError8 = false;
                    var _iteratorError8 = undefined;

                    try {
                        for (var _iterator8 = (0, _getIterator3.default)(handler.request.devices), _step8; !(_iteratorNormalCompletion8 = (_step8 = _iterator8.next()).done); _iteratorNormalCompletion8 = true) {
                            var devId = _step8.value;

                            if (devId !== deviceId) {
                                msgMap[devId] = message;
                            }
                        }
                    } catch (err) {
                        _didIteratorError8 = true;
                        _iteratorError8 = err;
                    } finally {
                        try {
                            if (!_iteratorNormalCompletion8 && _iterator8.return) {
                                _iterator8.return();
                            }
                        } finally {
                            if (_didIteratorError8) {
                                throw _iteratorError8;
                            }
                        }
                    }

                    this._baseApis.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, msgMap));

                    handler.request.resolve(verifier);
                }
            } else {
                // FIXME: make sure we're in a two-part verification, and the start matches the second part
            }
        }
        this._baseApis.emit("crypto.verification.start", verifier);
    }
};

/**
 * Handle a general key verification event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification start event
 */
Crypto.prototype._onKeyVerificationMessage = function (event) {
    var sender = event.getSender();
    var transactionId = event.getContent().transaction_id;
    var handler = this._verificationTransactions.has(sender) && this._verificationTransactions.get(sender).get(transactionId);
    if (!handler) {
        return;
    } else if (event.getType() === "m.key.verification.cancel") {
        _logger2.default.log(event);
        if (handler.verifier) {
            handler.verifier.cancel(event);
        } else if (handler.request && handler.request.cancel) {
            handler.request.cancel(event);
        }
    } else if (handler.verifier) {
        var verifier = handler.verifier;
        if (verifier.events && verifier.events.includes(event.getType())) {
            verifier.handleEvent(event);
        }
    }
};

/**
 * Handle a toDevice event that couldn't be decrypted
 *
 * @private
 * @param {module:models/event.MatrixEvent} event undecryptable event
 */
Crypto.prototype._onToDeviceBadEncrypted = function () {
    var _ref24 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee21(event) {
        var content, sender, algorithm, deviceKey, lastNewSessionForced, device, devicesByUser, encryptedContent, requestsToResend, _iteratorNormalCompletion9, _didIteratorError9, _iteratorError9, _iterator9, _step9, keyReq;

        return _regenerator2.default.wrap(function _callee21$(_context21) {
            while (1) {
                switch (_context21.prev = _context21.next) {
                    case 0:
                        content = event.getWireContent();
                        sender = event.getSender();
                        algorithm = content.algorithm;
                        deviceKey = content.sender_key;

                        if (!(sender === undefined || deviceKey === undefined || deviceKey === undefined)) {
                            _context21.next = 6;
                            break;
                        }

                        return _context21.abrupt('return');

                    case 6:

                        // check when we last forced a new session with this device: if we've already done so
                        // recently, don't do it again.
                        this._lastNewSessionForced[sender] = this._lastNewSessionForced[sender] || {};
                        lastNewSessionForced = this._lastNewSessionForced[sender][deviceKey] || 0;

                        if (!(lastNewSessionForced + MIN_FORCE_SESSION_INTERVAL_MS > Date.now())) {
                            _context21.next = 11;
                            break;
                        }

                        _logger2.default.debug("New session already forced with device " + sender + ":" + deviceKey + " at " + lastNewSessionForced + ": not forcing another");
                        return _context21.abrupt('return');

                    case 11:

                        // establish a new olm session with this device since we're failing to decrypt messages
                        // on a current session.
                        // Note that an undecryptable message from another device could easily be spoofed -
                        // is there anything we can do to mitigate this?
                        device = this._deviceList.getDeviceByIdentityKey(algorithm, deviceKey);

                        if (device) {
                            _context21.next = 15;
                            break;
                        }

                        _logger2.default.info("Couldn't find device for identity key " + deviceKey + ": not re-establishing session");
                        return _context21.abrupt('return');

                    case 15:
                        devicesByUser = {};

                        devicesByUser[sender] = [device];
                        _context21.next = 19;
                        return (0, _bluebird.resolve)(olmlib.ensureOlmSessionsForDevices(this._olmDevice, this._baseApis, devicesByUser, true));

                    case 19:

                        this._lastNewSessionForced[sender][deviceKey] = Date.now();

                        // Now send a blank message on that session so the other side knows about it.
                        // (The keyshare request is sent in the clear so that won't do)
                        // We send this first such that, as long as the toDevice messages arrive in the
                        // same order we sent them, the other end will get this first, set up the new session,
                        // then get the keyshare request and send the key over this new session (because it
                        // is the session it has most recently received a message on).
                        encryptedContent = {
                            algorithm: olmlib.OLM_ALGORITHM,
                            sender_key: this._olmDevice.deviceCurve25519Key,
                            ciphertext: {}
                        };
                        _context21.next = 23;
                        return (0, _bluebird.resolve)(olmlib.encryptMessageForDevice(encryptedContent.ciphertext, this._userId, this._deviceId, this._olmDevice, sender, device, { type: "m.dummy" }));

                    case 23:
                        _context21.next = 25;
                        return (0, _bluebird.resolve)(this._baseApis.sendToDevice("m.room.encrypted", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, device.deviceId, encryptedContent))));

                    case 25:
                        _context21.next = 27;
                        return (0, _bluebird.resolve)(this._outgoingRoomKeyRequestManager.getOutgoingSentRoomKeyRequest(sender, device.deviceId));

                    case 27:
                        requestsToResend = _context21.sent;
                        _iteratorNormalCompletion9 = true;
                        _didIteratorError9 = false;
                        _iteratorError9 = undefined;
                        _context21.prev = 31;

                        for (_iterator9 = (0, _getIterator3.default)(requestsToResend); !(_iteratorNormalCompletion9 = (_step9 = _iterator9.next()).done); _iteratorNormalCompletion9 = true) {
                            keyReq = _step9.value;

                            this.requestRoomKey(keyReq.requestBody, keyReq.recipients, true);
                        }
                        _context21.next = 39;
                        break;

                    case 35:
                        _context21.prev = 35;
                        _context21.t0 = _context21['catch'](31);
                        _didIteratorError9 = true;
                        _iteratorError9 = _context21.t0;

                    case 39:
                        _context21.prev = 39;
                        _context21.prev = 40;

                        if (!_iteratorNormalCompletion9 && _iterator9.return) {
                            _iterator9.return();
                        }

                    case 42:
                        _context21.prev = 42;

                        if (!_didIteratorError9) {
                            _context21.next = 45;
                            break;
                        }

                        throw _iteratorError9;

                    case 45:
                        return _context21.finish(42);

                    case 46:
                        return _context21.finish(39);

                    case 47:
                    case 'end':
                        return _context21.stop();
                }
            }
        }, _callee21, this, [[31, 35, 39, 47], [40,, 42, 46]]);
    }));

    return function (_x32) {
        return _ref24.apply(this, arguments);
    };
}();

/**
 * Handle a change in the membership state of a member of a room
 *
 * @private
 * @param {module:models/event.MatrixEvent} event  event causing the change
 * @param {module:models/room-member} member  user whose membership changed
 * @param {string=} oldMembership  previous membership
 */
Crypto.prototype._onRoomMembership = function (event, member, oldMembership) {
    // this event handler is registered on the *client* (as opposed to the room
    // member itself), which means it is only called on changes to the *live*
    // membership state (ie, it is not called when we back-paginate, nor when
    // we load the state in the initialsync).
    //
    // Further, it is automatically registered and called when new members
    // arrive in the room.

    var roomId = member.roomId;

    var alg = this._roomEncryptors[roomId];
    if (!alg) {
        // not encrypting in this room
        return;
    }
    // only mark users in this room as tracked if we already started tracking in this room
    // this way we don't start device queries after sync on behalf of this room which we won't use
    // the result of anyway, as we'll need to do a query again once all the members are fetched
    // by calling _trackRoomDevices
    if (this._roomDeviceTrackingState[roomId]) {
        if (member.membership == 'join') {
            _logger2.default.log('Join event for ' + member.userId + ' in ' + roomId);
            // make sure we are tracking the deviceList for this user
            this._deviceList.startTrackingDeviceList(member.userId);
        } else if (member.membership == 'invite' && this._clientStore.getRoom(roomId).shouldEncryptForInvitedMembers()) {
            _logger2.default.log('Invite event for ' + member.userId + ' in ' + roomId);
            this._deviceList.startTrackingDeviceList(member.userId);
        }
    }

    alg.onRoomMembership(event, member, oldMembership);
};

/**
 * Called when we get an m.room_key_request event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event key request event
 */
Crypto.prototype._onRoomKeyRequestEvent = function (event) {
    var content = event.getContent();
    if (content.action === "request") {
        // Queue it up for now, because they tend to arrive before the room state
        // events at initial sync, and we want to see if we know anything about the
        // room before passing them on to the app.
        var req = new IncomingRoomKeyRequest(event);
        this._receivedRoomKeyRequests.push(req);
    } else if (content.action === "request_cancellation") {
        var _req = new IncomingRoomKeyRequestCancellation(event);
        this._receivedRoomKeyRequestCancellations.push(_req);
    }
};

/**
 * Process any m.room_key_request events which were queued up during the
 * current sync.
 *
 * @private
 */
Crypto.prototype._processReceivedRoomKeyRequests = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee22() {
    var _this11 = this;

    var requests, cancellations;
    return _regenerator2.default.wrap(function _callee22$(_context22) {
        while (1) {
            switch (_context22.prev = _context22.next) {
                case 0:
                    if (!this._processingRoomKeyRequests) {
                        _context22.next = 2;
                        break;
                    }

                    return _context22.abrupt('return');

                case 2:
                    this._processingRoomKeyRequests = true;

                    _context22.prev = 3;

                    // we need to grab and clear the queues in the synchronous bit of this method,
                    // so that we don't end up racing with the next /sync.
                    requests = this._receivedRoomKeyRequests;

                    this._receivedRoomKeyRequests = [];
                    cancellations = this._receivedRoomKeyRequestCancellations;

                    this._receivedRoomKeyRequestCancellations = [];

                    // Process all of the requests, *then* all of the cancellations.
                    //
                    // This makes sure that if we get a request and its cancellation in the
                    // same /sync result, then we process the request before the
                    // cancellation (and end up with a cancelled request), rather than the
                    // cancellation before the request (and end up with an outstanding
                    // request which should have been cancelled.)
                    _context22.next = 10;
                    return (0, _bluebird.resolve)(_bluebird2.default.map(requests, function (req) {
                        return _this11._processReceivedRoomKeyRequest(req);
                    }));

                case 10:
                    _context22.next = 12;
                    return (0, _bluebird.resolve)(_bluebird2.default.map(cancellations, function (cancellation) {
                        return _this11._processReceivedRoomKeyRequestCancellation(cancellation);
                    }));

                case 12:
                    _context22.next = 17;
                    break;

                case 14:
                    _context22.prev = 14;
                    _context22.t0 = _context22['catch'](3);

                    _logger2.default.error('Error processing room key requsts: ' + _context22.t0);

                case 17:
                    _context22.prev = 17;

                    this._processingRoomKeyRequests = false;
                    return _context22.finish(17);

                case 20:
                case 'end':
                    return _context22.stop();
            }
        }
    }, _callee22, this, [[3, 14, 17, 20]]);
}));

/**
 * Helper for processReceivedRoomKeyRequests
 *
 * @param {IncomingRoomKeyRequest} req
 */
Crypto.prototype._processReceivedRoomKeyRequest = function () {
    var _ref26 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee23(req) {
        var userId, deviceId, body, roomId, alg, encryptor, _device, decryptor, device;

        return _regenerator2.default.wrap(function _callee23$(_context23) {
            while (1) {
                switch (_context23.prev = _context23.next) {
                    case 0:
                        userId = req.userId;
                        deviceId = req.deviceId;
                        body = req.requestBody;
                        roomId = body.room_id;
                        alg = body.algorithm;


                        _logger2.default.log('m.room_key_request from ' + userId + ':' + deviceId + (' for ' + roomId + ' / ' + body.session_id + ' (id ' + req.requestId + ')'));

                        if (!(userId !== this._userId)) {
                            _context23.next = 24;
                            break;
                        }

                        if (this._roomEncryptors[roomId]) {
                            _context23.next = 10;
                            break;
                        }

                        _logger2.default.debug('room key request for unencrypted room ' + roomId);
                        return _context23.abrupt('return');

                    case 10:
                        encryptor = this._roomEncryptors[roomId];
                        _device = this._deviceList.getStoredDevice(userId, deviceId);

                        if (_device) {
                            _context23.next = 15;
                            break;
                        }

                        _logger2.default.debug('Ignoring keyshare for unknown device ' + userId + ':' + deviceId);
                        return _context23.abrupt('return');

                    case 15:
                        _context23.prev = 15;
                        _context23.next = 18;
                        return (0, _bluebird.resolve)(encryptor.reshareKeyWithDevice(body.sender_key, body.session_id, userId, _device));

                    case 18:
                        _context23.next = 23;
                        break;

                    case 20:
                        _context23.prev = 20;
                        _context23.t0 = _context23['catch'](15);

                        _logger2.default.warn("Failed to re-share keys for session " + body.session_id + " with device " + userId + ":" + _device.deviceId, _context23.t0);

                    case 23:
                        return _context23.abrupt('return');

                    case 24:
                        if (this._roomDecryptors[roomId]) {
                            _context23.next = 27;
                            break;
                        }

                        _logger2.default.log('room key request for unencrypted room ' + roomId);
                        return _context23.abrupt('return');

                    case 27:
                        decryptor = this._roomDecryptors[roomId][alg];

                        if (decryptor) {
                            _context23.next = 31;
                            break;
                        }

                        _logger2.default.log('room key request for unknown alg ' + alg + ' in room ' + roomId);
                        return _context23.abrupt('return');

                    case 31:
                        _context23.next = 33;
                        return (0, _bluebird.resolve)(decryptor.hasKeysForKeyRequest(req));

                    case 33:
                        if (_context23.sent) {
                            _context23.next = 36;
                            break;
                        }

                        _logger2.default.log('room key request for unknown session ' + roomId + ' / ' + body.session_id);
                        return _context23.abrupt('return');

                    case 36:

                        req.share = function () {
                            decryptor.shareKeysWithDevice(req);
                        };

                        // if the device is is verified already, share the keys
                        device = this._deviceList.getStoredDevice(userId, deviceId);

                        if (!(device && device.isVerified())) {
                            _context23.next = 42;
                            break;
                        }

                        _logger2.default.log('device is already verified: sharing keys');
                        req.share();
                        return _context23.abrupt('return');

                    case 42:

                        this.emit("crypto.roomKeyRequest", req);

                    case 43:
                    case 'end':
                        return _context23.stop();
                }
            }
        }, _callee23, this, [[15, 20]]);
    }));

    return function (_x33) {
        return _ref26.apply(this, arguments);
    };
}();

/**
 * Helper for processReceivedRoomKeyRequests
 *
 * @param {IncomingRoomKeyRequestCancellation} cancellation
 */
Crypto.prototype._processReceivedRoomKeyRequestCancellation = function () {
    var _ref27 = (0, _bluebird.method)(function (cancellation) {
        _logger2.default.log('m.room_key_request cancellation for ' + cancellation.userId + ':' + (cancellation.deviceId + ' (id ' + cancellation.requestId + ')'));

        // we should probably only notify the app of cancellations we told it
        // about, but we don't currently have a record of that, so we just pass
        // everything through.
        this.emit("crypto.roomKeyRequestCancellation", cancellation);
    });

    return function (_x34) {
        return _ref27.apply(this, arguments);
    };
}();

/**
 * Get a decryptor for a given room and algorithm.
 *
 * If we already have a decryptor for the given room and algorithm, return
 * it. Otherwise try to instantiate it.
 *
 * @private
 *
 * @param {string?} roomId   room id for decryptor. If undefined, a temporary
 * decryptor is instantiated.
 *
 * @param {string} algorithm  crypto algorithm
 *
 * @return {module:crypto.algorithms.base.DecryptionAlgorithm}
 *
 * @raises {module:crypto.algorithms.DecryptionError} if the algorithm is
 * unknown
 */
Crypto.prototype._getRoomDecryptor = function (roomId, algorithm) {
    var decryptors = void 0;
    var alg = void 0;

    roomId = roomId || null;
    if (roomId) {
        decryptors = this._roomDecryptors[roomId];
        if (!decryptors) {
            this._roomDecryptors[roomId] = decryptors = {};
        }

        alg = decryptors[algorithm];
        if (alg) {
            return alg;
        }
    }

    var AlgClass = algorithms.DECRYPTION_CLASSES[algorithm];
    if (!AlgClass) {
        throw new algorithms.DecryptionError('UNKNOWN_ENCRYPTION_ALGORITHM', 'Unknown encryption algorithm "' + algorithm + '".');
    }
    alg = new AlgClass({
        userId: this._userId,
        crypto: this,
        olmDevice: this._olmDevice,
        baseApis: this._baseApis,
        roomId: roomId
    });

    if (decryptors) {
        decryptors[algorithm] = alg;
    }
    return alg;
};

/**
 * sign the given object with our ed25519 key
 *
 * @param {Object} obj  Object to which we will add a 'signatures' property
 */
Crypto.prototype._signObject = function () {
    var _ref28 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee24(obj) {
        var sigs;
        return _regenerator2.default.wrap(function _callee24$(_context24) {
            while (1) {
                switch (_context24.prev = _context24.next) {
                    case 0:
                        sigs = {};

                        sigs[this._userId] = {};
                        _context24.next = 4;
                        return (0, _bluebird.resolve)(this._olmDevice.sign(anotherjson.stringify(obj)));

                    case 4:
                        sigs[this._userId]["ed25519:" + this._deviceId] = _context24.sent;

                        obj.signatures = sigs;

                    case 6:
                    case 'end':
                        return _context24.stop();
                }
            }
        }, _callee24, this);
    }));

    return function (_x35) {
        return _ref28.apply(this, arguments);
    };
}();

/**
 * The parameters of a room key request. The details of the request may
 * vary with the crypto algorithm, but the management and storage layers for
 * outgoing requests expect it to have 'room_id' and 'session_id' properties.
 *
 * @typedef {Object} RoomKeyRequestBody
 */

/**
 * Represents a received m.room_key_request event
 *
 * @property {string} userId    user requesting the key
 * @property {string} deviceId  device requesting the key
 * @property {string} requestId unique id for the request
 * @property {module:crypto~RoomKeyRequestBody} requestBody
 * @property {function()} share  callback which, when called, will ask
 *    the relevant crypto algorithm implementation to share the keys for
 *    this request.
 */

var IncomingRoomKeyRequest = function IncomingRoomKeyRequest(event) {
    (0, _classCallCheck3.default)(this, IncomingRoomKeyRequest);

    var content = event.getContent();

    this.userId = event.getSender();
    this.deviceId = content.requesting_device_id;
    this.requestId = content.request_id;
    this.requestBody = content.body || {};
    this.share = function () {
        throw new Error("don't know how to share keys for this request yet");
    };
};

/**
 * Represents a received m.room_key_request cancellation
 *
 * @property {string} userId    user requesting the cancellation
 * @property {string} deviceId  device requesting the cancellation
 * @property {string} requestId unique id for the request to be cancelled
 */


var IncomingRoomKeyRequestCancellation = function IncomingRoomKeyRequestCancellation(event) {
    (0, _classCallCheck3.default)(this, IncomingRoomKeyRequestCancellation);

    var content = event.getContent();

    this.userId = event.getSender();
    this.deviceId = content.requesting_device_id;
    this.requestId = content.request_id;
};

/**
 * The result of a (successful) call to decryptEvent.
 *
 * @typedef {Object} EventDecryptionResult
 *
 * @property {Object} clearEvent The plaintext payload for the event
 *     (typically containing <tt>type</tt> and <tt>content</tt> fields).
 *
 * @property {?string} senderCurve25519Key Key owned by the sender of this
 *    event.  See {@link module:models/event.MatrixEvent#getSenderKey}.
 *
 * @property {?string} claimedEd25519Key ed25519 key claimed by the sender of
 *    this event. See
 *    {@link module:models/event.MatrixEvent#getClaimedEd25519Key}.
 *
 * @property {?Array<string>} forwardingCurve25519KeyChain list of curve25519
 *     keys involved in telling us about the senderCurve25519Key and
 *     claimedEd25519Key. See
 *     {@link module:models/event.MatrixEvent#getForwardingCurve25519KeyChain}.
 */

/**
 * Fires when we receive a room key request
 *
 * @event module:client~MatrixClient#"crypto.roomKeyRequest"
 * @param {module:crypto~IncomingRoomKeyRequest} req  request details
 */

/**
 * Fires when we receive a room key request cancellation
 *
 * @event module:client~MatrixClient#"crypto.roomKeyRequestCancellation"
 * @param {module:crypto~IncomingRoomKeyRequestCancellation} req
 */

/**
 * Fires when the app may wish to warn the user about something related
 * the end-to-end crypto.
 *
 * @event module:client~MatrixClient#"crypto.warning"
 * @param {string} type One of the strings listed above
 */
//# sourceMappingURL=index.js.map