---
title: "Owl Lisp v0.1.19 manual"
date: 2019-5-13
geometry: "left=3cm,right=3cm,top=2cm,bottom=2cm"
output: pdf_document
---


Owl Lisp is a simple purely functional programming language. 
It is essentially R7RS Scheme, 
   apart from having only immmutable data structures 
   and relying on multithreading for some core operations.
This document describes the language, 
   its implementation and builtin libraries. 

# LISP

LISP (or Lisp) is short for LISt Processor.
It is a mathematical formalism for reasoning about computations 
   invented by John McCarthy at MIT in 1958.
Although the system started out as a purely theoretical construct, 
   it soon ended up being implemented in practice.

One of the main features of Lisp is the syntax, -
   or to be more precise, the lack of it.
As the name suggests, Lisp is indeed good for processing lists.
Symbols are also a primitive data type.
List structures containing symbols and other primitive data types
   are colled S-expressions.
These have a simple textual representation.
A symbol is just the sequence of letters in its name, 
   and a list is just a sequence of other things separated by spaces 
      and delimited by parenthesis.
A key feature of Lisp is that 
   programs themselves are just Lisp data structures.
Therefore a separate syntax is not needed for the programming language - 
   it is just the textual representation of Lisp data.

The initial definition of Lisp in "Lisp 1.5 programmer's manual"
   contained a definition of Lisp itself in itself.
This evaluation function was the only thing that needed to be implemented
   to turn the initial version of Lisp from theory to practice.

There have been many major dialects of Lisp,
   and countless minor ones.
These days compiler construction and parsing are usually taught towards the end of computer science curriculum, 
   if at all. 
Parsers and compilers, 
   the building blocks of programming language implementations, 
   are considered to be dark magic only a few select pupils and devout lone programmers are fit to approach.
This has not always been the case. 
In a few programming language families 
   it has been customary to start, not end, studies by building a small version 
      of the language you are studying. 
This approach favored languages 
   which had a powerful small core of features, 
   on top of which you could build the rest of the system. 
Forth and Lisp are common examples of such languages.

Common Lisp and Scheme are some of the most widely used major Lisp dialects.
Scheme was also developed at MIT.
It took a step towards λ-calculus in its semantics
   by relying on one namespace and lexical scoping.
It was initially also an attempt to explore the actor model of computation, 
   but the actor part was dropped after it turned out to be equivalent 
   to functions in single-threaded context.
While it has good support for functional programming, 
   it also allows one define and use other programming paradigms.
One of the key concepts added in Scheme were continuations, 
   which allow the state of the program to be captured and restored.
This makes it possible define e.g. resumable error handling and 
   co-operative multithreading as library functions. 
There have at the time of writing been 7 revisions of the Scheme language standard.


## Owl Lisp

Owl Lisp is essentially a small step from the multi-paradigm world of Scheme
   into a purely functional one.
Scheme programs are usually mainly functional
   with some mutations sprinkled sparingly where needed.
In Owl, functional programming style is not just encouraged 
   - it is all you have.
Owl also brought back the actor-model -style operation, 
   but this time by using continuation-based threads to make them useufl.
Apart from being based on continuations, 
   the threading is modeled after that in Erlang.
   
The goal has not at any point been to become an ultimate Lisp 
   and take over the world. 
Ïn fact, this has been an anti-goal. 
The goal has been to remain simple 
   while incrementally growing only portable features required 
      to enable building the kinds of programs it is actually used for. 
While this is a somewhat circular definition, 
   it has worked surprisingly well.
Owl is shaped by minimalism and practical applications, 
   not by what seem like cool and powerful language features.




## Installation

To install system-wide to /usr
```
   $ make
   $ sudo make install
```

Alternatively you can try it out with
```
   $ make
   $ cp bin/ol /somewhere/convenient
   $ /somewhere/convenient/ol
   You see a prompt
   >
```


## Files

   bin/ol      - the owl interpreter/compiler
   c/ovm.c     - the virtual machine / shared owl lisp runtime
   owl/*.scm   - implementation of owl repl and compiler
   fasl/*.fasl - bytecode images for bin/vm used during boot
   bin/vm      - plain VM used during boot
   c/ol.c      - combined VM and REPL heap image


## Usage

Owl can be used either interactively, or to interpret code from files,
or compile programs to fasl-images or c-files. The difference between
an owl program and a plain script is that the program should just have
a function of one argument as the last value, which will be called with
the command line argument list when the program is executed.

In addition to being a regular interpreter, owl also tries to make it
easy to compile programs for different platforms. Owl programs can be
compiled with ol to C-files, which can be compiled to standalone binaries
without needing any owl-specific support files or libraries. The C files
also work on 32- and 64-bit systems, and compile as such at least on
Linux, OpenBSD, and macOS or can be cross-compiled to Windows executables
with MinGW.

For example, to build a hello world program:
```
  $ echo '(lambda (args) (print "Hello, world!"))' > hello.scm
  $ ol -o hello.c hello.scm
  $ gcc -o hello hello.c
  $ ./hello
  Hello, world!
```

Or simply:
```
  $ echo '(λ (args) (print "Hello, world!"))' \
     | ol -x c | gcc -x c -o hello - && ./hello
```

Parts of the compiled programs can be translated to C, instead of being
simply fasl-encoded, to increase speed. Using the --native flag compiles
most of the bytecode to C, and --usual-suspects compiles typically used
functions. To make programs run faster, one can use for example:

```
  $ ol -O2 -o test.c test.scm && gcc -O2 -o test test.c
```

# Libraries

Libraries are named by lists of symbols. For example `(owl lazy)` is
a library name. `ol` comes preloaded with many libraries, some of which
are loaded by default to REPL. If you want to use exported definitions from a
builtin library `(owl lazy)`, you can do so by issuing `(import (owl lazy))`.

Notice that `import` is not a function. It is one of the special forms used
by the REPL to handle management of visible definitions in your sessions. The
syntax is the same as in imports within a library.

If you try to import a library which is not currently loaded, say `(my test)`,
Owl would try to look for library definition from the file "my/test.scm". If
it is available and contains definition of a library called `(my test)`, it will
be loaded and added to your REPL.

You can get a listing of the currently loaded libraries with `,libraries` command.
Many of them are mainly needed for the implementation of the underlying system.
Some of the likely useful ones are documented below.


## (owl alpha)
This library is used internally to convert all variables to unique ones to 
avoid having to track variable shadowing.

Exported values:

- `alpha-convert`



## (owl args)
This library makes it easy to write tools which use command line arguments 
in the usual way.

Exported values:

- `process-arguments`
- `format-rules`
- `print-rules`
- `cl-rules`



## (owl assemble)
This library implements bytecode assembly.

Exported values:

- `assemble-code`
- `simple-value?`
- `small-value?`



## (owl ast)
After macro expansion the S-expressions are translated to a tree of tuples 
with checked structure to avoid having to constantly check S-expression structure.

Exported values:

- `call?`
- `var?`
- `value-of`
- `sexp->ast`
- `mkcall`
- `mklambda`
- `mkvarlambda`
- `mkvar`
- `mkval`



## (owl base)
This library defines the default toplevel. 

Exported values:

- `(exports (owl defmac))`
- `(exports (owl list))`
- `(exports (owl rlist))`
- `(exports (owl list-extra))`
- `(exports (owl ff))`
- `(exports (owl io))`
- `(exports (owl lazy))`
- `(exports (owl string))`
- `(exports (owl symbol))`
- `(exports (owl sort))`
- `(exports (owl bytevector))`
- `(exports (owl vector))`
- `(exports (owl equal))`
- `(exports (owl random))`
- `(exports (owl render))`
- `(exports (owl syscall))`
- `(exports (owl bisect))`
- `(exports (owl function))`
- `(exports (owl fasl))`
- `(exports (owl port))`
- `(exports (owl time))`
- `(exports (owl suffix))`
- `(exports (owl regex))`
- `(exports (owl math extra))`
- `(exports (owl math))`
- `(exports (owl tuple))`
- `(exports (owl digest))`
- `halt`
- `lets/cc`
- `read`
- `read-ll`
- `ref`
- `suspend`
- `wait`
- `(exports (scheme base))`
- `(exports (scheme cxr))`
- `(exports (scheme write))`



## (owl bisect)
Bisect makes a binary search on sorted data.

Exported values:

- `bisect`
- `bisect-unsorted`
- `bisect-range`
- `bisect-range-unsorted`



## (owl boolean)


Exported values:

- `boolean?`
- `boolean=?`



## (owl bytevector)
Byte vectors are vectors holding only numbers in the range 0-255. 
They are internally represented as chunks of memory in the garbage collected heap.
Typical use cases for them is packing data for an operating system call, 
  or receiving data from an external source.

Exported values:

- `bytevector`
- `bytevector?`
- `bytevector-length`
- `bytevector-u8-ref`
- `bytevector-append`
- `bytevector-concatenate`
- `bytevector-concatenate->list`
- `bytevector-copy`
- `bytevector->list`
- `list->bytevector`



## (owl cgen)


Exported values:

- `compile-to-c`



## (owl char)


Exported values:

- `char?`
- `char=?`
- `char<?`
- `char>?`
- `char<=?`
- `char>=?`
- `char->integer`
- `integer->char`



## (owl closure)


Exported values:

- `build-closures`
- `uncompiled-closure?`



## (owl codec)


Exported values:

- `hex-encode-list`
- `hex-encode`
- `hex-decode`
- `hex-decode-list`

### Examples

 - `(hex-decode (hex-encode ))` → ``


## (owl compile)


Exported values:

- `compile`



## (owl cps)


Exported values:

- `cps`



## (owl date)


Exported values:

- `date`
- `leap-year?`
- `valid-date?`
- `next-date`
- `next-date-with-week`
- `week-info`
- `day-names-fi`
- `day-names-en`
- `date-str`
- `date-str-tz`
- `year-start-day`
- `year-start-week-info`
- `minute`
- `hour`
- `day`
- `week`
- `year`
- `leap-year`

### Examples

 - `(next-date-with-week 31 12 1971 5 1)` → `(values 1 1 1972 6 1)`
 - `(date-str 0)` → `00:00:00 1.1.1970 UTC+00:00`


## (owl defmac)


Exported values:

- `λ`
- `syntax-error`
- `begin`
- `quasiquote`
- `letrec`
- `let`
- `letrec*`
- `if`
- `when`
- `unless`
- `cond`
- `case`
- `define`
- `define*`
- `lets`
- `or`
- `and`
- `list`
- `ilist`
- `tuple`
- `tuple-case`
- `define-library`
- `case-lambda`
- `define-values`
- `define-record-type`
- `_record-values`
- `not`
- `B`
- `C`
- `H`
- `I`
- `K`
- `self`
- `type-complex`
- `type-rational`
- `type-int+`
- `type-int-`
- `type-record`
- `immediate?`
- `allocated?`
- `raw?`
- `record?`
- `->`
- `->>`
- `if-lets`
- `type-bytecode`
- `type-proc`
- `type-clos`
- `type-fix+`
- `type-fix-`
- `type-pair`
- `type-vector-dispatch`
- `type-vector-leaf`
- `type-bytevector`
- `type-tuple`
- `type-symbol`
- `type-const`
- `type-rlist-spine`
- `type-rlist-node`
- `type-port`
- `type-string`
- `type-string-wide`
- `type-string-dispatch`
- `type-thread-state`
- `type-ff`
- `type-ff-r`
- `type-ff-red`
- `type-ff-red-r`
- `maybe`



## (owl digest)


Exported values:

- `sha1`
- `sha1-raw`
- `sha1-bytes`
- `sha256`
- `sha256-raw`
- `sha256-bytes`
- `hmac-sha1`
- `hmac-sha1-bytes`
- `hmac-sha256`
- `hmac-sha256-bytes`

### Examples

 - `(sha1 )` → `da39a3ee5e6b4b0d3255bfef95601890afd80709`


## (owl dump)


Exported values:

- `make-compiler`
- `dump-fasl`
- `load-fasl`
- `suspend`



## (owl env)


Exported values:

- `lookup`
- `env-bind`
- `empty-env`
- `apply-env`
- `env-fold`
- `verbose-vm-error`
- `prim-opcodes`
- `opcode->wrapper`
- `primop-of`
- `primitive?`
- `poll-tag`
- `name-tag`
- `link-tag`
- `buffer-tag`
- `signal-tag`
- `signal-halt`
- `thread-quantum`
- `meta-tag`
- `current-library-key`
- `env-set-macro`
- `*tabula-rasa*`
- `env-del`
- `env-get`
- `env-del`
- `env-set`
- `env-keep`
- `env-get-raw`
- `env-put-raw`
- `env-keys`



## (owl eof)


Exported values:

- `eof-object`
- `eof-object?`



## (owl equal-prim)


Exported values:

- `equal-prim?`
- `simple-equal?`



## (owl equal)


Exported values:

- `equal?`
- `eqv?`



## (owl eval)


Exported values:

- `evaluate`
- `exported-eval`



## (owl fasl)


Exported values:

- `fasl-encode`
- `fasl-encode-cooked`
- `fasl-encode-stream`
- `fasl-decode`
- `decode-or`
- `encode`
- `objects-below`
- `decode-stream`
- `sub-objects`

### Examples

 - `(fasl-decode (quote ()) (quote x))` → `(quote x)`


## (owl ff)
A typical way to make data structures for holding key-value mappings in
Lisp systems is to make an association list. An association list
is a list of pairs, where the car holds the key, and cdr holds the corresponding
value. While easy to define and use, they have the downside of slowing
down linearly as the size of the number of associations grows.

Another way to think about such a data structure is to view it as a
partial function. When the function is applied to a key, the associated
value if any is returned.

Owl has finite functions, or ffs, which behave like association
lists, but they slow down only logarithmically as they get more keys.
The semantics are exactly those of an association list using `eq?`
for comparing keys. Finite functions are internally represented as
red-black trees with compacted nodes, so a finite function will
not add any memory overhead compared to association lists.

`#empty` or `##()` can be used to refer to an empty finite function.
`put` adds or rewrites the value of a key, `get` fetches the value
or returns the third argument if the key is not found. `del` removes
a key from a ff.

```
  > (define f (put (put #empty 'foo 100) 'bar 42))
  > f
  ##(foo 100 bar 42)
  > (ff? f)
  #true
  > (get f 'foo #f)
  100
  > (get f 'x #f)
  #f
  > (get (del f 'foo) 'foo #f)
  #f
```

One can also apply a finite function to an argument. The result
will be the value mapped to that key, if any.
If the value is not defined in the function, an error is signaled.
As with many other such functions, you can also pass a second default
value for a finite function, in which case it is returned if the key
is not defined.
;;;
```
  > (f 'foo)
  100
  > (f 'x 'not-there)
  'not-there
  > (map f '(foo bar))
  '(100 42)
```

Many list functions have corresponding functions for ffs, where
usually a function receiving the list element just receives two
arguments, being a particular key and value pair. The name of the
function is typically prefixed with ff-.

```
  (get ##(a 1) 'a #f) → 1

  (get ##(a 1) 'x #f) → #f

  (put ##(a 1 b 2) 'c 3) → ##(a 1 b 2 c 3)

  (del ##(a 1 b 2) 'a) → ##(b 2)

  (fupd ff key value) → ff', like put, but for an existing key

  (keys ##(foo 1 bar 2)) → '(foo bar)

  (ff-union ##(a 100 b 200) ##(b 2 c 3) +) → ##(a 100 b 202 c 3)

  (ff-diff ##(a 1 b 2 c 3) ##(a 10 b 20)) → ##(c 3)

  (ff-fold (λ (o k v) (cons (cons k v) o)) #n ##(foo 1 bar 2) →
     '((bar . 2) (foo . 1))

  (ff-foldr (λ (o k v) (cons (cons k v) o)) #n ##(foo 1 bar 2) →
     '((foo . 1) (bar . 2))
  (ff-map ##(a 1 b 2 c 3) (λ (k v) (square v))) → ##(a 1 b 4 c 9)

  (ff-iter ff) → a lazy list of key-value pairs

  (list->ff '((a . 1) (b . 2))) → ##(a 1 b 2)

  (ff->list ##(a 1 b 2)) → '((a . 1) (b . 2))

```

Other programming languges typically use hashing to obtain a
similar data structure. Owl has an order preserving garbage
collector, which makes it possible to compare order as well as
equality of objects. This makes it easy to build efficient tree
data structures without complex key hashing and collision checks.


Exported values:

- `get`
- `put`
- `del`
- `keys`
- `ff-update`
- `fupd`
- `ff-union`
- `ff-diff`
- `ff-fold`
- `ff-foldr`
- `ff-map`
- `ff-iter`
- `ff?`
- `ff-singleton?`
- `ff-max`
- `ff-min`
- `list->ff`
- `ff->list`
- `ff->sexp`
- `ff-ok?`
- `empty`
- `empty?`
- `getf`



## (owl fixedpoint)
Owl does not allow you to use a special toplevel or use mutations to implement recursion.
Lambdas are the only way to make variable bindings.
Up to this point the compiler also has `rlambda` functions, 
   which will end up generating recursive bindings.
They are essentially to `letrec` what `lambda` is to `let`.
This compilation step gets rid of all the rlambdas
   turning them the ones we all know and love.
This is done by constructing the fixed points manually.

Exported values:

- `fix-points`



## (owl function)


Exported values:

- `bytecode?`
- `function?`
- `procedure?`



## (owl gensym)


Exported values:

- `fresh`
- `gensym`



## (owl iff)


Exported values:

- `iget`
- `iput`
- `ifold`
- `iff->list`



## (owl intern)


Exported values:

- `bytes->symbol`
- `initialize-interner`
- `string->uninterned-symbol`
- `string->interned-symbol`
- `put-symbol`
- `empty-symbol-tree`
- `intern-symbols`
- `start-dummy-interner`
- `start-symbol-interner`



## (owl io)


Exported values:

- `open-output-file`
- `open-input-file`
- `open-append-file`
- `open-socket`
- `open-tcp-socket`
- `open-udp-socket`
- `open-connection`
- `file-size`
- `port->fd`
- `fd->port`
- `port?`
- `close-port`
- `start-base-threads`
- `wait-write`
- `when-readable`
- `blocks->port`
- `closing-blocks->port`
- `tcp-client`
- `tcp-clients`
- `tcp-send`
- `udp-packets`
- `udp-client-socket`
- `wait-udp-packet`
- `check-udp-packet`
- `send-udp-packet`
- `file->vector`
- `file->list`
- `file->byte-stream`
- `vector->file`
- `write-vector`
- `port->meta-byte-stream`
- `port->byte-stream`
- `port->tail-byte-stream`
- `byte-stream->port`
- `port->block-stream`
- `block-stream->port`
- `display-to`
- `print-to`
- `print`
- `print*`
- `print*-to`
- `write`
- `writer-to`
- `write-to`
- `write-bytes`
- `write-bytevector`
- `read-bytevector`
- `try-get-block`
- `byte-stream->lines`
- `lines`
- `system-print`
- `system-println`
- `system-stderr`
- `fasl-save`
- `fasl-load`
- `start-muxer`
- `sleep`



## (owl lazy)


Exported values:

- `lfold`
- `lfoldr`
- `lmap`
- `lappend`
- `lfor`
- `liota`
- `liter`
- `lnums`
- `lzip`
- `ltake`
- `lsplit`
- `llast`
- `llen`
- `lcar`
- `lcdr`
- `lkeep`
- `lremove`
- `ldrop`
- `llref`
- `ledit`
- `pair`
- `tail`
- `uncons`
- `force-ll`
- `subsets`
- `permutations`
- `lunfold`
- `delay`
- `force`
- `avg`
- `lnull?`
- `lpair?`

### Examples

 - `(lsplit (quote (1 2 3 4)) 2)` → `(values (quote (1 2)) (quote (3 4)))`


## (owl list-extra)


Exported values:

- `lset`
- `ldel`
- `lget`
- `length`
- `led`
- `ledn`
- `lins`
- `take`
- `drop`
- `iota`
- `list-ref`
- `list-tail`
- `make-list`
- `split-at`

### Examples

 - `(list-ref (quote (a b c)) 1)` → `(quote b)`
 - `(length (quote (a b c)))` → `3`
 - `(iota 0 1 5)` → `(quote (0 1 2 3 4))`
 - `(list-tail (quote (a b c)) 1)` → `(quote (b c))`


## (owl list)


Exported values:

- `pair?`
- `null?`
- `caar`
- `cadr`
- `cdar`
- `cddr`
- `list?`
- `zip`
- `fold`
- `foldr`
- `map`
- `for-each`
- `memq`
- `assq`
- `last`
- `fold-map`
- `foldr-map`
- `append`
- `concatenate`
- `reverse`
- `filter`
- `remove`
- `every`
- `any`
- `unfold`
- `find`
- `find-tail`
- `take-while`
- `break`
- `fold2`
- `halve`
- `interleave`
- `diff`
- `union`
- `intersect`

### Examples

 - `(zip cons (quote (1 2 3)) (quote (a b c d)))` → `(quote ((1 . a) (2 . b) (3 . c)))`
 - `(foldr cons () (quote (a b c)))` → `(quote (a b c))`
 - `(map not (quote (#false #false #true)))` → `(quote (#true #true #false))`
 - `(assq (quote a) (quote ((a . 1) (b . 2))))` → `(quote (a . 1))`
 - `(last (quote (1 2 3)) (quote a))` → `3`
 - `(append (quote (1 2 3)) (quote (a b c)))` → `(quote (1 2 3 a b c))`
 - `(reverse (quote (1 2 3)))` → `(quote (3 2 1))`
 - `(find null? (quote (1 2 3)))` → `#false`
 - `(interleave (quote x) (quote (a b c)))` → `(quote (a x b x c))`


## (owl macro)


Exported values:

- `macro-expand`
- `match`
- `define-syntax-transformer`



## (owl math)
This library exports the usual arbitrary precision bignum arithmetic functions.
You can also use specific (owl math X) libraries if necessary.

Exported values:

- `quotient`
- `<<`
- `>>`
- `band`
- `bior`
- `bxor`
- `fx-width`
- `ncar`
- `ncdr`
- `fx-greatest`
- `zero?`
- `integer?`
- `truncate/`
- `even?`
- `odd?`
- `fixnum?`
- `ediv`
- `numerator`
- `denominator`
- `gcd`
- `(exports (owl math complex))`
- `(exports (owl math extra))`



## (owl math complex)
This library defines complex arbitrary precision math functions.

Exported values:

- `+`
- `-`
- `*`
- `/`
- `=`
- `complex`
- `negate`



## (owl math extra)

Exported values:

- `abs`
- `floor`
- `sum`
- `product`
- `render-number`
- `log2`
- `log`
- `lcm`
- `negative?`
- `positive?`
- `number?`
- `exact-integer-sqrt`
- `isqrt`
- `sqrt`
- `expt`
- `expt-mod`
- `round`
- `ncr`
- `npr`
- `truncate`
- `modulo`
- `remainder`
- `ceiling`
- `!`
- `factor`
- `prime?`
- `real?`
- `complex?`
- `rational?`
- `primes-between`
- `totient`
- `phi`
- `divisor-sum`
- `divisor-count`
- `dlog`
- `dlog-simple`
- `fib`
- `histogram`
- `negative?`
- `min`
- `max`
- `minl`
- `maxl`
- `<`
- `>`
- `>=`
- `<=`
- `+`
- `-`
- `*`
- `/`



## (owl math integer)
This library defines arbitrary precision integer arithmetic. Some of the
functions are only defined for integers, whereas others are typically
extended to handle also more complex kinds of numbers.

Operations to be extended typically export both the operation defined for
integers only, and a corresponding mk-* version which calls a given function
in case the types of inputs are not integers. This allows extending the
definitions in other modules while checking the most common cases of integer
inputs in first branches.

Owl currently has 24 bits available for encoding a value in an immediate
value, meaning a value that does not have to be stored in memory. A fixnum,
or a fixed size number, is a number which fits these bits. The sign of a
fixnum is stored the type of the immediate object.

When a number is bigger or smaller than the maximum fixnum, it is converted
to an allocated integer. In this case the representation of the number is a
linked sequence of base 2²⁴ digits of the number starting from the least
significant digits. Again the sign of the number is stored in the type.

A valid fixnum zero is always positive, and a valid negative bignum has the
negative type only at the root node.

Exported values:

- `fixnum?`
- `integer?`
- `mk-add`
- `mk-sub`
- `+`
- `-`
- `*`
- `=`
- `<<`
- `<`
- `<=`
- `=`
- `>=`
- `>`
- `>>`
- `band`
- `bior`
- `bxor`
- `quotient`
- `ediv`
- `truncate/`
- `nat-succ`
- `big-bad-args`
- `negate`
- `ncar`
- `ncdr`
- `even?`
- `odd?`
- `zero?`
- `negative?`
- `positive?`
- `rem`
- `mod`
- `ncons`
- `ncar`
- `ncdr`
- `big-digits-equal?`
- `fx-greatest`
- `fx-width`
- `to-int-`
- `to-int+`
- `to-fix+`
- `to-fix-`
- `add-big`
- `sub-big`
- `right-out`



## (owl math rational)
This library defines arbitrary precision rational arithmetic operations.
A rational number p/q is a typed pair of arbitrary precision integers. A
valid rational number has q != 0, q != 1, and gcd(p, q) = 1.

Exported values:

- `mk-rational-add`
- `mk-rational-sub`
- `+`
- `-`
- `<`
- `/`
- `gcd`
- `gcdl`
- `rational`
- `numerator`
- `denominator`
- `divide`
- `negate`



## (owl parse)


Exported values:

- `parses`
- `byte`
- `imm`
- `seq`
- `epsilon`
- `ε`
- `byte-if`
- `rune`
- `rune-if`
- `either`
- `one-of`
- `star`
- `plus`
- `greedy-star`
- `greedy-plus`
- `byte-between`
- `parse-head`
- `backtrack`
- `try-parse`
- `word`
- `fd->exp-stream`
- `file->exp-stream`
- `silent-syntax-fail`
- `resuming-syntax-fail`



## (owl port)


Exported values:

- `port?`
- `fd->port`
- `port->fd`
- `stdin`
- `stdout`
- `stderr`
- `stdio-port?`



## (owl primop)


Exported values:

- `primops`
- `primop-name`
- `special-bind-primop?`
- `run`
- `set-ticker`
- `bind`
- `mkt`
- `halt`
- `wait`
- `set-memory-limit`
- `get-word-size`
- `get-memory-limit`
- `apply`
- `call/cc`
- `lets/cc`
- `create-type`
- `object-size`
- `len`



## (owl proof)


Exported values:

- `theorem`
- `theorem-equal?`
- `example`

### Examples

 - `1` → `1`
 - `(cons 1 2)` → `(quote (1 . 2))`
 - `(values 1 2)` → `(values 1 2)`


## (owl queue)


Exported values:

- `qnull`
- `qnull?`
- `qlen`
- `qcons`
- `qsnoc`
- `quncons`
- `qunsnoc`
- `qcar`
- `qrac`
- `list->queue`
- `queue->list`



## (owl random)
Randomness is an interesting thing to work with in a purely
functional setting. Owl builds randomness around streams of
typically deterministically generated 24-bit fixnums. These
are usually called rands in the code.

A function involving randomness typically receives a rand
stream, and also returns one after possibly consuming some
rands. Behavior like this would be easy to hide using macros
or monadic code, but Owl generally strives to be explicit and
simple, so the rand streams are handled just like any other
value.

```
  > (define rs (seed->rands 9))
  > (rand rs 10000)
  '(values #<function> 3942) ;; the new rand stream and 3942
  > (lets ((rs a (rand rs 10000))) a)
  3942
  > (lets ((rs elem (rand-elem rs '(a b c d e f g)))) elem)
  'c
  > (lets ((rs sub (rand-subset rs '(a b c d e f g)))) sub)
  '(b e f)
  > (lets ((rs perm (random-permutation rs '(a b c d e f g)))) perm)
  '(g e c b d a f)
  > (lets ((rs ns (random-numbers rs 100 10))) ns)
  '(95 39 69 99 2 98 56 85 77 39)
```

Exported values:

- `lcg-rands`
- `seed->rands`
- `rands->bits`
- `seed->bits`
- `rands->bytes`
- `seed->bytes`
- `rand`
- `rand-nbit`
- `rand-log`
- `rand-elem`
- `rand-subset`
- `rand-range`
- `random-numbers`
- `reservoir-sample`
- `shuffle`
- `random-permutation`
- `random-subset`
- `rand-occurs?`



## (owl regex)


Exported values:

- `get-sexp-regex`
- `get-replace-regex`
- `string->regex`
- `string->replace-regex`
- `string->complete-match-regex`
- `rex-matches`



## (owl register)


Exported values:

- `allocate-registers`
- `n-registers`



## (owl render)


Exported values:

- `make-serializer`
- `render`
- `str`



## (owl repl)


Exported values:

- `repl-file`
- `repl-port`
- `repl-string`
- `repl-trampoline`
- `repl`
- `print-repl-error`
- `bind-toplevel`
- `library-import`
- `*owl-core*`



## (owl rlist)


Exported values:

- `rnull`
- `rcons`
- `rget`
- `rcar`
- `rcdr`
- `rset`
- `rlen`
- `rlist`
- `rfold`
- `rfoldr`
- `riter`
- `riterr`
- `rmap`
- `rnull?`
- `rpair?`
- `list->rlist`
- `rlist->list`

### Examples

 - `(rcar (rcons 11 rnull))` → `11`


## (owl rlist lc-forest)
Random access lists are a data structure similar to lists,
but with very different efficiency characteristics.
A regular list built out of cons cells is an optimal solution,
   if one needs to work mainly with the initial elements or the whole list at a time.
However, if you need to frequently find and maybe update values in the middle of the list,
   you have to perform operations taking time proportional to the length of the list.
In other words, those list operations are linear time, or have complexity O(n).
Cons, car and cdr on the other hand are very efficient for regular lists.
Regardless of the size of a list, it will always take a fixed amount of time to add, take or remove a value from it.
In other words, these operations are constant time, or have complexity O(1).

A random access list is a data structure,
   which unsurprisingly attempts to make random access and update efficient.

The performance characteristics of this random access list library are:
```
  car → O(1)
  cdr → O(log n)
  cons → O(log n)
  get → O(log n)
  set → O(log n)
  len → O(log n)
  fold → O(n)
  append → O(n)
  list->rlist → O(n log n)
  rlist->list → O(n)
```

The operation is based on two ideas.
Firstly, a random access list consists of a sequence of complete binary trees.
The binary trees are built out of cons cells when needed.
The first tree is always of height 1, meaning it just holds the value, much like a regular cons cell.
The next node always holds a binary tree either of the same or next height.
There can be at most two trees of the same height next to eachother.
Therefore, tree heights `(1 1)`, `(1 2 4)` and `(1 1 2 4 4)` are valid,
   whereas `(1 1 1)`, `(2 2 4)` and `(1 2 2 8)` are not.
`(5)` is right out.

Secondly, trees can be addressed directly with bits.
It takes a n-bit number address each node of a complete binary tree of height n.
Finding a value from a list works by first finding the tree in which the value is held,
  and then using the remaining bits to find the correct leaf node in the tree.

It is easy to see that it takes O(log n) steps to find the tree in which some particular value is held,
   and then another O(log n) steps to walk the tree to a given position,
Threfore we have a total complexity of O(log n) for access and update.

```
  (rcar (rcons 11 rnull)) → 11
  (rnull? (rcons 11 rnull)) → #false
  (rlist->list (rcons 1 (rcons 2 rnull))) → (1 2))
  (rget (list->rlist (iota 0 1 1000)) 123 #f) → 123
  (rget (list->rlist (iota 0 1 1000)) 1234 #f) → #false
```


Exported values:

- `rnull`
- `rcons`
- `rget`
- `rcar`
- `rcdr`
- `rset`
- `rlen`
- `rlist`
- `rfold`
- `rfoldr`
- `riter`
- `riterr`
- `rmap`
- `rnull?`
- `rpair?`
- `list->rlist`
- `rlist->list`



## (owl rlist linear)


Exported values:

- `rnull`
- `rcons`
- `rget`
- `rcar`
- `rcdr`
- `rset`
- `rlen`
- `rlist`
- `rfold`
- `rfoldr`
- `riter`
- `riterr`
- `rmap`
- `rnull?`
- `rpair?`
- `list->rlist`
- `rlist->list`



## (owl rlist old)


Exported values:

- `rnull`
- `rcons`
- `rcar`
- `rcdr`
- `rnull?`
- `rpair?`
- `rlist?`
- `rget`
- `rset`
- `rlen`
- `rfold`
- `rfoldr`
- `rmap`
- `riter`
- `riterr`
- `requal?`
- `rlist->list`
- `list->rlist`
- `rlist`
- `rrev`



## (owl sexp)


Exported values:

- `sexp-parser`
- `read-exps-from`
- `list->number`
- `get-sexps`
- `get-padded-sexps`
- `string->sexp`
- `vector->sexps`
- `list->sexps`
- `read`
- `read-ll`



## (owl sort)


Exported values:

- `sort`
- `isort`
- `quicksort`
- `mergesort`

### Examples

 - `(sort < (quote (1 4 2)))` → `(quote (1 2 4))`


## (owl string)


Exported values:

- `string?`
- `string-length`
- `runes->string`
- `bytes->string`
- `string->bytes`
- `string->runes`
- `list->string`
- `string->list`
- `string-append`
- `string-concatenate`
- `c-string`
- `null-terminate`
- `finish-string`
- `render-string`
- `render-quoted-string`
- `str-iter`
- `str-iterr`
- `str-fold`
- `str-foldr`
- `str-iter-bytes`
- `str-replace`
- `string-map`
- `str-rev`
- `string`
- `string-copy`
- `substring`
- `string-ref`
- `string=?`
- `string<?`
- `string>?`
- `string<=?`
- `string>=?`
- `string-ci=?`
- `string-ci<?`
- `string-ci>?`
- `string-ci<=?`
- `string-ci>=?`
- `unicode-fold-char`
- `make-string`



## (owl suffix)
Suffix arrays are a useful data structure for speeding up searches from strings.

Exported values:

- `suffix-array`
- `suffix-list`



## (owl sum)


Exported values:

- `eval`



## (owl symbol)
Functions for handling symbols.

Exported values:

- `string->symbol`
- `symbol?`
- `symbol=?`
- `symbol->string`
- `render-symbol`



## (owl sys)
This library defines various system calls and wrappers for calling them.

Exported values:

- `dir-fold`
- `dir->list`
- `dir->list-all`
- `errno`
- `strerror`
- `exec`
- `fork`
- `pipe`
- `wait`
- `waitpid`
- `kill`
- `getenv`
- `setenv`
- `unsetenv`
- `umask`
- `getcwd`
- `chdir`
- `readlink`
- `symlink`
- `link`
- `rename`
- `unlink`
- `rmdir`
- `mknod`
- `mkdir`
- `mkfifo`
- `stat`
- `directory?`
- `file?`
- `chmod`
- `chown`
- `lseek`
- `seek-current`
- `seek-set`
- `seek-end`
- `sighup`
- `signint`
- `sigquit`
- `sigill`
- `sigabrt`
- `sigfpe`
- `sigkill`
- `sigsegv`
- `sigpipe`
- `sigalrm`
- `sigterm`
- `O_RDONLY`
- `O_WRONLY`
- `O_APPEND`
- `O_CREAT`
- `O_TRUNC`
- `close`
- `fcntl`
- `open`
- `dupfd`
- `read`
- `write`
- `port->non-blocking`
- `CLOCK_REALTIME`
- `clock_gettime`
- `set-terminal-rawness`
- `mem-string`
- `mem-strings`
- `get-environment`
- `get-heap-bytes-written`
- `get-heap-max-live`
- `execvp`
- `system`



## (owl syscall)


Exported values:

- `syscall`
- `error`
- `interact`
- `accept-mail`
- `wait-mail`
- `check-mail`
- `exit-owl`
- `release-thread`
- `catch-thread`
- `set-signal-action`
- `single-thread?`
- `kill`
- `link`
- `mail`
- `return-mails`
- `fork-named`
- `exit-thread`
- `exit-owl`
- `poll-mail-from`
- `start-profiling`
- `stop-profiling`
- `running-threads`
- `par*`
- `par`
- `por*`
- `por`
- `thread`
- `thunk->thread`



## (owl thread)
This library defines the thread controller. It handles activation, suspension
and requested external actions of continuation-based threads.  It is much like
a very small kernel.

Exported values:

- `start-thread-controller`
- `thread-controller`
- `repl-signal-handler`
- `try-thunk`
- `try`



## (owl time)


Exported values:

- `elapsed-real-time`
- `timed`
- `time`
- `time-ms`
- `time-ns`



## (owl tuple)
Tuples are an early simple data structure for holding multiple values.
Values are indexed from 1 and there is little error detection
   apart from range checks.

```
  > (define x (list->tuple '(a b c)))
  > (ref x 1)
  'a
  > (size x)
  3
  > (equal? x (tuple 'a 'b 'c))
  #true
```

Exported values:

- `tuple?`
- `tuple-length`
- `list->tuple`
- `tuple->list`



## (owl unicode)
UTF-8 encoding and decoding.

Exported values:

- `utf8-encode`
- `utf8-decode`
- `utf8-sloppy-decode`
- `utf8-decoder`
- `encode-point`
- `two-byte-point`
- `three-byte-point`
- `four-byte-point`
- `last-code-point`
- `valid-code-point?`



## (owl variable)
You cannot mutate values, 
  but threads can also be used to encapsulate state.
This library introduces seemingly variable values 
  implemented threads which allow setting and getting a value.

Exported values:

- `make-variable`
- `link-variable`



## (owl vector)
Vectors are one-dimensional data structures indexable by natural numbers,
having O(n log_256 n) access and memory use (effectively O(1)). They are
mainly intended to be used for static data requiring efficient (modulo
owl) iteration and random access.

In owl, vectors are implemented as complete 256-ary trees. Small vectors
fitting to one node of the tree are of raw or allocated type 11, meaning
they usually take 8+4n or 4+n bytes of memory, depending on whether the
values are normal descriptors or fixnums in the range 0-255.

Large vectors are 256-ary trees. Each dispatch node in the tree handles
one byte of an index, and nodes starting from root each dispatch the
highest byte of an index. When only one byte is left, one reads the
reached leaf node, or the leaf node stored to the dispatch node.

Reading the vector in order corresponds to breadth-first walk
of the tree. Notice that since no number > 0 has 0 as the highest
byte, the first dispatch position of the root is always free. This
position contains the size of the vector, so that it is accessable
in O(1) without space overhead or special case handling. Leaf nodes
have the size as part of the normal owl object header.

Order example using binary trees:

```
           (0 1)                 bits 0 and 1, only 1 can have children
              |                  dispatch the top bit
            (2 3)                bits from top, 10 11, numbers ending here 2 and 3
            /   \                dispatch top and second bit
           /     \
       (4 5)     (6 7)           bits from top, (100 101) (110 111)
       /  |       |  \
      /   |       |   \
 (9 8) (10 11) (12 13) (14 15)   etc
```

Vectors use the same order, but with 256-ary trees.

Exported values:

- `vector`
- `vector?`
- `vector-length`
- `vector-ref`
- `list->vector`
- `vector->list`
- `vec-iter`
- `vec-iterr`
- `vec-fold`
- `vec-foldr`
- `vec-range`
- `vec-iter-range`
- `vector-map`
- `merge-chunks`
- `make-vector`
- `leaf-data`
- `vec-leaf-of`
- `vec-leaves`
- `vec-cat`
- `vec-rev`
- `*vec-leaf-size*`



## (scheme base)


Exported values:

- `*`
- `+`
- `-`
- `...`
- `/`
- `<`
- `<=`
- `=`
- `=>`
- `>`
- `>=`
- `abs`
- `and`
- `append`
- `apply`
- `assoc`
- `assq`
- `assv`
- `begin`
- `binary-port?`
- `boolean=?`
- `boolean?`
- `bytevector`
- `bytevector-append`
- `bytevector-copy`
- `bytevector-copy!`
- `bytevector-length`
- `bytevector-u8-ref`
- `bytevector-u8-set!`
- `bytevector?`
- `caar`
- `cadr`
- `call-with-current-continuation`
- `call-with-port`
- `call-with-values`
- `call/cc`
- `car`
- `case`
- `cdar`
- `cddr`
- `cdr`
- `ceiling`
- `char->integer`
- `char-ready?`
- `char<=?`
- `char<?`
- `char=?`
- `char>=?`
- `char>?`
- `char?`
- `close-input-port`
- `close-output-port`
- `close-port`
- `complex?`
- `cond`
- `cons`
- `current-error-port`
- `current-input-port`
- `current-output-port`
- `define`
- `define-library`
- `define-record-type`
- `define-syntax`
- `define-values`
- `denominator`
- `do`
- `dynamic-wind`
- `else`
- `eof-object`
- `eof-object?`
- `eq?`
- `equal?`
- `eqv?`
- `error`
- `error-object-irritants`
- `error-object-message`
- `error-object?`
- `even?`
- `exact`
- `exact-integer-sqrt`
- `exact-integer?`
- `exact?`
- `expt`
- `features`
- `file-error?`
- `floor`
- `floor-quotient`
- `floor-remainder`
- `floor/`
- `flush-output-port`
- `for-each`
- `gcd`
- `get-output-bytevector`
- `get-output-string`
- `guard`
- `if`
- `include`
- `include-ci`
- `inexact`
- `inexact?`
- `input-port-open?`
- `input-port?`
- `integer->char`
- `integer?`
- `lambda`
- `lcm`
- `length`
- `let`
- `let*`
- `let*-values`
- `let-syntax`
- `let-values`
- `letrec`
- `letrec*`
- `letrec-syntax`
- `list`
- `list->string`
- `list->vector`
- `list-copy`
- `list-ref`
- `list-set!`
- `list-tail`
- `list?`
- `make-bytevector`
- `make-list`
- `make-parameter`
- `make-string`
- `make-vector`
- `map`
- `max`
- `member`
- `memq`
- `memv`
- `min`
- `modulo`
- `negative?`
- `newline`
- `not`
- `null?`
- `number->string`
- `number?`
- `numerator`
- `odd?`
- `open-input-bytevector`
- `open-input-string`
- `open-output-bytevector`
- `open-output-string`
- `or`
- `output-port-open?`
- `output-port?`
- `pair?`
- `parameterize`
- `peek-char`
- `peek-u8`
- `port?`
- `positive?`
- `procedure?`
- `quasiquote`
- `quote`
- `quotient`
- `raise`
- `raise-continuable`
- `rational?`
- `rationalize`
- `read-bytevector`
- `read-bytevector!`
- `read-char`
- `read-error?`
- `read-line`
- `read-string`
- `read-u8`
- `real?`
- `remainder`
- `reverse`
- `round`
- `set!`
- `set-car!`
- `set-cdr!`
- `square`
- `string`
- `string->list`
- `string->number`
- `string->symbol`
- `string->utf8`
- `string->vector`
- `string-append`
- `string-copy`
- `string-copy!`
- `string-fill!`
- `string-for-each`
- `string-length`
- `string-map`
- `string-ref`
- `string-set!`
- `string<=?`
- `string<?`
- `string=?`
- `string>=?`
- `string>?`
- `string?`
- `substring`
- `symbol->string`
- `symbol=?`
- `symbol?`
- `syntax-error`
- `textual-port?`
- `truncate`
- `truncate-quotient`
- `truncate-remainder`
- `truncate/`
- `u8-ready?`
- `unless`
- `unquote`
- `unquote-splicing`
- `utf8->string`
- `values`
- `vector`
- `vector->list`
- `vector->string`
- `vector-append`
- `vector-copy`
- `vector-copy!`
- `vector-fill!`
- `vector-for-each`
- `vector-length`
- `vector-map`
- `vector-ref`
- `vector-set!`
- `vector?`
- `when`
- `with-exception-handler`
- `write-bytevector`
- `write-char`
- `write-string`
- `write-u8`
- `zero?`
- `string->integer`



## (scheme case-lambda)


Exported values:

- `case-lambda`



## (scheme complex)


Exported values:

- `angle`
- `imag-part`
- `magnitude`
- `make-polar`
- `make-rectangular`
- `real-part`



## (scheme cxr)


Exported values:

- `caaar`
- `caadr`
- `cadar`
- `caddr`
- `cdaar`
- `cdadr`
- `cddar`
- `cdddr`
- `caaaar`
- `caaadr`
- `caadar`
- `caaddr`
- `cadaar`
- `cadadr`
- `caddar`
- `cadddr`
- `cdaaar`
- `cdaadr`
- `cdadar`
- `cdaddr`
- `cddaar`
- `cddadr`
- `cdddar`
- `cddddr`



## (scheme process-context)


Exported values:

- `command-line`
- `emergency-exit`
- `exit`
- `get-environment-variable`
- `get-environment-variables`



## (scheme time)


Exported values:

- `current-jiffy`
- `current-second`
- `jiffies-per-second`



## (scheme write)


Exported values:

- `display`
- `write`
- `write-shared`
- `write-simple`


