/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import com.maxmind.geoip2.exception.AddressNotFoundException;
import com.maxmind.geoip2.model.AsnResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.model.CountryResponse;
import com.maxmind.geoip2.record.City;
import com.maxmind.geoip2.record.Continent;
import com.maxmind.geoip2.record.Country;
import com.maxmind.geoip2.record.Location;
import com.maxmind.geoip2.record.Subdivision;
import java.io.IOException;
import java.net.InetAddress;
import java.security.AccessController;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.common.network.InetAddresses;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.ingest.AbstractProcessor;
import org.elasticsearch.ingest.ConfigurationUtils;
import org.elasticsearch.ingest.IngestDocument;
import org.elasticsearch.ingest.Processor;
import org.elasticsearch.ingest.geoip.DatabaseReaderLazyLoader;
import org.elasticsearch.ingest.geoip.IngestGeoIpPlugin;

public final class GeoIpProcessor
extends AbstractProcessor {
    public static final String TYPE = "geoip";
    private static final String CITY_DB_SUFFIX = "-City";
    private static final String COUNTRY_DB_SUFFIX = "-Country";
    private static final String ASN_DB_SUFFIX = "-ASN";
    private final String field;
    private final String targetField;
    private final DatabaseReaderLazyLoader lazyLoader;
    private final Set<Property> properties;
    private final boolean ignoreMissing;
    private final IngestGeoIpPlugin.GeoIpCache cache;

    GeoIpProcessor(String tag, String field, DatabaseReaderLazyLoader lazyLoader, String targetField, Set<Property> properties, boolean ignoreMissing, IngestGeoIpPlugin.GeoIpCache cache) {
        super(tag);
        this.field = field;
        this.targetField = targetField;
        this.lazyLoader = lazyLoader;
        this.properties = properties;
        this.ignoreMissing = ignoreMissing;
        this.cache = cache;
    }

    boolean isIgnoreMissing() {
        return this.ignoreMissing;
    }

    public IngestDocument execute(IngestDocument ingestDocument) throws IOException {
        Map<String, Object> geoData;
        String ip = (String)ingestDocument.getFieldValue(this.field, String.class, this.ignoreMissing);
        if (ip == null && this.ignoreMissing) {
            return ingestDocument;
        }
        if (ip == null) {
            throw new IllegalArgumentException("field [" + this.field + "] is null, cannot extract geoip information.");
        }
        InetAddress ipAddress = InetAddresses.forString((String)ip);
        String databaseType = this.lazyLoader.getDatabaseType();
        if (databaseType.endsWith(CITY_DB_SUFFIX)) {
            try {
                geoData = this.retrieveCityGeoData(ipAddress);
            }
            catch (AddressNotFoundRuntimeException e) {
                geoData = Collections.emptyMap();
            }
        } else if (databaseType.endsWith(COUNTRY_DB_SUFFIX)) {
            try {
                geoData = this.retrieveCountryGeoData(ipAddress);
            }
            catch (AddressNotFoundRuntimeException e) {
                geoData = Collections.emptyMap();
            }
        } else if (databaseType.endsWith(ASN_DB_SUFFIX)) {
            try {
                geoData = this.retrieveAsnGeoData(ipAddress);
            }
            catch (AddressNotFoundRuntimeException e) {
                geoData = Collections.emptyMap();
            }
        } else {
            throw new ElasticsearchParseException("Unsupported database type [" + this.lazyLoader.getDatabaseType() + "]", (Throwable)new IllegalStateException(), new Object[0]);
        }
        if (!geoData.isEmpty()) {
            ingestDocument.setFieldValue(this.targetField, geoData);
        }
        return ingestDocument;
    }

    public String getType() {
        return TYPE;
    }

    String getField() {
        return this.field;
    }

    String getTargetField() {
        return this.targetField;
    }

    String getDatabaseType() throws IOException {
        return this.lazyLoader.getDatabaseType();
    }

    Set<Property> getProperties() {
        return this.properties;
    }

    private Map<String, Object> retrieveCityGeoData(InetAddress ipAddress) {
        SpecialPermission.check();
        CityResponse response = AccessController.doPrivileged(() -> this.cache.putIfAbsent(ipAddress, CityResponse.class, ip -> {
            try {
                return this.lazyLoader.get().city(ip);
            }
            catch (AddressNotFoundException e) {
                throw new AddressNotFoundRuntimeException(e);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }));
        Country country = response.getCountry();
        City city = response.getCity();
        Location location = response.getLocation();
        Continent continent = response.getContinent();
        Subdivision subdivision = response.getMostSpecificSubdivision();
        HashMap<String, Object> geoData = new HashMap<String, Object>();
        for (Property property : this.properties) {
            switch (property) {
                case IP: {
                    geoData.put("ip", NetworkAddress.format((InetAddress)ipAddress));
                    break;
                }
                case COUNTRY_ISO_CODE: {
                    String countryIsoCode = country.getIsoCode();
                    if (countryIsoCode == null) break;
                    geoData.put("country_iso_code", countryIsoCode);
                    break;
                }
                case COUNTRY_NAME: {
                    String countryName = country.getName();
                    if (countryName == null) break;
                    geoData.put("country_name", countryName);
                    break;
                }
                case CONTINENT_NAME: {
                    String continentName = continent.getName();
                    if (continentName == null) break;
                    geoData.put("continent_name", continentName);
                    break;
                }
                case REGION_ISO_CODE: {
                    String countryIso = country.getIsoCode();
                    String subdivisionIso = subdivision.getIsoCode();
                    if (countryIso == null || subdivisionIso == null) break;
                    String regionIsoCode = countryIso + "-" + subdivisionIso;
                    geoData.put("region_iso_code", regionIsoCode);
                    break;
                }
                case REGION_NAME: {
                    String subdivisionName = subdivision.getName();
                    if (subdivisionName == null) break;
                    geoData.put("region_name", subdivisionName);
                    break;
                }
                case CITY_NAME: {
                    String cityName = city.getName();
                    if (cityName == null) break;
                    geoData.put("city_name", cityName);
                    break;
                }
                case TIMEZONE: {
                    String locationTimeZone = location.getTimeZone();
                    if (locationTimeZone == null) break;
                    geoData.put("timezone", locationTimeZone);
                    break;
                }
                case LOCATION: {
                    Double latitude = location.getLatitude();
                    Double longitude = location.getLongitude();
                    if (latitude == null || longitude == null) break;
                    HashMap<String, Double> locationObject = new HashMap<String, Double>();
                    locationObject.put("lat", latitude);
                    locationObject.put("lon", longitude);
                    geoData.put("location", locationObject);
                }
            }
        }
        return geoData;
    }

    private Map<String, Object> retrieveCountryGeoData(InetAddress ipAddress) {
        SpecialPermission.check();
        CountryResponse response = AccessController.doPrivileged(() -> this.cache.putIfAbsent(ipAddress, CountryResponse.class, ip -> {
            try {
                return this.lazyLoader.get().country(ip);
            }
            catch (AddressNotFoundException e) {
                throw new AddressNotFoundRuntimeException(e);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }));
        Country country = response.getCountry();
        Continent continent = response.getContinent();
        HashMap<String, Object> geoData = new HashMap<String, Object>();
        for (Property property : this.properties) {
            switch (property) {
                case IP: {
                    geoData.put("ip", NetworkAddress.format((InetAddress)ipAddress));
                    break;
                }
                case COUNTRY_ISO_CODE: {
                    String countryIsoCode = country.getIsoCode();
                    if (countryIsoCode == null) break;
                    geoData.put("country_iso_code", countryIsoCode);
                    break;
                }
                case COUNTRY_NAME: {
                    String countryName = country.getName();
                    if (countryName == null) break;
                    geoData.put("country_name", countryName);
                    break;
                }
                case CONTINENT_NAME: {
                    String continentName = continent.getName();
                    if (continentName == null) break;
                    geoData.put("continent_name", continentName);
                }
            }
        }
        return geoData;
    }

    private Map<String, Object> retrieveAsnGeoData(InetAddress ipAddress) {
        SpecialPermission.check();
        AsnResponse response = AccessController.doPrivileged(() -> this.cache.putIfAbsent(ipAddress, AsnResponse.class, ip -> {
            try {
                return this.lazyLoader.get().asn(ip);
            }
            catch (AddressNotFoundException e) {
                throw new AddressNotFoundRuntimeException(e);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }));
        Integer asn = response.getAutonomousSystemNumber();
        String organization_name = response.getAutonomousSystemOrganization();
        HashMap<String, Object> geoData = new HashMap<String, Object>();
        for (Property property : this.properties) {
            switch (property) {
                case IP: {
                    geoData.put("ip", NetworkAddress.format((InetAddress)ipAddress));
                    break;
                }
                case ASN: {
                    if (asn == null) break;
                    geoData.put("asn", asn);
                    break;
                }
                case ORGANIZATION_NAME: {
                    if (organization_name == null) break;
                    geoData.put("organization_name", organization_name);
                }
            }
        }
        return geoData;
    }

    static enum Property {
        IP,
        COUNTRY_ISO_CODE,
        COUNTRY_NAME,
        CONTINENT_NAME,
        REGION_ISO_CODE,
        REGION_NAME,
        CITY_NAME,
        TIMEZONE,
        LOCATION,
        ASN,
        ORGANIZATION_NAME;

        static final EnumSet<Property> ALL_CITY_PROPERTIES;
        static final EnumSet<Property> ALL_COUNTRY_PROPERTIES;
        static final EnumSet<Property> ALL_ASN_PROPERTIES;

        public static Property parseProperty(String databaseType, String value) {
            EnumSet<Property> validProperties = EnumSet.noneOf(Property.class);
            if (databaseType.endsWith(GeoIpProcessor.CITY_DB_SUFFIX)) {
                validProperties = ALL_CITY_PROPERTIES;
            } else if (databaseType.endsWith(GeoIpProcessor.COUNTRY_DB_SUFFIX)) {
                validProperties = ALL_COUNTRY_PROPERTIES;
            } else if (databaseType.endsWith(GeoIpProcessor.ASN_DB_SUFFIX)) {
                validProperties = ALL_ASN_PROPERTIES;
            }
            try {
                Property property = Property.valueOf(value.toUpperCase(Locale.ROOT));
                if (!validProperties.contains((Object)property)) {
                    throw new IllegalArgumentException("invalid");
                }
                return property;
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("illegal property value [" + value + "]. valid values are " + Arrays.toString(validProperties.toArray()));
            }
        }

        static {
            ALL_CITY_PROPERTIES = EnumSet.of(IP, new Property[]{COUNTRY_ISO_CODE, COUNTRY_NAME, CONTINENT_NAME, REGION_ISO_CODE, REGION_NAME, CITY_NAME, TIMEZONE, LOCATION});
            ALL_COUNTRY_PROPERTIES = EnumSet.of(IP, CONTINENT_NAME, COUNTRY_NAME, COUNTRY_ISO_CODE);
            ALL_ASN_PROPERTIES = EnumSet.of(IP, ASN, ORGANIZATION_NAME);
        }
    }

    static final class AddressNotFoundRuntimeException
    extends RuntimeException {
        AddressNotFoundRuntimeException(Throwable cause) {
            super(cause);
        }
    }

    public static final class Factory
    implements Processor.Factory {
        static final Set<Property> DEFAULT_CITY_PROPERTIES = Collections.unmodifiableSet(EnumSet.of(Property.CONTINENT_NAME, new Property[]{Property.COUNTRY_ISO_CODE, Property.REGION_ISO_CODE, Property.REGION_NAME, Property.CITY_NAME, Property.LOCATION}));
        static final Set<Property> DEFAULT_COUNTRY_PROPERTIES = Collections.unmodifiableSet(EnumSet.of(Property.CONTINENT_NAME, Property.COUNTRY_ISO_CODE));
        static final Set<Property> DEFAULT_ASN_PROPERTIES = Collections.unmodifiableSet(EnumSet.of(Property.IP, Property.ASN, Property.ORGANIZATION_NAME));
        private final Map<String, DatabaseReaderLazyLoader> databaseReaders;
        private final IngestGeoIpPlugin.GeoIpCache cache;

        Map<String, DatabaseReaderLazyLoader> databaseReaders() {
            return Collections.unmodifiableMap(this.databaseReaders);
        }

        public Factory(Map<String, DatabaseReaderLazyLoader> databaseReaders, IngestGeoIpPlugin.GeoIpCache cache) {
            this.databaseReaders = databaseReaders;
            this.cache = cache;
        }

        public GeoIpProcessor create(Map<String, Processor.Factory> registry, String processorTag, Map<String, Object> config) throws IOException {
            Set<Property> properties;
            String ipField = ConfigurationUtils.readStringProperty((String)GeoIpProcessor.TYPE, (String)processorTag, config, (String)"field");
            String targetField = ConfigurationUtils.readStringProperty((String)GeoIpProcessor.TYPE, (String)processorTag, config, (String)"target_field", (String)GeoIpProcessor.TYPE);
            String databaseFile = ConfigurationUtils.readStringProperty((String)GeoIpProcessor.TYPE, (String)processorTag, config, (String)"database_file", (String)"GeoLite2-City.mmdb");
            List propertyNames = ConfigurationUtils.readOptionalList((String)GeoIpProcessor.TYPE, (String)processorTag, config, (String)"properties");
            boolean ignoreMissing = ConfigurationUtils.readBooleanProperty((String)GeoIpProcessor.TYPE, (String)processorTag, config, (String)"ignore_missing", (boolean)false);
            DatabaseReaderLazyLoader lazyLoader = this.databaseReaders.get(databaseFile);
            if (lazyLoader == null) {
                throw ConfigurationUtils.newConfigurationException((String)GeoIpProcessor.TYPE, (String)processorTag, (String)"database_file", (String)("database file [" + databaseFile + "] doesn't exist"));
            }
            String databaseType = lazyLoader.getDatabaseType();
            if (propertyNames != null) {
                EnumSet<Property> modifiableProperties = EnumSet.noneOf(Property.class);
                for (String fieldName : propertyNames) {
                    try {
                        modifiableProperties.add(Property.parseProperty(databaseType, fieldName));
                    }
                    catch (IllegalArgumentException e) {
                        throw ConfigurationUtils.newConfigurationException((String)GeoIpProcessor.TYPE, (String)processorTag, (String)"properties", (String)e.getMessage());
                    }
                }
                properties = Collections.unmodifiableSet(modifiableProperties);
            } else if (databaseType.endsWith(GeoIpProcessor.CITY_DB_SUFFIX)) {
                properties = DEFAULT_CITY_PROPERTIES;
            } else if (databaseType.endsWith(GeoIpProcessor.COUNTRY_DB_SUFFIX)) {
                properties = DEFAULT_COUNTRY_PROPERTIES;
            } else if (databaseType.endsWith(GeoIpProcessor.ASN_DB_SUFFIX)) {
                properties = DEFAULT_ASN_PROPERTIES;
            } else {
                throw ConfigurationUtils.newConfigurationException((String)GeoIpProcessor.TYPE, (String)processorTag, (String)"database_file", (String)("Unsupported database type [" + databaseType + "]"));
            }
            return new GeoIpProcessor(processorTag, ipField, lazyLoader, targetField, properties, ignoreMissing, this.cache);
        }
    }
}

