import { MatrixClient } from "./MatrixClient";
/**
 * Represents a profile for a group
 */
export interface GroupProfile {
    /**
     * The name of the group
     */
    name: string;
    /**
     * The avatar for the group. Must be a MSC URI.
     */
    avatar_url: string;
    /**
     * The short description for the group. Equivalent to a room's topic.
     */
    short_description: string;
    /**
     * The long description for the group. Most clients will support HTML
     * in this.
     */
    long_description: string;
}
/**
 * Unstable APIs that shouldn't be used in most circumstances.
 */
export declare class UnstableApis {
    private client;
    constructor(client: MatrixClient);
    /**
     * Creates a group.
     * @param {string} localpart The localpart for the group
     * @return {Promise<string>} Resolves to the created group ID.
     */
    createGroup(localpart: string): Promise<string>;
    /**
     * Invites a user to the group.
     * @param {string} groupId The group ID to invite the user to.
     * @param {string} userId The user ID to invite to the group.
     * @return {Promise<"join" | "invite" | "reject">} Resolves to the invite state for
     *  the user. This is normally "invite", but may be "join" or "reject" if the user's
     *  homeserver accepted/rejected the invite right away.
     */
    inviteUserToGroup(groupId: string, userId: string): Promise<"join" | "invite" | "reject">;
    /**
     * Updates a group's profile
     * @param {string} groupId The group ID to update.
     * @param {GroupProfile} profile The profile to update the group with.
     * @return {Promise<*>} Resolves when completed.
     */
    setGroupProfile(groupId: string, profile: GroupProfile): Promise<any>;
    /**
     * Sets a group's join policy to either be publicly joinable (open) or
     * require an invite (invite).
     * @param {string} groupId The group ID to set the policy for.
     * @param {"open" | "invite"} policy The policy to set.
     * @return {Promise<*>} Resolves when completed.
     */
    setGroupJoinPolicy(groupId: string, policy: "open" | "invite"): Promise<any>;
    /**
     * Adds a reaction to an event. The contract for this function may change in the future.
     * @param {string} roomId The room ID to react in
     * @param {string} eventId The event ID to react against, in the given room
     * @param {string} emoji The emoji to react with
     * @returns {Promise<string>} Resolves to the event ID of the reaction
     */
    addReactionToEvent(roomId: string, eventId: string, emoji: string): Promise<string>;
}
