package Lire::Report::Section;

use strict;

use Carp;

use Lire::Utils qw/ xml_encode check_object_param /;

=pod

=head1 NAME

Lire::Section - Interface to the content of the section's element.

=head1 SYNOPSIS

    my $section = Lire::ReportParser::ReportBuilder( "report.xml" );

    foreach my $s ( $report->sections() ) {
        print "Section: '", $s->title(), "' has ", scalar $s->subreports(),
            " subreports in it\n";
    }

=head1 DESCRIPTION

This class offers an API to the section's elements of a Lire report.

=head1 CONSTRUCTOR

=head2 new( [$title] )

Creates a new Lire::Report::Section. The section's title will be set
to the $title parameter if present.

=cut

sub new {
    my ( $class, $title ) = @_;

    return bless( { '_title'        => $title || "Untitled",
                    '_subreports'   => [] },
                  $class );
}

=pod

=head1 OBJECT METHODS

=head2 title( [$title] )

Returns the section's title, if it has one. The section's title will
be changed $title if that parameter is set.

=cut

sub title {
    $_[0]{'_title'} = $_[1] if ( @_ == 2 );

    return $_[0]{'_title'};
}

=pod

=pod

=head2 description( [$description] )

Returns this section's description. The description is encoded in
DocBook XML.

If the $description parameter is set, this method will set this
section's description to this new value. If the $description parameter
is undef, that description will be removed.

=cut

sub description {
    $_[0]{'_description'} = $_[1] if ( @_ == 2 );

    return $_[0]->{'_description'};
}

=pod

=head2 subreports()

Returns the subreport's included in that section. This will be an array of
Lire::Report::Subreport objects.

=cut

sub subreports {
    return @{$_[0]{'_subreports'}};
}

=pod

=head2 add_subreport( $subreport )

Adds a subreport to this report. The $subreport parameter should be a
Lire::Report::Subreport object.

=cut

sub add_subreport {
    my ( $self, $subreport ) = @_;

    check_object_param( $subreport, 'subreport', 'Lire::Report::Subreport' );
    croak "can't add a Subreport which doesn't have an id"
      unless $subreport->id();

    push @{$self->{'_subreports'}}, $subreport;

    return;
}

=pod

=head2 subreports_by_type( $type )

Returns all the subreports in this section of a the type $type. The
subreport's type is the id of the report specification used to compute
the reports. The subreports are returned as an array of
Lire::Report::Subreport objects.

=cut

sub subreports_by_type {
    return grep { $_->type() eq $_[1] } $_[0]->subreports();
}

sub write_report {
    my ( $self, $fh, $indent ) = @_;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    print $fh qq"$pfx<lire:section>\n";

    print $fh "$pfx <lire:title>", xml_encode( $self->{'_title'} ),
      "</lire:title>\n\n";

    if ( $self->description() ) {
        print $fh "$pfx <lire:description>";
        print $fh $self->description, "</lire:description>\n\n";
    }

    foreach my $r ( $self->subreports() ) {
        $r->write_report( $fh, $indent + 1 );
    }

    print $fh "$pfx</lire:section>\n";

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report(3pm)
Lire::Report::Subreport(3pm) Lire::Report::Entry(3pm)
Lire::Report::Group(3pm)

=head1 VERSION

$Id: Section.pm,v 1.16 2006/07/23 13:16:31 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html. 

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
