/* gutil2.c: Some more graph utilities. */

#include "gtools.h"
#include "gutils.h"

/**************************************************************************/

long
pathcount1(graph *g, int start, setword body, setword last)
/* Number of paths in g starting at start, lying within body and
   ending in last.  {start} and last should be disjoint subsets of body. */
{
    long count;
    setword gs,w;
    int i;

    gs = g[start];
    w = gs & last;
    count = POPCOUNT(w);

    body &= ~bit[start];
    w = gs & body;
    while (w)
    {
	TAKEBIT(i,w);
	count += pathcount1(g,i,body,last&~bit[i]);
    }

    return count;
}

/**************************************************************************/

long
cyclecount1(graph *g, int n)
/* The total number of cycles in g (assumed no loops), m=1 only */
{
    setword body,nbhd;
    long total;
    int i,j;

    body = ALLMASK(n);
    total = 0;

    for (i = 0; i < n-2; ++i)
    {
	body ^= bit[i];
	nbhd = g[i] & body;
	while (nbhd)
	{
	    TAKEBIT(j,nbhd);
	    total += pathcount1(g,j,body,nbhd);
	}
    }

    return total;
}

/**************************************************************************/

long
cyclecount(graph *g, int m, int n)
/* The total number of cycles in g (assumed no loops) */
{
    if (m == 1) return cyclecount1(g,n);

    gt_abort("cycle counting is only implemented for n <= WORDSIZE");
    return 0;
}

/**************************************************************************/

void
commonnbrs(graph *g, int *minadj, int *maxadj, int *minnon, int *maxnon,
       int m, int n)
/* Count the common neighbours of pairs of vertices, and give the minimum
   and maximum for adjacent and non-adjacent vertices.  Undirected only.
   Null minimums are n+1 and null maximums are -1.
*/
{
    int j,k;
    int mina,maxa,minn,maxn;
    int cn;
    set *gi,*gj;
    setword w;

    mina = minn = n+1;
    maxa = maxn = -1;

    for (j = 0, gj = g; j < n; ++j, gj += m)
    for (gi = g; gi != gj; gi += m)
    {
	cn = 0;
	for (k = 0; k < m; ++k)
	{
	    w = gi[k] & gj[k];
	    if (w) cn += POPCOUNT(w);
	}

	if (ISELEMENT(gi,j))
	{
	    if (cn < mina) mina = cn;
	    if (cn > maxa) maxa = cn;
	}
	else
	{
	    if (cn < minn) minn = cn;
	    if (cn > maxn) maxn = cn;
	}
    }

    *minadj = mina;
    *maxadj = maxa;
    *minnon = minn;
    *maxnon = maxn;
}

/**************************************************************************/

void
delete1(graph *g, graph *h, int v, int n)
/* Delete vertex v from g, result in h */
{
    setword mask1,mask2,gi;
    int i;

    mask1 = ALLMASK(v);
    mask2 = BITMASK(v);

    for (i = 0; i < v; ++i)
    {
	gi = g[i];
	h[i] = (gi & mask1) | ((gi & mask2) << 1);
    }
    for (i = v; i < n-1; ++i)
    {
	gi = g[i+1];
	h[i] = (gi & mask1) | ((gi & mask2) << 1);
    }
}

/**************************************************************************/

void
contract1(graph *g, graph *h, int v, int w, int n)  
/* Contract distinct vertices v and w (not necessarily adjacent)
   with result in h.  No loops are created. */
{
    int x,y;
    setword bitx,bity,mask1,mask2;
    int i;

    if (w < v)
    {
	x = w; 
	y = v;
    }
    else
    {
	x = v; 
	y = w;
    }

    bitx = bit[x];
    bity = bit[y];
    mask1 = ALLMASK(y);
    mask2 = BITMASK(y);

    for (i = 0; i < n; ++i)
        if (g[i] & bity)
	    h[i] = (g[i] & mask1) | bitx | ((g[i] & mask2) << 1);
	else
	    h[i] = (g[i] & mask1) | ((g[i] & mask2) << 1);

    h[x] |= h[y];
    for (i = y+1; i < n; ++i) h[i-1] = h[i];
    h[x] &= ~bitx;
}

/**************************************************************************/

static int knm[16][16];  /* knm[n,m] = conncontent(K_n - m*K_2) */
static boolean knm_computed = FALSE;

int
conncontent(graph *g, int m, int n)
/* number of connected spanning subgraphs with an even number
   of edges minus the number with an odd number of edges */
{
    graph h[WORDSIZE];
    setword gj;
    int i,j,v1,v2,x,y;
    int minv,mindeg,deg,goodv;
    long ne;
    
    if (m > 1) ABORT("conncontent only implemented for m=1");

 /* First handle tiny graphs */

    if (n <= 3)
    {
	if (n == 1) return 1;
	if (n == 2) return (g[0] ? -1 : 0);
	if (!g[0] || !g[1] || !g[2]) return 0;    /* disconnected */
	if (g[0]^g[1]^g[2]) return 1;             /* path */
	return 2;                                 /* triangle */
    }

 /* Now compute
      ne = number of edges
      mindeg = minimum degree
      minv = a vertex of minimum degree
      goodv = a vertex with a clique neighbourhood (-1 if none)
 */

    mindeg = n;
    ne = 0;
    goodv = -1;
    for (j = 0; j < n; ++j)
    {
	gj = g[j];
	deg = POPCOUNT(gj);
	ne += deg;
	if (deg < mindeg)
	{
	    mindeg = deg;
	    minv = j;
            if (deg == 1) goodv = j;
	}
	if (deg >= 3 && deg <= 4 && goodv < 0)
	{
	    while (gj)
	    {
		TAKEBIT(i,gj);
		if (gj & ~g[i]) break;
	    }
	    if (!gj) goodv = j;
	}
    }
    ne /= 2;

/* Cases of isolated vertex or tree */

    if (mindeg == 0) return 0;

#if 0
    if (mindeg == 1 && ne == n-1)
    {
	if (isconnected1(g,n)) return ((n&1) ? 1 : -1);
	else		       return 0;
    }
#endif

/* Cases of clique and near-clique */

    if (mindeg == n-1)
    {
	j = -1;
	for (i = 2; i < n; ++i) j *= -i;
        return j;
    }

    if (mindeg == n-2 && n < 16)
    {
	if (!knm_computed)
	{
	    knm_computed = TRUE;
	    knm[1][0] = 1;
	    for (i = 2; i < 16; ++i)
	    {
		knm[i][0] = -knm[i-1][0] * (i-1);
		for (j = 1; j+j <= i; ++j)
		    knm[i][j] = knm[i][j-1] + knm[i-1][j-1];
	    }
	}
	return knm[n][(n*n-n)/2-ne];
    }

/*  Case of vertex with clique neighbourhood */

    if (goodv >= 0)
    {
	delete1(g,h,goodv,n);
	return -POPCOUNT(g[goodv]) * conncontent(h,m,n-1);
    }

/* Case of minimum degree 2 */

    if (mindeg == 2)
    {
	x = FIRSTBIT(g[minv]);
	y = FIRSTBIT(g[minv]^bit[x]);
	if (x > minv) --x;
	if (y > minv) --y;
	delete1(g,h,minv,n);
	v1 = conncontent(h,m,n-1);
	if (h[x] & bit[y]) return -2*v1;   /* adjacent neighbours */

	h[x] |= bit[y];
	h[y] |= bit[x];
	v2 = conncontent(h,m,n-1);
	return -v1 - v2;
    }

/* Case of more than 2/3 dense but not complete */

    if (3*ne > n*n-n) 
    {
	j = FIRSTBIT(g[minv] ^ bit[minv] ^ ALLMASK(n));   /* non-neighbour */

	g[minv] ^= bit[j];
        g[j] ^= bit[minv];
        v1 = conncontent(g,m,n);
        g[minv] ^= bit[j];
        g[j] ^= bit[minv];

        contract1(g,h,minv,j,n);
        v2 = conncontent(h,m,n-1);
    
        return v1 + v2;
    }
 
/* All remaining cases */

    j = FIRSTBIT(g[minv]);     /* neighbour */

    g[minv] ^= bit[j];
    g[j] ^= bit[minv];
    v1 = conncontent(g,m,n);
    g[minv] ^= bit[j];
    g[j] ^= bit[minv];
    
    contract1(g,h,minv,j,n);
    v2 = conncontent(h,m,n-1);

    return v1 - v2;
}
