'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _regenerator = require('babel-runtime/regenerator');

var _regenerator2 = _interopRequireDefault(_regenerator);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _createClass2 = require('babel-runtime/helpers/createClass');

var _createClass3 = _interopRequireDefault(_createClass2);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _utils = require('../utils');

var _utils2 = _interopRequireDefault(_utils);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Internal module. Management of outgoing room key requests.
 *
 * See https://docs.google.com/document/d/1m4gQkcnJkxNuBmb5NoFCIadIY-DyqqNAS3lloE73BlQ
 * for draft documentation on what we're supposed to be implementing here.
 *
 * @module
 */

// delay between deciding we want some keys, and sending out the request, to
// allow for (a) it turning up anyway, (b) grouping requests together
var SEND_KEY_REQUESTS_DELAY_MS = 500;

/** possible states for a room key request
 *
 * The state machine looks like:
 *
 *     |         (cancellation sent)
 *     | .-------------------------------------------------.
 *     | |                                                 |
 *     V V       (cancellation requested)                  |
 *   UNSENT  -----------------------------+                |
 *     |                                  |                |
 *     |                                  |                |
 *     | (send successful)                |  CANCELLATION_PENDING_AND_WILL_RESEND
 *     V                                  |                Λ
 *    SENT                                |                |
 *     |--------------------------------  |  --------------'
 *     |                                  |  (cancellation requested with intent
 *     |                                  |   to resend the original request)
 *     |                                  |
 *     | (cancellation requested)         |
 *     V                                  |
 * CANCELLATION_PENDING                   |
 *     |                                  |
 *     | (cancellation sent)              |
 *     V                                  |
 * (deleted)  <---------------------------+
 *
 * @enum {number}
 */
/*
Copyright 2017 Vector Creations Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

var ROOM_KEY_REQUEST_STATES = {
    /** request not yet sent */
    UNSENT: 0,

    /** request sent, awaiting reply */
    SENT: 1,

    /** reply received, cancellation not yet sent */
    CANCELLATION_PENDING: 2,

    /**
     * Cancellation not yet sent and will transition to UNSENT instead of
     * being deleted once the cancellation has been sent.
     */
    CANCELLATION_PENDING_AND_WILL_RESEND: 3
};

var OutgoingRoomKeyRequestManager = function () {
    function OutgoingRoomKeyRequestManager(baseApis, deviceId, cryptoStore) {
        (0, _classCallCheck3.default)(this, OutgoingRoomKeyRequestManager);

        this._baseApis = baseApis;
        this._deviceId = deviceId;
        this._cryptoStore = cryptoStore;

        // handle for the delayed call to _sendOutgoingRoomKeyRequests. Non-null
        // if the callback has been set, or if it is still running.
        this._sendOutgoingRoomKeyRequestsTimer = null;

        // sanity check to ensure that we don't end up with two concurrent runs
        // of _sendOutgoingRoomKeyRequests
        this._sendOutgoingRoomKeyRequestsRunning = false;

        this._clientRunning = false;
    }

    /**
     * Called when the client is started. Sets background processes running.
     */


    (0, _createClass3.default)(OutgoingRoomKeyRequestManager, [{
        key: 'start',
        value: function start() {
            this._clientRunning = true;

            // set the timer going, to handle any requests which didn't get sent
            // on the previous run of the client.
            this._startTimer();
        }

        /**
         * Called when the client is stopped. Stops any running background processes.
         */

    }, {
        key: 'stop',
        value: function stop() {
            _logger2.default.log('stopping OutgoingRoomKeyRequestManager');
            // stop the timer on the next run
            this._clientRunning = false;
        }

        /**
         * Send off a room key request, if we haven't already done so.
         *
         * The `requestBody` is compared (with a deep-equality check) against
         * previous queued or sent requests and if it matches, no change is made.
         * Otherwise, a request is added to the pending list, and a job is started
         * in the background to send it.
         *
         * @param {module:crypto~RoomKeyRequestBody} requestBody
         * @param {Array<{userId: string, deviceId: string}>} recipients
         * @param {boolean} resend whether to resend the key request if there is
         *    already one
         *
         * @returns {Promise} resolves when the request has been added to the
         *    pending list (or we have established that a similar request already
         *    exists)
         */

    }, {
        key: 'sendRoomKeyRequest',
        value: function () {
            var _ref = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee(requestBody, recipients) {
                var resend = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

                var req, state, _state, updatedReq;

                return _regenerator2.default.wrap(function _callee$(_context) {
                    while (1) {
                        switch (_context.prev = _context.next) {
                            case 0:
                                _context.next = 2;
                                return (0, _bluebird.resolve)(this._cryptoStore.getOutgoingRoomKeyRequest(requestBody));

                            case 2:
                                req = _context.sent;

                                if (req) {
                                    _context.next = 8;
                                    break;
                                }

                                _context.next = 6;
                                return (0, _bluebird.resolve)(this._cryptoStore.getOrAddOutgoingRoomKeyRequest({
                                    requestBody: requestBody,
                                    recipients: recipients,
                                    requestId: this._baseApis.makeTxnId(),
                                    state: ROOM_KEY_REQUEST_STATES.UNSENT
                                }));

                            case 6:
                                _context.next = 35;
                                break;

                            case 8:
                                _context.t0 = req.state;
                                _context.next = _context.t0 === ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND ? 11 : _context.t0 === ROOM_KEY_REQUEST_STATES.UNSENT ? 11 : _context.t0 === ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING ? 12 : _context.t0 === ROOM_KEY_REQUEST_STATES.SENT ? 16 : 34;
                                break;

                            case 11:
                                return _context.abrupt('return');

                            case 12:
                                // existing request is about to be cancelled.  If we want to
                                // resend, then change the state so that it resends after
                                // cancelling.  Otherwise, just cancel the cancellation.
                                state = resend ? ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND : ROOM_KEY_REQUEST_STATES.SENT;
                                _context.next = 15;
                                return (0, _bluebird.resolve)(this._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING, {
                                    state: state,
                                    cancellationTxnId: this._baseApis.makeTxnId()
                                }));

                            case 15:
                                return _context.abrupt('break', 35);

                            case 16:
                                if (!resend) {
                                    _context.next = 33;
                                    break;
                                }

                                _state = ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND;
                                _context.next = 20;
                                return (0, _bluebird.resolve)(this._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.SENT, {
                                    state: _state,
                                    cancellationTxnId: this._baseApis.makeTxnId(),
                                    // need to use a new transaction ID so that
                                    // the request gets sent
                                    requestTxnId: this._baseApis.makeTxnId()
                                }));

                            case 20:
                                updatedReq = _context.sent;

                                if (updatedReq) {
                                    _context.next = 25;
                                    break;
                                }

                                _context.next = 24;
                                return (0, _bluebird.resolve)(this.sendRoomKeyRequest(requestBody, recipients, resend));

                            case 24:
                                return _context.abrupt('return', _context.sent);

                            case 25:
                                _context.prev = 25;
                                _context.next = 28;
                                return (0, _bluebird.resolve)(this._sendOutgoingRoomKeyRequestCancellation(updatedReq, true));

                            case 28:
                                _context.next = 33;
                                break;

                            case 30:
                                _context.prev = 30;
                                _context.t1 = _context['catch'](25);

                                _logger2.default.error("Error sending room key request cancellation;" + " will retry later.", _context.t1);

                            case 33:
                                return _context.abrupt('break', 35);

                            case 34:
                                throw new Error('unhandled state: ' + req.state);

                            case 35:
                                // some of the branches require the timer to be started.  Just start it
                                // all the time, because it doesn't hurt to start it.
                                this._startTimer();

                            case 36:
                            case 'end':
                                return _context.stop();
                        }
                    }
                }, _callee, this, [[25, 30]]);
            }));

            function sendRoomKeyRequest(_x, _x2, _x3) {
                return _ref.apply(this, arguments);
            }

            return sendRoomKeyRequest;
        }()

        /**
         * Cancel room key requests, if any match the given requestBody
         *
         * @param {module:crypto~RoomKeyRequestBody} requestBody
         *
         * @returns {Promise} resolves when the request has been updated in our
         *    pending list.
         */

    }, {
        key: 'cancelRoomKeyRequest',
        value: function cancelRoomKeyRequest(requestBody) {
            var _this = this;

            return this._cryptoStore.getOutgoingRoomKeyRequest(requestBody).then(function (req) {
                if (!req) {
                    // no request was made for this key
                    return;
                }
                switch (req.state) {
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING:
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND:
                        // nothing to do here
                        return;

                    case ROOM_KEY_REQUEST_STATES.UNSENT:
                        // just delete it

                        // FIXME: ghahah we may have attempted to send it, and
                        // not yet got a successful response. So the server
                        // may have seen it, so we still need to send a cancellation
                        // in that case :/

                        _logger2.default.log('deleting unnecessary room key request for ' + stringifyRequestBody(requestBody));
                        return _this._cryptoStore.deleteOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.UNSENT);

                    case ROOM_KEY_REQUEST_STATES.SENT:
                        {
                            // send a cancellation.
                            return _this._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.SENT, {
                                state: ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING,
                                cancellationTxnId: _this._baseApis.makeTxnId()
                            }).then(function (updatedReq) {
                                if (!updatedReq) {
                                    // updateOutgoingRoomKeyRequest couldn't find the
                                    // request in state ROOM_KEY_REQUEST_STATES.SENT,
                                    // so we must have raced with another tab to mark
                                    // the request cancelled. There is no point in
                                    // sending another cancellation since the other tab
                                    // will do it.
                                    _logger2.default.log('Tried to cancel room key request for ' + stringifyRequestBody(requestBody) + ' but it was already cancelled in another tab');
                                    return;
                                }

                                // We don't want to wait for the timer, so we send it
                                // immediately. (We might actually end up racing with the timer,
                                // but that's ok: even if we make the request twice, we'll do it
                                // with the same transaction_id, so only one message will get
                                // sent).
                                //
                                // (We also don't want to wait for the response from the server
                                // here, as it will slow down processing of received keys if we
                                // do.)
                                _this._sendOutgoingRoomKeyRequestCancellation(updatedReq).catch(function (e) {
                                    _logger2.default.error("Error sending room key request cancellation;" + " will retry later.", e);
                                    _this._startTimer();
                                });
                            });
                        }
                    default:
                        throw new Error('unhandled state: ' + req.state);
                }
            });
        }

        /**
         * Look for room key requests by target device and state
         *
         * @param {string} userId Target user ID
         * @param {string} deviceId Target device ID
         *
         * @return {Promise} resolves to a list of all the
         *    {@link module:crypto/store/base~OutgoingRoomKeyRequest}
         */

    }, {
        key: 'getOutgoingSentRoomKeyRequest',
        value: function getOutgoingSentRoomKeyRequest(userId, deviceId) {
            return this._cryptoStore.getOutgoingRoomKeyRequestsByTarget(userId, deviceId, [ROOM_KEY_REQUEST_STATES.SENT]);
        }

        // start the background timer to send queued requests, if the timer isn't
        // already running

    }, {
        key: '_startTimer',
        value: function _startTimer() {
            var _this2 = this;

            if (this._sendOutgoingRoomKeyRequestsTimer) {
                return;
            }

            var startSendingOutgoingRoomKeyRequests = function startSendingOutgoingRoomKeyRequests() {
                if (_this2._sendOutgoingRoomKeyRequestsRunning) {
                    throw new Error("RoomKeyRequestSend already in progress!");
                }
                _this2._sendOutgoingRoomKeyRequestsRunning = true;

                _this2._sendOutgoingRoomKeyRequests().finally(function () {
                    _this2._sendOutgoingRoomKeyRequestsRunning = false;
                }).catch(function (e) {
                    // this should only happen if there is an indexeddb error,
                    // in which case we're a bit stuffed anyway.
                    _logger2.default.warn('error in OutgoingRoomKeyRequestManager: ' + e);
                });
            };

            this._sendOutgoingRoomKeyRequestsTimer = global.setTimeout(startSendingOutgoingRoomKeyRequests, SEND_KEY_REQUESTS_DELAY_MS);
        }

        // look for and send any queued requests. Runs itself recursively until
        // there are no more requests, or there is an error (in which case, the
        // timer will be restarted before the promise resolves).

    }, {
        key: '_sendOutgoingRoomKeyRequests',
        value: function _sendOutgoingRoomKeyRequests() {
            var _this3 = this;

            if (!this._clientRunning) {
                this._sendOutgoingRoomKeyRequestsTimer = null;
                return _bluebird2.default.resolve();
            }

            _logger2.default.log("Looking for queued outgoing room key requests");

            return this._cryptoStore.getOutgoingRoomKeyRequestByState([ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND, ROOM_KEY_REQUEST_STATES.UNSENT]).then(function (req) {
                if (!req) {
                    _logger2.default.log("No more outgoing room key requests");
                    _this3._sendOutgoingRoomKeyRequestsTimer = null;
                    return;
                }

                var prom = void 0;
                switch (req.state) {
                    case ROOM_KEY_REQUEST_STATES.UNSENT:
                        prom = _this3._sendOutgoingRoomKeyRequest(req);
                        break;
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING:
                        prom = _this3._sendOutgoingRoomKeyRequestCancellation(req);
                        break;
                    case ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND:
                        prom = _this3._sendOutgoingRoomKeyRequestCancellation(req, true);
                        break;
                }

                return prom.then(function () {
                    // go around the loop again
                    return _this3._sendOutgoingRoomKeyRequests();
                }).catch(function (e) {
                    _logger2.default.error("Error sending room key request; will retry later.", e);
                    _this3._sendOutgoingRoomKeyRequestsTimer = null;
                    _this3._startTimer();
                });
            });
        }

        // given a RoomKeyRequest, send it and update the request record

    }, {
        key: '_sendOutgoingRoomKeyRequest',
        value: function _sendOutgoingRoomKeyRequest(req) {
            var _this4 = this;

            _logger2.default.log('Requesting keys for ' + stringifyRequestBody(req.requestBody) + (' from ' + stringifyRecipientList(req.recipients)) + ('(id ' + req.requestId + ')'));

            var requestMessage = {
                action: "request",
                requesting_device_id: this._deviceId,
                request_id: req.requestId,
                body: req.requestBody
            };

            return this._sendMessageToDevices(requestMessage, req.recipients, req.requestTxnId || req.requestId).then(function () {
                return _this4._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.UNSENT, { state: ROOM_KEY_REQUEST_STATES.SENT });
            });
        }

        // Given a RoomKeyRequest, cancel it and delete the request record unless
        // andResend is set, in which case transition to UNSENT.

    }, {
        key: '_sendOutgoingRoomKeyRequestCancellation',
        value: function _sendOutgoingRoomKeyRequestCancellation(req, andResend) {
            var _this5 = this;

            _logger2.default.log('Sending cancellation for key request for ' + (stringifyRequestBody(req.requestBody) + ' to ') + (stringifyRecipientList(req.recipients) + ' ') + ('(cancellation id ' + req.cancellationTxnId + ')'));

            var requestMessage = {
                action: "request_cancellation",
                requesting_device_id: this._deviceId,
                request_id: req.requestId
            };

            return this._sendMessageToDevices(requestMessage, req.recipients, req.cancellationTxnId).then(function () {
                if (andResend) {
                    // We want to resend, so transition to UNSENT
                    return _this5._cryptoStore.updateOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING_AND_WILL_RESEND, { state: ROOM_KEY_REQUEST_STATES.UNSENT });
                }
                return _this5._cryptoStore.deleteOutgoingRoomKeyRequest(req.requestId, ROOM_KEY_REQUEST_STATES.CANCELLATION_PENDING);
            });
        }

        // send a RoomKeyRequest to a list of recipients

    }, {
        key: '_sendMessageToDevices',
        value: function _sendMessageToDevices(message, recipients, txnId) {
            var contentMap = {};
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = (0, _getIterator3.default)(recipients), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var recip = _step.value;

                    if (!contentMap[recip.userId]) {
                        contentMap[recip.userId] = {};
                    }
                    contentMap[recip.userId][recip.deviceId] = message;
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            return this._baseApis.sendToDevice('m.room_key_request', contentMap, txnId);
        }
    }]);
    return OutgoingRoomKeyRequestManager;
}();

exports.default = OutgoingRoomKeyRequestManager;


function stringifyRequestBody(requestBody) {
    // we assume that the request is for megolm keys, which are identified by
    // room id and session id
    return requestBody.room_id + " / " + requestBody.session_id;
}

function stringifyRecipientList(recipients) {
    return '[' + _utils2.default.map(recipients, function (r) {
        return r.userId + ':' + r.deviceId;
    }).join(",") + ']';
}
//# sourceMappingURL=OutgoingRoomKeyRequestManager.js.map