"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
/**
 * An Intent is an intelligent client that tracks things like the user's membership
 * in rooms to ensure the action being performed is possible. This is very similar
 * to how Intents work in the matrix-js-sdk in that the Intent will ensure that the
 * user is joined to the room before posting a message, for example.
 */
class Intent {
    /**
     * Creates a new intent. Intended to be created by application services.
     * @param {IAppserviceOptions} options The options for the application service.
     * @param {string} impersonateUserId The user ID to impersonate.
     * @param {Appservice} appservice The application service itself.
     */
    constructor(options, impersonateUserId, appservice) {
        this.impersonateUserId = impersonateUserId;
        this.appservice = appservice;
        this.knownJoinedRooms = [];
        this.storage = options.storage;
        this.client = new __1.MatrixClient(options.homeserverUrl, options.registration.as_token);
        if (impersonateUserId !== appservice.botUserId)
            this.client.impersonateUserId(impersonateUserId);
        if (options.joinStrategy)
            this.client.setJoinStrategy(options.joinStrategy);
    }
    /**
     * Gets the user ID this intent is for.
     */
    get userId() {
        return this.impersonateUserId;
    }
    /**
     * Gets the underlying MatrixClient that powers this Intent.
     */
    get underlyingClient() {
        return this.client;
    }
    /**
     * Gets the joined rooms for the intent. Note that by working around
     * the intent to join rooms may yield inaccurate results.
     * @returns {Promise<string[]>} Resolves to an array of room IDs where
     * the intent is joined.
     */
    getJoinedRooms() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureRegistered();
            if (this.knownJoinedRooms.length === 0)
                yield this.refreshJoinedRooms();
            return this.knownJoinedRooms.map(r => r); // clone
        });
    }
    /**
     * Leaves the given room.
     * @param {string} roomId The room ID to leave
     * @returns {Promise<*>} Resolves when the room has been left.
     */
    leaveRoom(roomId) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureRegistered();
            return this.client.leaveRoom(roomId).then(() => __awaiter(this, void 0, void 0, function* () {
                // Recalculate joined rooms now that we've left a room
                yield this.refreshJoinedRooms();
            }));
        });
    }
    /**
     * Joins the given room
     * @param {string} roomIdOrAlias the room ID or alias to join
     * @returns {Promise<string>} resolves to the joined room ID
     */
    joinRoom(roomIdOrAlias) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureRegistered();
            return this.client.joinRoom(roomIdOrAlias).then((roomId) => __awaiter(this, void 0, void 0, function* () {
                // Recalculate joined rooms now that we've joined a room
                yield this.refreshJoinedRooms();
                return roomId;
            }));
        });
    }
    /**
     * Sends a text message to a room.
     * @param {string} roomId The room ID to send text to.
     * @param {string} body The message body to send.
     * @param {"m.text" | "m.emote" | "m.notice"} msgtype The message type to send.
     * @returns {Promise<string>} Resolves to the event ID of the sent message.
     */
    sendText(roomId, body, msgtype = "m.text") {
        return __awaiter(this, void 0, void 0, function* () {
            return this.sendEvent(roomId, { body: body, msgtype: msgtype });
        });
    }
    /**
     * Sends an event to a room.
     * @param {string} roomId The room ID to send the event to.
     * @param {*} content The content of the event.
     * @returns {Promise<string>} Resolves to the event ID of the sent event.
     */
    sendEvent(roomId, content) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureRegisteredAndJoined(roomId);
            return this.client.sendMessage(roomId, content);
        });
    }
    /**
     * Ensures the user is registered and joined to the given room.
     * @param {string} roomId The room ID to join
     * @returns {Promise<*>} Resolves when complete
     */
    ensureRegisteredAndJoined(roomId) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureRegistered();
            yield this.ensureJoined(roomId);
        });
    }
    /**
     * Ensures the user is joined to the given room
     * @param {string} roomId The room ID to join
     * @returns {Promise<*>} Resolves when complete
     */
    ensureJoined(roomId) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.knownJoinedRooms.indexOf(roomId) !== -1) {
                return;
            }
            yield this.refreshJoinedRooms();
            if (this.knownJoinedRooms.indexOf(roomId) !== -1) {
                return;
            }
            const returnedRoomId = yield this.client.joinRoom(roomId);
            if (!this.knownJoinedRooms.includes(returnedRoomId)) {
                this.knownJoinedRooms.push(returnedRoomId);
            }
            return returnedRoomId;
        });
    }
    /**
     * Refreshes which rooms the user is joined to, potentially saving time on
     * calls like ensureJoined()
     * @returns {Promise<string[]>} Resolves to the joined room IDs for the user.
     */
    refreshJoinedRooms() {
        return __awaiter(this, void 0, void 0, function* () {
            this.knownJoinedRooms = yield this.client.getJoinedRooms();
            return this.knownJoinedRooms.map(r => r); // clone
        });
    }
    /**
     * Ensures the user is registered
     * @returns {Promise<*>} Resolves when complete
     */
    ensureRegistered() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.storage.isUserRegistered(this.userId)) {
                try {
                    const result = yield this.client.doRequest("POST", "/_matrix/client/r0/register", null, {
                        type: "m.login.application_service",
                        username: this.userId.substring(1).split(":")[0],
                    });
                    // HACK: Workaround for unit tests
                    if (result['errcode']) {
                        // noinspection ExceptionCaughtLocallyJS
                        throw { body: result };
                    }
                }
                catch (err) {
                    if (typeof (err.body) === "string")
                        err.body = JSON.parse(err.body);
                    if (err.body && err.body["errcode"] === "M_USER_IN_USE") {
                        this.storage.addRegisteredUser(this.userId);
                        if (this.userId === this.appservice.botUserId) {
                            return null;
                        }
                        else {
                            __1.LogService.error("Appservice", "Error registering user: User ID is in use");
                            return null;
                        }
                    }
                    else {
                        __1.LogService.error("Appservice", "Encountered error registering user: ");
                        __1.LogService.error("Appservice", err);
                    }
                    throw err;
                }
                this.storage.addRegisteredUser(this.userId);
            }
        });
    }
}
exports.Intent = Intent;
