<?php
declare(strict_types=1);
/**
 *
 * @copyright Copyright (c) 2018, Daniel Calviño Sánchez (danxuliu@gmail.com)
 *
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

namespace OCA\Talk\PublicShareAuth;

use OCP\EventDispatcher\IEventDispatcher;
use OCP\Share\IShare;
use OCP\Util;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Helper class to extend the "publicshareauth" template from the server.
 *
 * The "loadRequestPasswordByTalkUi" method loads additional scripts that, when
 * run on the browser, adjust the page generated by the server to inject the
 * Talk UI as needed.
 */
class TemplateLoader {

	public static function register(IEventDispatcher $dispatcher): void {
		$listener = function(GenericEvent $event) {
			/** @var IShare $share */
			$share = $event->getArgument('share');
			self::loadRequestPasswordByTalkUi($share);
		};
		$dispatcher->addListener('OCA\Files_Sharing::loadAdditionalScripts::publicShareAuth', $listener);
	}

	/**
	 * Load the "Request password by Talk" UI in the public share authentication
	 * page for the given share.
	 *
	 * If the "Send Password by Talk" option is not set in the share then no UI
	 * to request the password is provided.
	 *
	 * This method should be called when loading additional scripts for the
	 * public share authentication page of the server.
	 *
	 * @param IShare $share
	 */
	public static function loadRequestPasswordByTalkUi(IShare $share): void {
		if (!$share->getSendPasswordByTalk()) {
			return;
		}

		Util::addStyle('spreed', 'merged-share-auth');
		Util::addScript('spreed', 'talk-public-share-auth-sidebar');

		// Needed to enable the screensharing extension in Chromium < 72.
		Util::addHeader('meta', ['id' => "app", 'class' => 'nc-enable-screensharing-extension']);
	}

}
