/*
Copyright 2017 Vector Creations Ltd
Copyright 2018, 2019 New Vector Ltd
Copyright 2019 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
"use strict";

/**
 * @module crypto/DeviceList
 *
 * Manages the list of other users' devices
 */

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _slicedToArray2 = require('babel-runtime/helpers/slicedToArray');

var _slicedToArray3 = _interopRequireDefault(_slicedToArray2);

var _entries = require('babel-runtime/core-js/object/entries');

var _entries2 = _interopRequireDefault(_entries);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _regenerator = require('babel-runtime/regenerator');

var _regenerator2 = _interopRequireDefault(_regenerator);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _getPrototypeOf = require('babel-runtime/core-js/object/get-prototype-of');

var _getPrototypeOf2 = _interopRequireDefault(_getPrototypeOf);

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _createClass2 = require('babel-runtime/helpers/createClass');

var _createClass3 = _interopRequireDefault(_createClass2);

var _possibleConstructorReturn2 = require('babel-runtime/helpers/possibleConstructorReturn');

var _possibleConstructorReturn3 = _interopRequireDefault(_possibleConstructorReturn2);

var _inherits2 = require('babel-runtime/helpers/inherits');

var _inherits3 = _interopRequireDefault(_inherits2);

var _updateStoredDeviceKeysForUser = function () {
    var _ref4 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee3(_olmDevice, userId, userStore, userResult) {
        var updated, deviceId, _deviceId, deviceResult;

        return _regenerator2.default.wrap(function _callee3$(_context3) {
            while (1) {
                switch (_context3.prev = _context3.next) {
                    case 0:
                        updated = false;

                        // remove any devices in the store which aren't in the response

                        _context3.t0 = _regenerator2.default.keys(userStore);

                    case 2:
                        if ((_context3.t1 = _context3.t0()).done) {
                            _context3.next = 9;
                            break;
                        }

                        deviceId = _context3.t1.value;

                        if (userStore.hasOwnProperty(deviceId)) {
                            _context3.next = 6;
                            break;
                        }

                        return _context3.abrupt('continue', 2);

                    case 6:

                        if (!(deviceId in userResult)) {
                            _logger2.default.log("Device " + userId + ":" + deviceId + " has been removed");
                            delete userStore[deviceId];
                            updated = true;
                        }
                        _context3.next = 2;
                        break;

                    case 9:
                        _context3.t2 = _regenerator2.default.keys(userResult);

                    case 10:
                        if ((_context3.t3 = _context3.t2()).done) {
                            _context3.next = 27;
                            break;
                        }

                        _deviceId = _context3.t3.value;

                        if (userResult.hasOwnProperty(_deviceId)) {
                            _context3.next = 14;
                            break;
                        }

                        return _context3.abrupt('continue', 10);

                    case 14:
                        deviceResult = userResult[_deviceId];

                        // check that the user_id and device_id in the response object are
                        // correct

                        if (!(deviceResult.user_id !== userId)) {
                            _context3.next = 18;
                            break;
                        }

                        _logger2.default.warn("Mismatched user_id " + deviceResult.user_id + " in keys from " + userId + ":" + _deviceId);
                        return _context3.abrupt('continue', 10);

                    case 18:
                        if (!(deviceResult.device_id !== _deviceId)) {
                            _context3.next = 21;
                            break;
                        }

                        _logger2.default.warn("Mismatched device_id " + deviceResult.device_id + " in keys from " + userId + ":" + _deviceId);
                        return _context3.abrupt('continue', 10);

                    case 21:
                        _context3.next = 23;
                        return (0, _bluebird.resolve)(_storeDeviceKeys(_olmDevice, userStore, deviceResult));

                    case 23:
                        if (!_context3.sent) {
                            _context3.next = 25;
                            break;
                        }

                        updated = true;

                    case 25:
                        _context3.next = 10;
                        break;

                    case 27:
                        return _context3.abrupt('return', updated);

                    case 28:
                    case 'end':
                        return _context3.stop();
                }
            }
        }, _callee3, this);
    }));

    return function _updateStoredDeviceKeysForUser(_x6, _x7, _x8, _x9) {
        return _ref4.apply(this, arguments);
    };
}();

/*
 * Process a device in a /query response, and add it to the userStore
 *
 * returns (a promise for) true if a change was made, else false
 */


var _storeDeviceKeys = function () {
    var _ref5 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee4(_olmDevice, userStore, deviceResult) {
        var deviceId, userId, signKeyId, signKey, unsigned, signatures, deviceStore;
        return _regenerator2.default.wrap(function _callee4$(_context4) {
            while (1) {
                switch (_context4.prev = _context4.next) {
                    case 0:
                        if (deviceResult.keys) {
                            _context4.next = 2;
                            break;
                        }

                        return _context4.abrupt('return', false);

                    case 2:
                        deviceId = deviceResult.device_id;
                        userId = deviceResult.user_id;
                        signKeyId = "ed25519:" + deviceId;
                        signKey = deviceResult.keys[signKeyId];

                        if (signKey) {
                            _context4.next = 9;
                            break;
                        }

                        _logger2.default.warn("Device " + userId + ":" + deviceId + " has no ed25519 key");
                        return _context4.abrupt('return', false);

                    case 9:
                        unsigned = deviceResult.unsigned || {};
                        signatures = deviceResult.signatures || {};
                        _context4.prev = 11;
                        _context4.next = 14;
                        return (0, _bluebird.resolve)(_olmlib2.default.verifySignature(_olmDevice, deviceResult, userId, deviceId, signKey));

                    case 14:
                        _context4.next = 20;
                        break;

                    case 16:
                        _context4.prev = 16;
                        _context4.t0 = _context4['catch'](11);

                        _logger2.default.warn("Unable to verify signature on device " + userId + ":" + deviceId + ":" + _context4.t0);
                        return _context4.abrupt('return', false);

                    case 20:

                        // DeviceInfo
                        deviceStore = void 0;

                        if (!(deviceId in userStore)) {
                            _context4.next = 28;
                            break;
                        }

                        // already have this device.
                        deviceStore = userStore[deviceId];

                        if (!(deviceStore.getFingerprint() != signKey)) {
                            _context4.next = 26;
                            break;
                        }

                        // this should only happen if the list has been MITMed; we are
                        // best off sticking with the original keys.
                        //
                        // Should we warn the user about it somehow?
                        _logger2.default.warn("Ed25519 key for device " + userId + ":" + deviceId + " has changed");
                        return _context4.abrupt('return', false);

                    case 26:
                        _context4.next = 29;
                        break;

                    case 28:
                        userStore[deviceId] = deviceStore = new _deviceinfo2.default(deviceId);

                    case 29:

                        deviceStore.keys = deviceResult.keys || {};
                        deviceStore.algorithms = deviceResult.algorithms || [];
                        deviceStore.unsigned = unsigned;
                        deviceStore.signatures = signatures;
                        return _context4.abrupt('return', true);

                    case 34:
                    case 'end':
                        return _context4.stop();
                }
            }
        }, _callee4, this, [[11, 16]]);
    }));

    return function _storeDeviceKeys(_x10, _x11, _x12) {
        return _ref5.apply(this, arguments);
    };
}();

var _events = require('events');

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _deviceinfo = require('./deviceinfo');

var _deviceinfo2 = _interopRequireDefault(_deviceinfo);

var _CrossSigning = require('./CrossSigning');

var _olmlib = require('./olmlib');

var _olmlib2 = _interopRequireDefault(_olmlib);

var _indexeddbCryptoStore = require('./store/indexeddb-crypto-store');

var _indexeddbCryptoStore2 = _interopRequireDefault(_indexeddbCryptoStore);

var _utils = require('../utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/* State transition diagram for DeviceList._deviceTrackingStatus
 *
 *                                |
 *     stopTrackingDeviceList     V
 *   +---------------------> NOT_TRACKED
 *   |                            |
 *   +<--------------------+      | startTrackingDeviceList
 *   |                     |      V
 *   |   +-------------> PENDING_DOWNLOAD <--------------------+-+
 *   |   |                      ^ |                            | |
 *   |   | restart     download | |  start download            | | invalidateUserDeviceList
 *   |   | client        failed | |                            | |
 *   |   |                      | V                            | |
 *   |   +------------ DOWNLOAD_IN_PROGRESS -------------------+ |
 *   |                    |       |                              |
 *   +<-------------------+       |  download successful         |
 *   ^                            V                              |
 *   +----------------------- UP_TO_DATE ------------------------+
 */

// constants for DeviceList._deviceTrackingStatus
var TRACKING_STATUS_NOT_TRACKED = 0;
var TRACKING_STATUS_PENDING_DOWNLOAD = 1;
var TRACKING_STATUS_DOWNLOAD_IN_PROGRESS = 2;
var TRACKING_STATUS_UP_TO_DATE = 3;

/**
 * @alias module:crypto/DeviceList
 */

var DeviceList = function (_EventEmitter) {
    (0, _inherits3.default)(DeviceList, _EventEmitter);

    function DeviceList(baseApis, cryptoStore, olmDevice) {
        (0, _classCallCheck3.default)(this, DeviceList);

        var _this = (0, _possibleConstructorReturn3.default)(this, (DeviceList.__proto__ || (0, _getPrototypeOf2.default)(DeviceList)).call(this));

        _this._cryptoStore = cryptoStore;

        // userId -> {
        //     deviceId -> {
        //         [device info]
        //     }
        // }
        _this._devices = {};

        // userId -> {
        //     [key info]
        // }
        _this._crossSigningInfo = {};

        // map of identity keys to the user who owns it
        _this._userByIdentityKey = {};

        // which users we are tracking device status for.
        // userId -> TRACKING_STATUS_*
        _this._deviceTrackingStatus = {}; // loaded from storage in load()

        // The 'next_batch' sync token at the point the data was writen,
        // ie. a token representing the point immediately after the
        // moment represented by the snapshot in the db.
        _this._syncToken = null;

        _this._serialiser = new DeviceListUpdateSerialiser(baseApis, olmDevice, _this);

        // userId -> promise
        _this._keyDownloadsInProgressByUser = {};

        // Set whenever changes are made other than setting the sync token
        _this._dirty = false;

        // Promise resolved when device data is saved
        _this._savePromise = null;
        // Function that resolves the save promise
        _this._resolveSavePromise = null;
        // The time the save is scheduled for
        _this._savePromiseTime = null;
        // The timer used to delay the save
        _this._saveTimer = null;
        return _this;
    }

    /**
     * Load the device tracking state from storage
     */


    (0, _createClass3.default)(DeviceList, [{
        key: 'load',
        value: function () {
            var _ref = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee() {
                var _this2 = this;

                var _iteratorNormalCompletion3, _didIteratorError3, _iteratorError3, _iterator3, _step3, u;

                return _regenerator2.default.wrap(function _callee$(_context) {
                    while (1) {
                        switch (_context.prev = _context.next) {
                            case 0:
                                _context.next = 2;
                                return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_DEVICE_DATA], function (txn) {
                                    _this2._cryptoStore.getEndToEndDeviceData(txn, function (deviceData) {
                                        _this2._devices = deviceData ? deviceData.devices : {}, _this2._ssks = deviceData ? deviceData.self_signing_keys || {} : {};
                                        _this2._deviceTrackingStatus = deviceData ? deviceData.trackingStatus : {};
                                        _this2._syncToken = deviceData ? deviceData.syncToken : null;
                                        _this2._userByIdentityKey = {};
                                        var _iteratorNormalCompletion = true;
                                        var _didIteratorError = false;
                                        var _iteratorError = undefined;

                                        try {
                                            for (var _iterator = (0, _getIterator3.default)((0, _keys2.default)(_this2._devices)), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                                                var user = _step.value;

                                                var userDevices = _this2._devices[user];
                                                var _iteratorNormalCompletion2 = true;
                                                var _didIteratorError2 = false;
                                                var _iteratorError2 = undefined;

                                                try {
                                                    for (var _iterator2 = (0, _getIterator3.default)((0, _keys2.default)(userDevices)), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                                                        var device = _step2.value;

                                                        var idKey = userDevices[device].keys['curve25519:' + device];
                                                        if (idKey !== undefined) {
                                                            _this2._userByIdentityKey[idKey] = user;
                                                        }
                                                    }
                                                } catch (err) {
                                                    _didIteratorError2 = true;
                                                    _iteratorError2 = err;
                                                } finally {
                                                    try {
                                                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                                                            _iterator2.return();
                                                        }
                                                    } finally {
                                                        if (_didIteratorError2) {
                                                            throw _iteratorError2;
                                                        }
                                                    }
                                                }
                                            }
                                        } catch (err) {
                                            _didIteratorError = true;
                                            _iteratorError = err;
                                        } finally {
                                            try {
                                                if (!_iteratorNormalCompletion && _iterator.return) {
                                                    _iterator.return();
                                                }
                                            } finally {
                                                if (_didIteratorError) {
                                                    throw _iteratorError;
                                                }
                                            }
                                        }
                                    });
                                }));

                            case 2:
                                _iteratorNormalCompletion3 = true;
                                _didIteratorError3 = false;
                                _iteratorError3 = undefined;
                                _context.prev = 5;


                                for (_iterator3 = (0, _getIterator3.default)((0, _keys2.default)(this._deviceTrackingStatus)); !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
                                    u = _step3.value;

                                    // if a download was in progress when we got shut down, it isn't any more.
                                    if (this._deviceTrackingStatus[u] == TRACKING_STATUS_DOWNLOAD_IN_PROGRESS) {
                                        this._deviceTrackingStatus[u] = TRACKING_STATUS_PENDING_DOWNLOAD;
                                    }
                                }
                                _context.next = 13;
                                break;

                            case 9:
                                _context.prev = 9;
                                _context.t0 = _context['catch'](5);
                                _didIteratorError3 = true;
                                _iteratorError3 = _context.t0;

                            case 13:
                                _context.prev = 13;
                                _context.prev = 14;

                                if (!_iteratorNormalCompletion3 && _iterator3.return) {
                                    _iterator3.return();
                                }

                            case 16:
                                _context.prev = 16;

                                if (!_didIteratorError3) {
                                    _context.next = 19;
                                    break;
                                }

                                throw _iteratorError3;

                            case 19:
                                return _context.finish(16);

                            case 20:
                                return _context.finish(13);

                            case 21:
                            case 'end':
                                return _context.stop();
                        }
                    }
                }, _callee, this, [[5, 9, 13, 21], [14,, 16, 20]]);
            }));

            function load() {
                return _ref.apply(this, arguments);
            }

            return load;
        }()
    }, {
        key: 'stop',
        value: function stop() {
            if (this._saveTimer !== null) {
                clearTimeout(this._saveTimer);
            }
        }

        /**
         * Save the device tracking state to storage, if any changes are
         * pending other than updating the sync token
         *
         * The actual save will be delayed by a short amount of time to
         * aggregate multiple writes to the database.
         *
         * @param {integer} delay Time in ms before which the save actually happens.
         *     By default, the save is delayed for a short period in order to batch
         *     multiple writes, but this behaviour can be disabled by passing 0.
         *
         * @return {Promise<bool>} true if the data was saved, false if
         *     it was not (eg. because no changes were pending). The promise
         *     will only resolve once the data is saved, so may take some time
         *     to resolve.
         */

    }, {
        key: 'saveIfDirty',
        value: function () {
            var _ref2 = (0, _bluebird.method)(function (delay) {
                var _this3 = this;

                if (!this._dirty) return _bluebird2.default.resolve(false);
                // Delay saves for a bit so we can aggregate multiple saves that happen
                // in quick succession (eg. when a whole room's devices are marked as known)
                if (delay === undefined) delay = 500;

                var targetTime = Date.now + delay;
                if (this._savePromiseTime && targetTime < this._savePromiseTime) {
                    // There's a save scheduled but for after we would like: cancel
                    // it & schedule one for the time we want
                    clearTimeout(this._saveTimer);
                    this._saveTimer = null;
                    this._savePromiseTime = null;
                    // (but keep the save promise since whatever called save before
                    // will still want to know when the save is done)
                }

                var savePromise = this._savePromise;
                if (savePromise === null) {
                    savePromise = new _bluebird2.default(function (resolve, reject) {
                        _this3._resolveSavePromise = resolve;
                    });
                    this._savePromise = savePromise;
                }

                if (this._saveTimer === null) {
                    var resolveSavePromise = this._resolveSavePromise;
                    this._savePromiseTime = targetTime;
                    this._saveTimer = setTimeout(function () {
                        _logger2.default.log('Saving device tracking data at token ' + _this3._syncToken);
                        // null out savePromise now (after the delay but before the write),
                        // otherwise we could return the existing promise when the save has
                        // actually already happened. Likewise for the dirty flag.
                        _this3._savePromiseTime = null;
                        _this3._saveTimer = null;
                        _this3._savePromise = null;
                        _this3._resolveSavePromise = null;

                        _this3._dirty = false;
                        _this3._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_DEVICE_DATA], function (txn) {
                            _this3._cryptoStore.storeEndToEndDeviceData({
                                devices: _this3._devices,
                                self_signing_keys: _this3._ssks,
                                trackingStatus: _this3._deviceTrackingStatus,
                                syncToken: _this3._syncToken
                            }, txn);
                        }).then(function () {
                            resolveSavePromise();
                        });
                    }, delay);
                }
                return savePromise;
            });

            function saveIfDirty(_x) {
                return _ref2.apply(this, arguments);
            }

            return saveIfDirty;
        }()

        /**
         * Gets the sync token last set with setSyncToken
         *
         * @return {string} The sync token
         */

    }, {
        key: 'getSyncToken',
        value: function getSyncToken() {
            return this._syncToken;
        }

        /**
         * Sets the sync token that the app will pass as the 'since' to the /sync
         * endpoint next time it syncs.
         * The sync token must always be set after any changes made as a result of
         * data in that sync since setting the sync token to a newer one will mean
         * those changed will not be synced from the server if a new client starts
         * up with that data.
         *
         * @param {string} st The sync token
         */

    }, {
        key: 'setSyncToken',
        value: function setSyncToken(st) {
            this._syncToken = st;
        }

        /**
         * Ensures up to date keys for a list of users are stored in the session store,
         * downloading and storing them if they're not (or if forceDownload is
         * true).
         * @param {Array} userIds The users to fetch.
         * @param {bool} forceDownload Always download the keys even if cached.
         *
         * @return {Promise} A promise which resolves to a map userId->deviceId->{@link
         * module:crypto/deviceinfo|DeviceInfo}.
         */

    }, {
        key: 'downloadKeys',
        value: function downloadKeys(userIds, forceDownload) {
            var _this4 = this;

            var usersToDownload = [];
            var promises = [];

            userIds.forEach(function (u) {
                var trackingStatus = _this4._deviceTrackingStatus[u];
                if (_this4._keyDownloadsInProgressByUser[u]) {
                    // already a key download in progress/queued for this user; its results
                    // will be good enough for us.
                    _logger2.default.log('downloadKeys: already have a download in progress for ' + (u + ': awaiting its result'));
                    promises.push(_this4._keyDownloadsInProgressByUser[u]);
                } else if (forceDownload || trackingStatus != TRACKING_STATUS_UP_TO_DATE) {
                    usersToDownload.push(u);
                }
            });

            if (usersToDownload.length != 0) {
                _logger2.default.log("downloadKeys: downloading for", usersToDownload);
                var downloadPromise = this._doKeyDownload(usersToDownload);
                promises.push(downloadPromise);
            }

            if (promises.length === 0) {
                _logger2.default.log("downloadKeys: already have all necessary keys");
            }

            return _bluebird2.default.all(promises).then(function () {
                return _this4._getDevicesFromStore(userIds);
            });
        }

        /**
         * Get the stored device keys for a list of user ids
         *
         * @param {string[]} userIds the list of users to list keys for.
         *
         * @return {Object} userId->deviceId->{@link module:crypto/deviceinfo|DeviceInfo}.
         */

    }, {
        key: '_getDevicesFromStore',
        value: function _getDevicesFromStore(userIds) {
            var stored = {};
            var self = this;
            userIds.map(function (u) {
                stored[u] = {};
                var devices = self.getStoredDevicesForUser(u) || [];
                devices.map(function (dev) {
                    stored[u][dev.deviceId] = dev;
                });
            });
            return stored;
        }

        /**
         * Get the stored device keys for a user id
         *
         * @param {string} userId the user to list keys for.
         *
         * @return {module:crypto/deviceinfo[]|null} list of devices, or null if we haven't
         * managed to get a list of devices for this user yet.
         */

    }, {
        key: 'getStoredDevicesForUser',
        value: function getStoredDevicesForUser(userId) {
            var devs = this._devices[userId];
            if (!devs) {
                return null;
            }
            var res = [];
            for (var deviceId in devs) {
                if (devs.hasOwnProperty(deviceId)) {
                    res.push(_deviceinfo2.default.fromStorage(devs[deviceId], deviceId));
                }
            }
            return res;
        }

        /**
         * Get the stored device data for a user, in raw object form
         *
         * @param {string} userId the user to get data for
         *
         * @return {Object} deviceId->{object} devices, or undefined if
         * there is no data for this user.
         */

    }, {
        key: 'getRawStoredDevicesForUser',
        value: function getRawStoredDevicesForUser(userId) {
            return this._devices[userId];
        }
    }, {
        key: 'getStoredCrossSigningForUser',
        value: function getStoredCrossSigningForUser(userId) {
            if (!this._crossSigningInfo[userId]) return null;

            return _CrossSigning.CrossSigningInfo.fromStorage(this._crossSigningInfo[userId], userId);
        }
    }, {
        key: 'storeCrossSigningForUser',
        value: function storeCrossSigningForUser(userId, info) {
            this._crossSigningInfo[userId] = info;
            this._dirty = true;
        }

        /**
         * Get the stored keys for a single device
         *
         * @param {string} userId
         * @param {string} deviceId
         *
         * @return {module:crypto/deviceinfo?} device, or undefined
         * if we don't know about this device
         */

    }, {
        key: 'getStoredDevice',
        value: function getStoredDevice(userId, deviceId) {
            var devs = this._devices[userId];
            if (!devs || !devs[deviceId]) {
                return undefined;
            }
            return _deviceinfo2.default.fromStorage(devs[deviceId], deviceId);
        }

        /**
         * Find a device by curve25519 identity key
         *
         * @param {string} algorithm  encryption algorithm
         * @param {string} senderKey  curve25519 key to match
         *
         * @return {module:crypto/deviceinfo?}
         */

    }, {
        key: 'getDeviceByIdentityKey',
        value: function getDeviceByIdentityKey(algorithm, senderKey) {
            var userId = this._userByIdentityKey[senderKey];
            if (!userId) {
                return null;
            }

            if (algorithm !== _olmlib2.default.OLM_ALGORITHM && algorithm !== _olmlib2.default.MEGOLM_ALGORITHM) {
                // we only deal in olm keys
                return null;
            }

            var devices = this._devices[userId];
            if (!devices) {
                return null;
            }

            for (var deviceId in devices) {
                if (!devices.hasOwnProperty(deviceId)) {
                    continue;
                }

                var device = devices[deviceId];
                for (var keyId in device.keys) {
                    if (!device.keys.hasOwnProperty(keyId)) {
                        continue;
                    }
                    if (keyId.indexOf("curve25519:") !== 0) {
                        continue;
                    }
                    var deviceKey = device.keys[keyId];
                    if (deviceKey == senderKey) {
                        return _deviceinfo2.default.fromStorage(device, deviceId);
                    }
                }
            }

            // doesn't match a known device
            return null;
        }

        /**
         * Replaces the list of devices for a user with the given device list
         *
         * @param {string} u The user ID
         * @param {Object} devs New device info for user
         */

    }, {
        key: 'storeDevicesForUser',
        value: function storeDevicesForUser(u, devs) {
            // remove previous devices from _userByIdentityKey
            if (this._devices[u] !== undefined) {
                var _iteratorNormalCompletion4 = true;
                var _didIteratorError4 = false;
                var _iteratorError4 = undefined;

                try {
                    for (var _iterator4 = (0, _getIterator3.default)((0, _entries2.default)(this._devices[u])), _step4; !(_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done); _iteratorNormalCompletion4 = true) {
                        var _step4$value = (0, _slicedToArray3.default)(_step4.value, 2),
                            deviceId = _step4$value[0],
                            dev = _step4$value[1];

                        var identityKey = dev.keys['curve25519:' + deviceId];

                        delete this._userByIdentityKey[identityKey];
                    }
                } catch (err) {
                    _didIteratorError4 = true;
                    _iteratorError4 = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion4 && _iterator4.return) {
                            _iterator4.return();
                        }
                    } finally {
                        if (_didIteratorError4) {
                            throw _iteratorError4;
                        }
                    }
                }
            }

            this._devices[u] = devs;

            // add new ones
            var _iteratorNormalCompletion5 = true;
            var _didIteratorError5 = false;
            var _iteratorError5 = undefined;

            try {
                for (var _iterator5 = (0, _getIterator3.default)((0, _entries2.default)(devs)), _step5; !(_iteratorNormalCompletion5 = (_step5 = _iterator5.next()).done); _iteratorNormalCompletion5 = true) {
                    var _step5$value = (0, _slicedToArray3.default)(_step5.value, 2),
                        deviceId = _step5$value[0],
                        dev = _step5$value[1];

                    var _identityKey = dev.keys['curve25519:' + deviceId];

                    this._userByIdentityKey[_identityKey] = u;
                }
            } catch (err) {
                _didIteratorError5 = true;
                _iteratorError5 = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion5 && _iterator5.return) {
                        _iterator5.return();
                    }
                } finally {
                    if (_didIteratorError5) {
                        throw _iteratorError5;
                    }
                }
            }

            this._dirty = true;
        }

        /**
         * flag the given user for device-list tracking, if they are not already.
         *
         * This will mean that a subsequent call to refreshOutdatedDeviceLists()
         * will download the device list for the user, and that subsequent calls to
         * invalidateUserDeviceList will trigger more updates.
         *
         * @param {String} userId
         */

    }, {
        key: 'startTrackingDeviceList',
        value: function startTrackingDeviceList(userId) {
            // sanity-check the userId. This is mostly paranoia, but if synapse
            // can't parse the userId we give it as an mxid, it 500s the whole
            // request and we can never update the device lists again (because
            // the broken userId is always 'invalid' and always included in any
            // refresh request).
            // By checking it is at least a string, we can eliminate a class of
            // silly errors.
            if (typeof userId !== 'string') {
                throw new Error('userId must be a string; was ' + userId);
            }
            if (!this._deviceTrackingStatus[userId]) {
                _logger2.default.log('Now tracking device list for ' + userId);
                this._deviceTrackingStatus[userId] = TRACKING_STATUS_PENDING_DOWNLOAD;
                // we don't yet persist the tracking status, since there may be a lot
                // of calls; we save all data together once the sync is done
                this._dirty = true;
            }
        }

        /**
         * Mark the given user as no longer being tracked for device-list updates.
         *
         * This won't affect any in-progress downloads, which will still go on to
         * complete; it will just mean that we don't think that we have an up-to-date
         * list for future calls to downloadKeys.
         *
         * @param {String} userId
         */

    }, {
        key: 'stopTrackingDeviceList',
        value: function stopTrackingDeviceList(userId) {
            if (this._deviceTrackingStatus[userId]) {
                _logger2.default.log('No longer tracking device list for ' + userId);
                this._deviceTrackingStatus[userId] = TRACKING_STATUS_NOT_TRACKED;

                // we don't yet persist the tracking status, since there may be a lot
                // of calls; we save all data together once the sync is done
                this._dirty = true;
            }
        }

        /**
         * Set all users we're currently tracking to untracked
         *
         * This will flag each user whose devices we are tracking as in need of an
         * update.
         */

    }, {
        key: 'stopTrackingAllDeviceLists',
        value: function stopTrackingAllDeviceLists() {
            var _iteratorNormalCompletion6 = true;
            var _didIteratorError6 = false;
            var _iteratorError6 = undefined;

            try {
                for (var _iterator6 = (0, _getIterator3.default)((0, _keys2.default)(this._deviceTrackingStatus)), _step6; !(_iteratorNormalCompletion6 = (_step6 = _iterator6.next()).done); _iteratorNormalCompletion6 = true) {
                    var userId = _step6.value;

                    this._deviceTrackingStatus[userId] = TRACKING_STATUS_NOT_TRACKED;
                }
            } catch (err) {
                _didIteratorError6 = true;
                _iteratorError6 = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion6 && _iterator6.return) {
                        _iterator6.return();
                    }
                } finally {
                    if (_didIteratorError6) {
                        throw _iteratorError6;
                    }
                }
            }

            this._dirty = true;
        }

        /**
         * Mark the cached device list for the given user outdated.
         *
         * If we are not tracking this user's devices, we'll do nothing. Otherwise
         * we flag the user as needing an update.
         *
         * This doesn't actually set off an update, so that several users can be
         * batched together. Call refreshOutdatedDeviceLists() for that.
         *
         * @param {String} userId
         */

    }, {
        key: 'invalidateUserDeviceList',
        value: function invalidateUserDeviceList(userId) {
            if (this._deviceTrackingStatus[userId]) {
                _logger2.default.log("Marking device list outdated for", userId);
                this._deviceTrackingStatus[userId] = TRACKING_STATUS_PENDING_DOWNLOAD;

                // we don't yet persist the tracking status, since there may be a lot
                // of calls; we save all data together once the sync is done
                this._dirty = true;
            }
        }

        /**
         * If we have users who have outdated device lists, start key downloads for them
         *
         * @returns {Promise} which completes when the download completes; normally there
         *    is no need to wait for this (it's mostly for the unit tests).
         */

    }, {
        key: 'refreshOutdatedDeviceLists',
        value: function refreshOutdatedDeviceLists() {
            this.saveIfDirty();

            var usersToDownload = [];
            var _iteratorNormalCompletion7 = true;
            var _didIteratorError7 = false;
            var _iteratorError7 = undefined;

            try {
                for (var _iterator7 = (0, _getIterator3.default)((0, _keys2.default)(this._deviceTrackingStatus)), _step7; !(_iteratorNormalCompletion7 = (_step7 = _iterator7.next()).done); _iteratorNormalCompletion7 = true) {
                    var userId = _step7.value;

                    var stat = this._deviceTrackingStatus[userId];
                    if (stat == TRACKING_STATUS_PENDING_DOWNLOAD) {
                        usersToDownload.push(userId);
                    }
                }
            } catch (err) {
                _didIteratorError7 = true;
                _iteratorError7 = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion7 && _iterator7.return) {
                        _iterator7.return();
                    }
                } finally {
                    if (_didIteratorError7) {
                        throw _iteratorError7;
                    }
                }
            }

            return this._doKeyDownload(usersToDownload);
        }

        /**
         * Set the stored device data for a user, in raw object form
         * Used only by internal class DeviceListUpdateSerialiser
         *
         * @param {string} userId the user to get data for
         *
         * @param {Object} devices deviceId->{object} the new devices
         */

    }, {
        key: '_setRawStoredDevicesForUser',
        value: function _setRawStoredDevicesForUser(userId, devices) {
            // remove old devices from _userByIdentityKey
            if (this._devices[userId] !== undefined) {
                var _iteratorNormalCompletion8 = true;
                var _didIteratorError8 = false;
                var _iteratorError8 = undefined;

                try {
                    for (var _iterator8 = (0, _getIterator3.default)((0, _entries2.default)(this._devices[userId])), _step8; !(_iteratorNormalCompletion8 = (_step8 = _iterator8.next()).done); _iteratorNormalCompletion8 = true) {
                        var _step8$value = (0, _slicedToArray3.default)(_step8.value, 2),
                            deviceId = _step8$value[0],
                            dev = _step8$value[1];

                        var identityKey = dev.keys['curve25519:' + deviceId];

                        delete this._userByIdentityKey[identityKey];
                    }
                } catch (err) {
                    _didIteratorError8 = true;
                    _iteratorError8 = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion8 && _iterator8.return) {
                            _iterator8.return();
                        }
                    } finally {
                        if (_didIteratorError8) {
                            throw _iteratorError8;
                        }
                    }
                }
            }

            this._devices[userId] = devices;

            // add new devices into _userByIdentityKey
            var _iteratorNormalCompletion9 = true;
            var _didIteratorError9 = false;
            var _iteratorError9 = undefined;

            try {
                for (var _iterator9 = (0, _getIterator3.default)((0, _entries2.default)(devices)), _step9; !(_iteratorNormalCompletion9 = (_step9 = _iterator9.next()).done); _iteratorNormalCompletion9 = true) {
                    var _step9$value = (0, _slicedToArray3.default)(_step9.value, 2),
                        deviceId = _step9$value[0],
                        dev = _step9$value[1];

                    var _identityKey2 = dev.keys['curve25519:' + deviceId];

                    this._userByIdentityKey[_identityKey2] = userId;
                }
            } catch (err) {
                _didIteratorError9 = true;
                _iteratorError9 = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion9 && _iterator9.return) {
                        _iterator9.return();
                    }
                } finally {
                    if (_didIteratorError9) {
                        throw _iteratorError9;
                    }
                }
            }
        }
    }, {
        key: 'setRawStoredCrossSigningForUser',
        value: function setRawStoredCrossSigningForUser(userId, info) {
            this._crossSigningInfo[userId] = info;
        }

        /**
         * Fire off download update requests for the given users, and update the
         * device list tracking status for them, and the
         * _keyDownloadsInProgressByUser map for them.
         *
         * @param {String[]} users  list of userIds
         *
         * @return {module:client.Promise} resolves when all the users listed have
         *     been updated. rejects if there was a problem updating any of the
         *     users.
         */

    }, {
        key: '_doKeyDownload',
        value: function _doKeyDownload(users) {
            var _this5 = this;

            if (users.length === 0) {
                // nothing to do
                return _bluebird2.default.resolve();
            }

            var prom = this._serialiser.updateDevicesForUsers(users, this._syncToken).then(function () {
                finished(true);
            }, function (e) {
                _logger2.default.error('Error downloading keys for ' + users + ":", e);
                finished(false);
                throw e;
            });

            users.forEach(function (u) {
                _this5._keyDownloadsInProgressByUser[u] = prom;
                var stat = _this5._deviceTrackingStatus[u];
                if (stat == TRACKING_STATUS_PENDING_DOWNLOAD) {
                    _this5._deviceTrackingStatus[u] = TRACKING_STATUS_DOWNLOAD_IN_PROGRESS;
                }
            });

            var finished = function finished(success) {
                users.forEach(function (u) {
                    _this5._dirty = true;

                    // we may have queued up another download request for this user
                    // since we started this request. If that happens, we should
                    // ignore the completion of the first one.
                    if (_this5._keyDownloadsInProgressByUser[u] !== prom) {
                        _logger2.default.log('Another update in the queue for', u, '- not marking up-to-date');
                        return;
                    }
                    delete _this5._keyDownloadsInProgressByUser[u];
                    var stat = _this5._deviceTrackingStatus[u];
                    if (stat == TRACKING_STATUS_DOWNLOAD_IN_PROGRESS) {
                        if (success) {
                            // we didn't get any new invalidations since this download started:
                            // this user's device list is now up to date.
                            _this5._deviceTrackingStatus[u] = TRACKING_STATUS_UP_TO_DATE;
                            _logger2.default.log("Device list for", u, "now up to date");
                        } else {
                            _this5._deviceTrackingStatus[u] = TRACKING_STATUS_PENDING_DOWNLOAD;
                        }
                    }
                });
                _this5.saveIfDirty();
                _this5.emit("crypto.devicesUpdated", users);
            };

            return prom;
        }
    }]);
    return DeviceList;
}(_events.EventEmitter);

/**
 * Serialises updates to device lists
 *
 * Ensures that results from /keys/query are not overwritten if a second call
 * completes *before* an earlier one.
 *
 * It currently does this by ensuring only one call to /keys/query happens at a
 * time (and queuing other requests up).
 */


exports.default = DeviceList;

var DeviceListUpdateSerialiser = function () {
    /*
     * @param {object} baseApis Base API object
     * @param {object} olmDevice The Olm Device
     * @param {object} deviceList The device list object
     */
    function DeviceListUpdateSerialiser(baseApis, olmDevice, deviceList) {
        (0, _classCallCheck3.default)(this, DeviceListUpdateSerialiser);

        this._baseApis = baseApis;
        this._olmDevice = olmDevice;
        this._deviceList = deviceList; // the device list to be updated

        this._downloadInProgress = false;

        // users which are queued for download
        // userId -> true
        this._keyDownloadsQueuedByUser = {};

        // deferred which is resolved when the queued users are downloaded.
        //
        // non-null indicates that we have users queued for download.
        this._queuedQueryDeferred = null;

        this._syncToken = null; // The sync token we send with the requests
    }

    /**
     * Make a key query request for the given users
     *
     * @param {String[]} users list of user ids
     *
     * @param {String} syncToken sync token to pass in the query request, to
     *     help the HS give the most recent results
     *
     * @return {module:client.Promise} resolves when all the users listed have
     *     been updated. rejects if there was a problem updating any of the
     *     users.
     */


    (0, _createClass3.default)(DeviceListUpdateSerialiser, [{
        key: 'updateDevicesForUsers',
        value: function updateDevicesForUsers(users, syncToken) {
            var _this6 = this;

            users.forEach(function (u) {
                _this6._keyDownloadsQueuedByUser[u] = true;
            });

            if (!this._queuedQueryDeferred) {
                this._queuedQueryDeferred = _bluebird2.default.defer();
            }

            // We always take the new sync token and just use the latest one we've
            // been given, since it just needs to be at least as recent as the
            // sync response the device invalidation message arrived in
            this._syncToken = syncToken;

            if (this._downloadInProgress) {
                // just queue up these users
                _logger2.default.log('Queued key download for', users);
                return this._queuedQueryDeferred.promise;
            }

            // start a new download.
            return this._doQueuedQueries();
        }
    }, {
        key: '_doQueuedQueries',
        value: function _doQueuedQueries() {
            var _this7 = this;

            if (this._downloadInProgress) {
                throw new Error("DeviceListUpdateSerialiser._doQueuedQueries called with request active");
            }

            var downloadUsers = (0, _keys2.default)(this._keyDownloadsQueuedByUser);
            this._keyDownloadsQueuedByUser = {};
            var deferred = this._queuedQueryDeferred;
            this._queuedQueryDeferred = null;

            _logger2.default.log('Starting key download for', downloadUsers);
            this._downloadInProgress = true;

            var opts = {};
            if (this._syncToken) {
                opts.token = this._syncToken;
            }

            this._baseApis.downloadKeysForUsers(downloadUsers, opts).then(function (res) {
                var dk = res.device_keys || {};
                var masterKeys = res.master_keys || {};
                var ssks = res.self_signing_keys || {};
                var usks = res.user_signing_keys || {};

                // do each user in a separate promise, to avoid wedging the CPU
                // (https://github.com/vector-im/riot-web/issues/3158)
                //
                // of course we ought to do this in a web worker or similar, but
                // this serves as an easy solution for now.
                var prom = _bluebird2.default.resolve();
                var _iteratorNormalCompletion10 = true;
                var _didIteratorError10 = false;
                var _iteratorError10 = undefined;

                try {
                    var _loop = function _loop() {
                        var userId = _step10.value;

                        prom = prom.then((0, _utils.sleep)(5)).then(function () {
                            return _this7._processQueryResponseForUser(userId, dk[userId], {
                                master: masterKeys[userId],
                                self_signing: ssks[userId],
                                user_signing: usks[userId]
                            });
                        });
                    };

                    for (var _iterator10 = (0, _getIterator3.default)(downloadUsers), _step10; !(_iteratorNormalCompletion10 = (_step10 = _iterator10.next()).done); _iteratorNormalCompletion10 = true) {
                        _loop();
                    }
                } catch (err) {
                    _didIteratorError10 = true;
                    _iteratorError10 = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion10 && _iterator10.return) {
                            _iterator10.return();
                        }
                    } finally {
                        if (_didIteratorError10) {
                            throw _iteratorError10;
                        }
                    }
                }

                return prom;
            }).done(function () {
                _logger2.default.log('Completed key download for ' + downloadUsers);

                _this7._downloadInProgress = false;
                deferred.resolve();

                // if we have queued users, fire off another request.
                if (_this7._queuedQueryDeferred) {
                    _this7._doQueuedQueries();
                }
            }, function (e) {
                _logger2.default.warn('Error downloading keys for ' + downloadUsers + ':', e);
                _this7._downloadInProgress = false;
                deferred.reject(e);
            });

            return deferred.promise;
        }
    }, {
        key: '_processQueryResponseForUser',
        value: function () {
            var _ref3 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee2(userId, dkResponse, crossSigningResponse, sskResponse) {
                var userStore, devs, storage, crossSigning;
                return _regenerator2.default.wrap(function _callee2$(_context2) {
                    while (1) {
                        switch (_context2.prev = _context2.next) {
                            case 0:
                                _logger2.default.log('got device keys for ' + userId + ':', dkResponse);
                                _logger2.default.log('got cross-signing keys for ' + userId + ':', crossSigningResponse);

                                // map from deviceid -> deviceinfo for this user
                                userStore = {};
                                devs = this._deviceList.getRawStoredDevicesForUser(userId);

                                if (devs) {
                                    (0, _keys2.default)(devs).forEach(function (deviceId) {
                                        var d = _deviceinfo2.default.fromStorage(devs[deviceId], deviceId);
                                        userStore[deviceId] = d;
                                    });
                                }

                                _context2.next = 7;
                                return (0, _bluebird.resolve)(_updateStoredDeviceKeysForUser(this._olmDevice, userId, userStore, dkResponse || {}));

                            case 7:

                                // put the updates into the object that will be returned as our results
                                storage = {};

                                (0, _keys2.default)(userStore).forEach(function (deviceId) {
                                    storage[deviceId] = userStore[deviceId].toStorage();
                                });

                                this._deviceList._setRawStoredDevicesForUser(userId, storage);

                                // FIXME: should we be ignoring empty cross-signing responses, or
                                // should we be dropping the keys?
                                if (crossSigningResponse && (crossSigningResponse.master || crossSigningResponse.self_signing || crossSigningResponse.user_signing)) {
                                    crossSigning = this._deviceList.getStoredCrossSigningForUser(userId) || new _CrossSigning.CrossSigningInfo(userId);


                                    crossSigning.setKeys(crossSigningResponse);

                                    this._deviceList.setRawStoredCrossSigningForUser(userId, crossSigning.toStorage());

                                    // NB. Unlike most events in the js-sdk, this one is internal to the
                                    // js-sdk and is not re-emitted
                                    this._deviceList.emit('userCrossSigningUpdated', userId);
                                }

                            case 11:
                            case 'end':
                                return _context2.stop();
                        }
                    }
                }, _callee2, this);
            }));

            function _processQueryResponseForUser(_x2, _x3, _x4, _x5) {
                return _ref3.apply(this, arguments);
            }

            return _processQueryResponseForUser;
        }()
    }]);
    return DeviceListUpdateSerialiser;
}();
//# sourceMappingURL=DeviceList.js.map