import * as LSP from 'vscode-languageserver';
import * as Parser from 'web-tree-sitter';
/**
 * The Analyzer uses the Abstract Syntax Trees (ASTs) that are provided by
 * tree-sitter to find definitions, reference, etc.
 */
export default class Analyzer {
    /**
     * Initialize the Analyzer based on a connection to the client and an optional
     * root path.
     *
     * If the rootPath is provided it will initialize all shell files it can find
     * anywhere on that path. This non-exhaustive glob is used to preload the parser.
     */
    static fromRoot({ connection, rootPath, parser, }: {
        connection: LSP.Connection;
        rootPath: LSP.InitializeParams['rootPath'];
        parser: Parser;
    }): Promise<Analyzer>;
    private parser;
    private uriToTextDocument;
    private uriToTreeSitterTrees;
    private uriToFileContent;
    private uriToDeclarations;
    private treeSitterTypeToLSPKind;
    constructor(parser: Parser);
    /**
     * Find all the locations where something named name has been defined.
     */
    findDefinition(name: string): LSP.Location[];
    /**
     * Find all the symbols matching the query using fuzzy search.
     */
    search(query: string): LSP.SymbolInformation[];
    getExplainshellDocumentation({ params, endpoint, }: {
        params: LSP.TextDocumentPositionParams;
        endpoint: string;
    }): Promise<any>;
    /**
     * Find all the locations where something named name has been defined.
     */
    findReferences(name: string): LSP.Location[];
    /**
     * Find all occurrences of name in the given file.
     * It's currently not scope-aware.
     */
    findOccurrences(uri: string, query: string): LSP.Location[];
    /**
     * Find all symbol definitions in the given file.
     */
    findSymbolsForFile({ uri }: {
        uri: string;
    }): LSP.SymbolInformation[];
    /**
     * Find symbol completions for the given word.
     */
    findSymbolsMatchingWord({ exactMatch, word, }: {
        exactMatch: boolean;
        word: string;
    }): LSP.SymbolInformation[];
    /**
     * Analyze the given document, cache the tree-sitter AST, and iterate over the
     * tree to find declarations.
     *
     * Returns all, if any, syntax errors that occurred while parsing the file.
     *
     */
    analyze(uri: string, document: LSP.TextDocument): LSP.Diagnostic[];
    /**
     * Find the full word at the given point.
     */
    wordAtPoint(uri: string, line: number, column: number): string | null;
    private getAllSymbols;
}
