"use strict";
/*
Copyright 2019 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const ircFormatting = __importStar(require("../irc/formatting"));
const log = require("../logging").get("IrcAction");
const ACTION_TYPES = ["message", "emote", "topic", "notice"];
class IrcAction {
    constructor(type, text, ts = 0) {
        this.type = type;
        this.text = text;
        this.ts = ts;
        if (ACTION_TYPES.indexOf(type) === -1) {
            throw new Error("Unknown IrcAction type: " + type);
        }
    }
    static fromMatrixAction(matrixAction) {
        switch (matrixAction.type) {
            case "message":
            case "emote":
            case "notice":
                if (matrixAction.text === null) {
                    break;
                }
                if (matrixAction.htmlText) {
                    const text = ircFormatting.htmlToIrc(matrixAction.htmlText);
                    const ircText = text !== null && text !== void 0 ? text : matrixAction.text; // fallback if needed.
                    if (ircText === null) {
                        throw Error("ircText is null");
                    }
                    // irc formatted text is the main text part
                    return new IrcAction(matrixAction.type, ircText, matrixAction.ts);
                }
                return new IrcAction(matrixAction.type, matrixAction.text, matrixAction.ts);
            case "image":
                return new IrcAction("emote", "uploaded an image: " + matrixAction.text, matrixAction.ts);
            case "video":
                return new IrcAction("emote", "uploaded a video: " + matrixAction.text, matrixAction.ts);
            case "file":
                return new IrcAction("emote", "posted a file: " + matrixAction.text, matrixAction.ts);
            case "topic":
                if (matrixAction.text === null) {
                    break;
                }
                return new IrcAction(matrixAction.type, matrixAction.text, matrixAction.ts);
            default:
                log.error("IrcAction.fromMatrixAction: Unknown action: %s", matrixAction.type);
        }
        return null;
    }
}
exports.IrcAction = IrcAction;
//# sourceMappingURL=IrcAction.js.map