"use strict";

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _getIterator2 = require("babel-runtime/core-js/get-iterator");

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _regenerator = require("babel-runtime/regenerator");

var _regenerator2 = _interopRequireDefault(_regenerator);

var _bluebird = require("bluebird");

var _bluebird2 = _interopRequireDefault(_bluebird);

var _slicedToArray2 = require("babel-runtime/helpers/slicedToArray");

var _slicedToArray3 = _interopRequireDefault(_slicedToArray2);

var _syncAccumulator = require("../sync-accumulator");

var _syncAccumulator2 = _interopRequireDefault(_syncAccumulator);

var _utils = require("../utils");

var _utils2 = _interopRequireDefault(_utils);

var _indexeddbHelpers = require("../indexeddb-helpers");

var IndexedDBHelpers = _interopRequireWildcard(_indexeddbHelpers);

var _logger = require("../logger");

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key]; } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var VERSION = 3; /*
                 Copyright 2017 Vector Creations Ltd
                 Copyright 2018 New Vector Ltd
                 
                 Licensed under the Apache License, Version 2.0 (the "License");
                 you may not use this file except in compliance with the License.
                 You may obtain a copy of the License at
                 
                     http://www.apache.org/licenses/LICENSE-2.0
                 
                 Unless required by applicable law or agreed to in writing, software
                 distributed under the License is distributed on an "AS IS" BASIS,
                 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
                 See the License for the specific language governing permissions and
                 limitations under the License.
                 */

function createDatabase(db) {
    // Make user store, clobber based on user ID. (userId property of User objects)
    db.createObjectStore("users", { keyPath: ["userId"] });

    // Make account data store, clobber based on event type.
    // (event.type property of MatrixEvent objects)
    db.createObjectStore("accountData", { keyPath: ["type"] });

    // Make /sync store (sync tokens, room data, etc), always clobber (const key).
    db.createObjectStore("sync", { keyPath: ["clobber"] });
}

function upgradeSchemaV2(db) {
    var oobMembersStore = db.createObjectStore("oob_membership_events", {
        keyPath: ["room_id", "state_key"]
    });
    oobMembersStore.createIndex("room", "room_id");
}

function upgradeSchemaV3(db) {
    db.createObjectStore("client_options", { keyPath: ["clobber"] });
}

/**
 * Helper method to collect results from a Cursor and promiseify it.
 * @param {ObjectStore|Index} store The store to perform openCursor on.
 * @param {IDBKeyRange=} keyRange Optional key range to apply on the cursor.
 * @param {Function} resultMapper A function which is repeatedly called with a
 * Cursor.
 * Return the data you want to keep.
 * @return {Promise<T[]>} Resolves to an array of whatever you returned from
 * resultMapper.
 */
function selectQuery(store, keyRange, resultMapper) {
    var query = store.openCursor(keyRange);
    return new _bluebird2.default(function (resolve, reject) {
        var results = [];
        query.onerror = function (event) {
            reject(new Error("Query failed: " + event.target.errorCode));
        };
        // collect results
        query.onsuccess = function (event) {
            var cursor = event.target.result;
            if (!cursor) {
                resolve(results);
                return; // end of results
            }
            results.push(resultMapper(cursor));
            cursor.continue();
        };
    });
}

function txnAsPromise(txn) {
    return new _bluebird2.default(function (resolve, reject) {
        txn.oncomplete = function (event) {
            resolve(event);
        };
        txn.onerror = function (event) {
            reject(event.target.error);
        };
    });
}

function reqAsEventPromise(req) {
    return new _bluebird2.default(function (resolve, reject) {
        req.onsuccess = function (event) {
            resolve(event);
        };
        req.onerror = function (event) {
            reject(event.target.error);
        };
    });
}

function reqAsPromise(req) {
    return new _bluebird2.default(function (resolve, reject) {
        req.onsuccess = function () {
            return resolve(req);
        };
        req.onerror = function (err) {
            return reject(err);
        };
    });
}

function reqAsCursorPromise(req) {
    return reqAsEventPromise(req).then(function (event) {
        return event.target.result;
    });
}

/**
 * Does the actual reading from and writing to the indexeddb
 *
 * Construct a new Indexed Database store backend. This requires a call to
 * <code>connect()</code> before this store can be used.
 * @constructor
 * @param {Object} indexedDBInterface The Indexed DB interface e.g
 * <code>window.indexedDB</code>
 * @param {string=} dbName Optional database name. The same name must be used
 * to open the same database.
 */
var LocalIndexedDBStoreBackend = function LocalIndexedDBStoreBackend(indexedDBInterface, dbName) {
    this.indexedDB = indexedDBInterface;
    this._dbName = "matrix-js-sdk:" + (dbName || "default");
    this.db = null;
    this._disconnected = true;
    this._syncAccumulator = new _syncAccumulator2.default();
    this._isNewlyCreated = false;
};

LocalIndexedDBStoreBackend.exists = function (indexedDB, dbName) {
    dbName = "matrix-js-sdk:" + (dbName || "default");
    return IndexedDBHelpers.exists(indexedDB, dbName);
};

LocalIndexedDBStoreBackend.prototype = {
    /**
     * Attempt to connect to the database. This can fail if the user does not
     * grant permission.
     * @return {Promise} Resolves if successfully connected.
     */
    connect: function connect() {
        var _this = this;

        if (!this._disconnected) {
            _logger2.default.log("LocalIndexedDBStoreBackend.connect: already connected or connecting");
            return _bluebird2.default.resolve();
        }

        this._disconnected = false;

        _logger2.default.log("LocalIndexedDBStoreBackend.connect: connecting...");
        var req = this.indexedDB.open(this._dbName, VERSION);
        req.onupgradeneeded = function (ev) {
            var db = ev.target.result;
            var oldVersion = ev.oldVersion;
            _logger2.default.log("LocalIndexedDBStoreBackend.connect: upgrading from " + oldVersion);
            if (oldVersion < 1) {
                // The database did not previously exist.
                _this._isNewlyCreated = true;
                createDatabase(db);
            }
            if (oldVersion < 2) {
                upgradeSchemaV2(db);
            }
            if (oldVersion < 3) {
                upgradeSchemaV3(db);
            }
            // Expand as needed.
        };

        req.onblocked = function () {
            _logger2.default.log("can't yet open LocalIndexedDBStoreBackend because it is open elsewhere");
        };

        _logger2.default.log("LocalIndexedDBStoreBackend.connect: awaiting connection...");
        return reqAsEventPromise(req).then(function (ev) {
            _logger2.default.log("LocalIndexedDBStoreBackend.connect: connected");
            _this.db = ev.target.result;

            // add a poorly-named listener for when deleteDatabase is called
            // so we can close our db connections.
            _this.db.onversionchange = function () {
                _this.db.close();
            };

            return _this._init();
        });
    },
    /** @return {bool} whether or not the database was newly created in this session. */
    isNewlyCreated: function isNewlyCreated() {
        return _bluebird2.default.resolve(this._isNewlyCreated);
    },

    /**
     * Having connected, load initial data from the database and prepare for use
     * @return {Promise} Resolves on success
     */
    _init: function _init() {
        var _this2 = this;

        return _bluebird2.default.all([this._loadAccountData(), this._loadSyncData()]).then(function (_ref) {
            var _ref2 = (0, _slicedToArray3.default)(_ref, 2),
                accountData = _ref2[0],
                syncData = _ref2[1];

            _logger2.default.log("LocalIndexedDBStoreBackend: loaded initial data");
            _this2._syncAccumulator.accumulate({
                next_batch: syncData.nextBatch,
                rooms: syncData.roomsData,
                groups: syncData.groupsData,
                account_data: {
                    events: accountData
                }
            });
        });
    },

    /**
     * Returns the out-of-band membership events for this room that
     * were previously loaded.
     * @param {string} roomId
     * @returns {Promise<event[]>} the events, potentially an empty array if OOB loading didn't yield any new members
     * @returns {null} in case the members for this room haven't been stored yet
     */
    getOutOfBandMembers: function getOutOfBandMembers(roomId) {
        var _this3 = this;

        return new _bluebird2.default(function (resolve, reject) {
            var tx = _this3.db.transaction(["oob_membership_events"], "readonly");
            var store = tx.objectStore("oob_membership_events");
            var roomIndex = store.index("room");
            var range = IDBKeyRange.only(roomId);
            var request = roomIndex.openCursor(range);

            var membershipEvents = [];
            // did we encounter the oob_written marker object
            // amongst the results? That means OOB member
            // loading already happened for this room
            // but there were no members to persist as they
            // were all known already
            var oobWritten = false;

            request.onsuccess = function (event) {
                var cursor = event.target.result;
                if (!cursor) {
                    // Unknown room
                    if (!membershipEvents.length && !oobWritten) {
                        return resolve(null);
                    }
                    return resolve(membershipEvents);
                }
                var record = cursor.value;
                if (record.oob_written) {
                    oobWritten = true;
                } else {
                    membershipEvents.push(record);
                }
                cursor.continue();
            };
            request.onerror = function (err) {
                reject(err);
            };
        }).then(function (events) {
            _logger2.default.log("LL: got " + (events && events.length) + (" membershipEvents from storage for room " + roomId + " ..."));
            return events;
        });
    },

    /**
     * Stores the out-of-band membership events for this room. Note that
     * it still makes sense to store an empty array as the OOB status for the room is
     * marked as fetched, and getOutOfBandMembers will return an empty array instead of null
     * @param {string} roomId
     * @param {event[]} membershipEvents the membership events to store
     */
    setOutOfBandMembers: function () {
        var _ref3 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee(roomId, membershipEvents) {
            var tx, store, markerObject;
            return _regenerator2.default.wrap(function _callee$(_context) {
                while (1) {
                    switch (_context.prev = _context.next) {
                        case 0:
                            _logger2.default.log("LL: backend about to store " + membershipEvents.length + (" members for " + roomId));
                            tx = this.db.transaction(["oob_membership_events"], "readwrite");
                            store = tx.objectStore("oob_membership_events");

                            membershipEvents.forEach(function (e) {
                                store.put(e);
                            });
                            // aside from all the events, we also write a marker object to the store
                            // to mark the fact that OOB members have been written for this room.
                            // It's possible that 0 members need to be written as all where previously know
                            // but we still need to know whether to return null or [] from getOutOfBandMembers
                            // where null means out of band members haven't been stored yet for this room
                            markerObject = {
                                room_id: roomId,
                                oob_written: true,
                                state_key: 0
                            };

                            store.put(markerObject);
                            _context.next = 8;
                            return (0, _bluebird.resolve)(txnAsPromise(tx));

                        case 8:
                            _logger2.default.log("LL: backend done storing for " + roomId + "!");

                        case 9:
                        case "end":
                            return _context.stop();
                    }
                }
            }, _callee, this);
        }));

        function setOutOfBandMembers(_x, _x2) {
            return _ref3.apply(this, arguments);
        }

        return setOutOfBandMembers;
    }(),

    clearOutOfBandMembers: function () {
        var _ref4 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee2(roomId) {
            var readTx, store, roomIndex, roomRange, minStateKeyProm, maxStateKeyProm, _ref5, _ref6, minStateKey, maxStateKey, writeTx, writeStore, membersKeyRange;

            return _regenerator2.default.wrap(function _callee2$(_context2) {
                while (1) {
                    switch (_context2.prev = _context2.next) {
                        case 0:
                            // the approach to delete all members for a room
                            // is to get the min and max state key from the index
                            // for that room, and then delete between those
                            // keys in the store.
                            // this should be way faster than deleting every member
                            // individually for a large room.
                            readTx = this.db.transaction(["oob_membership_events"], "readonly");
                            store = readTx.objectStore("oob_membership_events");
                            roomIndex = store.index("room");
                            roomRange = IDBKeyRange.only(roomId);
                            minStateKeyProm = reqAsCursorPromise(roomIndex.openKeyCursor(roomRange, "next")).then(function (cursor) {
                                return cursor && cursor.primaryKey[1];
                            });
                            maxStateKeyProm = reqAsCursorPromise(roomIndex.openKeyCursor(roomRange, "prev")).then(function (cursor) {
                                return cursor && cursor.primaryKey[1];
                            });
                            _context2.next = 8;
                            return (0, _bluebird.resolve)(_bluebird2.default.all([minStateKeyProm, maxStateKeyProm]));

                        case 8:
                            _ref5 = _context2.sent;
                            _ref6 = (0, _slicedToArray3.default)(_ref5, 2);
                            minStateKey = _ref6[0];
                            maxStateKey = _ref6[1];
                            writeTx = this.db.transaction(["oob_membership_events"], "readwrite");
                            writeStore = writeTx.objectStore("oob_membership_events");
                            membersKeyRange = IDBKeyRange.bound([roomId, minStateKey], [roomId, maxStateKey]);


                            _logger2.default.log("LL: Deleting all users + marker in storage for " + ("room " + roomId + ", with key range:"), [roomId, minStateKey], [roomId, maxStateKey]);
                            _context2.next = 18;
                            return (0, _bluebird.resolve)(reqAsPromise(writeStore.delete(membersKeyRange)));

                        case 18:
                        case "end":
                            return _context2.stop();
                    }
                }
            }, _callee2, this);
        }));

        function clearOutOfBandMembers(_x3) {
            return _ref4.apply(this, arguments);
        }

        return clearOutOfBandMembers;
    }(),

    /**
     * Clear the entire database. This should be used when logging out of a client
     * to prevent mixing data between accounts.
     * @return {Promise} Resolved when the database is cleared.
     */
    clearDatabase: function clearDatabase() {
        var _this4 = this;

        return new _bluebird2.default(function (resolve, reject) {
            _logger2.default.log("Removing indexeddb instance: " + _this4._dbName);
            var req = _this4.indexedDB.deleteDatabase(_this4._dbName);

            req.onblocked = function () {
                _logger2.default.log("can't yet delete indexeddb " + _this4._dbName + " because it is open elsewhere");
            };

            req.onerror = function (ev) {
                // in firefox, with indexedDB disabled, this fails with a
                // DOMError. We treat this as non-fatal, so that we can still
                // use the app.
                _logger2.default.warn("unable to delete js-sdk store indexeddb: " + ev.target.error);
                resolve();
            };

            req.onsuccess = function () {
                _logger2.default.log("Removed indexeddb instance: " + _this4._dbName);
                resolve();
            };
        });
    },

    /**
     * @param {boolean=} copy If false, the data returned is from internal
     * buffers and must not be mutated. Otherwise, a copy is made before
     * returning such that the data can be safely mutated. Default: true.
     *
     * @return {Promise} Resolves with a sync response to restore the
     * client state to where it was at the last save, or null if there
     * is no saved sync data.
     */
    getSavedSync: function getSavedSync(copy) {
        if (copy === undefined) copy = true;

        var data = this._syncAccumulator.getJSON();
        if (!data.nextBatch) return _bluebird2.default.resolve(null);
        if (copy) {
            // We must deep copy the stored data so that the /sync processing code doesn't
            // corrupt the internal state of the sync accumulator (it adds non-clonable keys)
            return _bluebird2.default.resolve(_utils2.default.deepCopy(data));
        } else {
            return _bluebird2.default.resolve(data);
        }
    },

    getNextBatchToken: function getNextBatchToken() {
        return _bluebird2.default.resolve(this._syncAccumulator.getNextBatchToken());
    },

    setSyncData: function setSyncData(syncData) {
        var _this5 = this;

        return _bluebird2.default.resolve().then(function () {
            _this5._syncAccumulator.accumulate(syncData);
        });
    },

    syncToDatabase: function syncToDatabase(userTuples) {
        var syncData = this._syncAccumulator.getJSON();

        return _bluebird2.default.all([this._persistUserPresenceEvents(userTuples), this._persistAccountData(syncData.accountData), this._persistSyncData(syncData.nextBatch, syncData.roomsData, syncData.groupsData)]);
    },

    /**
     * Persist rooms /sync data along with the next batch token.
     * @param {string} nextBatch The next_batch /sync value.
     * @param {Object} roomsData The 'rooms' /sync data from a SyncAccumulator
     * @param {Object} groupsData The 'groups' /sync data from a SyncAccumulator
     * @return {Promise} Resolves if the data was persisted.
     */
    _persistSyncData: function _persistSyncData(nextBatch, roomsData, groupsData) {
        var _this6 = this;

        _logger2.default.log("Persisting sync data up to ", nextBatch);
        return _bluebird2.default.try(function () {
            var txn = _this6.db.transaction(["sync"], "readwrite");
            var store = txn.objectStore("sync");
            store.put({
                clobber: "-", // constant key so will always clobber
                nextBatch: nextBatch,
                roomsData: roomsData,
                groupsData: groupsData
            }); // put == UPSERT
            return txnAsPromise(txn);
        });
    },

    /**
     * Persist a list of account data events. Events with the same 'type' will
     * be replaced.
     * @param {Object[]} accountData An array of raw user-scoped account data events
     * @return {Promise} Resolves if the events were persisted.
     */
    _persistAccountData: function _persistAccountData(accountData) {
        var _this7 = this;

        return _bluebird2.default.try(function () {
            var txn = _this7.db.transaction(["accountData"], "readwrite");
            var store = txn.objectStore("accountData");
            for (var i = 0; i < accountData.length; i++) {
                store.put(accountData[i]); // put == UPSERT
            }
            return txnAsPromise(txn);
        });
    },

    /**
     * Persist a list of [user id, presence event] they are for.
     * Users with the same 'userId' will be replaced.
     * Presence events should be the event in its raw form (not the Event
     * object)
     * @param {Object[]} tuples An array of [userid, event] tuples
     * @return {Promise} Resolves if the users were persisted.
     */
    _persistUserPresenceEvents: function _persistUserPresenceEvents(tuples) {
        var _this8 = this;

        return _bluebird2.default.try(function () {
            var txn = _this8.db.transaction(["users"], "readwrite");
            var store = txn.objectStore("users");
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = (0, _getIterator3.default)(tuples), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var tuple = _step.value;

                    store.put({
                        userId: tuple[0],
                        event: tuple[1]
                    }); // put == UPSERT
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            return txnAsPromise(txn);
        });
    },

    /**
     * Load all user presence events from the database. This is not cached.
     * FIXME: It would probably be more sensible to store the events in the
     * sync.
     * @return {Promise<Object[]>} A list of presence events in their raw form.
     */
    getUserPresenceEvents: function getUserPresenceEvents() {
        var _this9 = this;

        return _bluebird2.default.try(function () {
            var txn = _this9.db.transaction(["users"], "readonly");
            var store = txn.objectStore("users");
            return selectQuery(store, undefined, function (cursor) {
                return [cursor.value.userId, cursor.value.event];
            });
        });
    },

    /**
     * Load all the account data events from the database. This is not cached.
     * @return {Promise<Object[]>} A list of raw global account events.
     */
    _loadAccountData: function _loadAccountData() {
        var _this10 = this;

        _logger2.default.log("LocalIndexedDBStoreBackend: loading account data...");
        return _bluebird2.default.try(function () {
            var txn = _this10.db.transaction(["accountData"], "readonly");
            var store = txn.objectStore("accountData");
            return selectQuery(store, undefined, function (cursor) {
                return cursor.value;
            }).then(function (result) {
                _logger2.default.log("LocalIndexedDBStoreBackend: loaded account data");
                return result;
            });
        });
    },

    /**
     * Load the sync data from the database.
     * @return {Promise<Object>} An object with "roomsData" and "nextBatch" keys.
     */
    _loadSyncData: function _loadSyncData() {
        var _this11 = this;

        _logger2.default.log("LocalIndexedDBStoreBackend: loading sync data...");
        return _bluebird2.default.try(function () {
            var txn = _this11.db.transaction(["sync"], "readonly");
            var store = txn.objectStore("sync");
            return selectQuery(store, undefined, function (cursor) {
                return cursor.value;
            }).then(function (results) {
                _logger2.default.log("LocalIndexedDBStoreBackend: loaded sync data");
                if (results.length > 1) {
                    _logger2.default.warn("loadSyncData: More than 1 sync row found.");
                }
                return results.length > 0 ? results[0] : {};
            });
        });
    },

    getClientOptions: function getClientOptions() {
        var _this12 = this;

        return _bluebird2.default.resolve().then(function () {
            var txn = _this12.db.transaction(["client_options"], "readonly");
            var store = txn.objectStore("client_options");
            return selectQuery(store, undefined, function (cursor) {
                if (cursor.value && cursor.value && cursor.value.options) {
                    return cursor.value.options;
                }
            }).then(function (results) {
                return results[0];
            });
        });
    },

    storeClientOptions: function () {
        var _ref7 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee3(options) {
            var txn, store;
            return _regenerator2.default.wrap(function _callee3$(_context3) {
                while (1) {
                    switch (_context3.prev = _context3.next) {
                        case 0:
                            txn = this.db.transaction(["client_options"], "readwrite");
                            store = txn.objectStore("client_options");

                            store.put({
                                clobber: "-", // constant key so will always clobber
                                options: options
                            }); // put == UPSERT
                            _context3.next = 5;
                            return (0, _bluebird.resolve)(txnAsPromise(txn));

                        case 5:
                        case "end":
                            return _context3.stop();
                    }
                }
            }, _callee3, this);
        }));

        function storeClientOptions(_x4) {
            return _ref7.apply(this, arguments);
        }

        return storeClientOptions;
    }()
};

exports.default = LocalIndexedDBStoreBackend;
//# sourceMappingURL=indexeddb-local-backend.js.map