/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.translog;

import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.store.Directory;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.RemoveCorruptedShardDataCommand;
import org.elasticsearch.index.shard.ShardPath;
import org.elasticsearch.index.translog.Checkpoint;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.index.translog.TranslogConfig;
import org.elasticsearch.index.translog.TranslogCorruptedException;
import org.elasticsearch.index.translog.TranslogDeletionPolicy;
import org.elasticsearch.index.translog.TranslogHeader;
import org.elasticsearch.index.translog.TranslogReader;
import org.elasticsearch.index.translog.TranslogWriter;

public class TruncateTranslogAction {
    protected static final Logger logger = LogManager.getLogger(TruncateTranslogAction.class);
    private final NamedXContentRegistry namedXContentRegistry;

    public TruncateTranslogAction(NamedXContentRegistry namedXContentRegistry) {
        this.namedXContentRegistry = namedXContentRegistry;
    }

    public Tuple<RemoveCorruptedShardDataCommand.CleanStatus, String> getCleanStatus(ShardPath shardPath, ClusterState clusterState, Directory indexDirectory) throws IOException {
        Set<Path> translogFiles;
        List commits;
        Path indexPath = shardPath.resolveIndex();
        Path translogPath = shardPath.resolveTranslog();
        try {
            commits = DirectoryReader.listCommits((Directory)indexDirectory);
        }
        catch (IndexNotFoundException infe) {
            throw new ElasticsearchException("unable to find a valid shard at [" + indexPath + "]", (Throwable)infe, new Object[0]);
        }
        catch (IOException e) {
            throw new ElasticsearchException("unable to list commits at [" + indexPath + "]", (Throwable)e, new Object[0]);
        }
        HashMap commitData = new HashMap(((IndexCommit)commits.get(commits.size() - 1)).getUserData());
        String translogUUID = (String)commitData.get("translog_uuid");
        if (translogUUID == null) {
            throw new ElasticsearchException("shard must have a valid translog UUID but got: [null]", new Object[0]);
        }
        boolean clean = this.isTranslogClean(shardPath, clusterState, translogUUID);
        if (clean) {
            return Tuple.tuple((Object)((Object)RemoveCorruptedShardDataCommand.CleanStatus.CLEAN), null);
        }
        try {
            translogFiles = TruncateTranslogAction.filesInDirectory(translogPath);
        }
        catch (IOException e) {
            throw new ElasticsearchException("failed to find existing translog files", (Throwable)e, new Object[0]);
        }
        String details = this.deletingFilesDetails(translogPath, translogFiles);
        return Tuple.tuple((Object)((Object)RemoveCorruptedShardDataCommand.CleanStatus.CORRUPTED), (Object)details);
    }

    public void execute(Terminal terminal, ShardPath shardPath, Directory indexDirectory) throws IOException {
        List commits;
        Set<Path> translogFiles;
        Path indexPath = shardPath.resolveIndex();
        Path translogPath = shardPath.resolveTranslog();
        String historyUUID = UUIDs.randomBase64UUID();
        try {
            terminal.println("Checking existing translog files");
            translogFiles = TruncateTranslogAction.filesInDirectory(translogPath);
        }
        catch (IOException e) {
            terminal.println("encountered IOException while listing directory, aborting...");
            throw new ElasticsearchException("failed to find existing translog files", (Throwable)e, new Object[0]);
        }
        try {
            terminal.println("Reading translog UUID information from Lucene commit from shard at [" + indexPath + "]");
            commits = DirectoryReader.listCommits((Directory)indexDirectory);
        }
        catch (IndexNotFoundException infe) {
            throw new ElasticsearchException("unable to find a valid shard at [" + indexPath + "]", (Throwable)infe, new Object[0]);
        }
        Map commitData = ((IndexCommit)commits.get(commits.size() - 1)).getUserData();
        String translogUUID = (String)commitData.get("translog_uuid");
        if (translogUUID == null) {
            throw new ElasticsearchException("shard must have a valid translog UUID", new Object[0]);
        }
        long globalCheckpoint = commitData.containsKey("max_seq_no") ? Long.parseLong((String)commitData.get("max_seq_no")) : -2L;
        terminal.println("Translog UUID      : " + translogUUID);
        terminal.println("History UUID       : " + historyUUID);
        Path tempEmptyCheckpoint = translogPath.resolve("temp-translog.ckp");
        Path realEmptyCheckpoint = translogPath.resolve("translog.ckp");
        long gen = 1L;
        Path tempEmptyTranslog = translogPath.resolve("temp-translog-1.tlog");
        Path realEmptyTranslog = translogPath.resolve("translog-1.tlog");
        int translogLen = TruncateTranslogAction.writeEmptyTranslog(tempEmptyTranslog, translogUUID);
        TruncateTranslogAction.writeEmptyCheckpoint(tempEmptyCheckpoint, translogLen, 1L, globalCheckpoint);
        terminal.println("Removing existing translog files");
        IOUtils.rm((Path[])translogFiles.toArray(new Path[0]));
        terminal.println("Creating new empty checkpoint at [" + realEmptyCheckpoint + "]");
        Files.move(tempEmptyCheckpoint, realEmptyCheckpoint, StandardCopyOption.ATOMIC_MOVE);
        terminal.println("Creating new empty translog at [" + realEmptyTranslog + "]");
        Files.move(tempEmptyTranslog, realEmptyTranslog, StandardCopyOption.ATOMIC_MOVE);
        IOUtils.fsync((Path)translogPath, (boolean)true);
    }

    private boolean isTranslogClean(ShardPath shardPath, ClusterState clusterState, String translogUUID) throws IOException {
        try {
            Path translogPath = shardPath.resolveTranslog();
            long translogGlobalCheckpoint = Translog.readGlobalCheckpoint(translogPath, translogUUID);
            IndexMetaData indexMetaData = clusterState.metaData().getIndexSafe(shardPath.getShardId().getIndex());
            IndexSettings indexSettings = new IndexSettings(indexMetaData, Settings.EMPTY);
            TranslogConfig translogConfig = new TranslogConfig(shardPath.getShardId(), translogPath, indexSettings, BigArrays.NON_RECYCLING_INSTANCE);
            long primaryTerm = indexSettings.getIndexMetaData().primaryTerm(shardPath.getShardId().id());
            TranslogDeletionPolicy retainAllTranslogPolicy = new TranslogDeletionPolicy(Long.MAX_VALUE, Long.MAX_VALUE, Integer.MAX_VALUE){

                @Override
                long minTranslogGenRequired(List<TranslogReader> readers, TranslogWriter writer) {
                    long minGen = writer.generation;
                    for (TranslogReader reader : readers) {
                        minGen = Math.min(reader.generation, minGen);
                    }
                    return minGen;
                }
            };
            try (Translog translog = new Translog(translogConfig, translogUUID, retainAllTranslogPolicy, () -> translogGlobalCheckpoint, () -> primaryTerm, seqNo -> {});
                 Translog.Snapshot snapshot = translog.newSnapshot(0L, Long.MAX_VALUE);){
                while (snapshot.next() != null) {
                }
            }
            return true;
        }
        catch (TranslogCorruptedException e) {
            return false;
        }
    }

    private static void writeEmptyCheckpoint(Path filename, int translogLength, long translogGeneration, long globalCheckpoint) throws IOException {
        Checkpoint emptyCheckpoint = Checkpoint.emptyTranslogCheckpoint(translogLength, translogGeneration, globalCheckpoint, translogGeneration);
        Checkpoint.write(FileChannel::open, filename, emptyCheckpoint, StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE_NEW);
        IOUtils.fsync((Path)filename, (boolean)false);
    }

    private static int writeEmptyTranslog(Path filename, String translogUUID) throws IOException {
        try (FileChannel fc = FileChannel.open(filename, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);){
            TranslogHeader header = new TranslogHeader(translogUUID, 0L);
            header.write(fc);
            int n = header.sizeInBytes();
            return n;
        }
    }

    private String deletingFilesDetails(Path translogPath, Set<Path> files) {
        StringBuilder builder = new StringBuilder();
        builder.append("Documents inside of translog files will be lost.\n").append("  The following files will be DELETED at ").append(translogPath).append("\n\n");
        Iterator<Path> it = files.iterator();
        while (it.hasNext()) {
            builder.append("  --> ").append(it.next().getFileName());
            if (!it.hasNext()) continue;
            builder.append("\n");
        }
        return builder.toString();
    }

    private static Set<Path> filesInDirectory(Path directory) throws IOException {
        TreeSet<Path> files = new TreeSet<Path>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(directory);){
            for (Path file : stream) {
                files.add(file);
            }
        }
        return files;
    }
}

