/// <reference types="node" />
import { EventEmitter } from "events";
export declare class AppService extends EventEmitter {
    private config;
    /**
     * An HTTP log line.
     * @event AppService#http-log
     * @type {String}
     * @example
     * appService.on("http-log", function(logLine) {
     *   console.log(logLine);
     * });
     */
    /**
     * An incoming Matrix JSON event.
     * @event AppService#event
     * @type {Object}
     * @example
     * appService.on("event", function(ev) {
     *   console.log("ID: %s", ev.event_id);
     * });
     */
    /**
     * An incoming Matrix JSON event, filtered by <code>event.type</code>
     * @event AppService#type:event
     * @type {Object}
     * @example
     * appService.on("type:m.room.message", function(ev) {
     *   console.log("Body: %s", ev.content.body);
     * });
     */
    private app;
    private server?;
    private lastProcessedTxnId;
    /**
     * Construct a new application service.
     * @constructor
     * @param {Object} config Configuration for this service.
     * @param {String} config.homeserverToken The incoming HS token to expect. Must
     * be set prior to calling listen(port).
     * @param {Number} config.httpMaxSizeBytes The max number of bytes allowed on an
     * incoming HTTP request. Default: 5000000.
     * @throws If a homeserver token is not supplied.
     */
    constructor(config: {
        homeserverToken: string;
        httpMaxSizeBytes: number;
    });
    /***
     * Begin listening on the specified port.
     * @param {Number} port The port to listen on.
     * @param {String} hostname Optional hostname to listen on
     * @param {Number} backlog Maximum length of the queue of pending connections
     * @param {Function} callback The callback for the "listening" event. Optional.
     * @returns {Promise} When the server is listening, if a callback is not provided.
     */
    listen(port: number, hostname: string, backlog: number, callback?: () => void): Promise<unknown> | undefined;
    /**
     * Closes the HTTP server.
     * @returns {Promise} When the operation has completed
     * @throws If the server has not been started.
     */
    close(): Promise<void>;
    /**
     * Override this method to handle alias queries.
     * @param {string} alias The queried room alias
     * @param {Function} callback The callback to invoke when complete.
     * @return {Promise} A promise to resolve when complete (if callback isn't supplied)
     */
    onAliasQuery(alias: string, callback: () => void): PromiseLike<void> | null;
    /**
     * Override this method to handle user queries.
     * @param {string} userId The queried user ID.
     * @param {Function} callback The callback to invoke when complete.
     * @return {Promise} A promise to resolve when complete (if callback isn't supplied)
     */
    onUserQuery(userId: string, callback: () => void): PromiseLike<void> | null;
    /**
     * Set the token that should be used to verify incoming events.
     * @param {string} hsToken The home server token
     */
    setHomeserverToken(hsToken: string): void;
    private onMorganLog;
    private isInvalidToken;
    private onGetUsers;
    private onGetRoomAlias;
    private onTransaction;
}
