/*
Copyright 2016 OpenMarket Ltd
Copyright 2017 Vector Creations Ltd
Copyright 2018-2019 New Vector Ltd
Copyright 2019 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
"use strict";

/**
 * @module crypto
 */

Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.verificationMethods = undefined;

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _set = require('babel-runtime/core-js/set');

var _set2 = _interopRequireDefault(_set);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _stringify = require('babel-runtime/core-js/json/stringify');

var _stringify2 = _interopRequireDefault(_stringify);

var _toConsumableArray2 = require('babel-runtime/helpers/toConsumableArray');

var _toConsumableArray3 = _interopRequireDefault(_toConsumableArray2);

var _assign = require('babel-runtime/core-js/object/assign');

var _assign2 = _interopRequireDefault(_assign);

var _slicedToArray2 = require('babel-runtime/helpers/slicedToArray');

var _slicedToArray3 = _interopRequireDefault(_slicedToArray2);

var _entries = require('babel-runtime/core-js/object/entries');

var _entries2 = _interopRequireDefault(_entries);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _regenerator = require('babel-runtime/regenerator');

var _regenerator2 = _interopRequireDefault(_regenerator);

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _map = require('babel-runtime/core-js/map');

var _map2 = _interopRequireDefault(_map);

var _defineProperty2 = require('babel-runtime/helpers/defineProperty');

var _defineProperty3 = _interopRequireDefault(_defineProperty2);

var _defaultVerificationM;

// returns a promise which resolves to the response
var _uploadOneTimeKeys = function () {
    var _ref12 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee12(crypto) {
        var oneTimeKeys, oneTimeJson, promises, keyId, k, res;
        return _regenerator2.default.wrap(function _callee12$(_context12) {
            while (1) {
                switch (_context12.prev = _context12.next) {
                    case 0:
                        _context12.next = 2;
                        return (0, _bluebird.resolve)(crypto._olmDevice.getOneTimeKeys());

                    case 2:
                        oneTimeKeys = _context12.sent;
                        oneTimeJson = {};
                        promises = [];


                        for (keyId in oneTimeKeys.curve25519) {
                            if (oneTimeKeys.curve25519.hasOwnProperty(keyId)) {
                                k = {
                                    key: oneTimeKeys.curve25519[keyId]
                                };

                                oneTimeJson["signed_curve25519:" + keyId] = k;
                                promises.push(crypto._signObject(k));
                            }
                        }

                        _context12.next = 8;
                        return (0, _bluebird.resolve)(_bluebird2.default.all(promises));

                    case 8:
                        _context12.next = 10;
                        return (0, _bluebird.resolve)(crypto._baseApis.uploadKeysRequest({
                            one_time_keys: oneTimeJson
                        }, {
                            // for now, we set the device id explicitly, as we may not be using the
                            // same one as used in login.
                            device_id: crypto._deviceId
                        }));

                    case 10:
                        res = _context12.sent;
                        _context12.next = 13;
                        return (0, _bluebird.resolve)(crypto._olmDevice.markKeysAsPublished());

                    case 13:
                        return _context12.abrupt('return', res);

                    case 14:
                    case 'end':
                        return _context12.stop();
                }
            }
        }, _callee12, this);
    }));

    return function _uploadOneTimeKeys(_x12) {
        return _ref12.apply(this, arguments);
    };
}();

/**
 * Download the keys for a list of users and stores the keys in the session
 * store.
 * @param {Array} userIds The users to fetch.
 * @param {bool} forceDownload Always download the keys even if cached.
 *
 * @return {Promise} A promise which resolves to a map userId->deviceId->{@link
 * module:crypto/deviceinfo|DeviceInfo}.
 */


exports.isCryptoAvailable = isCryptoAvailable;
exports.default = Crypto;

var _events = require('events');

var _ReEmitter = require('../ReEmitter');

var _ReEmitter2 = _interopRequireDefault(_ReEmitter);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _randomstring = require('../randomstring');

var _CrossSigning = require('./CrossSigning');

var _SecretStorage = require('./SecretStorage');

var _SecretStorage2 = _interopRequireDefault(_SecretStorage);

var _OutgoingRoomKeyRequestManager = require('./OutgoingRoomKeyRequestManager');

var _OutgoingRoomKeyRequestManager2 = _interopRequireDefault(_OutgoingRoomKeyRequestManager);

var _indexeddbCryptoStore = require('./store/indexeddb-crypto-store');

var _indexeddbCryptoStore2 = _interopRequireDefault(_indexeddbCryptoStore);

var _QRCode = require('./verification/QRCode');

var _SAS = require('./verification/SAS');

var _SAS2 = _interopRequireDefault(_SAS);

var _Error = require('./verification/Error');

var _utils = require('../utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var anotherjson = require('another-json');

var utils = require("../utils");
var OlmDevice = require("./OlmDevice");
var olmlib = require("./olmlib");
var algorithms = require("./algorithms");
var DeviceInfo = require("./deviceinfo");
var DeviceVerification = DeviceInfo.DeviceVerification;
var DeviceList = require('./DeviceList').default;


var defaultVerificationMethods = (_defaultVerificationM = {}, (0, _defineProperty3.default)(_defaultVerificationM, _QRCode.ScanQRCode.NAME, _QRCode.ScanQRCode), (0, _defineProperty3.default)(_defaultVerificationM, _QRCode.ShowQRCode.NAME, _QRCode.ShowQRCode), (0, _defineProperty3.default)(_defaultVerificationM, _SAS2.default.NAME, _SAS2.default), _defaultVerificationM);

/**
 * verification method names
 */
var verificationMethods = exports.verificationMethods = {
    QR_CODE_SCAN: _QRCode.ScanQRCode.NAME,
    QR_CODE_SHOW: _QRCode.ShowQRCode.NAME,
    SAS: _SAS2.default.NAME
};

// the recommended amount of time before a verification request
// should be (automatically) cancelled without user interaction
// and ignored.
var VERIFICATION_REQUEST_TIMEOUT = 5 * 60 * 1000; //5m
// to avoid almost expired verification notifications
// from showing a notification and almost immediately
// disappearing, also ignore verification requests that
// are this amount of time away from expiring.
var VERIFICATION_REQUEST_MARGIN = 3 * 1000; //3s

function isCryptoAvailable() {
    return Boolean(global.Olm);
}

/* subscribes to timeline events / to_device events for SAS verification */
function listenForEvents(client, roomId, listener) {
    var isEncrypted = false;
    if (roomId) {
        isEncrypted = client.isRoomEncrypted(roomId);
    }

    if (isEncrypted) {
        client.on("Event.decrypted", listener);
    }
    client.on("event", listener);
    var subscribed = true;
    return function () {
        if (subscribed) {
            if (isEncrypted) {
                client.off("Event.decrypted", listener);
            }
            client.off("event", listener);
            subscribed = false;
        }
        return null;
    };
}

var MIN_FORCE_SESSION_INTERVAL_MS = 60 * 60 * 1000;
var KEY_BACKUP_KEYS_PER_REQUEST = 200;

/**
 * Cryptography bits
 *
 * This module is internal to the js-sdk; the public API is via MatrixClient.
 *
 * @constructor
 * @alias module:crypto
 *
 * @internal
 *
 * @param {module:base-apis~MatrixBaseApis} baseApis base matrix api interface
 *
 * @param {module:store/session/webstorage~WebStorageSessionStore} sessionStore
 *    Store to be used for end-to-end crypto session data
 *
 * @param {string} userId The user ID for the local user
 *
 * @param {string} deviceId The identifier for this device.
 *
 * @param {Object} clientStore the MatrixClient data store.
 *
 * @param {module:crypto/store/base~CryptoStore} cryptoStore
 *    storage for the crypto layer.
 *
 * @param {RoomList} roomList An initialised RoomList object
 *
 * @param {Array} verificationMethods Array of verification methods to use.
 *    Each element can either be a string from MatrixClient.verificationMethods
 *    or a class that implements a verification method.
 */
function Crypto(baseApis, sessionStore, userId, deviceId, clientStore, cryptoStore, roomList, verificationMethods) {
    this._onDeviceListUserCrossSigningUpdated = this._onDeviceListUserCrossSigningUpdated.bind(this);

    this._reEmitter = new _ReEmitter2.default(this);
    this._baseApis = baseApis;
    this._sessionStore = sessionStore;
    this._userId = userId;
    this._deviceId = deviceId;
    this._clientStore = clientStore;
    this._cryptoStore = cryptoStore;
    this._roomList = roomList;
    this._verificationMethods = new _map2.default();
    if (verificationMethods) {
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
            for (var _iterator = (0, _getIterator3.default)(verificationMethods), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                var method = _step.value;

                if (typeof method === "string") {
                    if (defaultVerificationMethods[method]) {
                        this._verificationMethods.set(method, defaultVerificationMethods[method]);
                    }
                } else if (method.NAME) {
                    this._verificationMethods.set(method.NAME, method);
                }
            }
        } catch (err) {
            _didIteratorError = true;
            _iteratorError = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion && _iterator.return) {
                    _iterator.return();
                }
            } finally {
                if (_didIteratorError) {
                    throw _iteratorError;
                }
            }
        }
    }

    // track whether this device's megolm keys are being backed up incrementally
    // to the server or not.
    // XXX: this should probably have a single source of truth from OlmAccount
    this.backupInfo = null; // The info dict from /room_keys/version
    this.backupKey = null; // The encryption key object
    this._checkedForBackup = false; // Have we checked the server for a backup we can use?
    this._sendingBackups = false; // Are we currently sending backups?

    this._olmDevice = new OlmDevice(cryptoStore);
    this._deviceList = new DeviceList(baseApis, cryptoStore, this._olmDevice);
    // XXX: This isn't removed at any point, but then none of the event listeners
    // this class sets seem to be removed at any point... :/
    this._deviceList.on('userCrossSigningUpdated', this._onDeviceListUserCrossSigningUpdated);
    this._reEmitter.reEmit(this._deviceList, ["crypto.devicesUpdated"]);

    // the last time we did a check for the number of one-time-keys on the
    // server.
    this._lastOneTimeKeyCheck = null;
    this._oneTimeKeyCheckInProgress = false;

    // EncryptionAlgorithm instance for each room
    this._roomEncryptors = {};

    // map from algorithm to DecryptionAlgorithm instance, for each room
    this._roomDecryptors = {};

    this._supportedAlgorithms = utils.keys(algorithms.DECRYPTION_CLASSES);

    this._deviceKeys = {};

    this._globalBlacklistUnverifiedDevices = false;

    this._outgoingRoomKeyRequestManager = new _OutgoingRoomKeyRequestManager2.default(baseApis, this._deviceId, this._cryptoStore);

    // list of IncomingRoomKeyRequests/IncomingRoomKeyRequestCancellations
    // we received in the current sync.
    this._receivedRoomKeyRequests = [];
    this._receivedRoomKeyRequestCancellations = [];
    // true if we are currently processing received room key requests
    this._processingRoomKeyRequests = false;
    // controls whether device tracking is delayed
    // until calling encryptEvent or trackRoomDevices,
    // or done immediately upon enabling room encryption.
    this._lazyLoadMembers = false;
    // in case _lazyLoadMembers is true,
    // track if an initial tracking of all the room members
    // has happened for a given room. This is delayed
    // to avoid loading room members as long as possible.
    this._roomDeviceTrackingState = {};

    // The timestamp of the last time we forced establishment
    // of a new session for each device, in milliseconds.
    // {
    //     userId: {
    //         deviceId: 1234567890000,
    //     },
    // }
    this._lastNewSessionForced = {};

    this._verificationTransactions = new _map2.default();

    this._crossSigningInfo = new _CrossSigning.CrossSigningInfo(userId, this._baseApis._cryptoCallbacks);

    this._secretStorage = new _SecretStorage2.default(baseApis, this._baseApis._cryptoCallbacks, this._crossSigningInfo);
}
utils.inherits(Crypto, _events.EventEmitter);

/**
 * Initialise the crypto module so that it is ready for use
 *
 * Returns a promise which resolves once the crypto module is ready for use.
 */
Crypto.prototype.init = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee() {
    var _this = this;

    var myDevices, deviceInfo;
    return _regenerator2.default.wrap(function _callee$(_context) {
        while (1) {
            switch (_context.prev = _context.next) {
                case 0:
                    _logger2.default.log("Crypto: initialising Olm...");
                    _context.next = 3;
                    return (0, _bluebird.resolve)(global.Olm.init());

                case 3:
                    _logger2.default.log("Crypto: initialising Olm device...");
                    _context.next = 6;
                    return (0, _bluebird.resolve)(this._olmDevice.init());

                case 6:
                    _logger2.default.log("Crypto: loading device list...");
                    _context.next = 9;
                    return (0, _bluebird.resolve)(this._deviceList.load());

                case 9:

                    // build our device keys: these will later be uploaded
                    this._deviceKeys["ed25519:" + this._deviceId] = this._olmDevice.deviceEd25519Key;
                    this._deviceKeys["curve25519:" + this._deviceId] = this._olmDevice.deviceCurve25519Key;

                    _logger2.default.log("Crypto: fetching own devices...");
                    myDevices = this._deviceList.getRawStoredDevicesForUser(this._userId);


                    if (!myDevices) {
                        myDevices = {};
                    }

                    if (!myDevices[this._deviceId]) {
                        // add our own deviceinfo to the cryptoStore
                        _logger2.default.log("Crypto: adding this device to the store...");
                        deviceInfo = {
                            keys: this._deviceKeys,
                            algorithms: this._supportedAlgorithms,
                            verified: DeviceVerification.VERIFIED,
                            known: true
                        };


                        myDevices[this._deviceId] = deviceInfo;
                        this._deviceList.storeDevicesForUser(this._userId, myDevices);
                        this._deviceList.saveIfDirty();
                    }

                    _context.next = 17;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                        _this._cryptoStore.getCrossSigningKeys(txn, function (keys) {
                            if (keys) {
                                _this._crossSigningInfo.setKeys(keys);
                            }
                        });
                    }));

                case 17:
                    // make sure we are keeping track of our own devices
                    // (this is important for key backups & things)
                    this._deviceList.startTrackingDeviceList(this._userId);

                    _logger2.default.log("Crypto: checking for key backup...");
                    this._checkAndStartKeyBackup();

                case 20:
                case 'end':
                    return _context.stop();
            }
        }
    }, _callee, this);
}));

Crypto.prototype.addSecretKey = function (algorithm, opts, keyID) {
    return this._secretStorage.addKey(algorithm, opts, keyID);
};

Crypto.prototype.storeSecret = function (name, secret, keys) {
    return this._secretStorage.store(name, secret, keys);
};

Crypto.prototype.getSecret = function (name) {
    return this._secretStorage.get(name);
};

Crypto.prototype.isSecretStored = function (name, checkKey) {
    return this._secretStorage.isStored(name, checkKey);
};

Crypto.prototype.requestSecret = function (name, devices) {
    if (!devices) {
        devices = (0, _keys2.default)(this._deviceList.getRawStoredDevicesForUser(this._userId));
    }
    return this._secretStorage.request(name, devices);
};

Crypto.prototype.getDefaultSecretStorageKeyId = function () {
    return this._secretStorage.getDefaultKeyId();
};

Crypto.prototype.setDefaultSecretStorageKeyId = function (k) {
    return this._secretStorage.setDefaultKeyId(k);
};

/**
 * Checks that a given private key matches a given public key
 * This can be used by the getCrossSigningKey callback to verify that the
 * private key it is about to supply is the one that was requested.
 *
 * @param {Uint8Array} privateKey The private key
 * @param {Uint8Array} expectedPublicKey The public key supplied by the getCrossSigningKey callback
 * @returns {boolean} true if the key matches, otherwise false
 */
Crypto.prototype.checkPrivateKey = function (privateKey, expectedPublicKey) {
    var signing = null;
    try {
        signing = new global.Olm.PkSigning();
        var gotPubkey = signing.init_with_seed(privateKey);
        // make sure it agrees with the given pubkey
        return gotPubkey === expectedPublicKey;
    } finally {
        if (signing) signing.free();
    }
};

/**
 * Check whether we already have cross-signing keys for the current user.
 *
 * @return {boolean} Whether we have keys.
 */
Crypto.prototype.doesCrossSigningHaveKeys = function () {
    return this._crossSigningInfo.hasKeys();
};

/**
 * Generate new cross-signing keys.
 *
 * @param {object} authDict Auth data to supply for User-Interactive auth.
 * @param {CrossSigningLevel} [level] the level of cross-signing to reset.  New
 * keys will be created for the given level and below.  Defaults to
 * regenerating all keys.
 */
Crypto.prototype.resetCrossSigningKeys = function () {
    var _ref2 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee2(authDict, level) {
        var _this2 = this;

        var keys, _iteratorNormalCompletion2, _didIteratorError2, _iteratorError2, _iterator2, _step2, _step2$value, name, key, device, signedDevice, users, _iteratorNormalCompletion3, _didIteratorError3, _iteratorError3, _iterator3, _step3, _step3$value, _userId2, crossSigningInfo, upgradeInfo, shouldUpgradeCb, usersToUpgrade, _iteratorNormalCompletion4, _didIteratorError4, _iteratorError4, _iterator4, _step4, userId;

        return _regenerator2.default.wrap(function _callee2$(_context2) {
            while (1) {
                switch (_context2.prev = _context2.next) {
                    case 0:
                        _context2.next = 2;
                        return (0, _bluebird.resolve)(this._crossSigningInfo.resetKeys(level));

                    case 2:
                        _context2.next = 4;
                        return (0, _bluebird.resolve)(this._signObject(this._crossSigningInfo.keys.master));

                    case 4:
                        _context2.next = 6;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                            _this2._cryptoStore.storeCrossSigningKeys(txn, _this2._crossSigningInfo.keys);
                        }));

                    case 6:

                        // send keys to server
                        keys = {};
                        _iteratorNormalCompletion2 = true;
                        _didIteratorError2 = false;
                        _iteratorError2 = undefined;
                        _context2.prev = 10;

                        for (_iterator2 = (0, _getIterator3.default)((0, _entries2.default)(this._crossSigningInfo.keys)); !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                            _step2$value = (0, _slicedToArray3.default)(_step2.value, 2), name = _step2$value[0], key = _step2$value[1];

                            keys[name + "_key"] = key;
                        }
                        _context2.next = 18;
                        break;

                    case 14:
                        _context2.prev = 14;
                        _context2.t0 = _context2['catch'](10);
                        _didIteratorError2 = true;
                        _iteratorError2 = _context2.t0;

                    case 18:
                        _context2.prev = 18;
                        _context2.prev = 19;

                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                            _iterator2.return();
                        }

                    case 21:
                        _context2.prev = 21;

                        if (!_didIteratorError2) {
                            _context2.next = 24;
                            break;
                        }

                        throw _iteratorError2;

                    case 24:
                        return _context2.finish(21);

                    case 25:
                        return _context2.finish(18);

                    case 26:
                        _context2.next = 28;
                        return (0, _bluebird.resolve)(this._baseApis.uploadDeviceSigningKeys(authDict || {}, keys));

                    case 28:
                        this._baseApis.emit("crossSigning.keysChanged", {});

                        // sign the current device with the new key, and upload to the server
                        device = this._deviceList.getStoredDevice(this._userId, this._deviceId);
                        _context2.next = 32;
                        return (0, _bluebird.resolve)(this._crossSigningInfo.signDevice(this._userId, device));

                    case 32:
                        signedDevice = _context2.sent;
                        _context2.next = 35;
                        return (0, _bluebird.resolve)(this._baseApis.uploadKeySignatures((0, _defineProperty3.default)({}, this._userId, (0, _defineProperty3.default)({}, this._deviceId, signedDevice))));

                    case 35:

                        // check all users for signatures
                        // FIXME: do this in batches
                        users = {};
                        _iteratorNormalCompletion3 = true;
                        _didIteratorError3 = false;
                        _iteratorError3 = undefined;
                        _context2.prev = 39;
                        _iterator3 = (0, _getIterator3.default)((0, _entries2.default)(this._deviceList._crossSigningInfo));

                    case 41:
                        if (_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done) {
                            _context2.next = 50;
                            break;
                        }

                        _step3$value = (0, _slicedToArray3.default)(_step3.value, 2), _userId2 = _step3$value[0], crossSigningInfo = _step3$value[1];
                        _context2.next = 45;
                        return (0, _bluebird.resolve)(this._checkForDeviceVerificationUpgrade(_userId2, _CrossSigning.CrossSigningInfo.fromStorage(crossSigningInfo, _userId2)));

                    case 45:
                        upgradeInfo = _context2.sent;

                        if (upgradeInfo) {
                            users[_userId2] = upgradeInfo;
                        }

                    case 47:
                        _iteratorNormalCompletion3 = true;
                        _context2.next = 41;
                        break;

                    case 50:
                        _context2.next = 56;
                        break;

                    case 52:
                        _context2.prev = 52;
                        _context2.t1 = _context2['catch'](39);
                        _didIteratorError3 = true;
                        _iteratorError3 = _context2.t1;

                    case 56:
                        _context2.prev = 56;
                        _context2.prev = 57;

                        if (!_iteratorNormalCompletion3 && _iterator3.return) {
                            _iterator3.return();
                        }

                    case 59:
                        _context2.prev = 59;

                        if (!_didIteratorError3) {
                            _context2.next = 62;
                            break;
                        }

                        throw _iteratorError3;

                    case 62:
                        return _context2.finish(59);

                    case 63:
                        return _context2.finish(56);

                    case 64:
                        shouldUpgradeCb = this._baseApis._cryptoCallbacks.shouldUpgradeDeviceVerifications;

                        if (!((0, _keys2.default)(users).length > 0 && shouldUpgradeCb)) {
                            _context2.next = 103;
                            break;
                        }

                        _context2.prev = 66;
                        _context2.next = 69;
                        return (0, _bluebird.resolve)(shouldUpgradeCb({ users: users }));

                    case 69:
                        usersToUpgrade = _context2.sent;

                        if (!usersToUpgrade) {
                            _context2.next = 98;
                            break;
                        }

                        _iteratorNormalCompletion4 = true;
                        _didIteratorError4 = false;
                        _iteratorError4 = undefined;
                        _context2.prev = 74;
                        _iterator4 = (0, _getIterator3.default)(usersToUpgrade);

                    case 76:
                        if (_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done) {
                            _context2.next = 84;
                            break;
                        }

                        userId = _step4.value;

                        if (!(userId in users)) {
                            _context2.next = 81;
                            break;
                        }

                        _context2.next = 81;
                        return (0, _bluebird.resolve)(this._baseApis.setDeviceVerified(userId, users[userId].crossSigningInfo.getId()));

                    case 81:
                        _iteratorNormalCompletion4 = true;
                        _context2.next = 76;
                        break;

                    case 84:
                        _context2.next = 90;
                        break;

                    case 86:
                        _context2.prev = 86;
                        _context2.t2 = _context2['catch'](74);
                        _didIteratorError4 = true;
                        _iteratorError4 = _context2.t2;

                    case 90:
                        _context2.prev = 90;
                        _context2.prev = 91;

                        if (!_iteratorNormalCompletion4 && _iterator4.return) {
                            _iterator4.return();
                        }

                    case 93:
                        _context2.prev = 93;

                        if (!_didIteratorError4) {
                            _context2.next = 96;
                            break;
                        }

                        throw _iteratorError4;

                    case 96:
                        return _context2.finish(93);

                    case 97:
                        return _context2.finish(90);

                    case 98:
                        _context2.next = 103;
                        break;

                    case 100:
                        _context2.prev = 100;
                        _context2.t3 = _context2['catch'](66);

                        _logger2.default.log("shouldUpgradeDeviceVerifications threw an error: not upgrading", _context2.t3);

                    case 103:
                    case 'end':
                        return _context2.stop();
                }
            }
        }, _callee2, this, [[10, 14, 18, 26], [19,, 21, 25], [39, 52, 56, 64], [57,, 59, 63], [66, 100], [74, 86, 90, 98], [91,, 93, 97]]);
    }));

    return function (_x, _x2) {
        return _ref2.apply(this, arguments);
    };
}();

/**
 * Check if a user's cross-signing key is a candidate for upgrading from device
 * verification.
 *
 * @param {string} userId the user whose cross-signing information is to be checked
 * @param {object} crossSigningInfo the cross-signing information to check
 */
Crypto.prototype._checkForDeviceVerificationUpgrade = function () {
    var _ref3 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee3(userId, crossSigningInfo) {
        var trustLevel, devices, deviceIds;
        return _regenerator2.default.wrap(function _callee3$(_context3) {
            while (1) {
                switch (_context3.prev = _context3.next) {
                    case 0:
                        // only upgrade if this is the first cross-signing key that we've seen for
                        // them, and if their cross-signing key isn't already verified
                        trustLevel = this._crossSigningInfo.checkUserTrust(crossSigningInfo);

                        if (!(crossSigningInfo.firstUse && !trustLevel.verified)) {
                            _context3.next = 8;
                            break;
                        }

                        devices = this._deviceList.getRawStoredDevicesForUser(userId);
                        _context3.next = 5;
                        return (0, _bluebird.resolve)(this._checkForValidDeviceSignature(userId, crossSigningInfo.keys.master, devices));

                    case 5:
                        deviceIds = _context3.sent;

                        if (!deviceIds.length) {
                            _context3.next = 8;
                            break;
                        }

                        return _context3.abrupt('return', {
                            devices: deviceIds.map(function (deviceId) {
                                return DeviceInfo.fromStorage(devices[deviceId], deviceId);
                            }),
                            crossSigningInfo: crossSigningInfo
                        });

                    case 8:
                    case 'end':
                        return _context3.stop();
                }
            }
        }, _callee3, this);
    }));

    return function (_x3, _x4) {
        return _ref3.apply(this, arguments);
    };
}();

/**
 * Check if the cross-signing key is signed by a verified device.
 *
 * @param {string} userId the user ID whose key is being checked
 * @param {object} key the key that is being checked
 * @param {object} devices the user's devices.  Should be a map from device ID
 *     to device info
 */
Crypto.prototype._checkForValidDeviceSignature = function () {
    var _ref4 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee4(userId, key, devices) {
        var deviceIds, _iteratorNormalCompletion5, _didIteratorError5, _iteratorError5, _iterator5, _step5, signame, _signame$split, _signame$split2, deviceId;

        return _regenerator2.default.wrap(function _callee4$(_context4) {
            while (1) {
                switch (_context4.prev = _context4.next) {
                    case 0:
                        deviceIds = [];

                        if (!(devices && key.signatures && key.signatures[userId])) {
                            _context4.next = 36;
                            break;
                        }

                        _iteratorNormalCompletion5 = true;
                        _didIteratorError5 = false;
                        _iteratorError5 = undefined;
                        _context4.prev = 5;
                        _iterator5 = (0, _getIterator3.default)((0, _keys2.default)(key.signatures[userId]));

                    case 7:
                        if (_iteratorNormalCompletion5 = (_step5 = _iterator5.next()).done) {
                            _context4.next = 22;
                            break;
                        }

                        signame = _step5.value;
                        _signame$split = signame.split(':', 2), _signame$split2 = (0, _slicedToArray3.default)(_signame$split, 2), deviceId = _signame$split2[1];

                        if (!(deviceId in devices && devices[deviceId].verified === DeviceVerification.VERIFIED)) {
                            _context4.next = 19;
                            break;
                        }

                        _context4.prev = 11;
                        _context4.next = 14;
                        return (0, _bluebird.resolve)(olmlib.verifySignature(this._olmDevice, key, userId, deviceId, devices[deviceId].keys[signame]));

                    case 14:
                        deviceIds.push(deviceId);
                        _context4.next = 19;
                        break;

                    case 17:
                        _context4.prev = 17;
                        _context4.t0 = _context4['catch'](11);

                    case 19:
                        _iteratorNormalCompletion5 = true;
                        _context4.next = 7;
                        break;

                    case 22:
                        _context4.next = 28;
                        break;

                    case 24:
                        _context4.prev = 24;
                        _context4.t1 = _context4['catch'](5);
                        _didIteratorError5 = true;
                        _iteratorError5 = _context4.t1;

                    case 28:
                        _context4.prev = 28;
                        _context4.prev = 29;

                        if (!_iteratorNormalCompletion5 && _iterator5.return) {
                            _iterator5.return();
                        }

                    case 31:
                        _context4.prev = 31;

                        if (!_didIteratorError5) {
                            _context4.next = 34;
                            break;
                        }

                        throw _iteratorError5;

                    case 34:
                        return _context4.finish(31);

                    case 35:
                        return _context4.finish(28);

                    case 36:
                        return _context4.abrupt('return', deviceIds);

                    case 37:
                    case 'end':
                        return _context4.stop();
                }
            }
        }, _callee4, this, [[5, 24, 28, 36], [11, 17], [29,, 31, 35]]);
    }));

    return function (_x5, _x6, _x7) {
        return _ref4.apply(this, arguments);
    };
}();

/**
 * Get the user's cross-signing key ID.
 *
 * @param {string} [type=master] The type of key to get the ID of.  One of
 *     "master", "self_signing", or "user_signing".  Defaults to "master".
 *
 * @returns {string} the key ID
 */
Crypto.prototype.getCrossSigningId = function (type) {
    return this._crossSigningInfo.getId(type);
};

/**
 * Get the cross signing information for a given user.
 *
 * @param {string} userId the user ID to get the cross-signing info for.
 *
 * @returns {CrossSigningInfo} the cross signing informmation for the user.
 */
Crypto.prototype.getStoredCrossSigningForUser = function (userId) {
    return this._deviceList.getStoredCrossSigningForUser(userId);
};

/**
 * Check whether a given user is trusted.
 *
 * @param {string} userId The ID of the user to check.
 *
 * @returns {UserTrustLevel}
 */
Crypto.prototype.checkUserTrust = function (userId) {
    var userCrossSigning = this._deviceList.getStoredCrossSigningForUser(userId);
    if (!userCrossSigning) {
        return new _CrossSigning.UserTrustLevel(false, false);
    }
    return this._crossSigningInfo.checkUserTrust(userCrossSigning);
};

/**
 * Check whether a given device is trusted.
 *
 * @param {string} userId The ID of the user whose devices is to be checked.
 * @param {string} deviceId The ID of the device to check
 *
 * @returns {DeviceTrustLevel}
 */
Crypto.prototype.checkDeviceTrust = function (userId, deviceId) {
    var device = this._deviceList.getStoredDevice(userId, deviceId);
    var trustedLocally = device && device.isVerified();

    var userCrossSigning = this._deviceList.getStoredCrossSigningForUser(userId);
    if (device && userCrossSigning) {
        return this._crossSigningInfo.checkDeviceTrust(userCrossSigning, device, trustedLocally);
    } else {
        return new _CrossSigning.DeviceTrustLevel(false, false, trustedLocally);
    }
};

/*
 * Event handler for DeviceList's userNewDevices event
 */
Crypto.prototype._onDeviceListUserCrossSigningUpdated = function () {
    var _ref5 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee5(userId) {
        var newCrossSigning, seenPubkey, currentPubkey, changed;
        return _regenerator2.default.wrap(function _callee5$(_context5) {
            while (1) {
                switch (_context5.prev = _context5.next) {
                    case 0:
                        if (!(userId === this._userId)) {
                            _context5.next = 13;
                            break;
                        }

                        // An update to our own cross-signing key.
                        // Get the new key first:
                        newCrossSigning = this._deviceList.getStoredCrossSigningForUser(userId);
                        seenPubkey = newCrossSigning ? newCrossSigning.getId() : null;
                        currentPubkey = this._crossSigningInfo.getId();
                        changed = currentPubkey !== seenPubkey;

                        if (!(currentPubkey && seenPubkey && !changed)) {
                            _context5.next = 10;
                            break;
                        }

                        _context5.next = 8;
                        return (0, _bluebird.resolve)(this.checkOwnCrossSigningTrust());

                    case 8:
                        _context5.next = 11;
                        break;

                    case 10:
                        this.emit("crossSigning.keysChanged", {});
                        // We'll now be in a state where cross-signing on the account is not trusted
                        // because our locally stored cross-signing keys will not match the ones
                        // on the server for our account. The app must call checkOwnCrossSigningTrust()
                        // to fix this.
                        // XXX: Do we need to do something to emit events saying every device has become
                        // untrusted?

                    case 11:
                        _context5.next = 16;
                        break;

                    case 13:
                        _context5.next = 15;
                        return (0, _bluebird.resolve)(this._checkDeviceVerifications(userId));

                    case 15:
                        this.emit("userTrustStatusChanged", userId, this.checkUserTrust(userId));

                    case 16:
                    case 'end':
                        return _context5.stop();
                }
            }
        }, _callee5, this);
    }));

    return function (_x8) {
        return _ref5.apply(this, arguments);
    };
}();

/*
 * Check the copy of our cross-signing key that we have in the device list and
 * see if we can get the private key. If so, mark it as trusted.
 */
Crypto.prototype.checkOwnCrossSigningTrust = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee6() {
    var _this3 = this;

    var userId, newCrossSigning, seenPubkey, changed, signing, ret, oldSelfSigningId, oldUserSigningId, keySignatures, device, signedDevice;
    return _regenerator2.default.wrap(function _callee6$(_context6) {
        while (1) {
            switch (_context6.prev = _context6.next) {
                case 0:
                    userId = this._userId;

                    // If we see an update to our own master key, check it against the master
                    // key we have and, if it matches, mark it as verified

                    // First, get the new cross-signing info

                    newCrossSigning = this._deviceList.getStoredCrossSigningForUser(userId);

                    if (newCrossSigning) {
                        _context6.next = 5;
                        break;
                    }

                    _logger2.default.error("Got cross-signing update event for user " + userId + " but no new cross-signing information found!");
                    return _context6.abrupt('return');

                case 5:
                    seenPubkey = newCrossSigning.getId();
                    changed = this._crossSigningInfo.getId() !== seenPubkey;

                    if (!changed) {
                        _context6.next = 19;
                        break;
                    }

                    // try to get the private key if the master key changed
                    _logger2.default.info("Got new master key", seenPubkey);

                    signing = null;
                    _context6.prev = 10;
                    _context6.next = 13;
                    return (0, _bluebird.resolve)(this._crossSigningInfo.getCrossSigningKey('master', seenPubkey));

                case 13:
                    ret = _context6.sent;

                    signing = ret[1];

                case 15:
                    _context6.prev = 15;

                    signing.free();
                    return _context6.finish(15);

                case 18:

                    _logger2.default.info("Got matching private key from callback for new public master key");

                case 19:
                    oldSelfSigningId = this._crossSigningInfo.getId("self_signing");
                    oldUserSigningId = this._crossSigningInfo.getId("user_signing");

                    // Update the version of our keys in our cross-signing object and the local store

                    this._crossSigningInfo.setKeys(newCrossSigning.keys);
                    _context6.next = 24;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                        _this3._cryptoStore.storeCrossSigningKeys(txn, _this3._crossSigningInfo.keys);
                    }));

                case 24:
                    keySignatures = {};

                    if (!(oldSelfSigningId !== newCrossSigning.getId("self_signing"))) {
                        _context6.next = 32;
                        break;
                    }

                    _logger2.default.info("Got new self-signing key", newCrossSigning.getId("self_signing"));

                    device = this._deviceList.getStoredDevice(this._userId, this._deviceId);
                    _context6.next = 30;
                    return (0, _bluebird.resolve)(this._crossSigningInfo.signDevice(this._userId, device));

                case 30:
                    signedDevice = _context6.sent;

                    keySignatures[this._deviceId] = signedDevice;

                case 32:
                    if (oldUserSigningId !== newCrossSigning.getId("user_signing")) {
                        _logger2.default.info("Got new user-signing key", newCrossSigning.getId("user_signing"));
                    }

                    if (!changed) {
                        _context6.next = 37;
                        break;
                    }

                    _context6.next = 36;
                    return (0, _bluebird.resolve)(this._signObject(this._crossSigningInfo.keys.master));

                case 36:
                    keySignatures[this._crossSigningInfo.getId()] = this._crossSigningInfo.keys.master;

                case 37:
                    if (!(0, _keys2.default)(keySignatures).length) {
                        _context6.next = 40;
                        break;
                    }

                    _context6.next = 40;
                    return (0, _bluebird.resolve)(this._baseApis.uploadKeySignatures((0, _defineProperty3.default)({}, this._userId, keySignatures)));

                case 40:

                    this.emit("userTrustStatusChanged", userId, this.checkUserTrust(userId));

                    // Now we may be able to trust our key backup
                    _context6.next = 43;
                    return (0, _bluebird.resolve)(this.checkKeyBackup());

                case 43:
                case 'end':
                    return _context6.stop();
            }
        }
    }, _callee6, this, [[10,, 15, 18]]);
}));

/**
 * Check if the master key is signed by a verified device, and if so, prompt
 * the application to mark it as verified.
 *
 * @param {string} userId the user ID whose key should be checked
 */
Crypto.prototype._checkDeviceVerifications = function () {
    var _ref7 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee7(userId) {
        var crossSigningInfo, upgradeInfo, shouldUpgradeCb, usersToUpgrade;
        return _regenerator2.default.wrap(function _callee7$(_context7) {
            while (1) {
                switch (_context7.prev = _context7.next) {
                    case 0:
                        if (!this._crossSigningInfo.keys.user_signing) {
                            _context7.next = 14;
                            break;
                        }

                        crossSigningInfo = this._deviceList.getStoredCrossSigningForUser(userId);

                        if (!crossSigningInfo) {
                            _context7.next = 14;
                            break;
                        }

                        _context7.next = 5;
                        return (0, _bluebird.resolve)(this._checkForDeviceVerificationUpgrade(userId, crossSigningInfo));

                    case 5:
                        upgradeInfo = _context7.sent;
                        shouldUpgradeCb = this._baseApis._cryptoCallbacks.shouldUpgradeDeviceVerifications;

                        if (!(upgradeInfo && shouldUpgradeCb)) {
                            _context7.next = 14;
                            break;
                        }

                        _context7.next = 10;
                        return (0, _bluebird.resolve)(shouldUpgradeCb({
                            users: (0, _defineProperty3.default)({}, userId, upgradeInfo)
                        }));

                    case 10:
                        usersToUpgrade = _context7.sent;

                        if (!usersToUpgrade.includes(userId)) {
                            _context7.next = 14;
                            break;
                        }

                        _context7.next = 14;
                        return (0, _bluebird.resolve)(this._baseApis.setDeviceVerified(userId, crossSigningInfo.getId()));

                    case 14:
                    case 'end':
                        return _context7.stop();
                }
            }
        }, _callee7, this);
    }));

    return function (_x9) {
        return _ref7.apply(this, arguments);
    };
}();

/**
 * Check the server for an active key backup and
 * if one is present and has a valid signature from
 * one of the user's verified devices, start backing up
 * to it.
 */
Crypto.prototype._checkAndStartKeyBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee8() {
    var backupInfo, trustInfo;
    return _regenerator2.default.wrap(function _callee8$(_context8) {
        while (1) {
            switch (_context8.prev = _context8.next) {
                case 0:
                    _logger2.default.log("Checking key backup status...");

                    if (!this._baseApis.isGuest()) {
                        _context8.next = 5;
                        break;
                    }

                    _logger2.default.log("Skipping key backup check since user is guest");
                    this._checkedForBackup = true;
                    return _context8.abrupt('return', null);

                case 5:
                    backupInfo = void 0;
                    _context8.prev = 6;
                    _context8.next = 9;
                    return (0, _bluebird.resolve)(this._baseApis.getKeyBackupVersion());

                case 9:
                    backupInfo = _context8.sent;
                    _context8.next = 17;
                    break;

                case 12:
                    _context8.prev = 12;
                    _context8.t0 = _context8['catch'](6);

                    _logger2.default.log("Error checking for active key backup", _context8.t0);
                    if (_context8.t0.httpStatus / 100 === 4) {
                        // well that's told us. we won't try again.
                        this._checkedForBackup = true;
                    }
                    return _context8.abrupt('return', null);

                case 17:
                    this._checkedForBackup = true;

                    _context8.next = 20;
                    return (0, _bluebird.resolve)(this.isKeyBackupTrusted(backupInfo));

                case 20:
                    trustInfo = _context8.sent;


                    if (trustInfo.usable && !this.backupInfo) {
                        _logger2.default.log("Found usable key backup v" + backupInfo.version + ": enabling key backups");
                        this._baseApis.enableKeyBackup(backupInfo);
                    } else if (!trustInfo.usable && this.backupInfo) {
                        _logger2.default.log("No usable key backup: disabling key backup");
                        this._baseApis.disableKeyBackup();
                    } else if (!trustInfo.usable && !this.backupInfo) {
                        _logger2.default.log("No usable key backup: not enabling key backup");
                    } else if (trustInfo.usable && this.backupInfo) {
                        // may not be the same version: if not, we should switch
                        if (backupInfo.version !== this.backupInfo.version) {
                            _logger2.default.log("On backup version " + this.backupInfo.version + " but found " + "version " + backupInfo.version + ": switching.");
                            this._baseApis.disableKeyBackup();
                            this._baseApis.enableKeyBackup(backupInfo);
                        } else {
                            _logger2.default.log("Backup version " + backupInfo.version + " still current");
                        }
                    }

                    return _context8.abrupt('return', { backupInfo: backupInfo, trustInfo: trustInfo });

                case 23:
                case 'end':
                    return _context8.stop();
            }
        }
    }, _callee8, this, [[6, 12]]);
}));

Crypto.prototype.setTrustedBackupPubKey = function () {
    var _ref9 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee9(trustedPubKey) {
        return _regenerator2.default.wrap(function _callee9$(_context9) {
            while (1) {
                switch (_context9.prev = _context9.next) {
                    case 0:
                        // This should be redundant post cross-signing is a thing, so just
                        // plonk it in localStorage for now.
                        this._sessionStore.setLocalTrustedBackupPubKey(trustedPubKey);
                        _context9.next = 3;
                        return (0, _bluebird.resolve)(this.checkKeyBackup());

                    case 3:
                    case 'end':
                        return _context9.stop();
                }
            }
        }, _callee9, this);
    }));

    return function (_x10) {
        return _ref9.apply(this, arguments);
    };
}();

/**
 * Forces a re-check of the key backup and enables/disables it
 * as appropriate.
 *
 * @return {Object} Object with backup info (as returned by
 *     getKeyBackupVersion) in backupInfo and
 *     trust information (as returned by isKeyBackupTrusted)
 *     in trustInfo.
 */
Crypto.prototype.checkKeyBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee10() {
    var returnInfo;
    return _regenerator2.default.wrap(function _callee10$(_context10) {
        while (1) {
            switch (_context10.prev = _context10.next) {
                case 0:
                    this._checkedForBackup = false;
                    _context10.next = 3;
                    return (0, _bluebird.resolve)(this._checkAndStartKeyBackup());

                case 3:
                    returnInfo = _context10.sent;
                    return _context10.abrupt('return', returnInfo);

                case 5:
                case 'end':
                    return _context10.stop();
            }
        }
    }, _callee10, this);
}));

/**
 * @param {object} backupInfo key backup info dict from /room_keys/version
 * @return {object} {
 *     usable: [bool], // is the backup trusted, true iff there is a sig that is valid & from a trusted device
 *     sigs: [
 *         valid: [bool || null], // true: valid, false: invalid, null: cannot attempt validation
 *         deviceId: [string],
 *         device: [DeviceInfo || null],
 *     ]
 * }
 */
Crypto.prototype.isKeyBackupTrusted = function () {
    var _ref11 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee11(backupInfo) {
        var ret, trustedPubkey, mySigs, _iteratorNormalCompletion6, _didIteratorError6, _iteratorError6, _iterator6, _step6, keyId, keyIdParts, sigInfo, crossSigningId, device;

        return _regenerator2.default.wrap(function _callee11$(_context11) {
            while (1) {
                switch (_context11.prev = _context11.next) {
                    case 0:
                        ret = {
                            usable: false,
                            trusted_locally: false,
                            sigs: []
                        };

                        if (!(!backupInfo || !backupInfo.algorithm || !backupInfo.auth_data || !backupInfo.auth_data.public_key || !backupInfo.auth_data.signatures)) {
                            _context11.next = 4;
                            break;
                        }

                        _logger2.default.info("Key backup is absent or missing required data");
                        return _context11.abrupt('return', ret);

                    case 4:
                        trustedPubkey = this._sessionStore.getLocalTrustedBackupPubKey();


                        if (backupInfo.auth_data.public_key === trustedPubkey) {
                            _logger2.default.info("Backup public key " + trustedPubkey + " is trusted locally");
                            ret.trusted_locally = true;
                        }

                        mySigs = backupInfo.auth_data.signatures[this._userId] || [];
                        _iteratorNormalCompletion6 = true;
                        _didIteratorError6 = false;
                        _iteratorError6 = undefined;
                        _context11.prev = 10;
                        _iterator6 = (0, _getIterator3.default)((0, _keys2.default)(mySigs));

                    case 12:
                        if (_iteratorNormalCompletion6 = (_step6 = _iterator6.next()).done) {
                            _context11.next = 55;
                            break;
                        }

                        keyId = _step6.value;
                        keyIdParts = keyId.split(':');

                        if (!(keyIdParts[0] !== 'ed25519')) {
                            _context11.next = 18;
                            break;
                        }

                        _logger2.default.log("Ignoring unknown signature type: " + keyIdParts[0]);
                        return _context11.abrupt('continue', 52);

                    case 18:
                        // Could be an SSK but just say this is the device ID for backwards compat
                        sigInfo = { deviceId: keyIdParts[1] }; // XXX: is this how we're supposed to get the device ID?

                        // first check to see if it's from our cross-signing key

                        crossSigningId = this._crossSigningInfo.getId();

                        if (!(crossSigningId === keyId)) {
                            _context11.next = 34;
                            break;
                        }

                        sigInfo.cross_signing_key = crossSigningId;
                        _context11.prev = 22;
                        _context11.next = 25;
                        return (0, _bluebird.resolve)(olmlib.verifySignature(this._olmDevice, backupInfo.auth_data, this._userId, sigInfo.deviceId, crossSigningId));

                    case 25:
                        sigInfo.valid = true;
                        _context11.next = 32;
                        break;

                    case 28:
                        _context11.prev = 28;
                        _context11.t0 = _context11['catch'](22);

                        _logger2.default.warning("Bad signature from cross signing key " + crossSigningId, _context11.t0);
                        sigInfo.valid = false;

                    case 32:
                        ret.sigs.push(sigInfo);
                        return _context11.abrupt('continue', 52);

                    case 34:

                        // Now look for a sig from a device
                        // At some point this can probably go away and we'll just support
                        // it being signed by the SSK
                        device = this._deviceList.getStoredDevice(this._userId, sigInfo.deviceId);

                        if (!device) {
                            _context11.next = 49;
                            break;
                        }

                        sigInfo.device = device;
                        _context11.prev = 37;
                        _context11.next = 40;
                        return (0, _bluebird.resolve)(olmlib.verifySignature(this._olmDevice,
                        // verifySignature modifies the object so we need to copy
                        // if we verify more than one sig
                        (0, _assign2.default)({}, backupInfo.auth_data), this._userId, device.deviceId, device.getFingerprint()));

                    case 40:
                        sigInfo.valid = true;
                        _context11.next = 47;
                        break;

                    case 43:
                        _context11.prev = 43;
                        _context11.t1 = _context11['catch'](37);

                        _logger2.default.info("Bad signature from key ID " + keyId + " userID " + this._userId + " device ID " + device.deviceId + " fingerprint: " + device.getFingerprint(), backupInfo.auth_data, _context11.t1);
                        sigInfo.valid = false;

                    case 47:
                        _context11.next = 51;
                        break;

                    case 49:
                        sigInfo.valid = null; // Can't determine validity because we don't have the signing device
                        _logger2.default.info("Ignoring signature from unknown key " + keyId);

                    case 51:
                        ret.sigs.push(sigInfo);

                    case 52:
                        _iteratorNormalCompletion6 = true;
                        _context11.next = 12;
                        break;

                    case 55:
                        _context11.next = 61;
                        break;

                    case 57:
                        _context11.prev = 57;
                        _context11.t2 = _context11['catch'](10);
                        _didIteratorError6 = true;
                        _iteratorError6 = _context11.t2;

                    case 61:
                        _context11.prev = 61;
                        _context11.prev = 62;

                        if (!_iteratorNormalCompletion6 && _iterator6.return) {
                            _iterator6.return();
                        }

                    case 64:
                        _context11.prev = 64;

                        if (!_didIteratorError6) {
                            _context11.next = 67;
                            break;
                        }

                        throw _iteratorError6;

                    case 67:
                        return _context11.finish(64);

                    case 68:
                        return _context11.finish(61);

                    case 69:

                        ret.usable = ret.sigs.some(function (s) {
                            return s.valid && (s.device && s.device.isVerified() || s.cross_signing_key);
                        });
                        ret.usable |= ret.trusted_locally;
                        return _context11.abrupt('return', ret);

                    case 72:
                    case 'end':
                        return _context11.stop();
                }
            }
        }, _callee11, this, [[10, 57, 61, 69], [22, 28], [37, 43], [62,, 64, 68]]);
    }));

    return function (_x11) {
        return _ref11.apply(this, arguments);
    };
}();

/**
 */
Crypto.prototype.enableLazyLoading = function () {
    this._lazyLoadMembers = true;
};

/**
 * Tell the crypto module to register for MatrixClient events which it needs to
 * listen for
 *
 * @param {external:EventEmitter} eventEmitter event source where we can register
 *    for event notifications
 */
Crypto.prototype.registerEventHandlers = function (eventEmitter) {
    var crypto = this;

    eventEmitter.on("RoomMember.membership", function (event, member, oldMembership) {
        try {
            crypto._onRoomMembership(event, member, oldMembership);
        } catch (e) {
            _logger2.default.error("Error handling membership change:", e);
        }
    });

    eventEmitter.on("toDeviceEvent", function (event) {
        crypto._onToDeviceEvent(event);
    });

    eventEmitter.on("Room.timeline", function (event) {
        crypto._onTimelineEvent(event);
    });

    eventEmitter.on("Event.decrypted", function (event) {
        crypto._onTimelineEvent(event);
    });
};

/** Start background processes related to crypto */
Crypto.prototype.start = function () {
    this._outgoingRoomKeyRequestManager.start();
};

/** Stop background processes related to crypto */
Crypto.prototype.stop = function () {
    this._outgoingRoomKeyRequestManager.stop();
    this._deviceList.stop();
};

/**
 * @return {string} The version of Olm.
 */
Crypto.getOlmVersion = function () {
    return OlmDevice.getOlmVersion();
};

/**
 * Get the Ed25519 key for this device
 *
 * @return {string} base64-encoded ed25519 key.
 */
Crypto.prototype.getDeviceEd25519Key = function () {
    return this._olmDevice.deviceEd25519Key;
};

/**
 * Set the global override for whether the client should ever send encrypted
 * messages to unverified devices.  This provides the default for rooms which
 * do not specify a value.
 *
 * @param {boolean} value whether to blacklist all unverified devices by default
 */
Crypto.prototype.setGlobalBlacklistUnverifiedDevices = function (value) {
    this._globalBlacklistUnverifiedDevices = value;
};

/**
 * @return {boolean} whether to blacklist all unverified devices by default
 */
Crypto.prototype.getGlobalBlacklistUnverifiedDevices = function () {
    return this._globalBlacklistUnverifiedDevices;
};

/**
 * Upload the device keys to the homeserver.
 * @return {object} A promise that will resolve when the keys are uploaded.
 */
Crypto.prototype.uploadDeviceKeys = function () {
    var crypto = this;
    var userId = crypto._userId;
    var deviceId = crypto._deviceId;

    var deviceKeys = {
        algorithms: crypto._supportedAlgorithms,
        device_id: deviceId,
        keys: crypto._deviceKeys,
        user_id: userId
    };

    return crypto._signObject(deviceKeys).then(function () {
        return crypto._baseApis.uploadKeysRequest({
            device_keys: deviceKeys
        }, {
            // for now, we set the device id explicitly, as we may not be using the
            // same one as used in login.
            device_id: deviceId
        });
    });
};

/**
 * Stores the current one_time_key count which will be handled later (in a call of
 * onSyncCompleted). The count is e.g. coming from a /sync response.
 *
 * @param {Number} currentCount The current count of one_time_keys to be stored
 */
Crypto.prototype.updateOneTimeKeyCount = function (currentCount) {
    if (isFinite(currentCount)) {
        this._oneTimeKeyCount = currentCount;
    } else {
        throw new TypeError("Parameter for updateOneTimeKeyCount has to be a number");
    }
};

// check if it's time to upload one-time keys, and do so if so.
function _maybeUploadOneTimeKeys(crypto) {
    // frequency with which to check & upload one-time keys
    var uploadPeriod = 1000 * 60; // one minute

    // max number of keys to upload at once
    // Creating keys can be an expensive operation so we limit the
    // number we generate in one go to avoid blocking the application
    // for too long.
    var maxKeysPerCycle = 5;

    if (crypto._oneTimeKeyCheckInProgress) {
        return;
    }

    var now = Date.now();
    if (crypto._lastOneTimeKeyCheck !== null && now - crypto._lastOneTimeKeyCheck < uploadPeriod) {
        // we've done a key upload recently.
        return;
    }

    crypto._lastOneTimeKeyCheck = now;

    // We need to keep a pool of one time public keys on the server so that
    // other devices can start conversations with us. But we can only store
    // a finite number of private keys in the olm Account object.
    // To complicate things further then can be a delay between a device
    // claiming a public one time key from the server and it sending us a
    // message. We need to keep the corresponding private key locally until
    // we receive the message.
    // But that message might never arrive leaving us stuck with duff
    // private keys clogging up our local storage.
    // So we need some kind of enginering compromise to balance all of
    // these factors.

    // Check how many keys we can store in the Account object.
    var maxOneTimeKeys = crypto._olmDevice.maxNumberOfOneTimeKeys();
    // Try to keep at most half that number on the server. This leaves the
    // rest of the slots free to hold keys that have been claimed from the
    // server but we haven't recevied a message for.
    // If we run out of slots when generating new keys then olm will
    // discard the oldest private keys first. This will eventually clean
    // out stale private keys that won't receive a message.
    var keyLimit = Math.floor(maxOneTimeKeys / 2);

    function uploadLoop(keyCount) {
        if (keyLimit <= keyCount) {
            // If we don't need to generate any more keys then we are done.
            return _bluebird2.default.resolve();
        }

        var keysThisLoop = Math.min(keyLimit - keyCount, maxKeysPerCycle);

        // Ask olm to generate new one time keys, then upload them to synapse.
        return crypto._olmDevice.generateOneTimeKeys(keysThisLoop).then(function () {
            return _uploadOneTimeKeys(crypto);
        }).then(function (res) {
            if (res.one_time_key_counts && res.one_time_key_counts.signed_curve25519) {
                // if the response contains a more up to date value use this
                // for the next loop
                return uploadLoop(res.one_time_key_counts.signed_curve25519);
            } else {
                throw new Error("response for uploading keys does not contain " + "one_time_key_counts.signed_curve25519");
            }
        });
    }

    crypto._oneTimeKeyCheckInProgress = true;
    _bluebird2.default.resolve().then(function () {
        if (crypto._oneTimeKeyCount !== undefined) {
            // We already have the current one_time_key count from a /sync response.
            // Use this value instead of asking the server for the current key count.
            return _bluebird2.default.resolve(crypto._oneTimeKeyCount);
        }
        // ask the server how many keys we have
        return crypto._baseApis.uploadKeysRequest({}, {
            device_id: crypto._deviceId
        }).then(function (res) {
            return res.one_time_key_counts.signed_curve25519 || 0;
        });
    }).then(function (keyCount) {
        // Start the uploadLoop with the current keyCount. The function checks if
        // we need to upload new keys or not.
        // If there are too many keys on the server then we don't need to
        // create any more keys.
        return uploadLoop(keyCount);
    }).catch(function (e) {
        _logger2.default.error("Error uploading one-time keys", e.stack || e);
    }).finally(function () {
        // reset _oneTimeKeyCount to prevent start uploading based on old data.
        // it will be set again on the next /sync-response
        crypto._oneTimeKeyCount = undefined;
        crypto._oneTimeKeyCheckInProgress = false;
    }).done();
}Crypto.prototype.downloadKeys = function (userIds, forceDownload) {
    return this._deviceList.downloadKeys(userIds, forceDownload);
};

/**
 * Get the stored device keys for a user id
 *
 * @param {string} userId the user to list keys for.
 *
 * @return {module:crypto/deviceinfo[]|null} list of devices, or null if we haven't
 * managed to get a list of devices for this user yet.
 */
Crypto.prototype.getStoredDevicesForUser = function (userId) {
    return this._deviceList.getStoredDevicesForUser(userId);
};

/**
 * Get the stored keys for a single device
 *
 * @param {string} userId
 * @param {string} deviceId
 *
 * @return {module:crypto/deviceinfo?} device, or undefined
 * if we don't know about this device
 */
Crypto.prototype.getStoredDevice = function (userId, deviceId) {
    return this._deviceList.getStoredDevice(userId, deviceId);
};

/**
 * Save the device list, if necessary
 *
 * @param {integer} delay Time in ms before which the save actually happens.
 *     By default, the save is delayed for a short period in order to batch
 *     multiple writes, but this behaviour can be disabled by passing 0.
 *
 * @return {Promise<bool>} true if the data was saved, false if
 *     it was not (eg. because no changes were pending). The promise
 *     will only resolve once the data is saved, so may take some time
 *     to resolve.
 */
Crypto.prototype.saveDeviceList = function (delay) {
    return this._deviceList.saveIfDirty(delay);
};

/**
 * Update the blocked/verified state of the given device
 *
 * @param {string} userId owner of the device
 * @param {string} deviceId unique identifier for the device
 *
 * @param {?boolean} verified whether to mark the device as verified. Null to
 *     leave unchanged.
 *
 * @param {?boolean} blocked whether to mark the device as blocked. Null to
 *      leave unchanged.
 *
 * @param {?boolean} known whether to mark that the user has been made aware of
 *      the existence of this device. Null to leave unchanged
 *
 * @return {Promise<module:crypto/deviceinfo>} updated DeviceInfo
 */
Crypto.prototype.setDeviceVerification = function () {
    var _ref13 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee13(userId, deviceId, verified, blocked, known) {
        var xsk, device, devices, dev, verificationStatus, knownStatus, _device, deviceObj;

        return _regenerator2.default.wrap(function _callee13$(_context13) {
            while (1) {
                switch (_context13.prev = _context13.next) {
                    case 0:
                        // get rid of any `undefined`s here so we can just check
                        // for null rather than null or undefined
                        if (verified === undefined) verified = null;
                        if (blocked === undefined) blocked = null;
                        if (known === undefined) known = null;

                        // Check if the 'device' is actually a cross signing key
                        // The js-sdk's verification treats cross-signing keys as devices
                        // and so uses this method to mark them verified.
                        xsk = this._deviceList.getStoredCrossSigningForUser(userId);

                        if (!(xsk && xsk.getId() === deviceId)) {
                            _context13.next = 16;
                            break;
                        }

                        if (!(blocked !== null || known !== null)) {
                            _context13.next = 7;
                            break;
                        }

                        throw new Error("Cannot set blocked or known for a cross-signing key");

                    case 7:
                        if (verified) {
                            _context13.next = 9;
                            break;
                        }

                        throw new Error("Cannot set a cross-signing key as unverified");

                    case 9:
                        _context13.next = 11;
                        return (0, _bluebird.resolve)(this._crossSigningInfo.signUser(xsk));

                    case 11:
                        device = _context13.sent;

                        if (!device) {
                            _context13.next = 15;
                            break;
                        }

                        _context13.next = 15;
                        return (0, _bluebird.resolve)(this._baseApis.uploadKeySignatures((0, _defineProperty3.default)({}, userId, (0, _defineProperty3.default)({}, deviceId, device))));

                    case 15:
                        return _context13.abrupt('return', device);

                    case 16:
                        devices = this._deviceList.getRawStoredDevicesForUser(userId);

                        if (!(!devices || !devices[deviceId])) {
                            _context13.next = 19;
                            break;
                        }

                        throw new Error("Unknown device " + userId + ":" + deviceId);

                    case 19:
                        dev = devices[deviceId];
                        verificationStatus = dev.verified;


                        if (verified) {
                            verificationStatus = DeviceVerification.VERIFIED;
                        } else if (verified !== null && verificationStatus == DeviceVerification.VERIFIED) {
                            verificationStatus = DeviceVerification.UNVERIFIED;
                        }

                        if (blocked) {
                            verificationStatus = DeviceVerification.BLOCKED;
                        } else if (blocked !== null && verificationStatus == DeviceVerification.BLOCKED) {
                            verificationStatus = DeviceVerification.UNVERIFIED;
                        }

                        knownStatus = dev.known;

                        if (known !== null) {
                            knownStatus = known;
                        }

                        if (dev.verified !== verificationStatus || dev.known !== knownStatus) {
                            dev.verified = verificationStatus;
                            dev.known = knownStatus;
                            this._deviceList.storeDevicesForUser(userId, devices);
                            this._deviceList.saveIfDirty();
                        }

                        // do cross-signing

                        if (!(verified && userId === this._userId)) {
                            _context13.next = 33;
                            break;
                        }

                        _context13.next = 29;
                        return (0, _bluebird.resolve)(this._crossSigningInfo.signDevice(userId, DeviceInfo.fromStorage(dev, deviceId)));

                    case 29:
                        _device = _context13.sent;

                        if (!_device) {
                            _context13.next = 33;
                            break;
                        }

                        _context13.next = 33;
                        return (0, _bluebird.resolve)(this._baseApis.uploadKeySignatures((0, _defineProperty3.default)({}, userId, (0, _defineProperty3.default)({}, deviceId, _device))));

                    case 33:
                        deviceObj = DeviceInfo.fromStorage(dev, deviceId);

                        this.emit("deviceVerificationChanged", userId, deviceId, deviceObj);
                        return _context13.abrupt('return', deviceObj);

                    case 36:
                    case 'end':
                        return _context13.stop();
                }
            }
        }, _callee13, this);
    }));

    return function (_x13, _x14, _x15, _x16, _x17) {
        return _ref13.apply(this, arguments);
    };
}();

function verificationEventHandler(target, userId, roomId, eventId) {
    return function (event) {
        // listen for events related to this verification
        if (event.getRoomId() !== roomId || event.getSender() !== userId) {
            return;
        }
        // ignore events that haven't been decrypted yet.
        // we also listen for undecrypted events, just in case
        // the other side would be sending unencrypted events in an e2ee room
        if (event.getType() === "m.room.encrypted") {
            return;
        }
        var relatesTo = event.getRelation();
        if (!relatesTo || !relatesTo.rel_type || relatesTo.rel_type !== "m.reference" || !relatesTo.event_id || relatesTo.event_id !== eventId) {
            return;
        }

        // the event seems to be related to this verification, so pass it on to
        // the verification handler
        target.handleEvent(event);
    };
}

Crypto.prototype.requestVerificationDM = function () {
    var _ref14 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee14(userId, roomId, methods) {
        var _this4 = this;

        var methodMap, _iteratorNormalCompletion7, _didIteratorError7, _iteratorError7, _iterator7, _step7, method, eventId, listenPromise, res;

        return _regenerator2.default.wrap(function _callee14$(_context14) {
            while (1) {
                switch (_context14.prev = _context14.next) {
                    case 0:
                        methodMap = void 0;

                        if (!methods) {
                            _context14.next = 24;
                            break;
                        }

                        methodMap = new _map2.default();
                        _iteratorNormalCompletion7 = true;
                        _didIteratorError7 = false;
                        _iteratorError7 = undefined;
                        _context14.prev = 6;
                        for (_iterator7 = (0, _getIterator3.default)(methods); !(_iteratorNormalCompletion7 = (_step7 = _iterator7.next()).done); _iteratorNormalCompletion7 = true) {
                            method = _step7.value;

                            if (typeof method === "string") {
                                methodMap.set(method, defaultVerificationMethods[method]);
                            } else if (method.NAME) {
                                methodMap.set(method.NAME, method);
                            }
                        }
                        _context14.next = 14;
                        break;

                    case 10:
                        _context14.prev = 10;
                        _context14.t0 = _context14['catch'](6);
                        _didIteratorError7 = true;
                        _iteratorError7 = _context14.t0;

                    case 14:
                        _context14.prev = 14;
                        _context14.prev = 15;

                        if (!_iteratorNormalCompletion7 && _iterator7.return) {
                            _iterator7.return();
                        }

                    case 17:
                        _context14.prev = 17;

                        if (!_didIteratorError7) {
                            _context14.next = 20;
                            break;
                        }

                        throw _iteratorError7;

                    case 20:
                        return _context14.finish(17);

                    case 21:
                        return _context14.finish(14);

                    case 22:
                        _context14.next = 25;
                        break;

                    case 24:
                        methodMap = this._baseApis._crypto._verificationMethods;

                    case 25:
                        eventId = undefined;
                        listenPromise = new _bluebird2.default(function (_resolve, _reject) {
                            var listener = function listener(event) {
                                // listen for events related to this verification
                                if (event.getRoomId() !== roomId || event.getSender() !== userId) {
                                    return;
                                }
                                var relatesTo = event.getRelation();
                                if (!relatesTo || !relatesTo.rel_type || relatesTo.rel_type !== "m.reference" || !relatesTo.event_id || relatesTo.event_id !== eventId) {
                                    return;
                                }

                                var content = event.getContent();
                                // the event seems to be related to this verification
                                switch (event.getType()) {
                                    case "m.key.verification.start":
                                        {
                                            var verifier = new (methodMap.get(content.method))(_this4._baseApis, userId, content.from_device, eventId, roomId, event);
                                            verifier.handler = verificationEventHandler(verifier, userId, roomId, eventId);
                                            // this handler gets removed when the verification finishes
                                            // (see the verify method of crypto/verification/Base.js)
                                            var subscription = listenForEvents(_this4._baseApis, roomId, verifier.handler);
                                            verifier.setEventsSubscription(subscription);
                                            resolve(verifier);
                                            break;
                                        }
                                    case "m.key.verification.cancel":
                                        {
                                            reject(event);
                                            break;
                                        }
                                }
                            };
                            var initialResponseSubscription = listenForEvents(_this4._baseApis, roomId, listener);

                            var resolve = function resolve() {
                                if (initialResponseSubscription) {
                                    initialResponseSubscription = initialResponseSubscription();
                                }
                                _resolve.apply(undefined, arguments);
                            };
                            var reject = function reject() {
                                if (initialResponseSubscription) {
                                    initialResponseSubscription = initialResponseSubscription();
                                }
                                _reject.apply(undefined, arguments);
                            };
                        });
                        _context14.next = 29;
                        return (0, _bluebird.resolve)(this._baseApis.sendEvent(roomId, "m.room.message", {
                            body: this._baseApis.getUserId() + " is requesting to verify " + "your key, but your client does not support in-chat key " + "verification.  You will need to use legacy key " + "verification to verify keys.",
                            msgtype: "m.key.verification.request",
                            to: userId,
                            from_device: this._baseApis.getDeviceId(),
                            methods: [].concat((0, _toConsumableArray3.default)(methodMap.keys()))
                        }));

                    case 29:
                        res = _context14.sent;

                        eventId = res.event_id;

                        return _context14.abrupt('return', listenPromise);

                    case 32:
                    case 'end':
                        return _context14.stop();
                }
            }
        }, _callee14, this, [[6, 10, 14, 22], [15,, 17, 21]]);
    }));

    return function (_x18, _x19, _x20) {
        return _ref14.apply(this, arguments);
    };
}();

Crypto.prototype.acceptVerificationDM = function (event, Method) {
    if (typeof Method === "string") {
        Method = defaultVerificationMethods[Method];
    }
    var content = event.getContent();
    var verifier = new Method(this._baseApis, event.getSender(), content.from_device, event.getId(), event.getRoomId());
    verifier.handler = verificationEventHandler(verifier, event.getSender(), event.getRoomId(), event.getId());
    var subscription = listenForEvents(this._baseApis, event.getRoomId(), verifier.handler);
    verifier.setEventsSubscription(subscription);
    return verifier;
};

Crypto.prototype.requestVerification = function (userId, methods, devices) {
    var _this5 = this;

    if (!methods) {
        // .keys() returns an iterator, so we need to explicitly turn it into an array
        methods = [].concat((0, _toConsumableArray3.default)(this._verificationMethods.keys()));
    }
    if (!devices) {
        devices = (0, _keys2.default)(this._deviceList.getRawStoredDevicesForUser(userId));
    }
    if (!this._verificationTransactions.has(userId)) {
        this._verificationTransactions.set(userId, new _map2.default());
    }

    var transactionId = (0, _randomstring.randomString)(32);

    var promise = new _bluebird2.default(function (resolve, reject) {
        _this5._verificationTransactions.get(userId).set(transactionId, {
            request: {
                methods: methods,
                devices: devices,
                resolve: resolve,
                reject: reject
            }
        });
    });

    var message = {
        transaction_id: transactionId,
        from_device: this._baseApis.deviceId,
        methods: methods,
        timestamp: Date.now()
    };
    var msgMap = {};
    var _iteratorNormalCompletion8 = true;
    var _didIteratorError8 = false;
    var _iteratorError8 = undefined;

    try {
        for (var _iterator8 = (0, _getIterator3.default)(devices), _step8; !(_iteratorNormalCompletion8 = (_step8 = _iterator8.next()).done); _iteratorNormalCompletion8 = true) {
            var deviceId = _step8.value;

            msgMap[deviceId] = message;
        }
    } catch (err) {
        _didIteratorError8 = true;
        _iteratorError8 = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion8 && _iterator8.return) {
                _iterator8.return();
            }
        } finally {
            if (_didIteratorError8) {
                throw _iteratorError8;
            }
        }
    }

    this._baseApis.sendToDevice("m.key.verification.request", (0, _defineProperty3.default)({}, userId, msgMap));

    return promise;
};

Crypto.prototype.beginKeyVerification = function (method, userId, deviceId, transactionId) {
    if (!this._verificationTransactions.has(userId)) {
        this._verificationTransactions.set(userId, new _map2.default());
    }
    transactionId = transactionId || (0, _randomstring.randomString)(32);
    if (this._verificationMethods.has(method)) {
        var verifier = new (this._verificationMethods.get(method))(this._baseApis, userId, deviceId, transactionId);
        if (!this._verificationTransactions.get(userId).has(transactionId)) {
            this._verificationTransactions.get(userId).set(transactionId, {});
        }
        this._verificationTransactions.get(userId).get(transactionId).verifier = verifier;
        return verifier;
    } else {
        throw (0, _Error.newUnknownMethodError)();
    }
};

/**
 * Get information on the active olm sessions with a user
 * <p>
 * Returns a map from device id to an object with keys 'deviceIdKey' (the
 * device's curve25519 identity key) and 'sessions' (an array of objects in the
 * same format as that returned by
 * {@link module:crypto/OlmDevice#getSessionInfoForDevice}).
 * <p>
 * This method is provided for debugging purposes.
 *
 * @param {string} userId id of user to inspect
 *
 * @return {Promise<Object.<string, {deviceIdKey: string, sessions: object[]}>>}
 */
Crypto.prototype.getOlmSessionsForUser = function () {
    var _ref15 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee15(userId) {
        var devices, result, j, device, deviceKey, sessions;
        return _regenerator2.default.wrap(function _callee15$(_context15) {
            while (1) {
                switch (_context15.prev = _context15.next) {
                    case 0:
                        devices = this.getStoredDevicesForUser(userId) || [];
                        result = {};
                        j = 0;

                    case 3:
                        if (!(j < devices.length)) {
                            _context15.next = 13;
                            break;
                        }

                        device = devices[j];
                        deviceKey = device.getIdentityKey();
                        _context15.next = 8;
                        return (0, _bluebird.resolve)(this._olmDevice.getSessionInfoForDevice(deviceKey));

                    case 8:
                        sessions = _context15.sent;


                        result[device.deviceId] = {
                            deviceIdKey: deviceKey,
                            sessions: sessions
                        };

                    case 10:
                        ++j;
                        _context15.next = 3;
                        break;

                    case 13:
                        return _context15.abrupt('return', result);

                    case 14:
                    case 'end':
                        return _context15.stop();
                }
            }
        }, _callee15, this);
    }));

    return function (_x21) {
        return _ref15.apply(this, arguments);
    };
}();

/**
 * Get the device which sent an event
 *
 * @param {module:models/event.MatrixEvent} event event to be checked
 *
 * @return {module:crypto/deviceinfo?}
 */
Crypto.prototype.getEventSenderDeviceInfo = function (event) {
    var senderKey = event.getSenderKey();
    var algorithm = event.getWireContent().algorithm;

    if (!senderKey || !algorithm) {
        return null;
    }

    var forwardingChain = event.getForwardingCurve25519KeyChain();
    if (forwardingChain.length > 0) {
        // we got this event from somewhere else
        // TODO: check if we can trust the forwarders.
        return null;
    }

    // senderKey is the Curve25519 identity key of the device which the event
    // was sent from. In the case of Megolm, it's actually the Curve25519
    // identity key of the device which set up the Megolm session.

    var device = this._deviceList.getDeviceByIdentityKey(algorithm, senderKey);

    if (device === null) {
        // we haven't downloaded the details of this device yet.
        return null;
    }

    // so far so good, but now we need to check that the sender of this event
    // hadn't advertised someone else's Curve25519 key as their own. We do that
    // by checking the Ed25519 claimed by the event (or, in the case of megolm,
    // the event which set up the megolm session), to check that it matches the
    // fingerprint of the purported sending device.
    //
    // (see https://github.com/vector-im/vector-web/issues/2215)

    var claimedKey = event.getClaimedEd25519Key();
    if (!claimedKey) {
        _logger2.default.warn("Event " + event.getId() + " claims no ed25519 key: " + "cannot verify sending device");
        return null;
    }

    if (claimedKey !== device.getFingerprint()) {
        _logger2.default.warn("Event " + event.getId() + " claims ed25519 key " + claimedKey + "but sender device has key " + device.getFingerprint());
        return null;
    }

    return device;
};

/**
 * Forces the current outbound group session to be discarded such
 * that another one will be created next time an event is sent.
 *
 * @param {string} roomId The ID of the room to discard the session for
 *
 * This should not normally be necessary.
 */
Crypto.prototype.forceDiscardSession = function (roomId) {
    var alg = this._roomEncryptors[roomId];
    if (alg === undefined) throw new Error("Room not encrypted");
    if (alg.forceDiscardSession === undefined) {
        throw new Error("Room encryption algorithm doesn't support session discarding");
    }
    alg.forceDiscardSession();
};

/**
 * Configure a room to use encryption (ie, save a flag in the cryptoStore).
 *
 * @param {string} roomId The room ID to enable encryption in.
 *
 * @param {object} config The encryption config for the room.
 *
 * @param {boolean=} inhibitDeviceQuery true to suppress device list query for
 *   users in the room (for now). In case lazy loading is enabled,
 *   the device query is always inhibited as the members are not tracked.
 */
Crypto.prototype.setRoomEncryption = function () {
    var _ref16 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee16(roomId, config, inhibitDeviceQuery) {
        var existingConfig, existingAlg, storeConfigPromise, AlgClass, alg;
        return _regenerator2.default.wrap(function _callee16$(_context16) {
            while (1) {
                switch (_context16.prev = _context16.next) {
                    case 0:
                        if (config.algorithm) {
                            _context16.next = 3;
                            break;
                        }

                        _logger2.default.log("Ignoring setRoomEncryption with no algorithm");
                        return _context16.abrupt('return');

                    case 3:

                        // if state is being replayed from storage, we might already have a configuration
                        // for this room as they are persisted as well.
                        // We just need to make sure the algorithm is initialized in this case.
                        // However, if the new config is different,
                        // we should bail out as room encryption can't be changed once set.
                        existingConfig = this._roomList.getRoomEncryption(roomId);

                        if (!existingConfig) {
                            _context16.next = 8;
                            break;
                        }

                        if (!((0, _stringify2.default)(existingConfig) != (0, _stringify2.default)(config))) {
                            _context16.next = 8;
                            break;
                        }

                        _logger2.default.error("Ignoring m.room.encryption event which requests " + "a change of config in " + roomId);
                        return _context16.abrupt('return');

                    case 8:
                        // if we already have encryption in this room, we should ignore this event,
                        // as it would reset the encryption algorithm.
                        // This is at least expected to be called twice, as sync calls onCryptoEvent
                        // for both the timeline and state sections in the /sync response,
                        // the encryption event would appear in both.
                        // If it's called more than twice though,
                        // it signals a bug on client or server.
                        existingAlg = this._roomEncryptors[roomId];

                        if (!existingAlg) {
                            _context16.next = 11;
                            break;
                        }

                        return _context16.abrupt('return');

                    case 11:

                        // _roomList.getRoomEncryption will not race with _roomList.setRoomEncryption
                        // because it first stores in memory. We should await the promise only
                        // after all the in-memory state (_roomEncryptors and _roomList) has been updated
                        // to avoid races when calling this method multiple times. Hence keep a hold of the promise.
                        storeConfigPromise = null;

                        if (!existingConfig) {
                            storeConfigPromise = this._roomList.setRoomEncryption(roomId, config);
                        }

                        AlgClass = algorithms.ENCRYPTION_CLASSES[config.algorithm];

                        if (AlgClass) {
                            _context16.next = 16;
                            break;
                        }

                        throw new Error("Unable to encrypt with " + config.algorithm);

                    case 16:
                        alg = new AlgClass({
                            userId: this._userId,
                            deviceId: this._deviceId,
                            crypto: this,
                            olmDevice: this._olmDevice,
                            baseApis: this._baseApis,
                            roomId: roomId,
                            config: config
                        });

                        this._roomEncryptors[roomId] = alg;

                        if (!storeConfigPromise) {
                            _context16.next = 21;
                            break;
                        }

                        _context16.next = 21;
                        return (0, _bluebird.resolve)(storeConfigPromise);

                    case 21:
                        if (this._lazyLoadMembers) {
                            _context16.next = 28;
                            break;
                        }

                        _logger2.default.log("Enabling encryption in " + roomId + "; " + "starting to track device lists for all users therein");

                        _context16.next = 25;
                        return (0, _bluebird.resolve)(this.trackRoomDevices(roomId));

                    case 25:
                        // TODO: this flag is only not used from MatrixClient::setRoomEncryption
                        // which is never used (inside riot at least)
                        // but didn't want to remove it as it technically would
                        // be a breaking change.
                        if (!this.inhibitDeviceQuery) {
                            this._deviceList.refreshOutdatedDeviceLists();
                        }
                        _context16.next = 29;
                        break;

                    case 28:
                        _logger2.default.log("Enabling encryption in " + roomId);

                    case 29:
                    case 'end':
                        return _context16.stop();
                }
            }
        }, _callee16, this);
    }));

    return function (_x22, _x23, _x24) {
        return _ref16.apply(this, arguments);
    };
}();

/**
 * Make sure we are tracking the device lists for all users in this room.
 *
 * @param {string} roomId The room ID to start tracking devices in.
 * @returns {Promise} when all devices for the room have been fetched and marked to track
 */
Crypto.prototype.trackRoomDevices = function (roomId) {
    var _this6 = this;

    var trackMembers = function () {
        var _ref17 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee17() {
            var room, members;
            return _regenerator2.default.wrap(function _callee17$(_context17) {
                while (1) {
                    switch (_context17.prev = _context17.next) {
                        case 0:
                            if (_this6._roomEncryptors[roomId]) {
                                _context17.next = 2;
                                break;
                            }

                            return _context17.abrupt('return');

                        case 2:
                            room = _this6._clientStore.getRoom(roomId);

                            if (room) {
                                _context17.next = 5;
                                break;
                            }

                            throw new Error('Unable to start tracking devices in unknown room ' + roomId);

                        case 5:
                            _logger2.default.log('Starting to track devices for room ' + roomId + ' ...');
                            _context17.next = 8;
                            return (0, _bluebird.resolve)(room.getEncryptionTargetMembers());

                        case 8:
                            members = _context17.sent;

                            members.forEach(function (m) {
                                _this6._deviceList.startTrackingDeviceList(m.userId);
                            });

                        case 10:
                        case 'end':
                            return _context17.stop();
                    }
                }
            }, _callee17, _this6);
        }));

        return function trackMembers() {
            return _ref17.apply(this, arguments);
        };
    }();

    var promise = this._roomDeviceTrackingState[roomId];
    if (!promise) {
        promise = trackMembers();
        this._roomDeviceTrackingState[roomId] = promise;
    }
    return promise;
};

/**
 * @typedef {Object} module:crypto~OlmSessionResult
 * @property {module:crypto/deviceinfo} device  device info
 * @property {string?} sessionId base64 olm session id; null if no session
 *    could be established
 */

/**
 * Try to make sure we have established olm sessions for all known devices for
 * the given users.
 *
 * @param {string[]} users list of user ids
 *
 * @return {module:client.Promise} resolves once the sessions are complete, to
 *    an Object mapping from userId to deviceId to
 *    {@link module:crypto~OlmSessionResult}
 */
Crypto.prototype.ensureOlmSessionsForUsers = function (users) {
    var devicesByUser = {};

    for (var i = 0; i < users.length; ++i) {
        var userId = users[i];
        devicesByUser[userId] = [];

        var devices = this.getStoredDevicesForUser(userId) || [];
        for (var j = 0; j < devices.length; ++j) {
            var deviceInfo = devices[j];

            var key = deviceInfo.getIdentityKey();
            if (key == this._olmDevice.deviceCurve25519Key) {
                // don't bother setting up session to ourself
                continue;
            }
            if (deviceInfo.verified == DeviceVerification.BLOCKED) {
                // don't bother setting up sessions with blocked users
                continue;
            }

            devicesByUser[userId].push(deviceInfo);
        }
    }

    return olmlib.ensureOlmSessionsForDevices(this._olmDevice, this._baseApis, devicesByUser);
};

/**
 * Get a list containing all of the room keys
 *
 * @return {module:crypto/OlmDevice.MegolmSessionData[]} a list of session export objects
 */
Crypto.prototype.exportRoomKeys = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee18() {
    var _this7 = this;

    var exportedSessions;
    return _regenerator2.default.wrap(function _callee18$(_context18) {
        while (1) {
            switch (_context18.prev = _context18.next) {
                case 0:
                    exportedSessions = [];
                    _context18.next = 3;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                        _this7._cryptoStore.getAllEndToEndInboundGroupSessions(txn, function (s) {
                            if (s === null) return;

                            var sess = _this7._olmDevice.exportInboundGroupSession(s.senderKey, s.sessionId, s.sessionData);
                            delete sess.first_known_index;
                            sess.algorithm = olmlib.MEGOLM_ALGORITHM;
                            exportedSessions.push(sess);
                        });
                    }));

                case 3:
                    return _context18.abrupt('return', exportedSessions);

                case 4:
                case 'end':
                    return _context18.stop();
            }
        }
    }, _callee18, this);
}));

/**
 * Import a list of room keys previously exported by exportRoomKeys
 *
 * @param {Object[]} keys a list of session export objects
 * @return {module:client.Promise} a promise which resolves once the keys have been imported
 */
Crypto.prototype.importRoomKeys = function (keys) {
    var _this8 = this;

    return _bluebird2.default.all(keys.map(function (key) {
        if (!key.room_id || !key.algorithm) {
            _logger2.default.warn("ignoring room key entry with missing fields", key);
            return null;
        }

        var alg = _this8._getRoomDecryptor(key.room_id, key.algorithm);
        return alg.importRoomKey(key);
    }));
};

/**
 * Schedules sending all keys waiting to be sent to the backup, if not already
 * scheduled. Retries if necessary.
 *
 * @param {number} maxDelay Maximum delay to wait in ms. 0 means no delay.
 */
Crypto.prototype.scheduleKeyBackupSend = function () {
    var _ref19 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee19() {
        var maxDelay = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 10000;
        var delay, numFailures, numBackedUp;
        return _regenerator2.default.wrap(function _callee19$(_context19) {
            while (1) {
                switch (_context19.prev = _context19.next) {
                    case 0:
                        if (!this._sendingBackups) {
                            _context19.next = 2;
                            break;
                        }

                        return _context19.abrupt('return');

                    case 2:

                        this._sendingBackups = true;

                        _context19.prev = 3;

                        // wait between 0 and `maxDelay` seconds, to avoid backup
                        // requests from different clients hitting the server all at
                        // the same time when a new key is sent
                        delay = Math.random() * maxDelay;
                        _context19.next = 7;
                        return (0, _bluebird.resolve)((0, _utils.sleep)(delay));

                    case 7:
                        numFailures = 0; // number of consecutive failures

                    case 8:
                        if (!1) {
                            _context19.next = 35;
                            break;
                        }

                        if (this.backupKey) {
                            _context19.next = 11;
                            break;
                        }

                        return _context19.abrupt('return');

                    case 11:
                        _context19.prev = 11;
                        _context19.next = 14;
                        return (0, _bluebird.resolve)(this._backupPendingKeys(KEY_BACKUP_KEYS_PER_REQUEST));

                    case 14:
                        numBackedUp = _context19.sent;

                        if (!(numBackedUp === 0)) {
                            _context19.next = 17;
                            break;
                        }

                        return _context19.abrupt('return');

                    case 17:
                        numFailures = 0;
                        _context19.next = 30;
                        break;

                    case 20:
                        _context19.prev = 20;
                        _context19.t0 = _context19['catch'](11);

                        numFailures++;
                        _logger2.default.log("Key backup request failed", _context19.t0);

                        if (!_context19.t0.data) {
                            _context19.next = 30;
                            break;
                        }

                        if (!(_context19.t0.data.errcode == 'M_NOT_FOUND' || _context19.t0.data.errcode == 'M_WRONG_ROOM_KEYS_VERSION')) {
                            _context19.next = 30;
                            break;
                        }

                        _context19.next = 28;
                        return (0, _bluebird.resolve)(this.checkKeyBackup());

                    case 28:
                        // Backup version has changed or this backup version
                        // has been deleted
                        this.emit("crypto.keyBackupFailed", _context19.t0.data.errcode);
                        throw _context19.t0;

                    case 30:
                        if (!numFailures) {
                            _context19.next = 33;
                            break;
                        }

                        _context19.next = 33;
                        return (0, _bluebird.resolve)((0, _utils.sleep)(1000 * Math.pow(2, Math.min(numFailures - 1, 4))));

                    case 33:
                        _context19.next = 8;
                        break;

                    case 35:
                        _context19.prev = 35;

                        this._sendingBackups = false;
                        return _context19.finish(35);

                    case 38:
                    case 'end':
                        return _context19.stop();
                }
            }
        }, _callee19, this, [[3,, 35, 38], [11, 20]]);
    }));

    return function (_x25) {
        return _ref19.apply(this, arguments);
    };
}();

/**
 * Take some e2e keys waiting to be backed up and send them
 * to the backup.
 *
 * @param {integer} limit Maximum number of keys to back up
 * @returns {integer} Number of sessions backed up
 */
Crypto.prototype._backupPendingKeys = function () {
    var _ref20 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee20(limit) {
        var sessions, remaining, data, _iteratorNormalCompletion9, _didIteratorError9, _iteratorError9, _iterator9, _step9, session, roomId, sessionData, firstKnownIndex, encrypted, forwardedCount, device;

        return _regenerator2.default.wrap(function _callee20$(_context20) {
            while (1) {
                switch (_context20.prev = _context20.next) {
                    case 0:
                        _context20.next = 2;
                        return (0, _bluebird.resolve)(this._cryptoStore.getSessionsNeedingBackup(limit));

                    case 2:
                        sessions = _context20.sent;

                        if (sessions.length) {
                            _context20.next = 5;
                            break;
                        }

                        return _context20.abrupt('return', 0);

                    case 5:
                        _context20.next = 7;
                        return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                    case 7:
                        remaining = _context20.sent;

                        this.emit("crypto.keyBackupSessionsRemaining", remaining);

                        data = {};
                        _iteratorNormalCompletion9 = true;
                        _didIteratorError9 = false;
                        _iteratorError9 = undefined;
                        _context20.prev = 13;
                        _iterator9 = (0, _getIterator3.default)(sessions);

                    case 15:
                        if (_iteratorNormalCompletion9 = (_step9 = _iterator9.next()).done) {
                            _context20.next = 34;
                            break;
                        }

                        session = _step9.value;
                        roomId = session.sessionData.room_id;

                        if (data[roomId] === undefined) {
                            data[roomId] = { sessions: {} };
                        }

                        _context20.next = 21;
                        return (0, _bluebird.resolve)(this._olmDevice.exportInboundGroupSession(session.senderKey, session.sessionId, session.sessionData));

                    case 21:
                        sessionData = _context20.sent;

                        sessionData.algorithm = olmlib.MEGOLM_ALGORITHM;
                        delete sessionData.session_id;
                        delete sessionData.room_id;
                        firstKnownIndex = sessionData.first_known_index;

                        delete sessionData.first_known_index;
                        encrypted = this.backupKey.encrypt((0, _stringify2.default)(sessionData));
                        forwardedCount = (sessionData.forwarding_curve25519_key_chain || []).length;
                        device = this._deviceList.getDeviceByIdentityKey(olmlib.MEGOLM_ALGORITHM, session.senderKey);


                        data[roomId]['sessions'][session.sessionId] = {
                            first_message_index: firstKnownIndex,
                            forwarded_count: forwardedCount,
                            is_verified: !!(device && device.isVerified()),
                            session_data: encrypted
                        };

                    case 31:
                        _iteratorNormalCompletion9 = true;
                        _context20.next = 15;
                        break;

                    case 34:
                        _context20.next = 40;
                        break;

                    case 36:
                        _context20.prev = 36;
                        _context20.t0 = _context20['catch'](13);
                        _didIteratorError9 = true;
                        _iteratorError9 = _context20.t0;

                    case 40:
                        _context20.prev = 40;
                        _context20.prev = 41;

                        if (!_iteratorNormalCompletion9 && _iterator9.return) {
                            _iterator9.return();
                        }

                    case 43:
                        _context20.prev = 43;

                        if (!_didIteratorError9) {
                            _context20.next = 46;
                            break;
                        }

                        throw _iteratorError9;

                    case 46:
                        return _context20.finish(43);

                    case 47:
                        return _context20.finish(40);

                    case 48:
                        _context20.next = 50;
                        return (0, _bluebird.resolve)(this._baseApis.sendKeyBackup(undefined, undefined, this.backupInfo.version, { rooms: data }));

                    case 50:
                        _context20.next = 52;
                        return (0, _bluebird.resolve)(this._cryptoStore.unmarkSessionsNeedingBackup(sessions));

                    case 52:
                        _context20.next = 54;
                        return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                    case 54:
                        remaining = _context20.sent;

                        this.emit("crypto.keyBackupSessionsRemaining", remaining);

                        return _context20.abrupt('return', sessions.length);

                    case 57:
                    case 'end':
                        return _context20.stop();
                }
            }
        }, _callee20, this, [[13, 36, 40, 48], [41,, 43, 47]]);
    }));

    return function (_x27) {
        return _ref20.apply(this, arguments);
    };
}();

Crypto.prototype.backupGroupSession = function () {
    var _ref21 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee21(roomId, senderKey, forwardingCurve25519KeyChain, sessionId, sessionKey, keysClaimed, exportFormat) {
        return _regenerator2.default.wrap(function _callee21$(_context21) {
            while (1) {
                switch (_context21.prev = _context21.next) {
                    case 0:
                        if (this.backupInfo) {
                            _context21.next = 2;
                            break;
                        }

                        throw new Error("Key backups are not enabled");

                    case 2:
                        _context21.next = 4;
                        return (0, _bluebird.resolve)(this._cryptoStore.markSessionsNeedingBackup([{
                            senderKey: senderKey,
                            sessionId: sessionId
                        }]));

                    case 4:

                        // don't wait for this to complete: it will delay so
                        // happens in the background
                        this.scheduleKeyBackupSend();

                    case 5:
                    case 'end':
                        return _context21.stop();
                }
            }
        }, _callee21, this);
    }));

    return function (_x28, _x29, _x30, _x31, _x32, _x33, _x34) {
        return _ref21.apply(this, arguments);
    };
}();

/**
 * Marks all group sessions as needing to be backed up and schedules them to
 * upload in the background as soon as possible.
 */
Crypto.prototype.scheduleAllGroupSessionsForBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee22() {
    return _regenerator2.default.wrap(function _callee22$(_context22) {
        while (1) {
            switch (_context22.prev = _context22.next) {
                case 0:
                    _context22.next = 2;
                    return (0, _bluebird.resolve)(this.flagAllGroupSessionsForBackup());

                case 2:

                    // Schedule keys to upload in the background as soon as possible.
                    this.scheduleKeyBackupSend(0 /* maxDelay */);

                case 3:
                case 'end':
                    return _context22.stop();
            }
        }
    }, _callee22, this);
}));

/**
 * Marks all group sessions as needing to be backed up without scheduling
 * them to upload in the background.
 * @returns {Promise<int>} Resolves to the number of sessions requiring a backup.
 */
Crypto.prototype.flagAllGroupSessionsForBackup = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee23() {
    var _this9 = this;

    var remaining;
    return _regenerator2.default.wrap(function _callee23$(_context23) {
        while (1) {
            switch (_context23.prev = _context23.next) {
                case 0:
                    _context23.next = 2;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS, _indexeddbCryptoStore2.default.STORE_BACKUP], function (txn) {
                        _this9._cryptoStore.getAllEndToEndInboundGroupSessions(txn, function (session) {
                            if (session !== null) {
                                _this9._cryptoStore.markSessionsNeedingBackup([session], txn);
                            }
                        });
                    }));

                case 2:
                    _context23.next = 4;
                    return (0, _bluebird.resolve)(this._cryptoStore.countSessionsNeedingBackup());

                case 4:
                    remaining = _context23.sent;

                    this.emit("crypto.keyBackupSessionsRemaining", remaining);
                    return _context23.abrupt('return', remaining);

                case 7:
                case 'end':
                    return _context23.stop();
            }
        }
    }, _callee23, this);
}));

/* eslint-disable valid-jsdoc */ //https://github.com/eslint/eslint/issues/7307
/**
 * Encrypt an event according to the configuration of the room.
 *
 * @param {module:models/event.MatrixEvent} event  event to be sent
 *
 * @param {module:models/room} room destination room.
 *
 * @return {module:client.Promise?} Promise which resolves when the event has been
 *     encrypted, or null if nothing was needed
 */
/* eslint-enable valid-jsdoc */
Crypto.prototype.encryptEvent = function () {
    var _ref24 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee24(event, room) {
        var roomId, alg, content, mRelatesTo, encryptedContent;
        return _regenerator2.default.wrap(function _callee24$(_context24) {
            while (1) {
                switch (_context24.prev = _context24.next) {
                    case 0:
                        if (room) {
                            _context24.next = 2;
                            break;
                        }

                        throw new Error("Cannot send encrypted messages in unknown rooms");

                    case 2:
                        roomId = event.getRoomId();
                        alg = this._roomEncryptors[roomId];

                        if (alg) {
                            _context24.next = 6;
                            break;
                        }

                        throw new Error("Room was previously configured to use encryption, but is " + "no longer. Perhaps the homeserver is hiding the " + "configuration event.");

                    case 6:

                        if (!this._roomDeviceTrackingState[roomId]) {
                            this.trackRoomDevices(roomId);
                        }
                        // wait for all the room devices to be loaded
                        _context24.next = 9;
                        return (0, _bluebird.resolve)(this._roomDeviceTrackingState[roomId]);

                    case 9:
                        content = event.getContent();
                        // If event has an m.relates_to then we need
                        // to put this on the wrapping event instead

                        mRelatesTo = content['m.relates_to'];

                        if (mRelatesTo) {
                            // Clone content here so we don't remove `m.relates_to` from the local-echo
                            content = (0, _assign2.default)({}, content);
                            delete content['m.relates_to'];
                        }

                        _context24.next = 14;
                        return (0, _bluebird.resolve)(alg.encryptMessage(room, event.getType(), content));

                    case 14:
                        encryptedContent = _context24.sent;


                        if (mRelatesTo) {
                            encryptedContent['m.relates_to'] = mRelatesTo;
                        }

                        event.makeEncrypted("m.room.encrypted", encryptedContent, this._olmDevice.deviceCurve25519Key, this._olmDevice.deviceEd25519Key);

                    case 17:
                    case 'end':
                        return _context24.stop();
                }
            }
        }, _callee24, this);
    }));

    return function (_x35, _x36) {
        return _ref24.apply(this, arguments);
    };
}();

/**
 * Decrypt a received event
 *
 * @param {MatrixEvent} event
 *
 * @return {Promise<module:crypto~EventDecryptionResult>} resolves once we have
 *  finished decrypting. Rejects with an `algorithms.DecryptionError` if there
 *  is a problem decrypting the event.
 */
Crypto.prototype.decryptEvent = function (event) {
    if (event.isRedacted()) {
        return _bluebird2.default.resolve({
            clearEvent: {
                room_id: event.getRoomId(),
                type: "m.room.message",
                content: {}
            }
        });
    }
    var content = event.getWireContent();
    var alg = this._getRoomDecryptor(event.getRoomId(), content.algorithm);
    return alg.decryptEvent(event);
};

/**
 * Handle the notification from /sync or /keys/changes that device lists have
 * been changed.
 *
 * @param {Object} syncData Object containing sync tokens associated with this sync
 * @param {Object} syncDeviceLists device_lists field from /sync, or response from
 * /keys/changes
 */
Crypto.prototype.handleDeviceListChanges = function () {
    var _ref25 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee25(syncData, syncDeviceLists) {
        return _regenerator2.default.wrap(function _callee25$(_context25) {
            while (1) {
                switch (_context25.prev = _context25.next) {
                    case 0:
                        if (syncData.oldSyncToken) {
                            _context25.next = 2;
                            break;
                        }

                        return _context25.abrupt('return');

                    case 2:
                        _context25.next = 4;
                        return (0, _bluebird.resolve)(this._evalDeviceListChanges(syncDeviceLists));

                    case 4:
                    case 'end':
                        return _context25.stop();
                }
            }
        }, _callee25, this);
    }));

    return function (_x37, _x38) {
        return _ref25.apply(this, arguments);
    };
}();

/**
 * Send a request for some room keys, if we have not already done so
 *
 * @param {module:crypto~RoomKeyRequestBody} requestBody
 * @param {Array<{userId: string, deviceId: string}>} recipients
 * @param {boolean} resend whether to resend the key request if there is
 *    already one
 *
 * @return {Promise} a promise that resolves when the key request is queued
 */
Crypto.prototype.requestRoomKey = function (requestBody, recipients) {
    var resend = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

    return this._outgoingRoomKeyRequestManager.sendRoomKeyRequest(requestBody, recipients, resend).catch(function (e) {
        // this normally means we couldn't talk to the store
        _logger2.default.error('Error requesting key for event', e);
    }).done();
};

/**
 * Cancel any earlier room key request
 *
 * @param {module:crypto~RoomKeyRequestBody} requestBody
 *    parameters to match for cancellation
 */
Crypto.prototype.cancelRoomKeyRequest = function (requestBody) {
    this._outgoingRoomKeyRequestManager.cancelRoomKeyRequest(requestBody).catch(function (e) {
        _logger2.default.warn("Error clearing pending room key requests", e);
    }).done();
};

/**
 * handle an m.room.encryption event
 *
 * @param {module:models/event.MatrixEvent} event encryption event
 */
Crypto.prototype.onCryptoEvent = function () {
    var _ref26 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee26(event) {
        var roomId, content;
        return _regenerator2.default.wrap(function _callee26$(_context26) {
            while (1) {
                switch (_context26.prev = _context26.next) {
                    case 0:
                        roomId = event.getRoomId();
                        content = event.getContent();
                        _context26.prev = 2;
                        _context26.next = 5;
                        return (0, _bluebird.resolve)(this.setRoomEncryption(roomId, content, true));

                    case 5:
                        _context26.next = 10;
                        break;

                    case 7:
                        _context26.prev = 7;
                        _context26.t0 = _context26['catch'](2);

                        _logger2.default.error("Error configuring encryption in room " + roomId + ":", _context26.t0);

                    case 10:
                    case 'end':
                        return _context26.stop();
                }
            }
        }, _callee26, this, [[2, 7]]);
    }));

    return function (_x40) {
        return _ref26.apply(this, arguments);
    };
}();

/**
 * Called before the result of a sync is procesed
 *
 * @param {Object} syncData  the data from the 'MatrixClient.sync' event
 */
Crypto.prototype.onSyncWillProcess = function () {
    var _ref27 = (0, _bluebird.method)(function (syncData) {
        if (!syncData.oldSyncToken) {
            // If there is no old sync token, we start all our tracking from
            // scratch, so mark everything as untracked. onCryptoEvent will
            // be called for all e2e rooms during the processing of the sync,
            // at which point we'll start tracking all the users of that room.
            _logger2.default.log("Initial sync performed - resetting device tracking state");
            this._deviceList.stopTrackingAllDeviceLists();
            // we always track our own device list (for key backups etc)
            this._deviceList.startTrackingDeviceList(this._userId);
            this._roomDeviceTrackingState = {};
        }
    });

    return function (_x41) {
        return _ref27.apply(this, arguments);
    };
}();

/**
 * handle the completion of a /sync
 *
 * This is called after the processing of each successful /sync response.
 * It is an opportunity to do a batch process on the information received.
 *
 * @param {Object} syncData  the data from the 'MatrixClient.sync' event
 */
Crypto.prototype.onSyncCompleted = function () {
    var _ref28 = (0, _bluebird.method)(function (syncData) {
        var nextSyncToken = syncData.nextSyncToken;

        this._deviceList.setSyncToken(syncData.nextSyncToken);
        this._deviceList.saveIfDirty();

        // catch up on any new devices we got told about during the sync.
        this._deviceList.lastKnownSyncToken = nextSyncToken;

        // we always track our own device list (for key backups etc)
        this._deviceList.startTrackingDeviceList(this._userId);

        this._deviceList.refreshOutdatedDeviceLists();

        // we don't start uploading one-time keys until we've caught up with
        // to-device messages, to help us avoid throwing away one-time-keys that we
        // are about to receive messages for
        // (https://github.com/vector-im/riot-web/issues/2782).
        if (!syncData.catchingUp) {
            _maybeUploadOneTimeKeys(this);
            this._processReceivedRoomKeyRequests();
        }
    });

    return function (_x42) {
        return _ref28.apply(this, arguments);
    };
}();

/**
 * Trigger the appropriate invalidations and removes for a given
 * device list
 *
 * @param {Object} deviceLists device_lists field from /sync, or response from
 * /keys/changes
 */
Crypto.prototype._evalDeviceListChanges = function () {
    var _ref29 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee27(deviceLists) {
        var _this10 = this;

        var e2eUserIds;
        return _regenerator2.default.wrap(function _callee27$(_context27) {
            while (1) {
                switch (_context27.prev = _context27.next) {
                    case 0:
                        if (deviceLists.changed && Array.isArray(deviceLists.changed)) {
                            deviceLists.changed.forEach(function (u) {
                                _this10._deviceList.invalidateUserDeviceList(u);
                            });
                        }

                        if (!(deviceLists.left && Array.isArray(deviceLists.left) && deviceLists.left.length)) {
                            _context27.next = 8;
                            break;
                        }

                        _context27.t0 = _set2.default;
                        _context27.next = 5;
                        return (0, _bluebird.resolve)(this._getTrackedE2eUsers());

                    case 5:
                        _context27.t1 = _context27.sent;
                        e2eUserIds = new _context27.t0(_context27.t1);


                        deviceLists.left.forEach(function (u) {
                            if (!e2eUserIds.has(u)) {
                                _this10._deviceList.stopTrackingDeviceList(u);
                            }
                        });

                    case 8:
                    case 'end':
                        return _context27.stop();
                }
            }
        }, _callee27, this);
    }));

    return function (_x43) {
        return _ref29.apply(this, arguments);
    };
}();

/**
 * Get a list of all the IDs of users we share an e2e room with
 * for which we are tracking devices already
 *
 * @returns {string[]} List of user IDs
 */
Crypto.prototype._getTrackedE2eUsers = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee28() {
    var e2eUserIds, _iteratorNormalCompletion10, _didIteratorError10, _iteratorError10, _iterator10, _step10, room, members, _iteratorNormalCompletion11, _didIteratorError11, _iteratorError11, _iterator11, _step11, member;

    return _regenerator2.default.wrap(function _callee28$(_context28) {
        while (1) {
            switch (_context28.prev = _context28.next) {
                case 0:
                    e2eUserIds = [];
                    _iteratorNormalCompletion10 = true;
                    _didIteratorError10 = false;
                    _iteratorError10 = undefined;
                    _context28.prev = 4;
                    _iterator10 = (0, _getIterator3.default)(this._getTrackedE2eRooms());

                case 6:
                    if (_iteratorNormalCompletion10 = (_step10 = _iterator10.next()).done) {
                        _context28.next = 33;
                        break;
                    }

                    room = _step10.value;
                    _context28.next = 10;
                    return (0, _bluebird.resolve)(room.getEncryptionTargetMembers());

                case 10:
                    members = _context28.sent;
                    _iteratorNormalCompletion11 = true;
                    _didIteratorError11 = false;
                    _iteratorError11 = undefined;
                    _context28.prev = 14;

                    for (_iterator11 = (0, _getIterator3.default)(members); !(_iteratorNormalCompletion11 = (_step11 = _iterator11.next()).done); _iteratorNormalCompletion11 = true) {
                        member = _step11.value;

                        e2eUserIds.push(member.userId);
                    }
                    _context28.next = 22;
                    break;

                case 18:
                    _context28.prev = 18;
                    _context28.t0 = _context28['catch'](14);
                    _didIteratorError11 = true;
                    _iteratorError11 = _context28.t0;

                case 22:
                    _context28.prev = 22;
                    _context28.prev = 23;

                    if (!_iteratorNormalCompletion11 && _iterator11.return) {
                        _iterator11.return();
                    }

                case 25:
                    _context28.prev = 25;

                    if (!_didIteratorError11) {
                        _context28.next = 28;
                        break;
                    }

                    throw _iteratorError11;

                case 28:
                    return _context28.finish(25);

                case 29:
                    return _context28.finish(22);

                case 30:
                    _iteratorNormalCompletion10 = true;
                    _context28.next = 6;
                    break;

                case 33:
                    _context28.next = 39;
                    break;

                case 35:
                    _context28.prev = 35;
                    _context28.t1 = _context28['catch'](4);
                    _didIteratorError10 = true;
                    _iteratorError10 = _context28.t1;

                case 39:
                    _context28.prev = 39;
                    _context28.prev = 40;

                    if (!_iteratorNormalCompletion10 && _iterator10.return) {
                        _iterator10.return();
                    }

                case 42:
                    _context28.prev = 42;

                    if (!_didIteratorError10) {
                        _context28.next = 45;
                        break;
                    }

                    throw _iteratorError10;

                case 45:
                    return _context28.finish(42);

                case 46:
                    return _context28.finish(39);

                case 47:
                    return _context28.abrupt('return', e2eUserIds);

                case 48:
                case 'end':
                    return _context28.stop();
            }
        }
    }, _callee28, this, [[4, 35, 39, 47], [14, 18, 22, 30], [23,, 25, 29], [40,, 42, 46]]);
}));

/**
 * Get a list of the e2e-enabled rooms we are members of,
 * and for which we are already tracking the devices
 *
 * @returns {module:models.Room[]}
 */
Crypto.prototype._getTrackedE2eRooms = function () {
    var _this11 = this;

    return this._clientStore.getRooms().filter(function (room) {
        // check for rooms with encryption enabled
        var alg = _this11._roomEncryptors[room.roomId];
        if (!alg) {
            return false;
        }
        if (!_this11._roomDeviceTrackingState[room.roomId]) {
            return false;
        }

        // ignore any rooms which we have left
        var myMembership = room.getMyMembership();
        return myMembership === "join" || myMembership === "invite";
    });
};

Crypto.prototype._onToDeviceEvent = function (event) {
    var _this12 = this;

    try {
        _logger2.default.log('received to_device ' + event.getType() + ' from: ' + (event.getSender() + ' id: ' + event.getId()));

        if (event.getType() == "m.room_key" || event.getType() == "m.forwarded_room_key") {
            this._onRoomKeyEvent(event);
        } else if (event.getType() == "m.room_key_request") {
            this._onRoomKeyRequestEvent(event);
        } else if (event.getType() === "m.secret.request") {
            this._secretStorage._onRequestReceived(event);
        } else if (event.getType() === "m.secret.send") {
            this._secretStorage._onSecretReceived(event);
        } else if (event.getType() === "m.key.verification.request") {
            this._onKeyVerificationRequest(event);
        } else if (event.getType() === "m.key.verification.start") {
            this._onKeyVerificationStart(event);
        } else if (event.getContent().transaction_id) {
            this._onKeyVerificationMessage(event);
        } else if (event.getContent().msgtype === "m.bad.encrypted") {
            this._onToDeviceBadEncrypted(event);
        } else if (event.isBeingDecrypted()) {
            // once the event has been decrypted, try again
            event.once('Event.decrypted', function (ev) {
                _this12._onToDeviceEvent(ev);
            });
        }
    } catch (e) {
        _logger2.default.error("Error handling toDeviceEvent:", e);
    }
};

/**
 * Handle a key event
 *
 * @private
 * @param {module:models/event.MatrixEvent} event key event
 */
Crypto.prototype._onRoomKeyEvent = function (event) {
    var content = event.getContent();

    if (!content.room_id || !content.algorithm) {
        _logger2.default.error("key event is missing fields");
        return;
    }

    if (!this._checkedForBackup) {
        // don't bother awaiting on this - the important thing is that we retry if we
        // haven't managed to check before
        this._checkAndStartKeyBackup();
    }

    var alg = this._getRoomDecryptor(content.room_id, content.algorithm);
    alg.onRoomKeyEvent(event);
};

/**
 * Handle a key verification request event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification request event
 */
Crypto.prototype._onKeyVerificationRequest = function (event) {
    var _this13 = this;

    if (event.isCancelled()) {
        _logger2.default.warn("Ignoring flagged verification request from " + event.getSender());
        return;
    }

    var content = event.getContent();
    if (!("from_device" in content) || typeof content.from_device !== "string" || !("transaction_id" in content) || !("methods" in content) || !Array.isArray(content.methods) || !("timestamp" in content) || typeof content.timestamp !== "number") {
        _logger2.default.warn("received invalid verification request from " + event.getSender());
        // ignore event if malformed
        return;
    }

    var now = Date.now();
    if (now < content.timestamp - 5 * 60 * 1000 || now > content.timestamp + 10 * 60 * 1000) {
        // ignore if event is too far in the past or too far in the future
        _logger2.default.log("received verification that is too old or from the future");
        return;
    }

    var sender = event.getSender();
    if (sender === this._userId && content.from_device === this._deviceId) {
        // ignore requests from ourselves, because it doesn't make sense for a
        // device to verify itself
        return;
    }
    if (this._verificationTransactions.has(sender)) {
        if (this._verificationTransactions.get(sender).has(content.transaction_id)) {
            // transaction already exists: cancel it and drop the existing
            // request because someone has gotten confused
            var err = (0, _Error.newUnexpectedMessageError)({
                transaction_id: content.transaction_id
            });
            if (this._verificationTransactions.get(sender).get(content.transaction_id).verifier) {
                this._verificationTransactions.get(sender).get(content.transaction_id).verifier.cancel(err);
            } else {
                this._verificationTransactions.get(sender).get(content.transaction_id).reject(err);
                this.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, err.getContent())));
            }
            this._verificationTransactions.get(sender).delete(content.transaction_id);
            return;
        }
    } else {
        this._verificationTransactions.set(sender, new _map2.default());
    }

    // determine what requested methods we support
    var methods = [];
    var _iteratorNormalCompletion12 = true;
    var _didIteratorError12 = false;
    var _iteratorError12 = undefined;

    try {
        for (var _iterator12 = (0, _getIterator3.default)(content.methods), _step12; !(_iteratorNormalCompletion12 = (_step12 = _iterator12.next()).done); _iteratorNormalCompletion12 = true) {
            var method = _step12.value;

            if (typeof method !== "string") {
                continue;
            }
            if (this._verificationMethods.has(method)) {
                methods.push(method);
            }
        }
    } catch (err) {
        _didIteratorError12 = true;
        _iteratorError12 = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion12 && _iterator12.return) {
                _iterator12.return();
            }
        } finally {
            if (_didIteratorError12) {
                throw _iteratorError12;
            }
        }
    }

    if (methods.length === 0) {
        this._baseApis.emit("crypto.verification.request.unknown", event.getSender(), function () {
            _this13.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, (0, _Error.newUserCancelledError)({
                transaction_id: content.transaction_id
            }).getContent())));
        });
    } else {
        // notify the application of the verification request, so it can
        // decide what to do with it
        var request = {
            timeout: VERIFICATION_REQUEST_TIMEOUT,
            event: event,
            methods: methods,
            beginKeyVerification: function beginKeyVerification(method) {
                var verifier = _this13.beginKeyVerification(method, sender, content.from_device, content.transaction_id);
                _this13._verificationTransactions.get(sender).get(content.transaction_id).verifier = verifier;
                return verifier;
            },
            cancel: function cancel() {
                _this13._baseApis.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, content.from_device, (0, _Error.newUserCancelledError)({
                    transaction_id: content.transaction_id
                }).getContent())));
            }
        };
        this._verificationTransactions.get(sender).set(content.transaction_id, {
            request: request
        });
        this._baseApis.emit("crypto.verification.request", request);
    }
};

/**
 * Handle a key verification start event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification start event
 */
Crypto.prototype._onKeyVerificationStart = function (event) {
    var _this14 = this;

    if (event.isCancelled()) {
        _logger2.default.warn("Ignoring flagged verification start from " + event.getSender());
        return;
    }

    var sender = event.getSender();
    var content = event.getContent();
    var transactionId = content.transaction_id;
    var deviceId = content.from_device;
    if (!transactionId || !deviceId) {
        // invalid request, and we don't have enough information to send a
        // cancellation, so just ignore it
        return;
    }

    var handler = this._verificationTransactions.has(sender) && this._verificationTransactions.get(sender).get(transactionId);
    // if the verification start message is invalid, send a cancel message to
    // the other side, and also send a cancellation event
    var cancel = function cancel(err) {
        if (handler.verifier) {
            handler.verifier.cancel(err);
        } else if (handler.request && handler.request.cancel) {
            handler.request.cancel(err);
        }
        _this14.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, deviceId, err.getContent())));
    };
    if (!this._verificationMethods.has(content.method)) {
        cancel((0, _Error.newUnknownMethodError)({
            transaction_id: content.transactionId
        }));
        return;
    } else {
        var verifier = new (this._verificationMethods.get(content.method))(this._baseApis, sender, deviceId, content.transaction_id, event, handler && handler.request);
        if (!handler) {
            if (!this._verificationTransactions.has(sender)) {
                this._verificationTransactions.set(sender, new _map2.default());
            }
            handler = this._verificationTransactions.get(sender).set(transactionId, {
                verifier: verifier
            });
        } else {
            if (!handler.verifier) {
                handler.verifier = verifier;
                if (handler.request) {
                    // the verification start was sent as a response to a
                    // verification request

                    if (!handler.request.devices.includes(deviceId)) {
                        // didn't send a request to that device, so it
                        // shouldn't have responded
                        cancel((0, _Error.newUnexpectedMessageError)({
                            transaction_id: content.transactionId
                        }));
                        return;
                    }
                    if (!handler.request.methods.includes(content.method)) {
                        // verification method wasn't one that was requested
                        cancel((0, _Error.newUnknownMethodError)({
                            transaction_id: content.transactionId
                        }));
                        return;
                    }

                    // send cancellation messages to all the other devices that
                    // the request was sent to
                    var message = {
                        transaction_id: transactionId,
                        code: "m.accepted",
                        reason: "Verification request accepted by another device"
                    };
                    var msgMap = {};
                    var _iteratorNormalCompletion13 = true;
                    var _didIteratorError13 = false;
                    var _iteratorError13 = undefined;

                    try {
                        for (var _iterator13 = (0, _getIterator3.default)(handler.request.devices), _step13; !(_iteratorNormalCompletion13 = (_step13 = _iterator13.next()).done); _iteratorNormalCompletion13 = true) {
                            var devId = _step13.value;

                            if (devId !== deviceId) {
                                msgMap[devId] = message;
                            }
                        }
                    } catch (err) {
                        _didIteratorError13 = true;
                        _iteratorError13 = err;
                    } finally {
                        try {
                            if (!_iteratorNormalCompletion13 && _iterator13.return) {
                                _iterator13.return();
                            }
                        } finally {
                            if (_didIteratorError13) {
                                throw _iteratorError13;
                            }
                        }
                    }

                    this._baseApis.sendToDevice("m.key.verification.cancel", (0, _defineProperty3.default)({}, sender, msgMap));

                    handler.request.resolve(verifier);
                }
            }
        }
        this._baseApis.emit("crypto.verification.start", verifier);

        // Riot does not implement `m.key.verification.request` while
        // verifying over to_device messages, but the protocol is made to
        // work when starting with a `m.key.verification.start` event straight
        // away as well. Verification over DM *does* start with a request event first,
        // and to expose a uniform api to monitor verification requests, we mock
        // the request api here for to_device messages.
        // "crypto.verification.start" is kept for backwards compatibility.

        // ahh, this will create 2 request notifications for clients that do support the request event
        // so maybe we should remove emitting the request when actually receiving it *sigh*
        var requestAdapter = {
            event: event,
            timeout: VERIFICATION_REQUEST_TIMEOUT,
            methods: [content.method],
            beginKeyVerification: function beginKeyVerification(method) {
                if (content.method === method) {
                    return verifier;
                }
            },
            cancel: function cancel() {
                return verifier.cancel("User cancelled");
            }
        };
        this._baseApis.emit("crypto.verification.request", requestAdapter);
    }
};

/**
 * Handle a general key verification event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event verification start event
 */
Crypto.prototype._onKeyVerificationMessage = function (event) {
    var sender = event.getSender();
    var transactionId = event.getContent().transaction_id;
    var handler = this._verificationTransactions.has(sender) && this._verificationTransactions.get(sender).get(transactionId);
    if (!handler) {
        return;
    } else if (event.getType() === "m.key.verification.cancel") {
        _logger2.default.log(event);
        if (handler.verifier) {
            handler.verifier.cancel(event);
        } else if (handler.request && handler.request.cancel) {
            handler.request.cancel(event);
        }
    } else if (handler.verifier) {
        var verifier = handler.verifier;
        if (verifier.events && verifier.events.includes(event.getType())) {
            verifier.handleEvent(event);
        }
    }
};

/**
 * Handle key verification requests sent as timeline events
 *
 * @private
 * @param {module:models/event.MatrixEvent} event the timeline event
 */
Crypto.prototype._onTimelineEvent = function (event) {
    var _this15 = this;

    if (event.getType() !== "m.room.message") {
        return;
    }
    var content = event.getContent();
    if (content.msgtype !== "m.key.verification.request") {
        return;
    }
    // ignore event if malformed
    if (!("from_device" in content) || typeof content.from_device !== "string" || !("methods" in content) || !Array.isArray(content.methods) || !("to" in content) || typeof content.to !== "string") {
        _logger2.default.warn("received invalid verification request over DM from " + event.getSender());
        return;
    }
    // check the request was directed to the syncing user
    if (content.to !== this._baseApis.getUserId()) {
        return;
    }

    var timeout = VERIFICATION_REQUEST_TIMEOUT - VERIFICATION_REQUEST_MARGIN;
    if (event.getLocalAge() >= timeout) {
        return;
    }
    var request = {
        event: event,
        timeout: VERIFICATION_REQUEST_TIMEOUT,
        methods: content.methods,
        beginKeyVerification: function beginKeyVerification(method) {
            var verifier = _this15.acceptVerificationDM(event, method);
            return verifier;
        },
        cancel: function cancel() {
            var verifier = _this15.acceptVerificationDM(event, content.methods[0]);
            verifier.cancel("User declined");
        }
    };
    this._baseApis.emit("crypto.verification.request", request);
};

/**
 * Handle a toDevice event that couldn't be decrypted
 *
 * @private
 * @param {module:models/event.MatrixEvent} event undecryptable event
 */
Crypto.prototype._onToDeviceBadEncrypted = function () {
    var _ref31 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee29(event) {
        var content, sender, algorithm, deviceKey, lastNewSessionForced, device, devicesByUser, encryptedContent, requestsToResend, _iteratorNormalCompletion14, _didIteratorError14, _iteratorError14, _iterator14, _step14, keyReq;

        return _regenerator2.default.wrap(function _callee29$(_context29) {
            while (1) {
                switch (_context29.prev = _context29.next) {
                    case 0:
                        content = event.getWireContent();
                        sender = event.getSender();
                        algorithm = content.algorithm;
                        deviceKey = content.sender_key;

                        if (!(sender === undefined || deviceKey === undefined || deviceKey === undefined)) {
                            _context29.next = 6;
                            break;
                        }

                        return _context29.abrupt('return');

                    case 6:

                        // check when we last forced a new session with this device: if we've already done so
                        // recently, don't do it again.
                        this._lastNewSessionForced[sender] = this._lastNewSessionForced[sender] || {};
                        lastNewSessionForced = this._lastNewSessionForced[sender][deviceKey] || 0;

                        if (!(lastNewSessionForced + MIN_FORCE_SESSION_INTERVAL_MS > Date.now())) {
                            _context29.next = 11;
                            break;
                        }

                        _logger2.default.debug("New session already forced with device " + sender + ":" + deviceKey + " at " + lastNewSessionForced + ": not forcing another");
                        return _context29.abrupt('return');

                    case 11:

                        // establish a new olm session with this device since we're failing to decrypt messages
                        // on a current session.
                        // Note that an undecryptable message from another device could easily be spoofed -
                        // is there anything we can do to mitigate this?
                        device = this._deviceList.getDeviceByIdentityKey(algorithm, deviceKey);

                        if (device) {
                            _context29.next = 15;
                            break;
                        }

                        _logger2.default.info("Couldn't find device for identity key " + deviceKey + ": not re-establishing session");
                        return _context29.abrupt('return');

                    case 15:
                        devicesByUser = {};

                        devicesByUser[sender] = [device];
                        _context29.next = 19;
                        return (0, _bluebird.resolve)(olmlib.ensureOlmSessionsForDevices(this._olmDevice, this._baseApis, devicesByUser, true));

                    case 19:

                        this._lastNewSessionForced[sender][deviceKey] = Date.now();

                        // Now send a blank message on that session so the other side knows about it.
                        // (The keyshare request is sent in the clear so that won't do)
                        // We send this first such that, as long as the toDevice messages arrive in the
                        // same order we sent them, the other end will get this first, set up the new session,
                        // then get the keyshare request and send the key over this new session (because it
                        // is the session it has most recently received a message on).
                        encryptedContent = {
                            algorithm: olmlib.OLM_ALGORITHM,
                            sender_key: this._olmDevice.deviceCurve25519Key,
                            ciphertext: {}
                        };
                        _context29.next = 23;
                        return (0, _bluebird.resolve)(olmlib.encryptMessageForDevice(encryptedContent.ciphertext, this._userId, this._deviceId, this._olmDevice, sender, device, { type: "m.dummy" }));

                    case 23:
                        _context29.next = 25;
                        return (0, _bluebird.resolve)(this._baseApis.sendToDevice("m.room.encrypted", (0, _defineProperty3.default)({}, sender, (0, _defineProperty3.default)({}, device.deviceId, encryptedContent))));

                    case 25:
                        _context29.next = 27;
                        return (0, _bluebird.resolve)(this._outgoingRoomKeyRequestManager.getOutgoingSentRoomKeyRequest(sender, device.deviceId));

                    case 27:
                        requestsToResend = _context29.sent;
                        _iteratorNormalCompletion14 = true;
                        _didIteratorError14 = false;
                        _iteratorError14 = undefined;
                        _context29.prev = 31;

                        for (_iterator14 = (0, _getIterator3.default)(requestsToResend); !(_iteratorNormalCompletion14 = (_step14 = _iterator14.next()).done); _iteratorNormalCompletion14 = true) {
                            keyReq = _step14.value;

                            this.requestRoomKey(keyReq.requestBody, keyReq.recipients, true);
                        }
                        _context29.next = 39;
                        break;

                    case 35:
                        _context29.prev = 35;
                        _context29.t0 = _context29['catch'](31);
                        _didIteratorError14 = true;
                        _iteratorError14 = _context29.t0;

                    case 39:
                        _context29.prev = 39;
                        _context29.prev = 40;

                        if (!_iteratorNormalCompletion14 && _iterator14.return) {
                            _iterator14.return();
                        }

                    case 42:
                        _context29.prev = 42;

                        if (!_didIteratorError14) {
                            _context29.next = 45;
                            break;
                        }

                        throw _iteratorError14;

                    case 45:
                        return _context29.finish(42);

                    case 46:
                        return _context29.finish(39);

                    case 47:
                    case 'end':
                        return _context29.stop();
                }
            }
        }, _callee29, this, [[31, 35, 39, 47], [40,, 42, 46]]);
    }));

    return function (_x44) {
        return _ref31.apply(this, arguments);
    };
}();

/**
 * Handle a change in the membership state of a member of a room
 *
 * @private
 * @param {module:models/event.MatrixEvent} event  event causing the change
 * @param {module:models/room-member} member  user whose membership changed
 * @param {string=} oldMembership  previous membership
 */
Crypto.prototype._onRoomMembership = function (event, member, oldMembership) {
    // this event handler is registered on the *client* (as opposed to the room
    // member itself), which means it is only called on changes to the *live*
    // membership state (ie, it is not called when we back-paginate, nor when
    // we load the state in the initialsync).
    //
    // Further, it is automatically registered and called when new members
    // arrive in the room.

    var roomId = member.roomId;

    var alg = this._roomEncryptors[roomId];
    if (!alg) {
        // not encrypting in this room
        return;
    }
    // only mark users in this room as tracked if we already started tracking in this room
    // this way we don't start device queries after sync on behalf of this room which we won't use
    // the result of anyway, as we'll need to do a query again once all the members are fetched
    // by calling _trackRoomDevices
    if (this._roomDeviceTrackingState[roomId]) {
        if (member.membership == 'join') {
            _logger2.default.log('Join event for ' + member.userId + ' in ' + roomId);
            // make sure we are tracking the deviceList for this user
            this._deviceList.startTrackingDeviceList(member.userId);
        } else if (member.membership == 'invite' && this._clientStore.getRoom(roomId).shouldEncryptForInvitedMembers()) {
            _logger2.default.log('Invite event for ' + member.userId + ' in ' + roomId);
            this._deviceList.startTrackingDeviceList(member.userId);
        }
    }

    alg.onRoomMembership(event, member, oldMembership);
};

/**
 * Called when we get an m.room_key_request event.
 *
 * @private
 * @param {module:models/event.MatrixEvent} event key request event
 */
Crypto.prototype._onRoomKeyRequestEvent = function (event) {
    var content = event.getContent();
    if (content.action === "request") {
        // Queue it up for now, because they tend to arrive before the room state
        // events at initial sync, and we want to see if we know anything about the
        // room before passing them on to the app.
        var req = new IncomingRoomKeyRequest(event);
        this._receivedRoomKeyRequests.push(req);
    } else if (content.action === "request_cancellation") {
        var _req = new IncomingRoomKeyRequestCancellation(event);
        this._receivedRoomKeyRequestCancellations.push(_req);
    }
};

/**
 * Process any m.room_key_request events which were queued up during the
 * current sync.
 *
 * @private
 */
Crypto.prototype._processReceivedRoomKeyRequests = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee30() {
    var _this16 = this;

    var requests, cancellations;
    return _regenerator2.default.wrap(function _callee30$(_context30) {
        while (1) {
            switch (_context30.prev = _context30.next) {
                case 0:
                    if (!this._processingRoomKeyRequests) {
                        _context30.next = 2;
                        break;
                    }

                    return _context30.abrupt('return');

                case 2:
                    this._processingRoomKeyRequests = true;

                    _context30.prev = 3;

                    // we need to grab and clear the queues in the synchronous bit of this method,
                    // so that we don't end up racing with the next /sync.
                    requests = this._receivedRoomKeyRequests;

                    this._receivedRoomKeyRequests = [];
                    cancellations = this._receivedRoomKeyRequestCancellations;

                    this._receivedRoomKeyRequestCancellations = [];

                    // Process all of the requests, *then* all of the cancellations.
                    //
                    // This makes sure that if we get a request and its cancellation in the
                    // same /sync result, then we process the request before the
                    // cancellation (and end up with a cancelled request), rather than the
                    // cancellation before the request (and end up with an outstanding
                    // request which should have been cancelled.)
                    _context30.next = 10;
                    return (0, _bluebird.resolve)(_bluebird2.default.all(requests.map(function (req) {
                        return _this16._processReceivedRoomKeyRequest(req);
                    })));

                case 10:
                    _context30.next = 12;
                    return (0, _bluebird.resolve)(_bluebird2.default.all(cancellations.map(function (cancellation) {
                        return _this16._processReceivedRoomKeyRequestCancellation(cancellation);
                    })));

                case 12:
                    _context30.next = 17;
                    break;

                case 14:
                    _context30.prev = 14;
                    _context30.t0 = _context30['catch'](3);

                    _logger2.default.error('Error processing room key requsts: ' + _context30.t0);

                case 17:
                    _context30.prev = 17;

                    this._processingRoomKeyRequests = false;
                    return _context30.finish(17);

                case 20:
                case 'end':
                    return _context30.stop();
            }
        }
    }, _callee30, this, [[3, 14, 17, 20]]);
}));

/**
 * Helper for processReceivedRoomKeyRequests
 *
 * @param {IncomingRoomKeyRequest} req
 */
Crypto.prototype._processReceivedRoomKeyRequest = function () {
    var _ref33 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee31(req) {
        var userId, deviceId, body, roomId, alg, encryptor, _device2, decryptor, device;

        return _regenerator2.default.wrap(function _callee31$(_context31) {
            while (1) {
                switch (_context31.prev = _context31.next) {
                    case 0:
                        userId = req.userId;
                        deviceId = req.deviceId;
                        body = req.requestBody;
                        roomId = body.room_id;
                        alg = body.algorithm;


                        _logger2.default.log('m.room_key_request from ' + userId + ':' + deviceId + (' for ' + roomId + ' / ' + body.session_id + ' (id ' + req.requestId + ')'));

                        if (!(userId !== this._userId)) {
                            _context31.next = 24;
                            break;
                        }

                        if (this._roomEncryptors[roomId]) {
                            _context31.next = 10;
                            break;
                        }

                        _logger2.default.debug('room key request for unencrypted room ' + roomId);
                        return _context31.abrupt('return');

                    case 10:
                        encryptor = this._roomEncryptors[roomId];
                        _device2 = this._deviceList.getStoredDevice(userId, deviceId);

                        if (_device2) {
                            _context31.next = 15;
                            break;
                        }

                        _logger2.default.debug('Ignoring keyshare for unknown device ' + userId + ':' + deviceId);
                        return _context31.abrupt('return');

                    case 15:
                        _context31.prev = 15;
                        _context31.next = 18;
                        return (0, _bluebird.resolve)(encryptor.reshareKeyWithDevice(body.sender_key, body.session_id, userId, _device2));

                    case 18:
                        _context31.next = 23;
                        break;

                    case 20:
                        _context31.prev = 20;
                        _context31.t0 = _context31['catch'](15);

                        _logger2.default.warn("Failed to re-share keys for session " + body.session_id + " with device " + userId + ":" + _device2.deviceId, _context31.t0);

                    case 23:
                        return _context31.abrupt('return');

                    case 24:
                        if (this._roomDecryptors[roomId]) {
                            _context31.next = 27;
                            break;
                        }

                        _logger2.default.log('room key request for unencrypted room ' + roomId);
                        return _context31.abrupt('return');

                    case 27:
                        decryptor = this._roomDecryptors[roomId][alg];

                        if (decryptor) {
                            _context31.next = 31;
                            break;
                        }

                        _logger2.default.log('room key request for unknown alg ' + alg + ' in room ' + roomId);
                        return _context31.abrupt('return');

                    case 31:
                        _context31.next = 33;
                        return (0, _bluebird.resolve)(decryptor.hasKeysForKeyRequest(req));

                    case 33:
                        if (_context31.sent) {
                            _context31.next = 36;
                            break;
                        }

                        _logger2.default.log('room key request for unknown session ' + roomId + ' / ' + body.session_id);
                        return _context31.abrupt('return');

                    case 36:

                        req.share = function () {
                            decryptor.shareKeysWithDevice(req);
                        };

                        // if the device is is verified already, share the keys
                        device = this._deviceList.getStoredDevice(userId, deviceId);

                        if (!(device && device.isVerified())) {
                            _context31.next = 42;
                            break;
                        }

                        _logger2.default.log('device is already verified: sharing keys');
                        req.share();
                        return _context31.abrupt('return');

                    case 42:

                        this.emit("crypto.roomKeyRequest", req);

                    case 43:
                    case 'end':
                        return _context31.stop();
                }
            }
        }, _callee31, this, [[15, 20]]);
    }));

    return function (_x45) {
        return _ref33.apply(this, arguments);
    };
}();

/**
 * Helper for processReceivedRoomKeyRequests
 *
 * @param {IncomingRoomKeyRequestCancellation} cancellation
 */
Crypto.prototype._processReceivedRoomKeyRequestCancellation = function () {
    var _ref34 = (0, _bluebird.method)(function (cancellation) {
        _logger2.default.log('m.room_key_request cancellation for ' + cancellation.userId + ':' + (cancellation.deviceId + ' (id ' + cancellation.requestId + ')'));

        // we should probably only notify the app of cancellations we told it
        // about, but we don't currently have a record of that, so we just pass
        // everything through.
        this.emit("crypto.roomKeyRequestCancellation", cancellation);
    });

    return function (_x46) {
        return _ref34.apply(this, arguments);
    };
}();

/**
 * Get a decryptor for a given room and algorithm.
 *
 * If we already have a decryptor for the given room and algorithm, return
 * it. Otherwise try to instantiate it.
 *
 * @private
 *
 * @param {string?} roomId   room id for decryptor. If undefined, a temporary
 * decryptor is instantiated.
 *
 * @param {string} algorithm  crypto algorithm
 *
 * @return {module:crypto.algorithms.base.DecryptionAlgorithm}
 *
 * @raises {module:crypto.algorithms.DecryptionError} if the algorithm is
 * unknown
 */
Crypto.prototype._getRoomDecryptor = function (roomId, algorithm) {
    var decryptors = void 0;
    var alg = void 0;

    roomId = roomId || null;
    if (roomId) {
        decryptors = this._roomDecryptors[roomId];
        if (!decryptors) {
            this._roomDecryptors[roomId] = decryptors = {};
        }

        alg = decryptors[algorithm];
        if (alg) {
            return alg;
        }
    }

    var AlgClass = algorithms.DECRYPTION_CLASSES[algorithm];
    if (!AlgClass) {
        throw new algorithms.DecryptionError('UNKNOWN_ENCRYPTION_ALGORITHM', 'Unknown encryption algorithm "' + algorithm + '".');
    }
    alg = new AlgClass({
        userId: this._userId,
        crypto: this,
        olmDevice: this._olmDevice,
        baseApis: this._baseApis,
        roomId: roomId
    });

    if (decryptors) {
        decryptors[algorithm] = alg;
    }
    return alg;
};

/**
 * sign the given object with our ed25519 key
 *
 * @param {Object} obj  Object to which we will add a 'signatures' property
 */
Crypto.prototype._signObject = function () {
    var _ref35 = (0, _bluebird.coroutine)( /*#__PURE__*/_regenerator2.default.mark(function _callee32(obj) {
        var sigs, unsigned;
        return _regenerator2.default.wrap(function _callee32$(_context32) {
            while (1) {
                switch (_context32.prev = _context32.next) {
                    case 0:
                        sigs = obj.signatures || {};
                        unsigned = obj.unsigned;


                        delete obj.signatures;
                        delete obj.unsigned;

                        sigs[this._userId] = sigs[this._userId] || {};
                        _context32.next = 7;
                        return (0, _bluebird.resolve)(this._olmDevice.sign(anotherjson.stringify(obj)));

                    case 7:
                        sigs[this._userId]["ed25519:" + this._deviceId] = _context32.sent;

                        obj.signatures = sigs;
                        if (unsigned !== undefined) obj.unsigned = unsigned;

                    case 10:
                    case 'end':
                        return _context32.stop();
                }
            }
        }, _callee32, this);
    }));

    return function (_x47) {
        return _ref35.apply(this, arguments);
    };
}();

/**
 * The parameters of a room key request. The details of the request may
 * vary with the crypto algorithm, but the management and storage layers for
 * outgoing requests expect it to have 'room_id' and 'session_id' properties.
 *
 * @typedef {Object} RoomKeyRequestBody
 */

/**
 * Represents a received m.room_key_request event
 *
 * @property {string} userId    user requesting the key
 * @property {string} deviceId  device requesting the key
 * @property {string} requestId unique id for the request
 * @property {module:crypto~RoomKeyRequestBody} requestBody
 * @property {function()} share  callback which, when called, will ask
 *    the relevant crypto algorithm implementation to share the keys for
 *    this request.
 */

var IncomingRoomKeyRequest = function IncomingRoomKeyRequest(event) {
    (0, _classCallCheck3.default)(this, IncomingRoomKeyRequest);

    var content = event.getContent();

    this.userId = event.getSender();
    this.deviceId = content.requesting_device_id;
    this.requestId = content.request_id;
    this.requestBody = content.body || {};
    this.share = function () {
        throw new Error("don't know how to share keys for this request yet");
    };
};

/**
 * Represents a received m.room_key_request cancellation
 *
 * @property {string} userId    user requesting the cancellation
 * @property {string} deviceId  device requesting the cancellation
 * @property {string} requestId unique id for the request to be cancelled
 */


var IncomingRoomKeyRequestCancellation = function IncomingRoomKeyRequestCancellation(event) {
    (0, _classCallCheck3.default)(this, IncomingRoomKeyRequestCancellation);

    var content = event.getContent();

    this.userId = event.getSender();
    this.deviceId = content.requesting_device_id;
    this.requestId = content.request_id;
};

/**
 * The result of a (successful) call to decryptEvent.
 *
 * @typedef {Object} EventDecryptionResult
 *
 * @property {Object} clearEvent The plaintext payload for the event
 *     (typically containing <tt>type</tt> and <tt>content</tt> fields).
 *
 * @property {?string} senderCurve25519Key Key owned by the sender of this
 *    event.  See {@link module:models/event.MatrixEvent#getSenderKey}.
 *
 * @property {?string} claimedEd25519Key ed25519 key claimed by the sender of
 *    this event. See
 *    {@link module:models/event.MatrixEvent#getClaimedEd25519Key}.
 *
 * @property {?Array<string>} forwardingCurve25519KeyChain list of curve25519
 *     keys involved in telling us about the senderCurve25519Key and
 *     claimedEd25519Key. See
 *     {@link module:models/event.MatrixEvent#getForwardingCurve25519KeyChain}.
 */

/**
 * Fires when we receive a room key request
 *
 * @event module:client~MatrixClient#"crypto.roomKeyRequest"
 * @param {module:crypto~IncomingRoomKeyRequest} req  request details
 */

/**
 * Fires when we receive a room key request cancellation
 *
 * @event module:client~MatrixClient#"crypto.roomKeyRequestCancellation"
 * @param {module:crypto~IncomingRoomKeyRequestCancellation} req
 */

/**
 * Fires when the app may wish to warn the user about something related
 * the end-to-end crypto.
 *
 * @event module:client~MatrixClient#"crypto.warning"
 * @param {string} type One of the strings listed above
 */
//# sourceMappingURL=index.js.map