import hl = require("../parser/highLevelAST");
import ll = require("../parser/lowLevelAST");
import def = require("raml-definition-system");
/**
 * Finds declaration of the entity located in the particular compilation unit at offset.
 *
 * @param unit - compilation unit.
 * @param offset - character offset in unit counting from 0.
 * @param nodePart - optionally specifies the relevant instruction part of the node.
 * @returns {ll.ICompilationUnit|hl.IParseResult} - either the declaration node or compilation unit
 */
export declare function findDeclaration(unit: ll.ICompilationUnit, offset: number, nodePart?: LocationKind): ll.ICompilationUnit | hl.IParseResult;
/**
 * Result of find usage call
 */
export interface FindUsagesResult {
    /**
     * High-level node at offset.
     */
    node: hl.IHighLevelNode;
    /**
     * High-level nodes, which use the entity defined by the node at offset.
     */
    results: hl.IParseResult[];
}
/**
 * Finds usages of the entity located in the particular compilation unit at offset.
 * @param unit - compilation unit.
 * @param offset - character offset in unit counting from 0.
 * @returns {FindUsagesResult}
 */
export declare function findUsages(unit: ll.ICompilationUnit, offset: number): FindUsagesResult;
/**
 * Finds global declarations (high-level nodes defined at the top-level of the unit) of the unit,
 * current node belongs to. If current unit is included by the external unit, also adds up declarations of the
 * external unit.
 * @param n
 * @returns {hl.IHighLevelNode[]}
 */
export declare function globalDeclarations(n: hl.IHighLevelNode): hl.IHighLevelNode[];
/**
 * Finds references for the node starting from a given root. In a general case its better to use findUsages method.
 * @param root - root of the unit to start from.
 * @param node - node to find references for,
 * @param result - array to put results to.
 */
export declare function refFinder(root: hl.IHighLevelNode, node: hl.IHighLevelNode, result: hl.IParseResult[]): void;
/**
 * Finds declaration of the entity set by a node.
 * @param node - node, which declaration to find
 * @param nodePart - optionally specifies the relevant instruction part of the node.
 * @returns {ll.ICompilationUnit|hl.IParseResult} - either the declaration node or compilation unit
 */
export declare function findDeclarationByNode(node: hl.IParseResult, nodePart?: LocationKind): ll.ICompilationUnit | hl.IParseResult;
/**
 * Location kind inside the instruction
 */
export declare enum LocationKind {
    /**
     * In value
     */
    VALUE_COMPLETION = 0,
    /**
     * In key
     */
    KEY_COMPLETION = 1,
    /**
     * In path
     */
    PATH_COMPLETION = 2,
    /**
     * In directive
     */
    DIRECTIVE_COMPLETION = 3,
    /**
     * In RAML version
     */
    VERSION_COMPLETION = 4,
    /**
     * In annotation
     */
    ANNOTATION_COMPLETION = 5,
    /**
     * In key of the sequence
     */
    SEQUENCE_KEY_COPLETION = 6,
    /**
     * In comment
     */
    INCOMMENT = 7
}
/**
 * Determines location kind inside the instruction.
 * @param text - RAML code
 * @param offset - offeset counting from 0.
 * @returns {any}
 */
export declare function determineCompletionKind(text: string, offset: number): LocationKind;
/**
 * Determines names, which can appear in a certain place of code defined by a node and a proprty.
 * @param property - definition property
 * @param node - node.
 * @returns {string[]} - list of names
 */
export declare function enumValues(property: hl.IProperty, node: hl.IHighLevelNode): string[];
/**
 * Returns a string representation of node name in a context. In case of deep unit dependencies returns contcatenated path.
 * @param node - node, which name to determine
 * @param context - sets up a node, which unit is used as a context to calculate potential unit paths from
 * @returns {string}
 */
export declare function qName(node: hl.IHighLevelNode, context: hl.IHighLevelNode): string;
/**
 * Return all sub types of a given type visible from the particular node.
 * @param type - type, which subtypes to find
 * @param context - context node to determine subtypes visibility.
 * @returns {ITypeDefinition[]} - list of subtypes
 */
export declare function subTypesWithLocals(type: hl.ITypeDefinition, context: hl.IHighLevelNode): hl.ITypeDefinition[];
/**
 * Finds nodes, which declare a type in the context of a specific node.
 * @param type - type, which declarations to find.
 * @param context - context node.
 * @returns {hl.IHighLevelNode[]} - list of results
 */
export declare function nodesDeclaringType(type: hl.ITypeDefinition, context: hl.IHighLevelNode): hl.IHighLevelNode[];
/**
 * Checks if a particular attribute declares an example.
 * @param attribute
 * @returns {boolean}
 */
export declare function isExampleNodeContent(attribute: hl.IAttribute): boolean;
/**
 * Finds the type, which declares the content of an example defined in the attribute.
 * @param attribute
 * @returns {hl.INodeDefinition}
 */
export declare function findExampleContentType(attribute: hl.IAttribute): hl.ITypeDefinition;
/**
 * Parses the value of example attribute
 * @param attribute - attribute, which value to parse
 * @param definition - type, which defines example contents.
 * @returns {hl.IHighLevelNode}
 */
export declare function parseDocumentationContent(attribute: hl.IAttribute, definition: hl.ITypeDefinition): hl.IHighLevelNode;
/**
 * Parses example node contents.
 * @param exampleNode - example node, which value to parse
 * @param definition - type, which defines example contents.
 * @returns {hl.IHighLevelNode}
 */
export declare function parseStructuredExample(exampleNode: hl.IHighLevelNode, definition: hl.INodeDefinition): hl.IHighLevelNode;
/**
 * Checks whether the node defines an example.
 * @param node
 * @returns {boolean}
 */
export declare function isExampleNode(node: hl.IHighLevelNode): boolean;
/**
 * Finds out, which references a particular node property targets.
 * @param property - node property.
 * @param node - node
 * @returns {hl.IHighLevelNode[]} - list of references
 */
export declare function referenceTargets(property: hl.IProperty, node: hl.IHighLevelNode): hl.IHighLevelNode[];
/**
 * This entity can provide its source as high-level node.
 */
export interface IHighLevelSourceProvider {
    getSource(): hl.IParseResult;
}
/**
 * Finds out the source of the nominal type.
 * @param nominalType - type, which source to find.
 * @returns {IHighLevelSourceProvider} - source provider. both provider and its getSource() can be null
 */
export declare function getNominalTypeSource(nominalType: def.ITypeDefinition): IHighLevelSourceProvider;
/**
 * Finds out all subtypes of the type being a range of the property, in the context of the specific node.
 * @param property - property, which range type's subtypes to find.
 * @param node - context node.
 * @returns {hl.ITypeDefinition[]} - list of subtypes.
 */
export declare function findAllSubTypes(property: hl.IProperty, node: hl.IHighLevelNode): hl.ITypeDefinition[];
/**
 * Finds the root of the unit of the particular node.
 * @param node
 * @returns {hl.IHighLevelNode}
 */
export declare function declRoot(node: hl.IHighLevelNode): hl.IHighLevelNode;
/**
 * Finds a node located in certain place of the code.
 * @param parent - searches inside this node
 * @param offset - start position
 * @param end - end position
 * @param goToOtherUnits - whether to check other units
 * @param returnAttrs - whether to return attributes, or only high level nodes
 * @returns {hl.IParseResult} - search result
 */
export declare function deepFindNode(parent: hl.IParseResult, offset: number, end: number, goToOtherUnits?: boolean, returnAttrs?: boolean): hl.IParseResult;
/**
 * Finds all children of the node, recursively.
 * @param node
 * @returns {hl.IParseResult[]}
 */
export declare function allChildren(node: hl.IHighLevelNode): hl.IParseResult[];
