#! /bin/sh
#
# This file is part of the LibreOffice project.
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
#
# This file incorporates work covered by the following license notice:
#
#   Licensed to the Apache Software Foundation (ASF) under one or more
#   contributor license agreements. See the NOTICE file distributed
#   with this work for additional information regarding copyright
#   ownership. The ASF licenses this file to you under the Apache
#   License, Version 2.0 (the "License"); you may not use this file
#   except in compliance with the License. You may obtain a copy of
#   the License at http://www.apache.org/licenses/LICENSE-2.0 .
#

# This script starts a new shell and sets all environment variables, which
# are necessary for building the examples of the Office Development Kit.
# The Script was developed for the operating systems Solaris, Linux and MacOS.

# The SDK name
OO_SDK_NAME=libreoffice6.3_sdk
export OO_SDK_NAME

# Installation directory of the Software Development Kit.
# Example: OO_SDK_HOME=/opt/libreoffice/basis3.4/sdk
OO_SDK_HOME='/usr/lib/libreoffice/sdk'

# Office installation directory.
# Example: OFFICE_HOME=/opt/libreoffice
OFFICE_HOME='/usr/lib/libreoffice'
export OFFICE_HOME

# Directory of the make command.
# Example: OO_SDK_MAKE_HOME=/usr/bin
OO_SDK_MAKE_HOME=/usr/bin

# Directory of the zip command.
# Example: OO_SDK_ZIP_HOME=/usr/bin
OO_SDK_ZIP_HOME=/usr/bin

# Directory of the cat command.
# Example: OO_SDK_CAT_HOME=/usr/bin
OO_SDK_CAT_HOME=/usr/bin

# Directory of the sed command.
# Example: OO_SDK_SED_HOME=/usr/bin
OO_SDK_SED_HOME=/usr/bin

# Directory of the C++ compiler.
# Example: OO_SDK_CPP_HOME=/usr/bin
OO_SDK_CPP_HOME=/usr/lib/distcc/bin

# Directory of the Java SDK.
# Example: OO_SDK_JAVA_HOME=/usr/jdk/jdk1.6.0_10
OO_SDK_JAVA_HOME=/usr/lib/jvm/default

# Special output directory
# Example: OO_SDK_OUTPUT_DIR=$HOME
OO_SDK_OUTPUT_DIR=$HOME

# Environment variable to enable auto deployment of example components
# Example: SDK_AUTO_DEPLOYMENT=YES
SDK_AUTO_DEPLOYMENT=NO
export SDK_AUTO_DEPLOYMENT

# Check installation path for the OpenOffice Development Kit.
if [ -z "${OO_SDK_HOME}" ]
then
    echo Error: Please insert a correct value for the variable OO_SDK_HOME.
    exit 0
fi

export OO_SDK_HOME

# Get the operating system.
sdk_platform=`/bin/sh ${OO_SDK_HOME}/config.guess | cut -d"-" -f3,4`

# Set the directory name.
programdir=program
sdk_lo_java_dir=program/classes
case ${sdk_platform} in
  darwin*)
    programdir="Contents/MacOS"
    sdk_lo_java_dir=Contents/Resources/java
    ;;
esac

# Set office program path (only set when using an Office).
OFFICE_PROGRAM_PATH=${OFFICE_HOME}/${programdir}
export OFFICE_PROGRAM_PATH

# Set UNO path, necessary to ensure that the cpp examples using the
# new UNO bootstrap mechanism use the configured office installation (only set
# when using an Office).
UNO_PATH=${OFFICE_PROGRAM_PATH}
export UNO_PATH

case ${sdk_platform} in
    darwin*)
        OO_SDK_URE_BIN_DIR=${OFFICE_PROGRAM_PATH}
        OO_SDK_URE_LIB_DIR=${OFFICE_HOME}/Contents/Frameworks
        OO_SDK_URE_JAVA_DIR=${OFFICE_HOME}/${sdk_lo_java_dir}
        ;;
    *)
        OO_SDK_URE_BIN_DIR=${OFFICE_PROGRAM_PATH}
        OO_SDK_URE_LIB_DIR=${OFFICE_PROGRAM_PATH}
        OO_SDK_URE_JAVA_DIR=${OFFICE_PROGRAM_PATH}/classes
        ;;
esac
export OO_SDK_URE_BIN_DIR
export OO_SDK_URE_LIB_DIR
export OO_SDK_URE_JAVA_DIR

OO_SDK_OUT=$OO_SDK_HOME
# Prepare appropriate output directory.
if [ -n "${OO_SDK_OUTPUT_DIR}" ]
then
    OO_SDK_OUT=${OO_SDK_OUTPUT_DIR}/${OO_SDK_NAME}
    export OO_SDK_OUT
fi

# Set the directory name.
case ${sdk_platform} in
  solaris*)
    sdk_proctype=`/bin/sh ${OO_SDK_HOME}/config.guess | cut -d"-" -f1`
    if [ "${sdk_proctype}" = "sparc" ]
    then
        directoryname=solsparc
	platform='Solaris Sparc'
    else
        directoryname=solintel
	platform='Solaris x86'
    fi
    comid=gcc3
    soext=so
    exampleout=SOLARISexample.out
    LD_LIBRARY_PATH=${OO_SDK_HOME}/lib:${OO_SDK_OUT}/${exampleout}/lib:${OO_SDK_URE_LIB_DIR}:.:${LD_LIBRARY_PATH}
    export LD_LIBRARY_PATH
    ;;

  darwin*)
    directoryname=macosx
    comid=gcc3
    soext=dylib
    exampleout=MACOSXexample.out
    platform=MacOSX
    DYLD_LIBRARY_PATH=${OO_SDK_OUT}/${directoryname}/lib:${OO_SDK_OUT}/${exampleout}/lib:${OO_SDK_URE_LIB_DIR}:.:${DYLD_LIBRARY_PATH}
    export DYLD_LIBRARY_PATH
    ;;

  linux-gnu*)
    directoryname=linux
    comid=gcc3
    soext=so
    exampleout=LINUXexample.out
    platform=Linux
    LD_LIBRARY_PATH=${OO_SDK_HOME}/lib:${OO_SDK_OUT}/${exampleout}/lib:${OO_SDK_URE_LIB_DIR}:.:${LD_LIBRARY_PATH}
    export LD_LIBRARY_PATH
    ;;

  freebsd*)
    directoryname=freebsd
    comid=gcc3
    soext=so
    exampleout=FREEBSDexample.out
    platform=FreeBSD
    LD_LIBRARY_PATH=${OO_SDK_HOME}/lib:${OO_SDK_OUT}/${exampleout}/lib:${OO_SDK_URE_LIB_DIR}:.:${LD_LIBRARY_PATH}
    export LD_LIBRARY_PATH

    if [ -e "/sbin/sysctl" ]
    then
	OSVERSION=`/sbin/sysctl  -n kern.osreldate`
    else
	OSVERSION=`/usr/sbin/sysctl  -n kern.osreldate`
    fi
    if [ $OSVERSION -lt 500016 ]
    then
	PTHREAD_CFLAGS=-D_THREAD_SAFE
	PTHREAD_LIBS=-pthread
	export PTHREAD_CFLAGS
	export PTHREAD_LIBS
    elif [ $OSVERSION -lt 502102 ]
    then
	PTHREAD_CFLAGS=-D_THREAD_SAFE
	PTHREAD_LIBS=-lc_r
	export PTHREAD_CFLAGS
	export PTHREAD_LIBS
    else
	PTHREAD_LIBS=-pthread
	export PTHREAD_LIBS
    fi
    ;;
esac

# Add directory of the SDK tools to the path.
PATH=${OO_SDK_HOME}/bin:${OO_SDK_OUT}/${exampleout}/bin:${OO_SDK_URE_BIN_DIR}:${OFFICE_PROGRAM_PATH}:${PATH}

# Set the classpath
CLASSPATH=${OO_SDK_URE_JAVA_DIR}/juh.jar:${OO_SDK_URE_JAVA_DIR}/jurt.jar:${OO_SDK_URE_JAVA_DIR}/ridl.jar:${OO_SDK_URE_JAVA_DIR}/unoloader.jar:${OFFICE_HOME}/$sdk_lo_java_dir/unoil.jar:${CLASSPATH}
export CLASSPATH


# Add directory of the command make to the path, if necessary.
if [ -n "${OO_SDK_MAKE_HOME}" ]
then
    PATH=${OO_SDK_MAKE_HOME}:${PATH}
    export OO_SDK_MAKE_HOME
fi

# Add directory of the zip tool to the path, if necessary.
if [ -n "${OO_SDK_ZIP_HOME}" ]
then
    PATH=${OO_SDK_ZIP_HOME}:${PATH}
    export OO_SDK_ZIP_HOME
fi

# Add directory of the sed tool to the path, if necessary.
if [ -n "${OO_SDK_SED_HOME}" ]
then
    PATH=${OO_SDK_SED_HOME}:${PATH}
    export OO_SDK_SED_HOME
fi

# Add directory of the cat tool to the path, if necessary.
if [ -n "${OO_SDK_CAT_HOME}" ]
then
    PATH=${OO_SDK_CAT_HOME}:${PATH}
    export OO_SDK_CAT_HOME
fi

# Add directory of the C++ tools to the path, if necessary.
if [ -n "${OO_SDK_CPP_HOME}" ]
then
    PATH=${OO_SDK_CPP_HOME}:${PATH}
    export OO_SDK_CPP_HOME
fi

# Add directory of the Java tools to the path, if necessary.
if [ -n "${OO_SDK_JAVA_HOME}" ]
then
    PATH=${OO_SDK_JAVA_HOME}/bin:${PATH}
#    JAVA_HOME=${OO_SDK_JAVA_HOME}
#    export JAVA_HOME
    export OO_SDK_JAVA_HOME

    export PATH
fi

export PATH

if [ "${platform}" = "MacOSX" ]
then
#    For URE, prepare symbolic links for libraries:
#    Only necessary on MacOSX, on other Unix systems the links are already prepared
#    in the SDK installation.

#    cleanup potential old links first
    rm -f "${OO_SDK_OUT}/${directoryname}/lib/libuno_cppu.${soext}" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_cppuhelper${comid}.${soext}" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_sal.${soext}" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_salhelper${comid}.${soext}" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_purpenvhelper${comid}.${soext}"

#    prepare links
    mkdir -p "${OO_SDK_OUT}/${directoryname}/lib"
    ln -s "${OO_SDK_URE_LIB_DIR}/libuno_cppu.${soext}.3" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_cppu.${soext}"
    ln -s "${OO_SDK_URE_LIB_DIR}/libuno_cppuhelper${comid}.${soext}.3" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_cppuhelper${comid}.${soext}"
    ln -s "${OO_SDK_URE_LIB_DIR}/libuno_sal.${soext}.3" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_sal.${soext}"
    ln -s "${OO_SDK_URE_LIB_DIR}/libuno_salhelper${comid}.${soext}.3" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_salhelper${comid}.${soext}"
    ln -s "${OO_SDK_URE_LIB_DIR}/libuno_purpenvhelper${comid}.${soext}.3" \
	"${OO_SDK_OUT}/${directoryname}/lib/libuno_purpenvhelper${comid}.${soext}"
fi


# Prepare shell with all necessary environment variables.
echo
echo " ************************************************************************"
echo " *"
echo " * SDK environment is prepared for ${platform}"
echo " *"
echo " * SDK = $OO_SDK_HOME"
echo " * Office = $OFFICE_HOME"
echo " * Make = $OO_SDK_MAKE_HOME"
echo " * Zip = $OO_SDK_ZIP_HOME"
echo " * cat = $OO_SDK_CAT_HOME"
echo " * sed = $OO_SDK_SED_HOME"
echo " * C++ Compiler = $OO_SDK_CPP_HOME"
echo " * Java = $OO_SDK_JAVA_HOME"
echo " * SDK Output directory = $OO_SDK_OUT"
echo " * Auto deployment = $SDK_AUTO_DEPLOYMENT"
echo " *"
echo " ************************************************************************"
