<?php
/**
 * Script to update list of supported lexers.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @file
 * @author Ori Livneh <ori@wikimedia.org>
 * @ingroup Maintenance
 */

use MediaWiki\Shell\Shell;
use Wikimedia\StaticArrayWriter;

$IP = getenv( 'MW_INSTALL_PATH' ) ?: __DIR__ . '/../../..';

require_once "$IP/maintenance/Maintenance.php";

class UpdateLexerList extends Maintenance {
	public function __construct() {
		parent::__construct();

		$this->requireExtension( 'SyntaxHighlight' );
		$this->addDescription( 'Update list of lexers supported by SyntaxHighlight_GeSHi' );
	}

	public function execute() {
		$header = 'Generated by ' . basename( __FILE__ );

		$lexers = [];

		$result = Shell::command(
			SyntaxHighlight::getPygmentizePath(),
			'-L', 'lexer'
		)
			->restrict( Shell::RESTRICT_DEFAULT | Shell::NO_NETWORK )
			->execute();

		if ( $result->getExitCode() != 0 ) {
			throw new \RuntimeException( $result->getStderr() );
		}

		$output = $result->getStdout();
		foreach ( explode( "\n", $output ) as $line ) {
			if ( substr( $line, 0, 1 ) === '*' ) {
				$newLexers = explode( ', ', trim( $line, "* :\n" ) );
				$lexers = array_merge( $lexers, $newLexers );
			}
		}
		$lexers = array_unique( $lexers );
		sort( $lexers );
		$data = [];
		foreach ( $lexers as $lexer ) {
			$data[$lexer] = true;
		}

		$writer = new StaticArrayWriter();
		$code = $writer->create( $data, $header );

		file_put_contents( __DIR__ . '/../SyntaxHighlight.lexers.php', $code );
		$this->output( "Updated language list written to SyntaxHighlight.lexers.php\n" );
	}
}

$maintClass = UpdateLexerList::class;
require_once RUN_MAINTENANCE_IF_MAIN;
