/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.service;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateObserver;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.LocalNodeMasterListener;
import org.elasticsearch.cluster.NodeConnectionsService;
import org.elasticsearch.cluster.TimeoutClusterStateListener;
import org.elasticsearch.cluster.metadata.ProcessClusterEventTimeoutException;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterApplier;
import org.elasticsearch.cluster.service.SourcePrioritizedRunnable;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.StopWatch;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.PrioritizedEsThreadPoolExecutor;
import org.elasticsearch.common.util.iterable.Iterables;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class ClusterApplierService
extends AbstractLifecycleComponent
implements ClusterApplier {
    private static final Logger logger = LogManager.getLogger(ClusterApplierService.class);
    public static final Setting<TimeValue> CLUSTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING = Setting.positiveTimeSetting("cluster.service.slow_task_logging_threshold", TimeValue.timeValueSeconds((long)30L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final String CLUSTER_UPDATE_THREAD_NAME = "clusterApplierService#updateTask";
    private final ClusterSettings clusterSettings;
    protected final ThreadPool threadPool;
    private volatile TimeValue slowTaskLoggingThreshold;
    private volatile PrioritizedEsThreadPoolExecutor threadPoolExecutor;
    private final Collection<ClusterStateApplier> highPriorityStateAppliers = new CopyOnWriteArrayList<ClusterStateApplier>();
    private final Collection<ClusterStateApplier> normalPriorityStateAppliers = new CopyOnWriteArrayList<ClusterStateApplier>();
    private final Collection<ClusterStateApplier> lowPriorityStateAppliers = new CopyOnWriteArrayList<ClusterStateApplier>();
    private final Iterable<ClusterStateApplier> clusterStateAppliers = Iterables.concat(this.highPriorityStateAppliers, this.normalPriorityStateAppliers, this.lowPriorityStateAppliers);
    private final Collection<ClusterStateListener> clusterStateListeners = new CopyOnWriteArrayList<ClusterStateListener>();
    private final Collection<TimeoutClusterStateListener> timeoutClusterStateListeners = Collections.newSetFromMap(new ConcurrentHashMap());
    private final LocalNodeMasterListeners localNodeMasterListeners;
    private final Queue<NotifyTimeout> onGoingTimeouts = ConcurrentCollections.newQueue();
    private final AtomicReference<ClusterState> state;
    private final String nodeName;
    private NodeConnectionsService nodeConnectionsService;

    public ClusterApplierService(String nodeName, Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.clusterSettings = clusterSettings;
        this.threadPool = threadPool;
        this.state = new AtomicReference();
        this.localNodeMasterListeners = new LocalNodeMasterListeners(threadPool);
        this.nodeName = nodeName;
        this.slowTaskLoggingThreshold = CLUSTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING.get(settings);
        this.clusterSettings.addSettingsUpdateConsumer(CLUSTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING, this::setSlowTaskLoggingThreshold);
    }

    private void setSlowTaskLoggingThreshold(TimeValue slowTaskLoggingThreshold) {
        this.slowTaskLoggingThreshold = slowTaskLoggingThreshold;
    }

    public synchronized void setNodeConnectionsService(NodeConnectionsService nodeConnectionsService) {
        assert (this.nodeConnectionsService == null) : "nodeConnectionsService is already set";
        this.nodeConnectionsService = nodeConnectionsService;
    }

    @Override
    public void setInitialState(ClusterState initialState) {
        if (this.lifecycle.started()) {
            throw new IllegalStateException("can't set initial state when started");
        }
        assert (this.state.get() == null) : "state is already set";
        this.state.set(initialState);
    }

    @Override
    protected synchronized void doStart() {
        Objects.requireNonNull(this.nodeConnectionsService, "please set the node connection service before starting");
        Objects.requireNonNull(this.state.get(), "please set initial state before starting");
        this.addListener(this.localNodeMasterListeners);
        this.threadPoolExecutor = this.createThreadPoolExecutor();
    }

    protected PrioritizedEsThreadPoolExecutor createThreadPoolExecutor() {
        return EsExecutors.newSinglePrioritizing(this.nodeName + "/" + CLUSTER_UPDATE_THREAD_NAME, EsExecutors.daemonThreadFactory(this.nodeName, CLUSTER_UPDATE_THREAD_NAME), this.threadPool.getThreadContext(), this.threadPool.scheduler());
    }

    @Override
    protected synchronized void doStop() {
        for (NotifyTimeout onGoingTimeout : this.onGoingTimeouts) {
            onGoingTimeout.cancel();
            try {
                onGoingTimeout.cancel();
                onGoingTimeout.listener.onClose();
            }
            catch (Exception ex) {
                logger.debug("failed to notify listeners on shutdown", (Throwable)ex);
            }
        }
        ThreadPool.terminate(this.threadPoolExecutor, 10L, TimeUnit.SECONDS);
        this.timeoutClusterStateListeners.forEach(TimeoutClusterStateListener::onClose);
        this.removeListener(this.localNodeMasterListeners);
    }

    @Override
    protected synchronized void doClose() {
    }

    public ClusterState state() {
        assert (ClusterApplierService.assertNotCalledFromClusterStateApplier("the applied cluster state is not yet available"));
        ClusterState clusterState = this.state.get();
        assert (clusterState != null) : "initial cluster state not set yet";
        return clusterState;
    }

    public void addHighPriorityApplier(ClusterStateApplier applier) {
        this.highPriorityStateAppliers.add(applier);
    }

    public void addLowPriorityApplier(ClusterStateApplier applier) {
        this.lowPriorityStateAppliers.add(applier);
    }

    public void addStateApplier(ClusterStateApplier applier) {
        this.normalPriorityStateAppliers.add(applier);
    }

    public void removeApplier(ClusterStateApplier applier) {
        this.normalPriorityStateAppliers.remove(applier);
        this.highPriorityStateAppliers.remove(applier);
        this.lowPriorityStateAppliers.remove(applier);
    }

    public void addListener(ClusterStateListener listener) {
        this.clusterStateListeners.add(listener);
    }

    public void removeListener(ClusterStateListener listener) {
        this.clusterStateListeners.remove(listener);
    }

    public void removeTimeoutListener(TimeoutClusterStateListener listener) {
        this.timeoutClusterStateListeners.remove(listener);
        Iterator it = this.onGoingTimeouts.iterator();
        while (it.hasNext()) {
            NotifyTimeout timeout = (NotifyTimeout)it.next();
            if (!timeout.listener.equals(listener)) continue;
            timeout.cancel();
            it.remove();
        }
    }

    public void addLocalNodeMasterListener(LocalNodeMasterListener listener) {
        this.localNodeMasterListeners.add(listener);
    }

    public void addTimeoutListener(final @Nullable TimeValue timeout, final TimeoutClusterStateListener listener) {
        if (this.lifecycle.stoppedOrClosed()) {
            listener.onClose();
            return;
        }
        try {
            this.threadPoolExecutor.execute(new SourcePrioritizedRunnable(Priority.HIGH, "_add_listener_"){

                @Override
                public void run() {
                    if (timeout != null) {
                        NotifyTimeout notifyTimeout = new NotifyTimeout(listener, timeout);
                        notifyTimeout.cancellable = ClusterApplierService.this.threadPool.schedule(notifyTimeout, timeout, "generic");
                        ClusterApplierService.this.onGoingTimeouts.add(notifyTimeout);
                    }
                    ClusterApplierService.this.timeoutClusterStateListeners.add(listener);
                    listener.postAdded();
                }
            });
        }
        catch (EsRejectedExecutionException e) {
            if (this.lifecycle.stoppedOrClosed()) {
                listener.onClose();
            }
            throw e;
        }
    }

    public void runOnApplierThread(String source, Consumer<ClusterState> clusterStateConsumer, ClusterApplier.ClusterApplyListener listener, Priority priority) {
        this.submitStateUpdateTask(source, ClusterStateTaskConfig.build(priority), clusterState -> {
            clusterStateConsumer.accept((ClusterState)clusterState);
            return clusterState;
        }, listener);
    }

    public void runOnApplierThread(String source, Consumer<ClusterState> clusterStateConsumer, ClusterApplier.ClusterApplyListener listener) {
        this.runOnApplierThread(source, clusterStateConsumer, listener, Priority.HIGH);
    }

    public ThreadPool threadPool() {
        return this.threadPool;
    }

    @Override
    public void onNewClusterState(String source, Supplier<ClusterState> clusterStateSupplier, ClusterApplier.ClusterApplyListener listener) {
        Function<ClusterState, ClusterState> applyFunction = currentState -> {
            ClusterState nextState = (ClusterState)clusterStateSupplier.get();
            if (nextState != null) {
                return nextState;
            }
            return currentState;
        };
        this.submitStateUpdateTask(source, ClusterStateTaskConfig.build(Priority.HIGH), applyFunction, listener);
    }

    private void submitStateUpdateTask(String source, ClusterStateTaskConfig config, Function<ClusterState, ClusterState> executor, ClusterApplier.ClusterApplyListener listener) {
        block5: {
            if (!this.lifecycle.started()) {
                return;
            }
            try {
                UpdateTask updateTask = new UpdateTask(config.priority(), source, new SafeClusterApplyListener(listener, logger), executor);
                if (config.timeout() != null) {
                    this.threadPoolExecutor.execute(updateTask, config.timeout(), () -> this.threadPool.generic().execute(() -> listener.onFailure(source, new ProcessClusterEventTimeoutException(config.timeout(), source))));
                } else {
                    this.threadPoolExecutor.execute(updateTask);
                }
            }
            catch (EsRejectedExecutionException e) {
                if (this.lifecycle.stoppedOrClosed()) break block5;
                throw e;
            }
        }
    }

    public static boolean assertNotClusterStateUpdateThread(String reason) {
        assert (!Thread.currentThread().getName().contains(CLUSTER_UPDATE_THREAD_NAME)) : "Expected current thread [" + Thread.currentThread() + "] to not be the cluster state update thread. Reason: [" + reason + "]";
        return true;
    }

    private static boolean assertNotCalledFromClusterStateApplier(String reason) {
        if (Thread.currentThread().getName().contains(CLUSTER_UPDATE_THREAD_NAME)) {
            for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
                String className = element.getClassName();
                String methodName = element.getMethodName();
                if (className.equals(ClusterStateObserver.class.getName())) {
                    return true;
                }
                if (className.equals(ClusterApplierService.class.getName()) && methodName.equals("callClusterStateAppliers")) {
                    throw new AssertionError((Object)("should not be called by a cluster state applier. reason [" + reason + "]"));
                }
            }
        }
        return true;
    }

    private void runTask(UpdateTask task) {
        TimeValue executionTime;
        ClusterState newClusterState;
        if (!this.lifecycle.started()) {
            logger.debug("processing [{}]: ignoring, cluster applier service not started", (Object)task.source);
            return;
        }
        logger.debug("processing [{}]: execute", (Object)task.source);
        ClusterState previousClusterState = this.state.get();
        long startTimeMS = this.currentTimeInMillis();
        StopWatch stopWatch = new StopWatch();
        try (Releasable ignored = stopWatch.timing("running task [" + task.source + ']');){
            newClusterState = task.apply(previousClusterState);
        }
        catch (Exception e) {
            TimeValue executionTime2 = TimeValue.timeValueMillis((long)Math.max(0L, this.currentTimeInMillis() - startTimeMS));
            logger.trace(() -> new ParameterizedMessage("failed to execute cluster state applier in [{}], state:\nversion [{}], source [{}]\n{}", new Object[]{executionTime2, previousClusterState.version(), task.source, previousClusterState}), (Throwable)e);
            this.warnAboutSlowTaskIfNeeded(executionTime2, task.source, stopWatch);
            task.listener.onFailure(task.source, e);
            return;
        }
        if (previousClusterState == newClusterState) {
            executionTime = TimeValue.timeValueMillis((long)Math.max(0L, this.currentTimeInMillis() - startTimeMS));
            logger.debug("processing [{}]: took [{}] no change in cluster state", (Object)task.source, (Object)executionTime);
            this.warnAboutSlowTaskIfNeeded(executionTime, task.source, stopWatch);
            task.listener.onSuccess(task.source);
        } else {
            if (logger.isTraceEnabled()) {
                logger.debug("cluster state updated, version [{}], source [{}]\n{}", (Object)newClusterState.version(), (Object)task.source, (Object)newClusterState);
            } else {
                logger.debug("cluster state updated, version [{}], source [{}]", (Object)newClusterState.version(), (Object)task.source);
            }
            try {
                this.applyChanges(task, previousClusterState, newClusterState, stopWatch);
                executionTime = TimeValue.timeValueMillis((long)Math.max(0L, this.currentTimeInMillis() - startTimeMS));
                logger.debug("processing [{}]: took [{}] done applying updated cluster state (version: {}, uuid: {})", (Object)task.source, (Object)executionTime, (Object)newClusterState.version(), (Object)newClusterState.stateUUID());
                this.warnAboutSlowTaskIfNeeded(executionTime, task.source, stopWatch);
                task.listener.onSuccess(task.source);
            }
            catch (Exception e) {
                TimeValue executionTime3 = TimeValue.timeValueMillis((long)Math.max(0L, this.currentTimeInMillis() - startTimeMS));
                if (logger.isTraceEnabled()) {
                    logger.warn((Message)new ParameterizedMessage("failed to apply updated cluster state in [{}]:\nversion [{}], uuid [{}], source [{}]\n{}", new Object[]{executionTime3, newClusterState.version(), newClusterState.stateUUID(), task.source, newClusterState}), (Throwable)e);
                } else {
                    logger.warn((Message)new ParameterizedMessage("failed to apply updated cluster state in [{}]:\nversion [{}], uuid [{}], source [{}]", new Object[]{executionTime3, newClusterState.version(), newClusterState.stateUUID(), task.source}), (Throwable)e);
                }
                assert (this.applicationMayFail());
                task.listener.onFailure(task.source, e);
            }
        }
    }

    private void applyChanges(UpdateTask task, ClusterState previousClusterState, ClusterState newClusterState, StopWatch stopWatch) {
        String summary;
        ClusterChangedEvent clusterChangedEvent = new ClusterChangedEvent(task.source, newClusterState, previousClusterState);
        DiscoveryNodes.Delta nodesDelta = clusterChangedEvent.nodesDelta();
        if (nodesDelta.hasChanges() && logger.isInfoEnabled() && (summary = nodesDelta.shortSummary()).length() > 0) {
            logger.info("{}, term: {}, version: {}, reason: {}", (Object)summary, (Object)newClusterState.term(), (Object)newClusterState.version(), (Object)task.source);
        }
        logger.trace("connecting to nodes of cluster state with version {}", (Object)newClusterState.version());
        try (Releasable ignored = stopWatch.timing("connecting to new nodes");){
            this.connectToNodesAndWait(newClusterState);
        }
        if (!clusterChangedEvent.state().blocks().disableStatePersistence() && clusterChangedEvent.metaDataChanged()) {
            logger.debug("applying settings from cluster state with version {}", (Object)newClusterState.version());
            Settings incomingSettings = clusterChangedEvent.state().metaData().settings();
            try (Releasable ignored = stopWatch.timing("applying settings");){
                this.clusterSettings.applySettings(incomingSettings);
            }
        }
        logger.debug("apply cluster state with version {}", (Object)newClusterState.version());
        this.callClusterStateAppliers(clusterChangedEvent, stopWatch);
        this.nodeConnectionsService.disconnectFromNodesExcept(newClusterState.nodes());
        assert (newClusterState.coordinationMetaData().getLastAcceptedConfiguration().equals(newClusterState.coordinationMetaData().getLastCommittedConfiguration())) : newClusterState.coordinationMetaData().getLastAcceptedConfiguration() + " vs " + newClusterState.coordinationMetaData().getLastCommittedConfiguration() + " on " + newClusterState.nodes().getLocalNode();
        logger.debug("set locally applied cluster state to version {}", (Object)newClusterState.version());
        this.state.set(newClusterState);
        this.callClusterStateListeners(clusterChangedEvent, stopWatch);
    }

    protected void connectToNodesAndWait(ClusterState newClusterState) {
        CountDownLatch countDownLatch = new CountDownLatch(1);
        this.nodeConnectionsService.connectToNodes(newClusterState.nodes(), countDownLatch::countDown);
        try {
            countDownLatch.await();
        }
        catch (InterruptedException e) {
            logger.debug("interrupted while connecting to nodes, continuing", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private void callClusterStateAppliers(ClusterChangedEvent clusterChangedEvent, StopWatch stopWatch) {
        this.clusterStateAppliers.forEach(applier -> {
            logger.trace("calling [{}] with change to version [{}]", applier, (Object)clusterChangedEvent.state().version());
            try (Releasable ignored = stopWatch.timing("running applier [" + applier + "]");){
                applier.applyClusterState(clusterChangedEvent);
            }
        });
    }

    private void callClusterStateListeners(ClusterChangedEvent clusterChangedEvent, StopWatch stopWatch) {
        Stream.concat(this.clusterStateListeners.stream(), this.timeoutClusterStateListeners.stream()).forEach(listener -> {
            try {
                logger.trace("calling [{}] with change to version [{}]", listener, (Object)clusterChangedEvent.state().version());
                try (Releasable ignored = stopWatch.timing("notifying listener [" + listener + "]");){
                    listener.clusterChanged(clusterChangedEvent);
                }
            }
            catch (Exception ex) {
                logger.warn("failed to notify ClusterStateListener", (Throwable)ex);
            }
        });
    }

    private void warnAboutSlowTaskIfNeeded(TimeValue executionTime, String source, StopWatch stopWatch) {
        if (executionTime.getMillis() > this.slowTaskLoggingThreshold.getMillis()) {
            logger.warn("cluster state applier task [{}] took [{}] which is above the warn threshold of [{}]: {}", (Object)source, (Object)executionTime, (Object)this.slowTaskLoggingThreshold, (Object)Arrays.stream(stopWatch.taskInfo()).map(ti -> '[' + ti.getTaskName() + "] took [" + ti.getTime().millis() + "ms]").collect(Collectors.joining(", ")));
        }
    }

    protected long currentTimeInMillis() {
        return this.threadPool.relativeTimeInMillis();
    }

    protected boolean applicationMayFail() {
        return false;
    }

    class UpdateTask
    extends SourcePrioritizedRunnable
    implements Function<ClusterState, ClusterState> {
        final ClusterApplier.ClusterApplyListener listener;
        final Function<ClusterState, ClusterState> updateFunction;

        UpdateTask(Priority priority, String source, ClusterApplier.ClusterApplyListener listener, Function<ClusterState, ClusterState> updateFunction) {
            super(priority, source);
            this.listener = listener;
            this.updateFunction = updateFunction;
        }

        @Override
        public ClusterState apply(ClusterState clusterState) {
            return this.updateFunction.apply(clusterState);
        }

        @Override
        public void run() {
            ClusterApplierService.this.runTask(this);
        }
    }

    private static class LocalNodeMasterListeners
    implements ClusterStateListener {
        private final List<LocalNodeMasterListener> listeners = new CopyOnWriteArrayList<LocalNodeMasterListener>();
        private final ThreadPool threadPool;
        private volatile boolean master = false;

        private LocalNodeMasterListeners(ThreadPool threadPool) {
            this.threadPool = threadPool;
        }

        @Override
        public void clusterChanged(ClusterChangedEvent event) {
            if (!this.master && event.localNodeMaster()) {
                this.master = true;
                for (LocalNodeMasterListener listener : this.listeners) {
                    ExecutorService executor = this.threadPool.executor(listener.executorName());
                    executor.execute(new OnMasterRunnable(listener));
                }
                return;
            }
            if (this.master && !event.localNodeMaster()) {
                this.master = false;
                for (LocalNodeMasterListener listener : this.listeners) {
                    ExecutorService executor = this.threadPool.executor(listener.executorName());
                    executor.execute(new OffMasterRunnable(listener));
                }
            }
        }

        private void add(LocalNodeMasterListener listener) {
            this.listeners.add(listener);
        }
    }

    class NotifyTimeout
    implements Runnable {
        final TimeoutClusterStateListener listener;
        final TimeValue timeout;
        volatile Scheduler.Cancellable cancellable;

        NotifyTimeout(TimeoutClusterStateListener listener, TimeValue timeout) {
            this.listener = listener;
            this.timeout = timeout;
        }

        public void cancel() {
            if (this.cancellable != null) {
                this.cancellable.cancel();
            }
        }

        @Override
        public void run() {
            if (this.cancellable != null && this.cancellable.isCancelled()) {
                return;
            }
            if (ClusterApplierService.this.lifecycle.stoppedOrClosed()) {
                this.listener.onClose();
            } else {
                this.listener.onTimeout(this.timeout);
            }
        }
    }

    private static class SafeClusterApplyListener
    implements ClusterApplier.ClusterApplyListener {
        private final ClusterApplier.ClusterApplyListener listener;
        private final Logger logger;

        SafeClusterApplyListener(ClusterApplier.ClusterApplyListener listener, Logger logger) {
            this.listener = listener;
            this.logger = logger;
        }

        @Override
        public void onFailure(String source, Exception e) {
            try {
                this.listener.onFailure(source, e);
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                this.logger.error((Message)new ParameterizedMessage("exception thrown by listener notifying of failure from [{}]", (Object)source), (Throwable)inner);
            }
        }

        @Override
        public void onSuccess(String source) {
            try {
                this.listener.onSuccess(source);
            }
            catch (Exception e) {
                this.logger.error((Message)new ParameterizedMessage("exception thrown by listener while notifying of cluster state processed from [{}]", (Object)source), (Throwable)e);
            }
        }
    }

    private static class OffMasterRunnable
    implements Runnable {
        private final LocalNodeMasterListener listener;

        private OffMasterRunnable(LocalNodeMasterListener listener) {
            this.listener = listener;
        }

        @Override
        public void run() {
            this.listener.offMaster();
        }
    }

    private static class OnMasterRunnable
    implements Runnable {
        private final LocalNodeMasterListener listener;

        private OnMasterRunnable(LocalNodeMasterListener listener) {
            this.listener = listener;
        }

        @Override
        public void run() {
            this.listener.onMaster();
        }
    }
}

