/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.io;

import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonNumber;
import javax.json.JsonObject;
import javax.json.JsonString;
import javax.json.JsonValue;
import javax.json.stream.JsonParser;
import javax.json.stream.JsonParsingException;
import org.openstreetmap.josm.data.coor.EastNorth;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.osm.BBox;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.UploadPolicy;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.data.projection.Projection;
import org.openstreetmap.josm.data.projection.Projections;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.io.AbstractReader;
import org.openstreetmap.josm.io.IllegalDataException;
import org.openstreetmap.josm.tools.I18n;
import org.openstreetmap.josm.tools.Logging;

public class GeoJSONReader
extends AbstractReader {
    private static final String CRS = "crs";
    private static final String NAME = "name";
    private static final String LINK = "link";
    private static final String COORDINATES = "coordinates";
    private static final String FEATURES = "features";
    private static final String PROPERTIES = "properties";
    private static final String GEOMETRY = "geometry";
    private static final String TYPE = "type";
    private JsonParser parser;
    private Projection projection = Projections.getProjectionByCode("EPSG:4326");

    GeoJSONReader() {
    }

    private void setParser(JsonParser parser) {
        this.parser = parser;
    }

    private void parse() throws IllegalDataException {
        while (this.parser.hasNext()) {
            JsonParser.Event event = this.parser.next();
            if (event != JsonParser.Event.START_OBJECT) continue;
            this.parseRoot(this.parser.getObject());
        }
        this.parser.close();
    }

    private void parseRoot(JsonObject object) throws IllegalDataException {
        this.parseCrs(object.getJsonObject(CRS));
        switch (Optional.ofNullable(object.getJsonString(TYPE)).orElseThrow(() -> new IllegalDataException("No type")).getString()) {
            case "FeatureCollection": {
                this.parseFeatureCollection(object.getJsonArray(FEATURES));
                break;
            }
            case "Feature": {
                this.parseFeature(object);
                break;
            }
            case "GeometryCollection": {
                this.parseGeometryCollection(null, object);
                break;
            }
            default: {
                this.parseGeometry(null, object);
            }
        }
    }

    private void parseCrs(JsonObject crs) throws IllegalDataException {
        JsonObject properties;
        if (crs != null && (properties = crs.getJsonObject(PROPERTIES)) != null) {
            switch (crs.getString(TYPE)) {
                case "name": {
                    String crsName = properties.getString(NAME);
                    if ("urn:ogc:def:crs:OGC:1.3:CRS84".equals(crsName)) {
                        crsName = "EPSG:4326";
                    } else if (crsName.startsWith("urn:ogc:def:crs:EPSG:")) {
                        crsName = crsName.replace("urn:ogc:def:crs:", "");
                    }
                    this.projection = Optional.ofNullable(Projections.getProjectionByCode(crsName)).orElse(Projections.getProjectionByCode("EPSG:4326"));
                    break;
                }
                default: {
                    throw new IllegalDataException(crs.toString());
                }
            }
        }
    }

    private void parseFeatureCollection(JsonArray features) {
        for (JsonValue feature : features) {
            if (!(feature instanceof JsonObject)) continue;
            this.parseFeature((JsonObject)feature);
        }
    }

    private void parseFeature(JsonObject feature) {
        JsonValue geometry = (JsonValue)feature.get(GEOMETRY);
        if (geometry != null && geometry.getValueType() == JsonValue.ValueType.OBJECT) {
            this.parseGeometry(feature, geometry.asJsonObject());
        } else {
            JsonValue properties = (JsonValue)feature.get(PROPERTIES);
            if (properties != null && properties.getValueType() == JsonValue.ValueType.OBJECT) {
                this.parseNonGeometryFeature(feature, properties.asJsonObject());
            } else {
                Logging.warn(I18n.tr("Relation/non-geometry feature without properties found: {0}", feature));
            }
        }
    }

    private void parseNonGeometryFeature(JsonObject feature, JsonObject properties) {
        JsonValue type = (JsonValue)properties.get(TYPE);
        if (type == null || properties.getValueType() == JsonValue.ValueType.STRING) {
            Logging.warn(I18n.tr("Relation/non-geometry feature without type found: {0}", feature));
            return;
        }
        Relation relation = new Relation();
        relation.put(TYPE, type.toString());
        GeoJSONReader.fillTagsFromFeature(feature, relation);
        this.getDataSet().addPrimitive(relation);
    }

    private void parseGeometryCollection(JsonObject feature, JsonObject geometry) {
        for (JsonValue jsonValue : geometry.getJsonArray("geometries")) {
            this.parseGeometry(feature, jsonValue.asJsonObject());
        }
    }

    private void parseGeometry(JsonObject feature, JsonObject geometry) {
        if (geometry == null) {
            GeoJSONReader.parseNullGeometry(feature);
            return;
        }
        switch (geometry.getString(TYPE)) {
            case "Point": {
                this.parsePoint(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiPoint": {
                this.parseMultiPoint(feature, geometry);
                break;
            }
            case "LineString": {
                this.parseLineString(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiLineString": {
                this.parseMultiLineString(feature, geometry);
                break;
            }
            case "Polygon": {
                this.parsePolygon(feature, geometry.getJsonArray(COORDINATES));
                break;
            }
            case "MultiPolygon": {
                this.parseMultiPolygon(feature, geometry);
                break;
            }
            case "GeometryCollection": {
                this.parseGeometryCollection(feature, geometry);
                break;
            }
            default: {
                GeoJSONReader.parseUnknown(geometry);
            }
        }
    }

    private LatLon getLatLon(JsonArray coordinates) {
        return this.projection.eastNorth2latlon(new EastNorth(GeoJSONReader.parseCoordinate((JsonValue)coordinates.get(0)), GeoJSONReader.parseCoordinate((JsonValue)coordinates.get(1))));
    }

    private static double parseCoordinate(JsonValue coordinate) {
        if (coordinate instanceof JsonString) {
            return Double.parseDouble(((JsonString)coordinate).getString());
        }
        if (coordinate instanceof JsonNumber) {
            return ((JsonNumber)coordinate).doubleValue();
        }
        throw new IllegalArgumentException(Objects.toString(coordinate));
    }

    private void parsePoint(JsonObject feature, JsonArray coordinates) {
        GeoJSONReader.fillTagsFromFeature(feature, this.createNode(this.getLatLon(coordinates)));
    }

    private void parseMultiPoint(JsonObject feature, JsonObject geometry) {
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parsePoint(feature, coordinate.asJsonArray());
        }
    }

    private void parseLineString(JsonObject feature, JsonArray coordinates) {
        if (!coordinates.isEmpty()) {
            this.createWay(coordinates, false).ifPresent(way -> GeoJSONReader.fillTagsFromFeature(feature, way));
        }
    }

    private void parseMultiLineString(JsonObject feature, JsonObject geometry) {
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parseLineString(feature, coordinate.asJsonArray());
        }
    }

    private void parsePolygon(JsonObject feature, JsonArray coordinates) {
        int size = coordinates.size();
        if (size == 1) {
            this.createWay(coordinates.getJsonArray(0), true).ifPresent(way -> GeoJSONReader.fillTagsFromFeature(feature, way));
        } else if (size > 1) {
            Relation multipolygon = new Relation();
            multipolygon.put(TYPE, "multipolygon");
            this.createWay(coordinates.getJsonArray(0), true).ifPresent(way -> multipolygon.addMember(new RelationMember("outer", (OsmPrimitive)way)));
            for (JsonValue interiorRing : coordinates.subList(1, size)) {
                this.createWay(interiorRing.asJsonArray(), true).ifPresent(way -> multipolygon.addMember(new RelationMember("inner", (OsmPrimitive)way)));
            }
            GeoJSONReader.fillTagsFromFeature(feature, multipolygon);
            this.getDataSet().addPrimitive(multipolygon);
        }
    }

    private void parseMultiPolygon(JsonObject feature, JsonObject geometry) {
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parsePolygon(feature, coordinate.asJsonArray());
        }
    }

    private Node createNode(LatLon latlon) {
        List<Node> existingNodes = this.getDataSet().searchNodes(new BBox(latlon, latlon));
        if (!existingNodes.isEmpty()) {
            return existingNodes.get(0);
        }
        Node node = new Node(latlon);
        this.getDataSet().addPrimitive(node);
        return node;
    }

    private Optional<Way> createWay(JsonArray coordinates, boolean autoClose) {
        boolean doAutoclose;
        if (coordinates.isEmpty()) {
            return Optional.empty();
        }
        List latlons = coordinates.stream().map(coordinate -> this.getLatLon(coordinate.asJsonArray())).collect(Collectors.toList());
        int size = latlons.size();
        if (size > 1) {
            if (((LatLon)latlons.get(0)).equals(latlons.get(size - 1))) {
                latlons.remove(size - 1);
                doAutoclose = true;
            } else {
                doAutoclose = autoClose;
            }
        } else {
            doAutoclose = false;
        }
        Way way = new Way();
        this.getDataSet().addPrimitive(way);
        way.setNodes(latlons.stream().map(this::createNode).collect(Collectors.toList()));
        if (doAutoclose) {
            way.addNode(way.getNode(0));
        }
        return Optional.of(way);
    }

    private static void fillTagsFromFeature(JsonObject feature, OsmPrimitive primitive) {
        if (feature != null) {
            primitive.setKeys(GeoJSONReader.getTags(feature));
        }
    }

    private static void parseUnknown(JsonObject object) {
        Logging.warn(I18n.tr("Unknown json object found {0}", object));
    }

    private static void parseNullGeometry(JsonObject feature) {
        Logging.warn(I18n.tr("Geometry of feature {0} is null", feature));
    }

    private static Map<String, String> getTags(JsonObject feature) {
        JsonValue properties;
        TreeMap<String, String> tags = new TreeMap<String, String>();
        if (feature.containsKey(PROPERTIES) && !feature.isNull(PROPERTIES) && (properties = (JsonValue)feature.get(PROPERTIES)) != null && properties.getValueType() == JsonValue.ValueType.OBJECT) {
            for (Map.Entry stringJsonValueEntry : properties.asJsonObject().entrySet()) {
                JsonValue value = (JsonValue)stringJsonValueEntry.getValue();
                if (value instanceof JsonString) {
                    tags.put((String)stringJsonValueEntry.getKey(), ((JsonString)value).getString());
                    continue;
                }
                if (value instanceof JsonObject) {
                    Logging.warn("The GeoJSON contains an object with property '" + (String)stringJsonValueEntry.getKey() + "' whose value has the unsupported type '" + value.getClass().getSimpleName() + "'. That key-value pair is ignored!");
                    continue;
                }
                if (value.getValueType() == JsonValue.ValueType.NULL) continue;
                tags.put((String)stringJsonValueEntry.getKey(), value.toString());
            }
        }
        return tags;
    }

    @Override
    protected DataSet doParseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        this.setParser(Json.createParser(source));
        this.ds.setUploadPolicy(UploadPolicy.DISCOURAGED);
        try {
            this.parse();
        }
        catch (JsonParsingException e) {
            throw new IllegalDataException(e);
        }
        return this.getDataSet();
    }

    public static DataSet parseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        return new GeoJSONReader().doParseDataSet(source, progressMonitor);
    }
}

