"""Testing other aspects of the implementation and API."""

import os
import re
import sys
import time
import pytest
import builtins

from io import StringIO
from random import randint
from pathlib import Path
from datetime import timedelta
from tempfile import TemporaryDirectory
from contextlib import suppress, ExitStack, contextmanager
from flufl.lock import SEP, Lock, LockState, TimeOutError, NotLockedError
from unittest.mock import patch
from multiprocessing import Queue, Process


EMOCKEDFAILURE = 99
EOTHERMOCKEDFAILURE = 98


@pytest.fixture
def lock():
    with TemporaryDirectory() as lock_dir:
        lock = Lock(os.path.join(lock_dir, 'test.lck'))
        yield lock
        with suppress(NotLockedError):
            lock.unlock()


def test_retry_errno_property(lock):
    assert lock.retry_errnos == []
    lock.retry_errnos = [EMOCKEDFAILURE, EOTHERMOCKEDFAILURE]
    assert lock.retry_errnos == [EMOCKEDFAILURE, EOTHERMOCKEDFAILURE]
    del lock.retry_errnos
    assert lock.retry_errnos == []


class RetryOpen:
    def __init__(self, failure_countdown=0, retry_count=0):
        self.failure_countdown = failure_countdown
        self.retry_count = retry_count
        self._open = builtins.open
        self._errno = EMOCKEDFAILURE

    def __call__(self, *args, **kws):
        if self.failure_countdown <= 0:
            return self._open(*args, **kws)
        self.failure_countdown -= 1
        self.retry_count += 1
        raise OSError(self._errno, 'test exception')


def test_retries(lock):
    # Test that _read() will retry when a given errno is encountered.
    lock.lock()
    lock.retry_errnos = [EMOCKEDFAILURE]
    retry_open = RetryOpen(failure_countdown=3)
    with patch('builtins.open', retry_open):
        # This should trigger exactly 3 retries.
        assert lock.is_locked
    assert retry_open.retry_count == 3


def test_is_locked_permission_error(lock):
    with ExitStack() as resources:
        resources.enter_context(patch('os.utime', side_effect=PermissionError))
        log_mock = resources.enter_context(patch('flufl.lock._lockfile.log'))
        assert not lock.is_locked
        log_mock.error.assert_called_once_with(
            'No permission to refresh the log')


def test_nondefault_lifetime(tmpdir):
    lock_file = os.path.join(tmpdir, 'test.lck')
    assert Lock(lock_file, lifetime=77).lifetime.seconds == 77


def test_lockfile_repr(lock):
    # Handle both POSIX and Windows paths.
    assert re.match(
        r'<Lock .*test.lck \[unlocked: \d{1,2}:\d{2}:\d{2}] pid=\d+ at .+>',
        repr(lock))
    lock.lock()
    assert re.match(
        r'<Lock .*test.lck \[locked: \d{1,2}:\d{2}:\d{2}] pid=\d+ at .+>',
        repr(lock))
    lock.unlock()
    assert re.match(
        r'<Lock .*test.lck \[unlocked: \d{1,2}:\d{2}:\d{2}] pid=\d+ at .+>',
        repr(lock))


def test_lockfile_repr_does_not_refresh(lock):
    with lock:
        expiration = lock.expiration
        time.sleep(1)
        repr(lock)
        assert lock.expiration == expiration


def test_details(lock):
    # No details are available if the lock is not locked.
    with pytest.raises(NotLockedError):
        lock.details()
    with lock:
        hostname, pid, filename = lock.details
        assert hostname == lock.hostname
        assert pid == os.getpid()
        assert Path(filename).name == 'test.lck'


def test_expiration(lock):
    with lock:
        expiration = lock.expiration
        time.sleep(1)
        lock.refresh()
        assert lock.expiration > expiration


class FailingOpen:
    def __init__(self, errno=EMOCKEDFAILURE):
        self._errno = errno

    def __call__(self, *args, **kws):
        raise OSError(self._errno, 'test exception')


def test_details_weird_open_failure(lock):
    lock.lock()
    with ExitStack() as resources:
        # Force open() to fail with our unexpected errno.
        resources.enter_context(patch('builtins.open', FailingOpen()))
        # Capture the OSError with the unexpected errno that will occur when
        # .details tries to open the lock file.
        error = resources.enter_context(pytest.raises(OSError))
        lock.details
        assert error.errno == EMOCKEDFAILURE


@contextmanager
def corrupt_open(*args, **kws):
    yield StringIO('bad claim file name')


def test_details_with_corrupt_filename(lock):
    lock.lock()
    with patch('builtins.open', corrupt_open):
        with pytest.raises(NotLockedError, match='Details are unavailable'):
            lock.details


def test_lifetime_property(lock):
    assert lock.lifetime.seconds == 15
    lock.lifetime = timedelta(seconds=31)
    assert lock.lifetime.seconds == 31
    lock.lifetime = 42
    assert lock.lifetime.seconds == 42


def test_refresh(lock):
    with pytest.raises(NotLockedError):
        lock.refresh()
    # With a lifetime parameter, the lock's lifetime is set.
    lock.lock()
    lock.refresh(31)
    assert lock.lifetime.seconds == 31
    # No exception is raised when we try to refresh an unlocked lock
    # unconditionally.
    lock.unlock()
    lock.refresh(unconditionally=True)


def child_locker(filename, queue):
    with suppress(NotLockedError):
        with Lock(filename, lifetime=15):
            queue.put(True)
            time.sleep(3)


def test_lock_with_explicit_timeout(lock):
    queue = Queue()
    Process(target=child_locker, args=(lock.lockfile, queue)).start()
    # Wait for the child process to acquire the lock.
    queue.get()
    with pytest.raises(TimeOutError):
        lock.lock(timeout=1)


def test_lock_with_explicit_timeout_as_timedelta(lock):
    queue = Queue()
    Process(target=child_locker, args=(lock.lockfile, queue)).start()
    # Wait for the child process to acquire the lock.
    queue.get()
    with pytest.raises(TimeOutError):
        lock.lock(timeout=timedelta(seconds=1))


def test_lock_state_with_corrupt_lockfile(lock):
    # Since we're deliberately corrupting the contents of the lock file,
    # unlocking at context manager exit will not work.
    with suppress(NotLockedError):
        with lock:
            with open(lock.lockfile, 'w') as fp:
                fp.write('xxx')
            assert lock.state == LockState.unknown


def test_lock_state_on_other_host(lock):
    # Since we're going to corrupt the lock contents, ignore the exception
    # when we leave the context manager and unlock the lock.
    with suppress(NotLockedError):
        with lock:
            hostname, pid, lockfile = lock.details
            with open(lock.lockfile, 'w') as fp:
                claimfile = SEP.join((
                    lockfile,
                    # Corrupt the hostname to emulate the lock being acquired
                    # on some other host.
                    f'   {hostname}   ',
                    str(pid),
                    str(randint(0, sys.maxsize)),
                    ))
                fp.write(claimfile)
            assert lock.state == LockState.unknown
