/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.GetStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptCache;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptContextInfo;
import org.elasticsearch.script.ScriptEngine;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptLanguagesInfo;
import org.elasticsearch.script.ScriptMetaData;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.ScriptType;
import org.elasticsearch.script.StoredScriptSource;

public class ScriptService
implements Closeable,
ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(ScriptService.class);
    static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    static final Tuple<Integer, TimeValue> USE_CONTEXT_RATE_VALUE = new Tuple((Object)-1, (Object)TimeValue.MINUS_ONE);
    static final String USE_CONTEXT_RATE_KEY = "use-context";
    static final Function<String, Tuple<Integer, TimeValue>> MAX_COMPILATION_RATE_FUNCTION = value -> {
        if (!value.contains("/") || value.startsWith("/") || value.endsWith("/")) {
            throw new IllegalArgumentException("parameter must contain a positive integer and a timevalue, i.e. 10/1m, but was [" + value + "]");
        }
        int idx = value.indexOf("/");
        String count = value.substring(0, idx);
        String time = value.substring(idx + 1);
        try {
            int rate = Integer.parseInt(count);
            if (rate < 0) {
                throw new IllegalArgumentException("rate [" + rate + "] must be positive");
            }
            TimeValue timeValue = TimeValue.parseTimeValue((String)time, (String)"script.max_compilations_rate");
            if (timeValue.nanos() <= 0L) {
                throw new IllegalArgumentException("time value [" + time + "] must be positive");
            }
            if (timeValue.seconds() < 60L) {
                throw new IllegalArgumentException("time value [" + time + "] must be at least on a one minute resolution");
            }
            return Tuple.tuple((Object)rate, (Object)timeValue);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("could not parse [" + count + "] as integer in value [" + value + "]", e);
        }
    };
    public static final Setting<Integer> SCRIPT_GENERAL_CACHE_SIZE_SETTING = Setting.intSetting("script.cache.max_size", 100, 0, Setting.Property.NodeScope);
    public static final Setting<TimeValue> SCRIPT_GENERAL_CACHE_EXPIRE_SETTING = Setting.positiveTimeSetting("script.cache.expire", TimeValue.timeValueMillis((long)0L), Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_SIZE_IN_BYTES = Setting.intSetting("script.max_size_in_bytes", 65535, 0, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Tuple<Integer, TimeValue>> SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING = new Setting<Tuple>("script.max_compilations_rate", "75/5m", value -> value.equals(USE_CONTEXT_RATE_KEY) ? USE_CONTEXT_RATE_VALUE : MAX_COMPILATION_RATE_FUNCTION.apply((String)value), Setting.Property.Dynamic, Setting.Property.NodeScope);
    static final String CONTEXT_PREFIX = "script.context.";
    public static final Setting.AffixSetting<Integer> SCRIPT_CACHE_SIZE_SETTING = Setting.affixKeySetting("script.context.", "cache_max_size", key -> Setting.intSetting(key, SCRIPT_GENERAL_CACHE_SIZE_SETTING, 0, Setting.Property.NodeScope, Setting.Property.Dynamic), new Setting.AffixSettingDependency[0]);
    public static final Setting.AffixSetting<TimeValue> SCRIPT_CACHE_EXPIRE_SETTING = Setting.affixKeySetting("script.context.", "cache_expire", key -> Setting.positiveTimeSetting(key, SCRIPT_GENERAL_CACHE_EXPIRE_SETTING, TimeValue.timeValueMillis((long)0L), Setting.Property.NodeScope, Setting.Property.Dynamic), new Setting.AffixSettingDependency[0]);
    static final String UNLIMITED_COMPILATION_RATE_KEY = "unlimited";
    public static final Setting.AffixSetting<Tuple<Integer, TimeValue>> SCRIPT_MAX_COMPILATIONS_RATE_SETTING = Setting.affixKeySetting("script.context.", "max_compilations_rate", key -> new Setting<Tuple>((String)key, "75/5m", value -> value.equals(UNLIMITED_COMPILATION_RATE_KEY) ? ScriptCache.UNLIMITED_COMPILATION_RATE : MAX_COMPILATION_RATE_FUNCTION.apply((String)value), Setting.Property.NodeScope, Setting.Property.Dynamic), new Setting.AffixSettingDependency[0]);
    private static final Tuple<Integer, TimeValue> SCRIPT_COMPILATION_RATE_ZERO = new Tuple((Object)0, (Object)TimeValue.ZERO);
    public static final String ALLOW_NONE = "none";
    public static final Setting<List<String>> TYPES_ALLOWED_SETTING = Setting.listSetting("script.allowed_types", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> CONTEXTS_ALLOWED_SETTING = Setting.listSetting("script.allowed_contexts", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    private final Set<String> typesAllowed;
    private final Set<String> contextsAllowed;
    private final Map<String, ScriptEngine> engines;
    private final Map<String, ScriptContext<?>> contexts;
    private ClusterState clusterState;
    private int maxSizeInBytes;
    private final AtomicReference<CacheHolder> cacheHolder;

    public ScriptService(Settings settings, Map<String, ScriptEngine> engines, Map<String, ScriptContext<?>> contexts) {
        this.engines = Objects.requireNonNull(engines);
        this.contexts = Objects.requireNonNull(contexts);
        if (Strings.hasLength(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING))) {
            throw new IllegalArgumentException("script.disable_dynamic is not a supported setting, replace with fine-grained script settings. \n Dynamic scripts can be enabled for all languages and all operations not using `script.disable_dynamic: false` in elasticsearch.yml");
        }
        HashSet hashSet = this.typesAllowed = TYPES_ALLOWED_SETTING.exists(settings) ? new HashSet() : null;
        if (this.typesAllowed != null) {
            List<String> typesAllowedList = TYPES_ALLOWED_SETTING.get(settings);
            if (typesAllowedList.isEmpty()) {
                throw new IllegalArgumentException("must specify at least one script type or none for setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
            }
            for (String settingType : typesAllowedList) {
                if (ALLOW_NONE.equals(settingType)) {
                    if (typesAllowedList.size() == 1) break;
                    throw new IllegalArgumentException("cannot specify both [none] and other script types for setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
                }
                boolean found = false;
                for (ScriptType scriptType : ScriptType.values()) {
                    if (!scriptType.getName().equals(settingType)) continue;
                    found = true;
                    this.typesAllowed.add(settingType);
                    break;
                }
                if (found) continue;
                throw new IllegalArgumentException("unknown script type [" + settingType + "] found in setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
            }
        }
        HashSet hashSet2 = this.contextsAllowed = CONTEXTS_ALLOWED_SETTING.exists(settings) ? new HashSet() : null;
        if (this.contextsAllowed != null) {
            List<String> contextsAllowedList = CONTEXTS_ALLOWED_SETTING.get(settings);
            if (contextsAllowedList.isEmpty()) {
                throw new IllegalArgumentException("must specify at least one script context or none for setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
            }
            for (String settingContext : contextsAllowedList) {
                if (ALLOW_NONE.equals(settingContext)) {
                    if (contextsAllowedList.size() == 1) break;
                    throw new IllegalArgumentException("cannot specify both [none] and other script contexts for setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
                }
                if (contexts.containsKey(settingContext)) {
                    this.contextsAllowed.add(settingContext);
                    continue;
                }
                throw new IllegalArgumentException("unknown script context [" + settingContext + "] found in setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
            }
        }
        this.setMaxSizeInBytes(SCRIPT_MAX_SIZE_IN_BYTES.get(settings));
        this.validateCacheSettings(settings);
        this.cacheHolder = new AtomicReference<CacheHolder>(new CacheHolder(settings, contexts.values(), this.compilationLimitsEnabled()));
    }

    boolean compilationLimitsEnabled() {
        return true;
    }

    void registerClusterSettingsListeners(ClusterSettings clusterSettings) {
        clusterSettings.addSettingsUpdateConsumer(SCRIPT_MAX_SIZE_IN_BYTES, this::setMaxSizeInBytes);
        for (ScriptContext<?> context : this.contexts.values()) {
            clusterSettings.addSettingsUpdateConsumer(settings -> this.cacheHolder.get().updateContextSettings((Settings)settings, context), Arrays.asList(SCRIPT_CACHE_SIZE_SETTING.getConcreteSettingForNamespace(context.name), SCRIPT_CACHE_EXPIRE_SETTING.getConcreteSettingForNamespace(context.name), SCRIPT_MAX_COMPILATIONS_RATE_SETTING.getConcreteSettingForNamespace(context.name), SCRIPT_GENERAL_CACHE_EXPIRE_SETTING, SCRIPT_GENERAL_CACHE_SIZE_SETTING));
        }
        clusterSettings.addSettingsUpdateConsumer(settings -> this.cacheHolder.set(this.cacheHolder.get().withUpdatedCacheSettings((Settings)settings)), Arrays.asList(SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING, SCRIPT_GENERAL_CACHE_EXPIRE_SETTING, SCRIPT_GENERAL_CACHE_SIZE_SETTING, SCRIPT_MAX_COMPILATIONS_RATE_SETTING, SCRIPT_CACHE_EXPIRE_SETTING, SCRIPT_CACHE_SIZE_SETTING), this::validateCacheSettings);
    }

    void validateCacheSettings(Settings settings) {
        boolean useContext = SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings).equals(USE_CONTEXT_RATE_VALUE);
        List<Setting.AffixSetting> affixes = Arrays.asList(SCRIPT_MAX_COMPILATIONS_RATE_SETTING, SCRIPT_CACHE_EXPIRE_SETTING, SCRIPT_CACHE_SIZE_SETTING);
        ArrayList<String> keys = new ArrayList<String>();
        for (Setting.AffixSetting affix : affixes) {
            keys.addAll(this.getConcreteSettingKeys(affix, settings));
        }
        if (!useContext && !keys.isEmpty()) {
            throw new IllegalArgumentException("Context cache settings [" + String.join((CharSequence)", ", keys) + "] requires [" + SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.getKey() + "] to be [" + USE_CONTEXT_RATE_KEY + "]");
        }
    }

    List<String> getConcreteSettingKeys(Setting.AffixSetting<?> setting, Settings settings) {
        ArrayList<String> concreteKeys = new ArrayList<String>();
        for (String context : setting.getAsMap(settings).keySet()) {
            String s = setting.getConcreteSettingForNamespace(context).getKey();
            if (!this.contexts.containsKey(context)) {
                throw new IllegalArgumentException("Context [" + context + "] doesn't exist for setting [" + s + "]");
            }
            concreteKeys.add(s);
        }
        concreteKeys.sort(Comparator.naturalOrder());
        return concreteKeys;
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.engines.values());
    }

    private ScriptEngine getEngine(String lang) {
        ScriptEngine scriptEngine = this.engines.get(lang);
        if (scriptEngine == null) {
            throw new IllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        return scriptEngine;
    }

    void setMaxSizeInBytes(int newMaxSizeInBytes) {
        for (Map.Entry<String, StoredScriptSource> source : this.getScriptsFromClusterState().entrySet()) {
            if (source.getValue().getSource().getBytes(StandardCharsets.UTF_8).length <= newMaxSizeInBytes) continue;
            throw new IllegalArgumentException("script.max_size_in_bytes cannot be set to [" + newMaxSizeInBytes + "], stored script [" + source.getKey() + "] exceeds the new value with a size of [" + source.getValue().getSource().getBytes(StandardCharsets.UTF_8).length + "]");
        }
        this.maxSizeInBytes = newMaxSizeInBytes;
    }

    public <FactoryType> FactoryType compile(Script script, ScriptContext<FactoryType> context) {
        Objects.requireNonNull(script);
        Objects.requireNonNull(context);
        ScriptType type = script.getType();
        String lang = script.getLang();
        String idOrCode = script.getIdOrCode();
        Map<String, String> options = script.getOptions();
        String id = idOrCode;
        if (type == ScriptType.STORED) {
            StoredScriptSource source = this.getScriptFromClusterState(id);
            lang = source.getLang();
            idOrCode = source.getSource();
            options = source.getOptions();
        }
        ScriptEngine scriptEngine = this.getEngine(lang);
        if (!this.isTypeEnabled(type)) {
            throw new IllegalArgumentException("cannot execute [" + type + "] scripts");
        }
        if (!this.contexts.containsKey(context.name)) {
            throw new IllegalArgumentException("script context [" + context.name + "] not supported");
        }
        if (!this.isContextEnabled(context)) {
            throw new IllegalArgumentException("cannot execute scripts using [" + context.name + "] context");
        }
        if (type == ScriptType.INLINE && idOrCode.getBytes(StandardCharsets.UTF_8).length > this.maxSizeInBytes) {
            throw new IllegalArgumentException("exceeded max allowed inline script size in bytes [" + this.maxSizeInBytes + "] with size [" + idOrCode.getBytes(StandardCharsets.UTF_8).length + "] for script [" + idOrCode + "]");
        }
        if (logger.isTraceEnabled()) {
            logger.trace("compiling lang: [{}] type: [{}] script: {}", (Object)lang, (Object)type, (Object)idOrCode);
        }
        ScriptCache scriptCache = this.cacheHolder.get().get(context.name);
        assert (scriptCache != null) : "script context [" + context.name + "] has no script cache";
        return scriptCache.compile(context, scriptEngine, id, idOrCode, type, options);
    }

    public boolean isLangSupported(String lang) {
        Objects.requireNonNull(lang);
        return this.engines.containsKey(lang);
    }

    public boolean isTypeEnabled(ScriptType scriptType) {
        return this.typesAllowed == null || this.typesAllowed.contains(scriptType.getName());
    }

    public boolean isContextEnabled(ScriptContext scriptContext) {
        return this.contextsAllowed == null || this.contextsAllowed.contains(scriptContext.name);
    }

    public boolean isAnyContextEnabled() {
        return this.contextsAllowed == null || !this.contextsAllowed.isEmpty();
    }

    Map<String, StoredScriptSource> getScriptsFromClusterState() {
        if (this.clusterState == null) {
            return Collections.emptyMap();
        }
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            return Collections.emptyMap();
        }
        return scriptMetadata.getStoredScripts();
    }

    StoredScriptSource getScriptFromClusterState(String id) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "] in cluster state", new Object[0]);
        }
        StoredScriptSource source = scriptMetadata.getStoredScript(id);
        if (source == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "] in cluster state", new Object[0]);
        }
        return source;
    }

    public void putStoredScript(ClusterService clusterService, final PutStoredScriptRequest request, ActionListener<AcknowledgedResponse> listener) {
        if (request.content().length() > this.maxSizeInBytes) {
            throw new IllegalArgumentException("exceeded max allowed stored script size in bytes [" + this.maxSizeInBytes + "] with size [" + request.content().length() + "] for script [" + request.id() + "]");
        }
        final StoredScriptSource source = request.source();
        if (!this.isLangSupported(source.getLang())) {
            throw new IllegalArgumentException("unable to put stored script with unsupported lang [" + source.getLang() + "]");
        }
        try {
            ScriptEngine scriptEngine = this.getEngine(source.getLang());
            if (!this.isTypeEnabled(ScriptType.STORED)) {
                throw new IllegalArgumentException("cannot put [" + ScriptType.STORED + "] script, [" + ScriptType.STORED + "] scripts are not enabled");
            }
            if (!this.isAnyContextEnabled()) {
                throw new IllegalArgumentException("cannot put [" + ScriptType.STORED + "] script, no script contexts are enabled");
            }
            if (request.context() != null) {
                ScriptContext<?> context = this.contexts.get(request.context());
                if (context == null) {
                    throw new IllegalArgumentException("Unknown context [" + request.context() + "]");
                }
                scriptEngine.compile(request.id(), source.getSource(), context, Collections.emptyMap());
            }
        }
        catch (ScriptException good) {
            throw good;
        }
        catch (Exception exception) {
            throw new IllegalArgumentException("failed to parse/compile stored script [" + request.id() + "]", exception);
        }
        clusterService.submitStateUpdateTask("put-script-" + request.id(), new AckedClusterStateUpdateTask<AcknowledgedResponse>((AckedRequest)request, listener){

            @Override
            protected AcknowledgedResponse newResponse(boolean acknowledged) {
                return new AcknowledgedResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.putStoredScript(smd, request.id(), source);
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public void deleteStoredScript(ClusterService clusterService, final DeleteStoredScriptRequest request, ActionListener<AcknowledgedResponse> listener) {
        clusterService.submitStateUpdateTask("delete-script-" + request.id(), new AckedClusterStateUpdateTask<AcknowledgedResponse>((AckedRequest)request, listener){

            @Override
            protected AcknowledgedResponse newResponse(boolean acknowledged) {
                return new AcknowledgedResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.deleteStoredScript(smd, request.id());
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public StoredScriptSource getStoredScript(ClusterState state, GetStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)state.metaData().custom("stored_scripts");
        if (scriptMetadata != null) {
            return scriptMetadata.getStoredScript(request.id());
        }
        return null;
    }

    public Set<ScriptContextInfo> getContextInfos() {
        HashSet<ScriptContextInfo> infos = new HashSet<ScriptContextInfo>(this.contexts.size());
        for (ScriptContext<?> context : this.contexts.values()) {
            infos.add(new ScriptContextInfo(context.name, context.instanceClazz));
        }
        return infos;
    }

    public ScriptLanguagesInfo getScriptLanguages() {
        Set<String> types = this.typesAllowed;
        if (types == null) {
            types = new HashSet<String>();
            for (ScriptType type : ScriptType.values()) {
                types.add(type.getName());
            }
        }
        Set<String> contexts = this.contextsAllowed != null ? this.contextsAllowed : this.contexts.keySet();
        HashMap<String, Set<String>> languageContexts = new HashMap<String, Set<String>>();
        this.engines.forEach((key, value) -> languageContexts.put((String)key, value.getSupportedContexts().stream().map(c -> c.name).filter(contexts::contains).collect(Collectors.toSet())));
        return new ScriptLanguagesInfo(types, languageContexts);
    }

    public ScriptStats stats() {
        return this.cacheHolder.get().stats();
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        this.clusterState = event.state();
    }

    static class CacheHolder {
        final ScriptCache general;
        final Map<String, AtomicReference<ScriptCache>> contextCache;
        final Set<ScriptContext<?>> contexts;
        final boolean compilationLimitsEnabled;

        CacheHolder(Settings settings, Collection<ScriptContext<?>> contexts, boolean compilationLimitsEnabled) {
            this.compilationLimitsEnabled = compilationLimitsEnabled;
            this.contexts = Collections.unmodifiableSet(new HashSet(contexts));
            if (SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings).equals(USE_CONTEXT_RATE_VALUE)) {
                this.general = null;
                HashMap<String, AtomicReference<ScriptCache>> contextCache = new HashMap<String, AtomicReference<ScriptCache>>(this.contexts.size());
                for (ScriptContext<?> context : this.contexts) {
                    contextCache.put(context.name, new AtomicReference<ScriptCache>(CacheHolder.contextFromSettings(settings, context, this.compilationLimitsEnabled)));
                }
                this.contextCache = Collections.unmodifiableMap(contextCache);
            } else {
                this.contextCache = null;
                this.general = new ScriptCache(SCRIPT_GENERAL_CACHE_SIZE_SETTING.get(settings), SCRIPT_GENERAL_CACHE_EXPIRE_SETTING.get(settings), compilationLimitsEnabled ? SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings) : SCRIPT_COMPILATION_RATE_ZERO);
            }
        }

        private static ScriptCache contextFromSettings(Settings settings, ScriptContext<?> context, boolean compilationLimitsEnabled) {
            String name = context.name;
            Setting<Tuple<Integer, TimeValue>> rateSetting = SCRIPT_MAX_COMPILATIONS_RATE_SETTING.getConcreteSettingForNamespace(name);
            Tuple<Integer, TimeValue> compileRate = !compilationLimitsEnabled ? SCRIPT_COMPILATION_RATE_ZERO : (rateSetting.existsOrFallbackExists(settings) ? rateSetting.get(settings) : context.maxCompilationRateDefault);
            Setting<TimeValue> cacheExpire = SCRIPT_CACHE_EXPIRE_SETTING.getConcreteSettingForNamespace(name);
            Setting<Integer> cacheSize = SCRIPT_CACHE_SIZE_SETTING.getConcreteSettingForNamespace(name);
            return new ScriptCache(cacheSize.existsOrFallbackExists(settings) ? cacheSize.get(settings) : context.cacheSizeDefault, cacheExpire.existsOrFallbackExists(settings) ? cacheExpire.get(settings) : context.cacheExpireDefault, compileRate);
        }

        CacheHolder withUpdatedCacheSettings(Settings settings) {
            if (SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings).equals(USE_CONTEXT_RATE_VALUE)) {
                if (this.general != null) {
                    logger.debug("Switching to context cache from general cache");
                    return new CacheHolder(settings, this.contexts, this.compilationLimitsEnabled);
                }
            } else {
                if (this.general == null) {
                    logger.debug("Switching from context cache to general cache");
                    return new CacheHolder(settings, this.contexts, this.compilationLimitsEnabled);
                }
                if (!this.general.rate.equals(SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings))) {
                    logger.debug("General compilation rate changed from [" + this.general.rate + "] to [" + SCRIPT_GENERAL_MAX_COMPILATIONS_RATE_SETTING.get(settings) + "], creating new general cache");
                    return new CacheHolder(settings, this.contexts, this.compilationLimitsEnabled);
                }
            }
            return this;
        }

        ScriptCache get(String context) {
            if (this.general != null) {
                return this.general;
            }
            AtomicReference<ScriptCache> ref = this.contextCache.get(context);
            if (ref == null) {
                return null;
            }
            return ref.get();
        }

        ScriptStats stats() {
            if (this.general != null) {
                return this.general.stats();
            }
            return ScriptStats.sum(this.contextCache.values().stream().map(AtomicReference::get).map(ScriptCache::stats)::iterator);
        }

        void updateContextSettings(Settings settings, ScriptContext<?> context) {
            if (this.general != null) {
                return;
            }
            AtomicReference<ScriptCache> ref = this.contextCache.get(context.name);
            assert (ref != null) : "expected script cache to exist for context [" + context.name + "]";
            ScriptCache cache = ref.get();
            assert (cache != null) : "expected script cache to be non-null for context [" + context.name + "]";
            ref.set(CacheHolder.contextFromSettings(settings, context, this.compilationLimitsEnabled));
            logger.debug("Replaced context [" + context.name + "] with new settings");
        }
    }
}

