/*
  ==============================================================================

   This file is part of the JUCE library.
   Copyright (c) 2017 - ROLI Ltd.

   JUCE is an open source library subject to commercial or open-source
   licensing.

   By using JUCE, you agree to the terms of both the JUCE 5 End-User License
   Agreement and JUCE 5 Privacy Policy (both updated and effective as of the
   27th April 2017).

   End User License Agreement: www.juce.com/juce-5-licence
   Privacy Policy: www.juce.com/juce-5-privacy-policy

   Or: You may also use this code under the terms of the GPL v3 (see
   www.gnu.org/licenses).

   JUCE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES, WHETHER
   EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR PURPOSE, ARE
   DISCLAIMED.

  ==============================================================================
*/

namespace juce
{
// This byte-code is generated from native/java/com/roli/juce/ComponentPeerView.java with min sdk version 16
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const uint8 javaComponentPeerView[] =
{31,139,8,8,1,253,119,92,0,3,74,97,118,97,68,101,120,66,121,116,101,67,111,100,101,46,100,101,120,0,165,154,11,124,92,85,157,
199,255,231,220,121,36,147,201,100,50,73,147,54,153,180,147,164,73,167,109,154,76,120,53,37,41,244,25,72,155,210,71,166,149,38,
69,184,153,185,73,166,157,220,59,157,153,36,141,226,82,144,109,171,86,165,110,69,150,151,168,149,15,98,87,237,46,32,91,193,237,
242,80,116,81,1,69,235,10,88,93,92,1,31,31,100,203,218,207,234,210,253,157,115,207,36,183,47,81,55,253,124,231,255,63,255,243,
63,239,115,254,231,76,154,164,177,203,23,187,248,82,170,47,249,108,247,151,235,94,184,235,163,155,215,44,254,82,229,169,135,223,
24,25,51,30,92,117,226,206,101,11,136,50,68,180,107,203,37,33,82,63,55,206,39,250,21,217,246,118,16,118,17,45,134,60,1,89,
9,153,247,18,25,144,149,69,68,26,100,87,9,209,161,217,68,119,67,254,176,154,232,56,56,1,78,129,63,128,211,128,207,36,242,128,
98,80,10,154,193,124,176,8,180,131,203,192,53,224,86,240,105,112,20,188,8,222,6,51,102,17,45,3,253,32,7,246,129,123,193,87,193,
247,193,59,160,162,6,117,1,3,236,1,143,128,151,65,113,45,81,7,216,13,190,6,94,1,60,76,212,0,86,131,235,65,22,220,14,30,3,207,
131,147,160,168,142,104,30,232,4,219,193,30,176,31,220,9,238,1,159,1,15,128,195,224,33,112,20,28,3,79,131,111,129,239,130,239,
131,227,224,21,240,115,240,6,248,13,120,11,156,2,255,11,24,230,205,3,124,32,0,26,192,98,176,18,172,7,3,96,24,228,192,110,176,
15,124,18,124,26,28,5,143,131,231,192,113,112,2,188,6,126,7,78,1,247,28,162,16,104,0,237,224,114,176,28,92,5,214,128,141,224,
122,176,29,220,12,110,3,119,130,251,193,67,224,95,192,15,192,79,193,155,224,29,224,141,16,5,65,29,104,3,87,130,56,72,128,28,56,
0,62,5,238,1,247,129,35,224,73,240,34,120,21,252,14,120,235,137,102,130,249,160,3,172,6,235,193,8,200,130,93,224,253,96,47,
216,15,238,1,135,193,227,224,25,240,67,240,10,120,3,156,4,167,65,81,3,81,61,136,129,197,160,27,244,130,56,216,6,174,7,6,48,65,
30,188,15,124,20,124,22,60,4,190,9,94,5,191,7,190,70,162,70,176,16,92,14,214,128,205,32,9,70,193,4,216,3,14,130,187,193,231,
193,151,193,35,224,9,240,45,240,44,120,19,4,230,98,255,130,26,176,0,116,128,141,64,7,22,248,16,184,15,60,14,190,13,126,12,126,
1,222,6,90,19,214,15,52,128,118,176,22,108,4,91,129,14,134,65,26,100,193,141,224,38,176,7,124,28,124,18,220,15,142,128,39,193,
247,192,11,224,69,240,99,240,114,147,125,118,221,0,199,153,2,160,12,4,65,57,16,193,160,130,236,243,62,3,84,1,28,99,194,113,36,
28,55,194,209,162,90,17,27,0,182,49,97,187,17,182,7,97,121,9,203,65,152,70,194,240,9,195,32,52,69,205,96,30,136,2,132,23,66,
216,161,133,160,5,44,2,173,160,13,196,84,204,185,8,92,12,46,1,151,130,14,112,21,88,7,174,1,235,193,6,178,199,80,248,113,41,121,
107,149,61,46,166,210,30,165,11,187,24,43,87,246,34,165,239,135,221,167,252,196,143,95,229,29,84,246,128,178,251,212,188,20,
244,25,14,127,49,47,119,87,217,186,152,147,67,170,108,189,242,41,81,115,33,250,80,170,230,227,176,242,23,250,67,202,63,170,252,
75,213,28,61,86,101,143,125,161,178,47,81,186,168,231,114,165,63,5,189,83,233,207,65,239,82,250,113,232,75,149,254,26,244,43,
148,126,18,250,149,133,58,177,168,203,148,94,4,125,133,210,131,14,251,76,135,30,129,190,74,233,81,135,253,18,135,222,229,208,
87,57,234,236,117,216,227,208,87,42,125,155,195,158,175,154,214,197,28,46,87,122,210,81,79,218,225,47,230,109,117,161,44,236,
221,74,191,209,225,179,223,161,31,172,182,247,81,139,154,207,171,149,126,55,236,61,74,63,4,125,141,210,191,2,189,87,233,143,66,
95,171,244,99,14,251,51,14,251,115,213,98,77,25,93,71,182,252,137,156,226,8,93,171,228,199,165,100,244,9,37,15,42,249,73,37,
111,87,242,83,202,255,62,18,103,173,133,62,45,101,45,253,152,196,185,11,211,113,37,127,38,101,13,189,74,98,223,113,186,77,202,
90,186,87,202,22,250,154,148,37,244,117,185,255,234,233,50,18,103,163,142,158,32,177,223,61,52,42,101,144,222,7,89,140,19,207,
164,92,68,79,147,216,143,245,50,93,162,236,37,56,61,223,144,227,178,211,101,104,47,45,101,57,153,42,253,126,177,119,148,61,136,
19,185,83,165,199,164,212,104,156,236,24,51,161,228,46,41,25,237,86,242,102,18,103,140,83,86,202,6,186,137,68,28,154,45,219,
171,64,84,120,88,202,118,122,68,202,25,244,207,36,206,97,51,245,43,249,36,137,88,213,36,253,103,33,130,124,76,202,70,250,150,
148,151,210,247,229,60,206,147,249,53,168,225,33,57,127,243,101,186,22,246,141,74,110,146,210,79,125,82,86,208,15,148,124,81,
206,231,28,233,31,198,76,198,165,140,210,102,41,43,104,155,146,186,148,49,249,78,10,163,165,29,82,122,228,155,42,172,198,23,70,
212,203,73,89,76,147,82,250,232,70,149,255,1,41,139,232,111,164,180,231,33,140,145,125,80,201,91,149,252,91,41,103,210,30,37,
247,42,251,62,41,171,233,67,74,126,88,201,143,72,57,139,246,75,217,70,119,41,121,183,148,97,122,80,237,159,47,42,121,88,201,127,
80,249,95,82,233,47,43,249,21,181,191,142,72,25,161,127,148,178,137,254,73,74,123,189,194,106,189,68,250,171,106,95,62,42,
165,189,126,97,68,249,163,82,46,164,111,42,249,140,148,115,233,223,164,172,164,239,72,185,128,190,171,210,223,83,126,207,41,249,
188,202,127,65,141,255,231,36,98,111,136,242,82,94,70,119,202,253,31,160,19,36,226,173,189,127,155,17,101,45,18,49,87,163,
65,41,57,125,142,68,220,45,163,191,39,251,142,35,178,227,179,136,75,226,174,88,5,121,64,5,233,90,181,135,155,149,159,200,143,
35,255,144,202,23,247,64,57,217,247,232,141,170,124,18,242,217,232,180,255,117,208,147,81,251,30,51,33,199,192,77,81,251,190,
218,39,237,92,234,39,154,196,185,196,123,48,84,66,153,224,13,176,133,104,91,208,39,239,166,34,164,68,249,215,225,51,32,218,243,
84,82,3,247,195,175,8,165,227,185,18,138,95,236,39,51,120,17,188,252,44,68,75,88,215,180,53,114,25,78,199,180,79,12,62,62,109,
19,174,170,77,77,140,142,196,15,144,230,186,244,134,26,218,156,243,81,135,86,69,33,109,27,180,130,127,60,95,66,59,131,77,56,231,
126,215,206,224,124,200,18,215,206,216,2,90,233,22,254,75,52,47,45,190,193,77,161,10,81,38,128,182,102,96,45,204,96,37,250,
123,118,27,209,239,138,251,91,147,115,18,104,182,223,34,3,33,191,26,107,3,214,34,19,137,96,182,7,130,1,26,168,40,149,227,102,
242,31,222,30,205,246,221,159,9,138,149,8,76,217,155,167,236,243,164,157,171,23,65,107,179,61,223,241,96,169,92,31,13,118,209,
238,101,205,246,125,28,143,148,162,151,226,245,129,17,214,139,126,215,160,229,192,148,223,210,11,250,213,74,191,0,226,130,24,
201,42,248,137,125,31,242,132,102,100,34,85,244,5,106,112,137,213,243,160,87,3,7,17,221,92,181,148,193,131,245,96,241,192,193,
32,190,101,185,81,195,123,233,16,249,92,29,174,235,104,142,79,164,241,34,167,254,187,203,160,245,83,138,182,221,133,72,27,233,
163,173,248,140,203,207,205,216,169,13,216,25,153,160,24,81,133,171,146,26,151,47,162,102,38,226,152,25,113,161,63,113,209,146,
219,135,249,115,147,168,209,139,152,21,210,204,160,142,82,126,87,141,251,219,20,106,110,28,106,163,144,119,176,168,130,214,21,
121,188,161,234,112,81,137,212,204,216,117,20,247,248,181,14,173,146,66,60,52,183,113,85,140,66,238,157,193,235,49,58,191,103,
157,199,229,14,205,8,73,105,198,146,244,119,46,204,65,16,107,139,122,231,184,236,222,31,164,232,103,74,93,209,147,224,45,240,
38,248,119,112,28,75,46,223,150,211,63,187,175,164,191,40,125,246,143,157,47,238,237,42,196,35,131,236,251,132,107,141,183,178,
134,91,216,252,221,140,207,219,199,154,246,48,123,189,197,207,131,106,31,244,69,202,228,57,22,251,79,164,143,20,246,77,172,139,
234,53,177,154,246,249,58,170,246,101,223,178,50,170,226,242,157,202,151,32,39,32,45,153,216,18,114,177,232,255,76,215,255,
68,179,253,14,21,111,231,126,180,17,80,109,136,159,103,154,237,243,31,10,150,77,237,203,23,84,127,54,7,203,101,59,92,237,227,
31,53,219,111,67,51,120,137,56,57,120,3,186,100,185,16,69,223,209,84,125,47,55,219,49,35,32,203,217,47,220,95,56,108,30,216,132,
231,175,223,101,12,139,213,24,10,254,111,191,139,127,199,212,152,237,126,188,115,158,126,120,230,157,107,43,117,216,92,170,173,
202,121,246,123,57,196,68,228,29,136,113,178,165,70,91,219,17,69,222,220,26,243,210,214,152,71,89,139,177,127,240,98,15,110,
141,185,144,95,132,83,80,133,254,224,14,98,229,106,188,98,238,27,231,217,49,248,252,253,143,175,8,81,102,211,50,114,109,140,158,
18,223,13,52,185,238,11,47,88,230,226,213,167,79,203,113,47,107,37,151,46,202,20,195,46,218,233,152,103,127,119,9,5,251,6,
81,10,213,180,51,238,90,194,176,82,152,191,22,175,248,238,36,114,204,96,9,226,181,143,101,54,94,78,205,185,232,127,89,177,106,
26,47,242,81,244,183,102,112,33,122,29,125,67,76,146,221,143,194,119,45,241,157,163,84,225,67,142,104,111,211,60,251,123,83,
136,77,181,199,209,30,74,150,177,37,222,98,180,83,140,190,249,120,168,241,162,139,188,100,45,155,73,227,119,248,88,244,45,19,
119,6,206,61,239,224,191,57,93,208,205,200,32,218,240,201,254,101,54,160,95,90,168,60,250,178,61,31,162,173,244,60,251,187,206,
153,243,97,143,76,216,196,152,60,104,43,179,169,19,51,25,10,70,223,182,223,150,226,103,210,177,206,197,228,150,245,237,85,243,
107,93,59,139,226,227,206,90,59,208,166,189,175,182,97,95,217,223,73,185,252,46,118,7,202,132,81,81,223,243,240,214,196,236,
50,77,206,46,43,99,102,208,143,250,125,197,102,80,220,11,37,222,23,15,156,166,133,236,191,229,236,68,127,215,247,2,78,10,51,113,
254,102,35,178,154,65,17,161,125,110,19,103,12,210,227,125,126,188,104,169,184,77,203,163,47,189,187,231,149,182,231,55,223,
221,243,10,120,154,65,113,23,248,74,66,229,139,107,23,80,168,190,49,220,138,152,216,142,183,76,168,230,210,163,245,36,106,17,
117,124,73,212,17,9,9,201,66,238,85,110,191,251,150,205,223,152,97,70,42,108,75,213,42,143,223,115,203,208,55,170,10,245,30,247,
97,37,79,160,31,31,172,100,116,154,232,225,3,63,243,177,162,37,190,153,244,231,142,96,62,13,157,118,246,231,175,233,137,168,41,
250,147,191,188,7,75,85,15,154,254,223,61,88,42,123,128,101,102,94,185,215,196,222,18,177,161,77,73,17,83,196,251,62,39,247,
30,151,119,114,105,212,254,93,3,250,136,27,208,139,221,95,163,61,75,161,202,198,85,184,1,221,131,30,220,128,242,54,187,142,46,
114,249,217,18,212,28,192,185,139,190,3,254,24,10,53,54,224,254,211,196,253,87,132,59,110,157,139,107,226,222,219,201,163,191,
45,229,209,95,131,55,192,107,98,191,151,163,111,226,125,232,23,145,136,207,171,106,154,169,53,70,27,90,230,47,112,196,192,197,
209,233,179,161,41,235,149,81,251,157,215,193,189,56,95,102,228,90,120,4,40,180,44,250,7,49,62,59,78,94,29,181,127,55,98,6,219,
228,139,174,134,223,1,79,142,216,233,199,55,147,89,212,129,200,97,6,19,56,161,62,182,148,85,96,182,205,216,28,138,97,196,151,
74,127,51,54,155,130,174,120,123,16,111,183,70,89,127,33,39,196,205,88,29,5,185,157,55,87,204,239,91,246,239,117,157,63,15,156,
149,126,244,172,180,24,83,37,217,247,91,57,122,193,148,205,254,206,100,203,34,37,171,149,172,85,254,205,136,162,34,29,85,233,
40,86,211,69,133,183,158,125,31,114,181,214,133,123,147,171,181,46,220,151,182,238,149,191,115,225,146,22,135,93,200,82,153,118,
169,182,61,42,207,3,63,174,108,94,37,139,149,244,171,178,1,244,200,94,95,117,127,171,50,97,181,126,226,59,146,200,111,81,125,
108,113,244,91,176,72,201,86,85,158,169,183,133,144,101,83,182,50,213,150,93,46,56,213,150,122,215,138,190,118,165,204,84,254,
10,170,90,105,141,102,44,211,48,243,27,12,35,187,37,101,76,180,110,215,199,117,98,221,196,187,123,136,245,16,239,129,88,67,124,
77,47,85,175,53,38,7,45,61,155,92,149,202,141,166,114,185,222,84,46,111,152,70,150,88,47,241,94,184,246,246,146,214,139,143,
170,94,221,76,102,173,84,178,77,207,100,218,150,39,242,169,241,84,126,178,147,46,57,211,158,201,164,83,9,61,159,178,204,185,5,
159,222,212,144,145,152,76,164,141,149,122,58,61,168,39,118,228,58,105,214,133,74,57,179,18,150,137,190,228,219,86,10,185,43,
239,204,26,206,234,153,145,84,34,215,182,82,55,199,117,84,56,251,60,89,86,218,202,118,167,210,121,35,123,225,252,117,122,62,155,
218,213,73,243,255,100,254,25,85,205,60,215,117,131,158,50,209,191,234,115,115,54,25,9,100,84,76,101,88,185,182,21,99,102,
50,109,116,82,165,211,216,179,34,101,38,69,237,211,117,140,99,229,218,176,60,171,199,13,81,121,205,153,25,235,44,49,93,42,111,
254,153,121,98,205,231,174,55,187,173,196,88,110,229,136,110,14,27,133,101,117,118,101,202,213,57,164,41,227,85,89,107,44,211,
73,151,157,155,19,207,26,198,250,193,156,145,29,55,178,104,229,170,180,53,168,167,123,245,73,107,44,63,221,204,156,63,93,174,
147,90,207,116,72,153,153,177,252,168,145,31,177,146,109,43,244,156,209,35,210,88,120,19,243,39,183,69,211,133,253,87,39,83,121,
43,219,99,14,89,157,180,224,194,110,231,84,185,232,93,124,215,73,125,157,110,234,195,162,199,221,189,9,107,180,45,107,165,83,
109,219,199,18,70,219,57,199,108,238,5,206,210,220,51,71,190,228,175,173,167,147,234,223,173,104,39,53,244,38,245,244,120,106,
71,155,110,154,86,94,158,169,182,213,102,34,109,229,82,230,240,202,180,158,147,135,229,92,159,30,204,75,86,229,215,159,39,127,
157,49,58,168,28,140,156,216,49,34,166,180,165,177,185,218,176,197,178,125,198,206,49,195,76,96,91,151,59,115,236,250,26,28,
166,158,116,218,24,214,211,203,19,9,35,151,91,189,43,97,100,236,197,152,123,30,159,236,240,216,40,6,231,240,170,112,122,33,42,
12,219,179,50,109,188,198,234,27,75,140,216,43,231,40,23,114,184,172,31,220,46,15,101,157,195,214,103,36,198,178,136,85,23,40,
210,135,32,96,14,139,29,51,109,203,26,67,105,212,131,110,140,91,118,236,138,235,217,97,195,217,219,154,243,184,219,93,195,171,
56,190,117,195,106,242,59,183,6,177,45,196,183,244,144,123,75,15,126,160,174,33,207,150,53,61,221,221,107,200,5,217,35,62,69,
56,222,178,166,31,153,66,233,93,35,62,164,214,143,220,222,126,4,245,45,253,40,213,47,107,96,253,164,245,139,114,248,232,21,42,
98,122,191,80,68,96,31,192,117,48,208,67,161,129,115,215,171,98,224,60,211,229,211,229,146,205,141,197,98,83,122,187,67,191,
200,161,95,236,208,47,129,94,98,235,221,105,125,56,71,30,93,158,63,97,20,178,87,31,52,210,84,164,171,219,130,102,233,201,228,
249,163,10,177,65,42,23,23,200,138,177,124,222,50,55,100,81,165,145,36,207,160,133,228,40,164,12,172,228,73,200,59,129,188,9,
25,250,146,228,198,221,163,103,169,36,97,37,141,13,22,34,245,242,188,72,76,69,119,10,200,68,60,171,155,185,33,43,59,74,165,226,
230,65,72,207,73,111,84,100,95,64,168,200,26,67,218,149,76,13,13,17,51,200,109,136,240,75,254,161,233,64,155,164,98,236,129,
229,246,8,103,8,117,250,110,83,247,24,149,194,44,206,12,134,108,100,115,84,36,146,98,226,201,39,52,229,228,23,91,73,84,31,79,
141,26,178,210,171,141,212,240,72,158,42,160,246,170,29,183,222,236,75,96,3,153,50,223,222,89,84,6,85,246,27,167,85,118,215,63,
109,232,73,146,23,169,77,250,196,181,5,101,43,149,8,197,178,242,34,126,80,16,137,190,73,204,247,104,31,246,100,42,97,80,0,150,
205,102,74,76,158,232,180,108,255,236,112,46,135,180,37,149,75,13,166,210,98,13,69,153,247,224,66,179,38,226,214,14,116,47,60,
149,150,78,105,3,145,45,147,214,39,187,179,58,6,231,66,238,181,242,115,43,177,17,42,195,68,98,29,49,113,27,244,49,177,192,193,
41,195,38,35,135,136,48,101,89,49,181,23,168,212,182,32,114,174,178,38,176,183,166,146,155,51,84,57,149,144,81,245,234,84,50,
137,62,169,102,214,89,104,67,150,57,195,144,213,135,11,117,74,3,170,81,117,202,187,158,138,70,244,156,188,95,169,122,36,149,52,
250,172,161,188,188,51,186,179,214,168,61,82,184,192,49,46,214,210,53,98,229,242,196,82,228,195,90,174,151,193,33,71,90,106,
116,148,202,196,147,45,165,167,87,234,153,220,58,204,48,149,42,67,159,145,94,109,38,167,242,145,236,203,235,217,60,21,203,91,
42,62,153,49,200,47,213,235,237,27,139,60,41,196,161,29,6,154,200,245,152,185,188,142,88,76,69,169,220,250,140,142,192,140,98,
57,53,243,228,221,97,76,174,20,77,85,239,184,192,211,175,180,144,209,55,34,230,197,157,150,39,180,20,235,101,100,69,203,215,224,
202,33,87,218,24,202,147,39,109,152,195,249,17,242,168,94,48,147,92,166,88,83,175,105,76,92,35,23,215,74,39,71,228,231,4,149,
91,102,225,81,184,50,107,232,121,172,100,197,180,105,149,145,203,103,173,73,177,188,211,70,181,5,28,37,11,123,160,118,218,212
,167,143,27,133,65,99,154,242,134,211,95,206,219,153,85,244,229,173,76,6,166,106,156,73,217,143,179,158,6,228,177,76,236,128,
9,42,181,156,143,40,10,88,103,196,37,42,182,204,194,134,43,149,234,186,177,116,62,149,17,147,44,147,216,52,69,34,192,73,103,120,
244,165,222,103,20,194,132,31,247,133,133,171,74,158,113,52,104,175,147,215,150,248,50,137,34,203,243,121,132,7,119,70,110,
56,95,70,207,194,83,30,82,119,70,198,37,150,165,134,172,49,44,150,45,123,225,199,54,133,179,198,168,53,110,216,61,95,111,158,
21,83,221,89,25,85,180,156,145,167,64,78,196,159,169,167,46,249,145,150,227,215,197,198,169,118,166,122,236,222,203,61,43,138,
57,30,76,178,88,111,97,179,208,76,164,206,251,22,165,25,185,66,172,217,156,114,4,143,218,243,154,197,83,71,71,220,206,217,209,
71,238,173,210,220,25,81,199,87,72,166,237,62,189,39,149,78,95,99,229,229,74,250,115,216,203,133,8,128,130,72,77,157,89,56,139,
61,98,247,11,247,60,178,177,65,166,147,51,115,118,111,122,166,219,82,35,117,229,71,82,184,208,196,231,220,152,146,237,176,138,
144,173,161,14,168,98,10,138,198,242,67,29,50,120,178,113,114,143,235,105,177,210,82,172,31,34,151,120,124,82,153,248,116,238,
143,98,97,136,91,155,115,6,5,199,207,14,183,190,241,233,81,179,9,98,187,136,239,138,129,118,98,147,116,132,51,242,6,250,187,232,
81,124,227,109,25,208,248,19,172,116,175,198,142,177,138,57,26,165,120,235,174,223,111,239,98,229,229,169,46,158,171,235,162,
251,56,167,87,153,55,192,47,223,202,175,154,88,68,247,112,246,58,146,231,200,251,81,89,224,33,250,162,45,102,107,195,143,179,
3,204,219,194,159,167,78,254,91,54,193,159,252,192,196,62,198,221,190,229,139,186,90,187,186,174,24,208,40,233,187,81,99,70,107,
215,253,245,154,246,121,182,144,85,207,136,205,214,248,103,25,103,229,213,110,206,55,214,185,201,205,220,154,199,199,23,30,114,
251,60,228,97,30,238,209,22,44,224,227,45,110,190,128,231,90,104,150,221,244,44,254,58,123,67,40,39,196,192,62,28,38,92,94,
129,181,244,136,198,95,99,191,18,246,237,72,211,97,77,124,190,164,241,254,235,33,127,168,177,126,136,99,82,160,200,47,165,242,
145,48,61,167,28,126,110,59,124,219,22,255,193,217,215,153,183,110,237,218,69,253,107,251,91,73,103,55,200,82,183,104,252,167,
236,99,104,226,35,181,139,232,117,198,239,101,59,188,117,123,121,168,142,103,234,120,89,39,31,59,180,141,79,174,161,253,92,187,
151,141,32,139,151,238,225,227,117,183,15,108,223,171,209,43,140,247,211,45,92,150,9,236,141,132,111,167,63,112,183,254,57,
246,18,123,154,189,23,77,190,197,93,191,100,31,100,15,178,47,160,254,165,123,251,105,82,58,242,103,168,142,31,189,169,110,173,
86,124,45,239,213,188,63,97,26,127,146,182,242,101,45,172,162,44,102,19,236,213,74,238,103,124,105,151,230,255,87,214,182,148,
49,205,247,9,198,23,177,112,233,149,110,159,219,223,238,46,217,238,241,181,178,138,42,254,254,206,46,143,127,41,171,155,33,236,
103,26,121,55,171,11,208,1,141,253,39,218,143,104,236,71,88,162,80,21,247,183,240,209,58,108,148,237,209,217,110,18,178,185,
209,77,63,139,45,160,143,106,236,1,49,217,71,52,182,205,27,72,133,233,59,140,189,12,195,83,26,29,100,53,173,219,215,238,186,97,
214,94,226,117,236,167,108,102,45,159,205,183,184,248,253,172,122,177,109,8,75,67,8,134,98,62,7,134,48,155,89,83,80,106,137,
51,230,227,236,182,72,100,247,110,215,177,234,122,246,106,53,105,30,242,223,22,193,94,225,13,55,239,118,29,158,197,246,69,94,
18,31,39,197,199,254,26,198,15,129,103,107,200,85,54,171,140,51,249,47,10,199,147,53,200,62,56,27,31,135,103,187,110,225,84,12,
216,159,65,24,44,100,79,205,102,236,36,184,109,78,37,59,52,135,177,199,192,9,112,18,236,139,48,246,40,120,13,252,17,220,86,207,
216,33,112,10,28,110,128,31,120,174,17,101,230,50,215,41,176,191,137,177,99,77,46,182,111,1,99,7,23,114,246,0,120,118,97,208,
241,251,176,130,44,252,93,165,248,125,78,225,111,43,197,239,127,10,127,95,233,162,233,191,177,20,191,63,42,252,157,101,225,119,
84,226,111,45,181,160,173,139,223,203,177,136,253,119,69,199,160,123,34,182,93,252,31,53,11,218,191,67,147,255,111,29,177,219,
21,127,155,169,41,127,241,255,201,174,136,93,175,248,63,104,82,101,229,255,109,7,237,190,138,191,3,253,63,65,69,100,73,64,42,0,0,0,0};

//==============================================================================
#if JUCE_PUSH_NOTIFICATIONS && JUCE_MODULE_AVAILABLE_juce_gui_extra
 // Returns true if the intent was handled.
 extern bool juce_handleNotificationIntent (void*);
 extern void juce_firebaseDeviceNotificationsTokenRefreshed (void*);
 extern void juce_firebaseRemoteNotificationReceived (void*);
 extern void juce_firebaseRemoteMessagesDeleted();
 extern void juce_firebaseRemoteMessageSent(void*);
 extern void juce_firebaseRemoteMessageSendError (void*, void*);
#endif

extern void juce_contentSharingCompleted (int);

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create,                      "<init>",                      "(II)V")

DECLARE_JNI_CLASS (AndroidLayoutParams, "android/view/ViewGroup$LayoutParams")
#undef JNI_CLASS_MEMBERS

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (addView,       "addView",             "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V") \
 METHOD (removeView, "removeView", "(Landroid/view/View;)V") \
 METHOD (updateViewLayout, "updateViewLayout", "(Landroid/view/View;Landroid/view/ViewGroup$LayoutParams;)V")

DECLARE_JNI_CLASS (AndroidViewManager, "android/view/ViewManager")
#undef JNI_CLASS_MEMBERS

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create,           "<init>",             "(IIIIIII)V") \
 FIELD  (gravity,          "gravity",            "I") \
 FIELD  (windowAnimations, "windowAnimations",   "I")

DECLARE_JNI_CLASS (AndroidWindowManagerLayoutParams, "android/view/WindowManager$LayoutParams")
#undef JNI_CLASS_MEMBERS

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (getDecorView, "getDecorView",       "()Landroid/view/View;") \
 METHOD (setFlags,     "setFlags",           "(II)V") \
 METHOD (clearFlags,   "clearFlags",         "(I)V")

DECLARE_JNI_CLASS (AndroidWindow, "android/view/Window")
#undef JNI_CLASS_MEMBERS

namespace
{
    enum
    {
        SYSTEM_UI_FLAG_VISIBLE = 0,
        SYSTEM_UI_FLAG_LOW_PROFILE = 1,
        SYSTEM_UI_FLAG_HIDE_NAVIGATION = 2,
        SYSTEM_UI_FLAG_FULLSCREEN = 4,
        SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION = 512,
        SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN = 1024,
        SYSTEM_UI_FLAG_IMMERSIVE = 2048,
        SYSTEM_UI_FLAG_IMMERSIVE_STICKY = 4096
    };

    constexpr int fullScreenFlags = SYSTEM_UI_FLAG_HIDE_NAVIGATION | SYSTEM_UI_FLAG_FULLSCREEN | SYSTEM_UI_FLAG_IMMERSIVE_STICKY;

    constexpr int FLAG_NOT_FOCUSABLE = 0x8;
}

//==============================================================================
class AndroidComponentPeer  : public ComponentPeer,
                              private Timer
{
public:
    AndroidComponentPeer (Component& comp, int windowStyleFlags, void* nativeViewHandle)
        : ComponentPeer (comp, windowStyleFlags),
          fullScreen (false),
          navBarsHidden (false),
          sizeAllocated (0),
          scale ((float) Desktop::getInstance().getDisplays().getMainDisplay().scale)
    {
        auto* env = getEnv();

        // NB: must not put this in the initialiser list, as it invokes a callback,
        // which will fail if the peer is only half-constructed.
        view = GlobalRef (LocalRef<jobject> (env->NewObject (ComponentPeerView, ComponentPeerView.create,
                                                             getAppContext().get(), (jboolean) component.isOpaque(),
                                                             (jlong) this)));

        if (nativeViewHandle != nullptr)
        {
            viewGroupIsWindow = false;

            // we don't know if the user is holding on to a local ref to this, so
            // explicitly create a new one
            auto nativeView = LocalRef<jobject>(env->NewLocalRef(static_cast<jobject> (nativeViewHandle)));

            if (env->IsInstanceOf (nativeView.get(), AndroidActivity))
            {
                viewGroup = GlobalRef (nativeView);
                env->CallVoidMethod (viewGroup.get(), AndroidActivity.setContentView, view.get());
            }
            else if (env->IsInstanceOf (nativeView.get(), AndroidViewGroup))
            {
                viewGroup = GlobalRef (nativeView);
                LocalRef<jobject> layoutParams (env->NewObject (AndroidLayoutParams, AndroidLayoutParams.create, -2, -2));

                env->CallVoidMethod (view.get(), AndroidView.setLayoutParams, layoutParams.get());
                env->CallVoidMethod ((jobject) viewGroup.get(), AndroidViewGroup.addView, view.get());
            }
            else
            {
                // the native handle you passed as a second argument to Component::addToDesktop must
                // either be an Activity or a ViewGroup
                jassertfalse;
            }
        }
        else
        {
            viewGroupIsWindow = true;

            LocalRef<jobject> viewLayoutParams (env->NewObject (AndroidLayoutParams, AndroidLayoutParams.create, -2, -2));

            env->CallVoidMethod (view.get(), AndroidView.setLayoutParams, viewLayoutParams.get());

            Rectangle<int> physicalBounds = comp.getBoundsInParent() * scale;

            view.callVoidMethod (AndroidView.layout,
                                 physicalBounds.getX(), physicalBounds.getY(), physicalBounds.getRight(), physicalBounds.getBottom());

            LocalRef<jobject> windowLayoutParams (env->NewObject (AndroidWindowManagerLayoutParams, AndroidWindowManagerLayoutParams.create,
                                                                  physicalBounds.getWidth(), physicalBounds.getHeight(),
                                                                  physicalBounds.getX(), physicalBounds.getY(),
                                                                  TYPE_APPLICATION, FLAG_NOT_TOUCH_MODAL | FLAG_LAYOUT_IN_SCREEN | FLAG_LAYOUT_NO_LIMITS | FLAG_NOT_FOCUSABLE,
                                                                  component.isOpaque() ? PIXEL_FORMAT_OPAQUE : PIXEL_FORMAT_TRANSPARENT));
            env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.gravity, GRAVITY_LEFT | GRAVITY_TOP);
            env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.windowAnimations, 0x01030000 /* android.R.style.Animation */);

            if (Desktop::getInstance().getKioskModeComponent() != nullptr)
                setNavBarsHidden (true);

            LocalRef<jobject> activity (getCurrentActivity());

            if (activity == nullptr)
                activity = getMainActivity();

            viewGroup = GlobalRef (LocalRef<jobject> (env->CallObjectMethod (activity.get(), AndroidContext.getSystemService, javaString ("window").get())));
            env->CallVoidMethod (viewGroup.get(), AndroidViewManager.addView, view.get(), windowLayoutParams.get());
        }

        if (isFocused())
            handleFocusGain();
    }

    ~AndroidComponentPeer() override
    {
        auto* env = getEnv();

        env->CallVoidMethod (view, ComponentPeerView.clear);
        frontWindow = nullptr;

        if (MessageManager::getInstance()->isThisTheMessageThread())
        {
            if (env->IsInstanceOf (viewGroup.get(), AndroidActivity))
                env->CallVoidMethod (viewGroup.get(), AndroidActivity.setContentView, nullptr);
            else
                env->CallVoidMethod (viewGroup.get(), AndroidViewManager.removeView, view.get());
        }
        else
        {
            struct ViewDeleter  : public CallbackMessage
            {
                ViewDeleter (const GlobalRef& view_, const GlobalRef& viewGroup_) : view (view_), group (viewGroup_) {}

                void messageCallback() override
                {
                    auto* callbackEnv = getEnv();

                    if (callbackEnv->IsInstanceOf (group.get(), AndroidActivity))
                        callbackEnv->CallVoidMethod (group.get(), AndroidActivity.setContentView, nullptr);
                    else
                        callbackEnv->CallVoidMethod (group.get(), AndroidViewManager.removeView, view.get());
                }

            private:
                GlobalRef view, group;
            };

            (new ViewDeleter (view, viewGroup))->post();
        }
    }

    void* getNativeHandle() const override
    {
        return (void*) view.get();
    }

    void setVisible (bool shouldBeVisible) override
    {
        if (MessageManager::getInstance()->isThisTheMessageThread())
        {
            view.callVoidMethod (ComponentPeerView.setVisible, shouldBeVisible);
        }
        else
        {
            struct VisibilityChanger  : public CallbackMessage
            {
                VisibilityChanger (const GlobalRef& view_, bool shouldBeVisible_)
                    : view (view_), shouldBeVisible (shouldBeVisible_)
                {}

                void messageCallback() override
                {
                    view.callVoidMethod (ComponentPeerView.setVisible, shouldBeVisible);
                }

                GlobalRef view;
                bool shouldBeVisible;
            };

            (new VisibilityChanger (view, shouldBeVisible))->post();
        }
    }

    void setTitle (const String& title) override
    {
        view.callVoidMethod (ComponentPeerView.setViewName, javaString (title).get());
    }

    void setBounds (const Rectangle<int>& userRect, bool isNowFullScreen) override
    {
        Rectangle<int> r = (userRect.toFloat() * scale).toNearestInt();

        if (MessageManager::getInstance()->isThisTheMessageThread())
        {
            auto* env = getEnv();

            fullScreen = isNowFullScreen;

            {
                view.callVoidMethod (AndroidView.layout,
                                     r.getX(), r.getY(), r.getRight(), r.getBottom());

                if (viewGroup != nullptr && viewGroupIsWindow)
                {
                    LocalRef<jobject> windowLayoutParams (env->NewObject (AndroidWindowManagerLayoutParams, AndroidWindowManagerLayoutParams.create,
                                                                          r.getWidth(), r.getHeight(),
                                                                          r.getX(), r.getY(),
                                                                          TYPE_APPLICATION, FLAG_NOT_TOUCH_MODAL | FLAG_LAYOUT_IN_SCREEN | FLAG_LAYOUT_NO_LIMITS,
                                                                          component.isOpaque() ? PIXEL_FORMAT_OPAQUE : PIXEL_FORMAT_TRANSPARENT));
                    env->SetIntField (windowLayoutParams.get(), AndroidWindowManagerLayoutParams.gravity, 0x3 /* LEFT */ | 0x30 /* TOP */);
                    env->CallVoidMethod (viewGroup.get(), AndroidViewManager.updateViewLayout, view.get(), windowLayoutParams.get());
                }
            }
        }
        else
        {
            class ViewMover  : public CallbackMessage
            {
            public:
                ViewMover (const GlobalRef& v, Rectangle<int> boundsToUse)  : view (v), bounds (boundsToUse) {}

                void messageCallback() override
                {
                    view.callVoidMethod (AndroidView.layout,
                                         bounds.getX(), bounds.getY(), bounds.getRight(), bounds.getBottom());
                }

            private:
                GlobalRef view;
                Rectangle<int> bounds;
            };

            (new ViewMover (view, r))->post();
        }
    }

    Rectangle<int> getBounds() const override
    {
        Rectangle<int> r (view.callIntMethod (AndroidView.getLeft),
                          view.callIntMethod (AndroidView.getTop),
                          view.callIntMethod (AndroidView.getWidth),
                          view.callIntMethod (AndroidView.getHeight));

        return r / scale;
    }

    void handleScreenSizeChange() override
    {
        ComponentPeer::handleScreenSizeChange();

        if (isFullScreen())
            setFullScreen (true);
    }

    Point<int> getScreenPosition() const
    {
        auto* env = getEnv();

        LocalRef<jintArray> position (env->NewIntArray (2));
        env->CallVoidMethod (view.get(), AndroidView.getLocationOnScreen, position.get());

        jint* const screenPosition = env->GetIntArrayElements (position.get(), nullptr);
        Point<int> pos (screenPosition[0], screenPosition[1]);
        env->ReleaseIntArrayElements (position.get(), screenPosition, 0);

        return pos;
    }

    Point<float> localToGlobal (Point<float> relativePosition) override
    {
        return relativePosition + (getScreenPosition().toFloat() / scale);
    }

    using ComponentPeer::localToGlobal;

    Point<float> globalToLocal (Point<float> screenPosition) override
    {
        return screenPosition - (getScreenPosition().toFloat() / scale);
    }

    using ComponentPeer::globalToLocal;

    void setMinimised (bool /*shouldBeMinimised*/) override
    {
        // n/a
    }

    bool isMinimised() const override
    {
        return false;
    }

    bool shouldNavBarsBeHidden (bool shouldBeFullScreen) const
    {
        if (shouldBeFullScreen)
            if (Component* kiosk = Desktop::getInstance().getKioskModeComponent())
                if (kiosk->getPeer() == this)
                    return true;

        return false;
    }

    void setNavBarsHidden (bool hidden)
    {
        view.callVoidMethod (ComponentPeerView.setSystemUiVisibilityCompat,
                             hidden ? (jint) (fullScreenFlags)
                                    : (jint) (SYSTEM_UI_FLAG_VISIBLE));

        navBarsHidden = hidden;
    }

    void setFullScreen (bool shouldBeFullScreen) override
    {
        // updating the nav bar visibility is a bit odd on Android - need to wait for
        if (shouldNavBarsBeHidden (shouldBeFullScreen))
        {
            if (! isTimerRunning())
            {
                startTimer (500);
            }
        }
        else
        {
            setNavBarsHidden (false);
        }

        Rectangle<int> r (shouldBeFullScreen ? Desktop::getInstance().getDisplays().getMainDisplay().userArea
                                             : lastNonFullscreenBounds);

        if ((! shouldBeFullScreen) && r.isEmpty())
            r = getBounds();

        // (can't call the component's setBounds method because that'll reset our fullscreen flag)
        if (! r.isEmpty())
            setBounds (r, shouldBeFullScreen);

        component.repaint();
    }

    bool isFullScreen() const override
    {
        return fullScreen;
    }

    void timerCallback() override
    {
        setNavBarsHidden (shouldNavBarsBeHidden (fullScreen));
        setFullScreen (fullScreen);
        stopTimer();
    }

    void setIcon (const Image& /*newIcon*/) override
    {
        // n/a
    }

    bool contains (Point<int> localPos, bool trueIfInAChildWindow) const override
    {
        return isPositiveAndBelow (localPos.x, component.getWidth())
            && isPositiveAndBelow (localPos.y, component.getHeight())
            && ((! trueIfInAChildWindow) || view.callBooleanMethod (ComponentPeerView.containsPoint,
                                                                    localPos.x * scale,
                                                                    localPos.y * scale));
    }

    BorderSize<int> getFrameSize() const override
    {
        // TODO
        return BorderSize<int>();
    }

    bool setAlwaysOnTop (bool /*alwaysOnTop*/) override
    {
        // TODO
        return false;
    }

    void toFront (bool makeActive) override
    {
        // Avoid calling bringToFront excessively: it's very slow
        if (frontWindow != this)
        {
            view.callVoidMethod (AndroidView.bringToFront);

            frontWindow = this;
        }

        if (makeActive)
            grabFocus();

        handleBroughtToFront();
    }

    void toBehind (ComponentPeer*) override
    {
        // TODO
    }

    //==============================================================================
    void handleMouseDownCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        Point<float> pos = globalToLocal (lastMousePos);

        // this forces a mouse-enter/up event, in case for some reason we didn't get a mouse-up before.
        handleMouseEvent (MouseInputSource::InputSourceType::touch, pos, ModifierKeys::currentModifiers.withoutMouseButtons(),
                          MouseInputSource::invalidPressure, MouseInputSource::invalidOrientation, time, {}, index);

        if (isValidPeer (this))
            handleMouseDragCallback (index, sysPos, time);
    }

    void handleMouseDragCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        Point<float> pos = globalToLocal (lastMousePos);

        jassert (index < 64);
        touchesDown = (touchesDown | (1 << (index & 63)));
        ModifierKeys::currentModifiers = ModifierKeys::currentModifiers.withoutMouseButtons().withFlags (ModifierKeys::leftButtonModifier);
        handleMouseEvent (MouseInputSource::InputSourceType::touch, pos, ModifierKeys::currentModifiers.withoutMouseButtons().withFlags (ModifierKeys::leftButtonModifier),
                          MouseInputSource::invalidPressure, MouseInputSource::invalidOrientation, time, {}, index);
    }

    void handleMouseUpCallback (int index, Point<float> sysPos, int64 time)
    {
        lastMousePos = sysPos / scale;
        Point<float> pos = globalToLocal (lastMousePos);

        jassert (index < 64);
        touchesDown = (touchesDown & ~(1 << (index & 63)));

        if (touchesDown == 0)
            ModifierKeys::currentModifiers = ModifierKeys::currentModifiers.withoutMouseButtons();

        handleMouseEvent (MouseInputSource::InputSourceType::touch, pos, ModifierKeys::currentModifiers.withoutMouseButtons(), MouseInputSource::invalidPressure,
                          MouseInputSource::invalidOrientation, time, {}, index);
    }

    void handleKeyDownCallback (int k, int kc)
    {
        handleKeyPress (k, static_cast<juce_wchar> (kc));
    }

    void handleKeyUpCallback (int /*k*/, int /*kc*/)
    {
    }

    void handleBackButtonCallback()
    {
        bool handled = false;

        if (auto* app = JUCEApplicationBase::getInstance())
            handled = app->backButtonPressed();

        if (Component* kiosk = Desktop::getInstance().getKioskModeComponent())
            if (kiosk->getPeer() == this)
                setNavBarsHidden (navBarsHidden);

        if (! handled)
        {
            auto* env = getEnv();
            LocalRef<jobject> activity (getCurrentActivity());

            if (activity != nullptr)
            {
                jmethodID finishMethod = env->GetMethodID (AndroidActivity, "finish", "()V");

                if (finishMethod != nullptr)
                    env->CallVoidMethod (activity.get(), finishMethod);
            }
        }

    }

    void handleKeyboardHiddenCallback()
    {
        Component::unfocusAllComponents();
    }

    void handleAppPausedCallback() {}

    void handleAppResumedCallback()
    {
        if (Component* kiosk = Desktop::getInstance().getKioskModeComponent())
            if (kiosk->getPeer() == this)
                setNavBarsHidden (navBarsHidden);
    }

    //==============================================================================
    bool isFocused() const override
    {
        if (view != nullptr)
            return view.callBooleanMethod (AndroidView.hasFocus);

        return false;
    }

    void grabFocus() override
    {
        if (view != nullptr)
            view.callBooleanMethod (AndroidView.requestFocus);
    }

    void handleFocusChangeCallback (bool hasFocus)
    {
        if (isFullScreen())
            setFullScreen (true);

        if (hasFocus)
            handleFocusGain();
        else
            handleFocusLoss();
    }

    static const char* getVirtualKeyboardType (TextInputTarget::VirtualKeyboardType type) noexcept
    {
        switch (type)
        {
            case TextInputTarget::textKeyboard:          return "text";
            case TextInputTarget::numericKeyboard:       return "number";
            case TextInputTarget::decimalKeyboard:       return "numberDecimal";
            case TextInputTarget::urlKeyboard:           return "textUri";
            case TextInputTarget::emailAddressKeyboard:  return "textEmailAddress";
            case TextInputTarget::phoneNumberKeyboard:   return "phone";
            default:                                     jassertfalse; break;
        }

        return "text";
    }

    void textInputRequired (Point<int>, TextInputTarget& target) override
    {
        view.callVoidMethod (ComponentPeerView.showKeyboard,
                             javaString (getVirtualKeyboardType (target.getKeyboardType())).get());
    }

    void dismissPendingTextInput() override
    {
        view.callVoidMethod (ComponentPeerView.showKeyboard, javaString ("").get());

        // updating the nav bar visibility is a bit odd on Android - need to wait for
        if (! isTimerRunning())
            hideNavBarDelayed();
     }

    void hideNavBarDelayed()
    {
        startTimer (500);
    }

    //==============================================================================
    void handlePaintCallback (jobject canvas, jobject paint)
    {
        auto* env = getEnv();

        jobject rect = env->CallObjectMethod (canvas, AndroidCanvas.getClipBounds);
        const int left   = env->GetIntField (rect, AndroidRect.left);
        const int top    = env->GetIntField (rect, AndroidRect.top);
        const int right  = env->GetIntField (rect, AndroidRect.right);
        const int bottom = env->GetIntField (rect, AndroidRect.bottom);
        env->DeleteLocalRef (rect);

        const Rectangle<int> clip (left, top, right - left, bottom - top);

        const int sizeNeeded = clip.getWidth() * clip.getHeight();
        if (sizeAllocated < sizeNeeded)
        {
            buffer.clear();
            sizeAllocated = sizeNeeded;
            buffer = GlobalRef (LocalRef<jobject> ((jobject) env->NewIntArray (sizeNeeded)));
        }
        else if (sizeNeeded == 0)
        {
            return;
        }

        if (jint* dest = env->GetIntArrayElements ((jintArray) buffer.get(), nullptr))
        {
            {
                Image temp (new PreallocatedImage (clip.getWidth(), clip.getHeight(),
                                                   dest, ! component.isOpaque()));

                {
                    LowLevelGraphicsSoftwareRenderer g (temp);
                    g.setOrigin (-clip.getPosition());
                    g.addTransform (AffineTransform::scale (scale));
                    handlePaint (g);
                }
            }

            env->ReleaseIntArrayElements ((jintArray) buffer.get(), dest, 0);

            env->CallVoidMethod (canvas, AndroidCanvas.drawBitmap, (jintArray) buffer.get(), 0, clip.getWidth(),
                                 (jfloat) clip.getX(), (jfloat) clip.getY(),
                                 clip.getWidth(), clip.getHeight(), true, paint);
        }
    }

    void repaint (const Rectangle<int>& userArea) override
    {
        Rectangle<int> area = userArea * scale;

        if (MessageManager::getInstance()->isThisTheMessageThread())
        {
            view.callVoidMethod (AndroidView.invalidate, area.getX(), area.getY(), area.getRight(), area.getBottom());
        }
        else
        {
            struct ViewRepainter  : public CallbackMessage
            {
                ViewRepainter (const GlobalRef& view_, const Rectangle<int>& area_)
                    : view (view_), area (area_) {}

                void messageCallback() override
                {
                    view.callVoidMethod (AndroidView.invalidate, area.getX(), area.getY(),
                                         area.getRight(), area.getBottom());
                }

            private:
                GlobalRef view;
                const Rectangle<int> area;
            };

            (new ViewRepainter (view, area))->post();
        }
    }

    void performAnyPendingRepaintsNow() override
    {
        // TODO
    }

    void setAlpha (float /*newAlpha*/) override
    {
        // TODO
    }

    StringArray getAvailableRenderingEngines() override
    {
        return StringArray ("Software Renderer");
    }

    //==============================================================================
    static Point<float> lastMousePos;
    static int64 touchesDown;

    //==============================================================================
    struct StartupActivityCallbackListener : ActivityLifecycleCallbacks
    {
        void onActivityStarted (jobject /*activity*/) override
        {
            auto* env = getEnv();
            LocalRef<jobject> appContext (getAppContext());

            if (appContext.get() != nullptr)
            {

                env->CallVoidMethod (appContext.get(),
                                     AndroidApplication.unregisterActivityLifecycleCallbacks,
                                     activityCallbackListener.get());
                clear();
                activityCallbackListener.clear();

                const_cast<Displays &> (Desktop::getInstance().getDisplays()).refresh();
            }
        }
    };

private:
    //==============================================================================
    GlobalRef view, viewGroup;
    bool viewGroupIsWindow = false;
    GlobalRef buffer;
    bool fullScreen;
    bool navBarsHidden;
    int sizeAllocated;
    float scale;

    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD   (create,                      "<init>",                      "(Landroid/content/Context;ZJ)V") \
     METHOD   (clear,                       "clear",                       "()V") \
     METHOD   (setViewName,                 "setViewName",                 "(Ljava/lang/String;)V") \
     METHOD   (setVisible,                  "setVisible",                  "(Z)V") \
     METHOD   (isVisible,                   "isVisible",                   "()Z") \
     METHOD   (containsPoint,               "containsPoint",               "(II)Z") \
     METHOD   (showKeyboard,                "showKeyboard",                "(Ljava/lang/String;)V") \
     METHOD   (setSystemUiVisibilityCompat, "setSystemUiVisibilityCompat", "(I)V") \
     CALLBACK (handlePaintJni,              "handlePaint",                 "(JLandroid/graphics/Canvas;Landroid/graphics/Paint;)V") \
     CALLBACK (handleMouseDownJni,          "handleMouseDown",             "(JIFFJ)V") \
     CALLBACK (handleMouseDragJni,          "handleMouseDrag",             "(JIFFJ)V") \
     CALLBACK (handleMouseUpJni,            "handleMouseUp",               "(JIFFJ)V") \
     CALLBACK (handleKeyDownJni,            "handleKeyDown",               "(JII)V") \
     CALLBACK (handleKeyUpJni,              "handleKeyUp",                 "(JII)V") \
     CALLBACK (handleBackButtonJni,         "handleBackButton",            "(J)V") \
     CALLBACK (handleKeyboardHiddenJni,     "handleKeyboardHidden",        "(J)V") \
     CALLBACK (viewSizeChangedJni,          "viewSizeChanged",             "(J)V") \
     CALLBACK (focusChangedJni,             "focusChanged",                "(JZ)V") \
     CALLBACK (handleAppPausedJni,          "handleAppPaused",             "(J)V") \
     CALLBACK (handleAppResumedJni,         "handleAppResumed",            "(J)V") \

    DECLARE_JNI_CLASS_WITH_BYTECODE (ComponentPeerView, "com/roli/juce/ComponentPeerView", 16, javaComponentPeerView, sizeof (javaComponentPeerView))
    #undef JNI_CLASS_MEMBERS

    static void JNICALL handlePaintJni          (JNIEnv*, jobject /*view*/, jlong host, jobject canvas, jobject paint)           { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handlePaintCallback (canvas, paint); }
    static void JNICALL handleMouseDownJni      (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseDownCallback (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL handleMouseDragJni      (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseDragCallback (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL handleMouseUpJni        (JNIEnv*, jobject /*view*/, jlong host, jint i, jfloat x, jfloat y, jlong time)  { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMouseUpCallback   (i, Point<float> ((float) x, (float) y), (int64) time); }
    static void JNICALL viewSizeChangedJni      (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleMovedOrResized(); }
    static void JNICALL focusChangedJni         (JNIEnv*, jobject /*view*/, jlong host, jboolean hasFocus)                       { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleFocusChangeCallback (hasFocus); }
    static void JNICALL handleKeyDownJni        (JNIEnv*, jobject /*view*/, jlong host, jint k, jint kc)                         { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyDownCallback ((int) k, (int) kc); }
    static void JNICALL handleKeyUpJni          (JNIEnv*, jobject /*view*/, jlong host, jint k, jint kc)                         { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyUpCallback ((int) k, (int) kc); }
    static void JNICALL handleBackButtonJni     (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleBackButtonCallback(); }
    static void JNICALL handleKeyboardHiddenJni (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleKeyboardHiddenCallback(); }
    static void JNICALL handleAppPausedJni      (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleAppPausedCallback(); }
    static void JNICALL handleAppResumedJni     (JNIEnv*, jobject /*view*/, jlong host)                                          { if (auto* myself = reinterpret_cast<AndroidComponentPeer*> (host)) myself->handleAppResumedCallback(); }

    //==============================================================================
    friend class Displays;
    static AndroidComponentPeer* frontWindow;
    static GlobalRef activityCallbackListener;

    //==============================================================================
    static constexpr int GRAVITY_LEFT = 0x3, GRAVITY_TOP = 0x30;
    static constexpr int TYPE_APPLICATION = 0x2;
    static constexpr int FLAG_NOT_TOUCH_MODAL = 0x20, FLAG_LAYOUT_IN_SCREEN = 0x100, FLAG_LAYOUT_NO_LIMITS = 0x200;
    static constexpr int PIXEL_FORMAT_OPAQUE = -1, PIXEL_FORMAT_TRANSPARENT = -2;

    struct PreallocatedImage  : public ImagePixelData
    {
        PreallocatedImage (int width_, int height_, jint* data_, bool hasAlpha_)
            : ImagePixelData (Image::ARGB, width_, height_), data (data_), hasAlpha (hasAlpha_)
        {
            if (hasAlpha_)
                zeromem (data_, static_cast<size_t> (width * height) * sizeof (jint));
        }

        ~PreallocatedImage() override
        {
            if (hasAlpha)
            {
                auto pix = (PixelARGB*) data;

                for (int i = width * height; --i >= 0;)
                {
                    pix->unpremultiply();
                    ++pix;
                }
            }
        }

        std::unique_ptr<ImageType> createType() const override
        {
            return std::make_unique<SoftwareImageType>();
        }

        std::unique_ptr<LowLevelGraphicsContext> createLowLevelContext() override
        {
            return std::make_unique<LowLevelGraphicsSoftwareRenderer> (Image (this));
        }

        void initialiseBitmapData (Image::BitmapData& bm, int x, int y, Image::BitmapData::ReadWriteMode /*mode*/) override
        {
            bm.lineStride = width * static_cast<int> (sizeof (jint));
            bm.pixelStride = static_cast<int> (sizeof (jint));
            bm.pixelFormat = Image::ARGB;
            bm.data = (uint8*) (data + x + y * width);
        }

        ImagePixelData::Ptr clone() override
        {
            auto s = new PreallocatedImage (width, height, nullptr, hasAlpha);
            s->allocatedData.malloc (sizeof (jint) * static_cast<size_t> (width * height));
            s->data = s->allocatedData;
            memcpy (s->data, data, sizeof (jint) * static_cast<size_t> (width * height));
            return s;
        }

    private:
        jint* data;
        HeapBlock<jint> allocatedData;
        bool hasAlpha;

        JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (PreallocatedImage)
    };

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (AndroidComponentPeer)
};

Point<float> AndroidComponentPeer::lastMousePos;
int64 AndroidComponentPeer::touchesDown = 0;
AndroidComponentPeer* AndroidComponentPeer::frontWindow = nullptr;
GlobalRef AndroidComponentPeer::activityCallbackListener;
AndroidComponentPeer::ComponentPeerView_Class AndroidComponentPeer::ComponentPeerView;

//==============================================================================
ComponentPeer* Component::createNewPeer (int styleFlags, void* nativeWindow)
{
    return new AndroidComponentPeer (*this, styleFlags, nativeWindow);
}

//==============================================================================
bool Desktop::canUseSemiTransparentWindows() noexcept
{
    return true;
}

double Desktop::getDefaultMasterScale()
{
    return 1.0;
}

Desktop::DisplayOrientation Desktop::getCurrentOrientation() const
{
    enum
    {
        ROTATION_0   = 0,
        ROTATION_90  = 1,
        ROTATION_180 = 2,
        ROTATION_270 = 3
    };

    JNIEnv* env = getEnv();
    LocalRef<jstring> windowServiceString (javaString ("window"));


    LocalRef<jobject> windowManager = LocalRef<jobject> (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, windowServiceString.get()));

    if (windowManager.get() != nullptr)
    {
        LocalRef<jobject> display = LocalRef<jobject> (env->CallObjectMethod (windowManager, AndroidWindowManager.getDefaultDisplay));

        if (display.get() != nullptr)
        {
            int rotation = env->CallIntMethod (display, AndroidDisplay.getRotation);

            switch (rotation)
            {
                case ROTATION_0:   return upright;
                case ROTATION_90:  return rotatedAntiClockwise;
                case ROTATION_180: return upsideDown;
                case ROTATION_270: return rotatedClockwise;
            }
        }
    }

    jassertfalse;
    return upright;
}

bool MouseInputSource::SourceList::addSource()
{
    addSource (sources.size(), MouseInputSource::InputSourceType::touch);
    return true;
}

bool MouseInputSource::SourceList::canUseTouch()
{
    return true;
}

Point<float> MouseInputSource::getCurrentRawMousePosition()
{
    return AndroidComponentPeer::lastMousePos;
}

void MouseInputSource::setRawMousePosition (Point<float>)
{
    // not needed
}

//==============================================================================
bool KeyPress::isKeyCurrentlyDown (int /*keyCode*/)
{
    // TODO
    return false;
}

JUCE_API void JUCE_CALLTYPE Process::hide()
{
    auto* env = getEnv();
    LocalRef<jobject> currentActivity (getCurrentActivity().get());

    if (env->CallBooleanMethod (currentActivity.get(), AndroidActivity.moveTaskToBack, true) == 0)
    {
        GlobalRef intent (LocalRef<jobject> (env->NewObject (AndroidIntent, AndroidIntent.constructor)));
        env->CallObjectMethod (intent, AndroidIntent.setAction,   javaString ("android.intent.action.MAIN")  .get());
        env->CallObjectMethod (intent, AndroidIntent.addCategory, javaString ("android.intent.category.HOME").get());

        env->CallVoidMethod (currentActivity.get(), AndroidContext.startActivity, intent.get());
    }
}

//==============================================================================
// TODO
JUCE_API bool JUCE_CALLTYPE Process::isForegroundProcess() { return true; }
JUCE_API void JUCE_CALLTYPE Process::makeForegroundProcess() {}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (show,                   "show",                 "()V") \
 METHOD (getWindow,              "getWindow",            "()Landroid/view/Window;")

DECLARE_JNI_CLASS (AndroidDialog, "android/app/Dialog")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (construct,                   "<init>",                 "(Landroid/content/Context;)V") \
 METHOD (create,                      "create",                 "()Landroid/app/AlertDialog;") \
 METHOD (setTitle,                    "setTitle",               "(Ljava/lang/CharSequence;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setMessage,                  "setMessage",             "(Ljava/lang/CharSequence;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setCancelable,               "setCancelable",          "(Z)Landroid/app/AlertDialog$Builder;") \
 METHOD (setOnCancelListener,         "setOnCancelListener",    "(Landroid/content/DialogInterface$OnCancelListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setPositiveButton,           "setPositiveButton",      "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setNegativeButton,           "setNegativeButton",      "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;") \
 METHOD (setNeutralButton,            "setNeutralButton",       "(Ljava/lang/CharSequence;Landroid/content/DialogInterface$OnClickListener;)Landroid/app/AlertDialog$Builder;")

DECLARE_JNI_CLASS (AndroidAlertDialogBuilder, "android/app/AlertDialog$Builder")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (dismiss,    "dismiss",  "()V")

DECLARE_JNI_CLASS (AndroidDialogInterface, "android/content/DialogInterface")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \

DECLARE_JNI_CLASS (AndroidDialogOnClickListener, "android/content/DialogInterface$OnClickListener")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class DialogListener  : public juce::AndroidInterfaceImplementer
{
public:
    DialogListener (ModalComponentManager::Callback* callbackToUse, int resultToUse)
        : callback (callbackToUse), result (resultToUse)
    {}

    void onResult (jobject dialog)
    {
        auto* env = getEnv();
        env->CallVoidMethod (dialog, AndroidDialogInterface.dismiss);

        if (callback != nullptr)
            callback->modalStateFinished (result);

        callback = nullptr;
    }

private:
    jobject invoke (jobject proxy, jobject method, jobjectArray args) override
    {
        auto* env = getEnv();
        auto methodName = juce::juceString ((jstring) env->CallObjectMethod (method, JavaMethod.getName));

        if (methodName == "onCancel" || methodName == "onClick")
        {
            onResult (env->GetObjectArrayElement (args, 0));
            return nullptr;
        }

        // invoke base class
        return AndroidInterfaceImplementer::invoke (proxy, method, args);
    }

    std::unique_ptr<ModalComponentManager::Callback> callback;
    int result;
};

//==============================================================================
static void createAndroidDialog (const String& title, const String& message,
                                 ModalComponentManager::Callback* callback,
                                 const String& positiveButton = {}, const String& negativeButton = {},
                                 const String& neutralButton = {})
{
    auto* env = getEnv();

    LocalRef<jobject> builder (env->NewObject (AndroidAlertDialogBuilder, AndroidAlertDialogBuilder.construct, getMainActivity().get()));

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setTitle,   javaString (title).get()));
    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setMessage, javaString (message).get()));
    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setCancelable, true));

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setOnCancelListener,
                                     CreateJavaInterface (new DialogListener (callback, 0),
                                                          "android/content/DialogInterface$OnCancelListener").get()));

    auto positiveButtonText = positiveButton.isEmpty() ? String ("OK") : positiveButton;

    builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setPositiveButton,
                                     javaString (positiveButtonText).get(),
                                     CreateJavaInterface (new DialogListener (callback, positiveButton.isEmpty() ? 0 : 1),
                                                          "android/content/DialogInterface$OnClickListener").get()));

    if (negativeButton.isNotEmpty())
        builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setNegativeButton,
                                         javaString (negativeButton).get(),
                                         CreateJavaInterface (new DialogListener (callback, neutralButton.isEmpty() ? 0 : 2),
                                                              "android/content/DialogInterface$OnClickListener").get()));

    if (neutralButton.isNotEmpty())
        builder = LocalRef<jobject> (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.setNegativeButton,
                                         javaString (neutralButton).get(),
                                         CreateJavaInterface (new DialogListener (callback, 0),
                                                              "android/content/DialogInterface$OnClickListener").get()));

    LocalRef<jobject> dialog (env->CallObjectMethod (builder.get(), AndroidAlertDialogBuilder.create));

    LocalRef<jobject> window (env->CallObjectMethod (dialog.get(), AndroidDialog.getWindow));

    if (Desktop::getInstance().getKioskModeComponent() != nullptr)
    {
        env->CallVoidMethod (window.get(), AndroidWindow.setFlags, FLAG_NOT_FOCUSABLE, FLAG_NOT_FOCUSABLE);
        LocalRef<jobject> decorView (env->CallObjectMethod (window.get(), AndroidWindow.getDecorView));
        env->CallVoidMethod (decorView.get(), AndroidView.setSystemUiVisibility, fullScreenFlags);
    }

    env->CallVoidMethod (dialog.get(), AndroidDialog.show);

    if (Desktop::getInstance().getKioskModeComponent() != nullptr)
        env->CallVoidMethod (window.get(), AndroidWindow.clearFlags, FLAG_NOT_FOCUSABLE);
}

void JUCE_CALLTYPE NativeMessageBox::showMessageBoxAsync (AlertWindow::AlertIconType /*iconType*/,
                                                          const String& title, const String& message,
                                                          Component* /*associatedComponent*/,
                                                          ModalComponentManager::Callback* callback)
{
    createAndroidDialog (title, message, callback);
}

bool JUCE_CALLTYPE NativeMessageBox::showOkCancelBox (AlertWindow::AlertIconType /*iconType*/,
                                                      const String& title, const String& message,
                                                      Component* /*associatedComponent*/,
                                                      ModalComponentManager::Callback* callback)
{
    jassert (callback != nullptr); // on android, all alerts must be non-modal!!

    createAndroidDialog (title, message, callback, "OK", "Cancel");
    return false;
}

int JUCE_CALLTYPE NativeMessageBox::showYesNoCancelBox (AlertWindow::AlertIconType /*iconType*/,
                                                        const String& title, const String& message,
                                                        Component* /*associatedComponent*/,
                                                        ModalComponentManager::Callback* callback)
{
    jassert (callback != nullptr); // on android, all alerts must be non-modal!!

    createAndroidDialog (title, message, callback, "Yes", "No", "Cancel");
    return 0;
}

int JUCE_CALLTYPE NativeMessageBox::showYesNoBox (AlertWindow::AlertIconType /*iconType*/,
                                                   const String& title, const String& message,
                                                   Component* /*associatedComponent*/,
                                                   ModalComponentManager::Callback* callback)
{
    jassert (callback != nullptr); // on android, all alerts must be non-modal!!

    createAndroidDialog (title, message, callback, "Yes", "No");
    return 0;
}

//==============================================================================
static bool androidScreenSaverEnabled = false;

void Desktop::setScreenSaverEnabled (bool shouldEnable)
{
    constexpr auto FLAG_KEEP_SCREEN_ON = 0x80;

    if (shouldEnable != androidScreenSaverEnabled)
    {
        LocalRef<jobject> activity (getMainActivity());

        if (activity != nullptr)
        {
            auto* env = getEnv();

            LocalRef<jobject> mainWindow (env->CallObjectMethod (activity.get(), AndroidActivity.getWindow));
            env->CallVoidMethod(mainWindow.get(), AndroidWindow.setFlags, shouldEnable ? FLAG_KEEP_SCREEN_ON : 0, FLAG_KEEP_SCREEN_ON);
        }

        androidScreenSaverEnabled = shouldEnable;
    }
}

bool Desktop::isScreenSaverEnabled()
{
    return androidScreenSaverEnabled;
}

//==============================================================================
void Desktop::setKioskComponent (Component* kioskComp, bool enableOrDisable, bool allowMenusAndBars)
{
    ignoreUnused (allowMenusAndBars);

    if (AndroidComponentPeer* peer = dynamic_cast<AndroidComponentPeer*> (kioskComp->getPeer()))
        peer->setFullScreen (enableOrDisable);
    else
        jassertfalse; // (this should have been checked by the caller)
}

//==============================================================================
static jint getAndroidOrientationFlag (int orientations) noexcept
{
    enum
    {
        SCREEN_ORIENTATION_LANDSCAPE          = 0,
        SCREEN_ORIENTATION_PORTRAIT           = 1,
        SCREEN_ORIENTATION_USER               = 2,
        SCREEN_ORIENTATION_REVERSE_LANDSCAPE  = 8,
        SCREEN_ORIENTATION_REVERSE_PORTRAIT   = 9,
        SCREEN_ORIENTATION_USER_LANDSCAPE     = 11,
        SCREEN_ORIENTATION_USER_PORTRAIT      = 12,
    };

    switch (orientations)
    {
        case Desktop::upright:                                          return (jint) SCREEN_ORIENTATION_PORTRAIT;
        case Desktop::upsideDown:                                       return (jint) SCREEN_ORIENTATION_REVERSE_PORTRAIT;
        case Desktop::upright + Desktop::upsideDown:                    return (jint) SCREEN_ORIENTATION_USER_PORTRAIT;
        case Desktop::rotatedAntiClockwise:                             return (jint) SCREEN_ORIENTATION_LANDSCAPE;
        case Desktop::rotatedClockwise:                                 return (jint) SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
        case Desktop::rotatedClockwise + Desktop::rotatedAntiClockwise: return (jint) SCREEN_ORIENTATION_USER_LANDSCAPE;
        default:                                                        return (jint) SCREEN_ORIENTATION_USER;
    }
}

void Desktop::allowedOrientationsChanged()
{
    LocalRef<jobject> activity (getMainActivity());

    if (activity != nullptr)
        getEnv()->CallVoidMethod (activity.get(), AndroidActivity.setRequestedOrientation, getAndroidOrientationFlag (allowedOrientations));
}

//==============================================================================
bool juce_areThereAnyAlwaysOnTopWindows()
{
    return false;
}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (create,          "<init>",         "()V") \
 FIELD  (density,         "density",        "F") \
 FIELD  (widthPixels,     "widthPixels",    "I") \
 FIELD  (heightPixels,    "heightPixels",   "I")

DECLARE_JNI_CLASS (AndroidDisplayMetrics, "android/util/DisplayMetrics")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class LayoutChangeListener  : public juce::AndroidInterfaceImplementer
{
public:
    virtual void onLayoutChange (LocalRef<jobject> view, int left, int top, int right, int bottom,
                                 int oldLeft, int oldTop, int oldRight, int oldBottom) = 0;

private:
    jobject invoke (jobject proxy, jobject method, jobjectArray args) override
    {
        auto* env = getEnv();
        auto methodName = juce::juceString ((jstring) env->CallObjectMethod (method, JavaMethod.getName));

        if (methodName == "onLayoutChange")
        {
            jassert (env->GetArrayLength (args) == 9);

            LocalRef<jobject> view (env->GetObjectArrayElement (args, 0));
            int dims[8];

            for (int i = 1; i < 9; ++i)
            {
                LocalRef<jobject> integer (env->GetObjectArrayElement (args, i));
                dims[i - 1] = env->CallIntMethod (integer.get(), JavaInteger.intValue);
            }

            onLayoutChange (std::move (view), dims[0], dims[1], dims[2], dims[3],
                            dims[4], dims[5], dims[6], dims[7]);

            return nullptr;
        }

        // invoke base class
        return AndroidInterfaceImplementer::invoke (proxy, method, args);
    }

    std::unique_ptr<ModalComponentManager::Callback> callback;
};

//==============================================================================
class MainActivityWindowLayoutListener   : public LayoutChangeListener
{
public:
    void onLayoutChange (LocalRef<jobject> /*view*/, int left, int top, int right, int bottom,
                         int oldLeft, int oldTop, int oldRight, int oldBottom) override
    {
        auto newBounds = Rectangle<int>::leftTopRightBottom (left, top, right, bottom);
        auto oldBounds = Rectangle<int>::leftTopRightBottom (oldLeft, oldTop, oldRight, oldBottom);

        if (newBounds != oldBounds)
        {
            auto& displays = Desktop::getInstance().getDisplays();
            auto& mainDisplay = displays.getMainDisplay();

            Rectangle<int> userArea = newBounds / mainDisplay.scale;

            if (userArea != mainDisplay.userArea)
                const_cast<Displays&> (displays).refresh();
        }
    }
};

//==============================================================================
void Displays::findDisplays (float masterScale)
{
    auto* env = getEnv();

    LocalRef<jobject> usableSize (env->NewObject (AndroidPoint, AndroidPoint.create, 0, 0));
    LocalRef<jstring> windowServiceString (javaString ("window"));
    LocalRef<jobject> displayMetrics (env->NewObject (AndroidDisplayMetrics, AndroidDisplayMetrics.create));
    LocalRef<jobject> windowManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, windowServiceString.get()));
    LocalRef <jobject> display (env->CallObjectMethod (windowManager, AndroidWindowManager.getDefaultDisplay));

    jmethodID getRealMetricsMethod = env->GetMethodID (AndroidDisplay, "getRealMetrics", "(Landroid/util/DisplayMetrics;)V");

    if (getRealMetricsMethod != nullptr)
        env->CallVoidMethod (display.get(), getRealMetricsMethod, displayMetrics.get());
    else
        env->CallVoidMethod (display.get(), AndroidDisplay.getMetrics, displayMetrics.get());

    env->CallVoidMethod (display.get(), AndroidDisplay.getSize, usableSize.get());

    Display d;

    d.isMain = true;
    d.scale = env->GetFloatField (displayMetrics.get(), AndroidDisplayMetrics.density);
    d.dpi = (d.scale * 160.f);
    d.scale *= masterScale;

    d.totalArea = Rectangle<int> (env->GetIntField (displayMetrics.get(), AndroidDisplayMetrics.widthPixels),
                                  env->GetIntField (displayMetrics.get(), AndroidDisplayMetrics.heightPixels)) / d.scale;

    d.userArea = Rectangle<int> (env->GetIntField (usableSize.get(), AndroidPoint.x),
                                 env->GetIntField (usableSize.get(), AndroidPoint.y)) / d.scale;

    // unfortunately usableSize still contains the nav bar
    // the best workaround is to try to get the size of the top-level view of
    // the main activity
    LocalRef<jobject> activity (getMainActivity());

    if (activity != nullptr)
    {
        LocalRef<jobject> mainWindow (env->CallObjectMethod (activity.get(), AndroidActivity.getWindow));
        LocalRef<jobject> decorView (env->CallObjectMethod (mainWindow.get(), AndroidWindow.getDecorView));
        LocalRef<jobject> contentView (env->CallObjectMethod (decorView.get(), AndroidView.findViewById, 0x01020002 /* android.R.id.content */));

        if (contentView != nullptr)
        {
            Rectangle<int> activityArea (env->CallIntMethod (contentView.get(), AndroidView.getLeft),
                                         env->CallIntMethod (contentView.get(), AndroidView.getTop),
                                         env->CallIntMethod (contentView.get(), AndroidView.getWidth),
                                         env->CallIntMethod (contentView.get(), AndroidView.getHeight));

            if (! activityArea.isEmpty())
                d.userArea = activityArea / d.scale;

            static bool hasAddedMainActivityListener = false;

            if (! hasAddedMainActivityListener)
            {
                hasAddedMainActivityListener = true;

                env->CallVoidMethod (contentView.get(), AndroidView.addOnLayoutChangeListener,
                                     CreateJavaInterface (new MainActivityWindowLayoutListener,
                                                          "android/view/View$OnLayoutChangeListener").get());
            }
        }
    }
    else
    {
        // the main activity may have not started yet so add an activity listener
        if (AndroidComponentPeer::activityCallbackListener == nullptr)
        {
            LocalRef<jobject> appContext (getAppContext());

            if (appContext.get() != nullptr)
            {
                AndroidComponentPeer::activityCallbackListener = GlobalRef (CreateJavaInterface (
                        new AndroidComponentPeer::StartupActivityCallbackListener,
                        "android/app/Application$ActivityLifecycleCallbacks"));

                env->CallVoidMethod (appContext.get(),
                                     AndroidApplication.registerActivityLifecycleCallbacks,
                                     AndroidComponentPeer::activityCallbackListener.get());
            }
        }
    }

    displays.add (d);
}

//==============================================================================
Image juce_createIconForFile (const File& /*file*/)
{
    return Image();
}

//==============================================================================
void* CustomMouseCursorInfo::create() const                                         { return nullptr; }
void* MouseCursor::createStandardMouseCursor (MouseCursor::StandardCursorType)      { return nullptr; }
void MouseCursor::deleteMouseCursor (void* /*cursorHandle*/, bool /*isStandard*/)   {}

//==============================================================================
void MouseCursor::showInWindow (ComponentPeer*) const   {}

//==============================================================================
bool DragAndDropContainer::performExternalDragDropOfFiles (const StringArray& /*files*/, bool /*canMove*/,
                                                           Component* /*srcComp*/, std::function<void()> /*callback*/)
{
    jassertfalse;    // no such thing on Android!
    return false;
}

bool DragAndDropContainer::performExternalDragDropOfText (const String& /*text*/, Component* /*srcComp*/,
                                                          std::function<void()> /*callback*/)
{
    jassertfalse;    // no such thing on Android!
    return false;
}

//==============================================================================
void LookAndFeel::playAlertSound()
{
}

//==============================================================================
#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
 METHOD (getText,      "getText",            "()Ljava/lang/CharSequence;") \
 METHOD (setText,      "setText",            "(Ljava/lang/CharSequence;)V")

DECLARE_JNI_CLASS (AndroidClipboardManager, "android/content/ClipboardManager")
#undef JNI_CLASS_MEMBERS

//==============================================================================
void SystemClipboard::copyTextToClipboard (const String& text)
{
    auto* env = getEnv();

    LocalRef<jobject> clipboardManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, javaString ("clipboard").get()));
    env->CallVoidMethod (clipboardManager.get(), AndroidClipboardManager.setText, javaString(text).get());
}

String SystemClipboard::getTextFromClipboard()
{
    auto* env = getEnv();

    LocalRef<jobject> clipboardManager (env->CallObjectMethod (getAppContext().get(), AndroidContext.getSystemService, javaString ("clipboard").get()));
    LocalRef<jobject> charSequence (env->CallObjectMethod (clipboardManager.get(), AndroidClipboardManager.getText));

    if (charSequence == nullptr)
        return {};

    return juceString(LocalRef<jstring> ((jstring) env->CallObjectMethod(charSequence.get(), JavaCharSequence.toString)));
}

//==============================================================================
const int extendedKeyModifier       = 0x10000;

const int KeyPress::spaceKey        = ' ';
const int KeyPress::returnKey       = 66;
const int KeyPress::escapeKey       = 4;
const int KeyPress::backspaceKey    = 67;
const int KeyPress::leftKey         = extendedKeyModifier + 1;
const int KeyPress::rightKey        = extendedKeyModifier + 2;
const int KeyPress::upKey           = extendedKeyModifier + 3;
const int KeyPress::downKey         = extendedKeyModifier + 4;
const int KeyPress::pageUpKey       = extendedKeyModifier + 5;
const int KeyPress::pageDownKey     = extendedKeyModifier + 6;
const int KeyPress::endKey          = extendedKeyModifier + 7;
const int KeyPress::homeKey         = extendedKeyModifier + 8;
const int KeyPress::deleteKey       = extendedKeyModifier + 9;
const int KeyPress::insertKey       = -1;
const int KeyPress::tabKey          = 61;
const int KeyPress::F1Key           = extendedKeyModifier + 10;
const int KeyPress::F2Key           = extendedKeyModifier + 11;
const int KeyPress::F3Key           = extendedKeyModifier + 12;
const int KeyPress::F4Key           = extendedKeyModifier + 13;
const int KeyPress::F5Key           = extendedKeyModifier + 14;
const int KeyPress::F6Key           = extendedKeyModifier + 16;
const int KeyPress::F7Key           = extendedKeyModifier + 17;
const int KeyPress::F8Key           = extendedKeyModifier + 18;
const int KeyPress::F9Key           = extendedKeyModifier + 19;
const int KeyPress::F10Key          = extendedKeyModifier + 20;
const int KeyPress::F11Key          = extendedKeyModifier + 21;
const int KeyPress::F12Key          = extendedKeyModifier + 22;
const int KeyPress::F13Key          = extendedKeyModifier + 23;
const int KeyPress::F14Key          = extendedKeyModifier + 24;
const int KeyPress::F15Key          = extendedKeyModifier + 25;
const int KeyPress::F16Key          = extendedKeyModifier + 26;
const int KeyPress::F17Key          = extendedKeyModifier + 50;
const int KeyPress::F18Key          = extendedKeyModifier + 51;
const int KeyPress::F19Key          = extendedKeyModifier + 52;
const int KeyPress::F20Key          = extendedKeyModifier + 53;
const int KeyPress::F21Key          = extendedKeyModifier + 54;
const int KeyPress::F22Key          = extendedKeyModifier + 55;
const int KeyPress::F23Key          = extendedKeyModifier + 56;
const int KeyPress::F24Key          = extendedKeyModifier + 57;
const int KeyPress::F25Key          = extendedKeyModifier + 58;
const int KeyPress::F26Key          = extendedKeyModifier + 59;
const int KeyPress::F27Key          = extendedKeyModifier + 60;
const int KeyPress::F28Key          = extendedKeyModifier + 61;
const int KeyPress::F29Key          = extendedKeyModifier + 62;
const int KeyPress::F30Key          = extendedKeyModifier + 63;
const int KeyPress::F31Key          = extendedKeyModifier + 64;
const int KeyPress::F32Key          = extendedKeyModifier + 65;
const int KeyPress::F33Key          = extendedKeyModifier + 66;
const int KeyPress::F34Key          = extendedKeyModifier + 67;
const int KeyPress::F35Key          = extendedKeyModifier + 68;
const int KeyPress::numberPad0      = extendedKeyModifier + 27;
const int KeyPress::numberPad1      = extendedKeyModifier + 28;
const int KeyPress::numberPad2      = extendedKeyModifier + 29;
const int KeyPress::numberPad3      = extendedKeyModifier + 30;
const int KeyPress::numberPad4      = extendedKeyModifier + 31;
const int KeyPress::numberPad5      = extendedKeyModifier + 32;
const int KeyPress::numberPad6      = extendedKeyModifier + 33;
const int KeyPress::numberPad7      = extendedKeyModifier + 34;
const int KeyPress::numberPad8      = extendedKeyModifier + 35;
const int KeyPress::numberPad9      = extendedKeyModifier + 36;
const int KeyPress::numberPadAdd            = extendedKeyModifier + 37;
const int KeyPress::numberPadSubtract       = extendedKeyModifier + 38;
const int KeyPress::numberPadMultiply       = extendedKeyModifier + 39;
const int KeyPress::numberPadDivide         = extendedKeyModifier + 40;
const int KeyPress::numberPadSeparator      = extendedKeyModifier + 41;
const int KeyPress::numberPadDecimalPoint   = extendedKeyModifier + 42;
const int KeyPress::numberPadEquals         = extendedKeyModifier + 43;
const int KeyPress::numberPadDelete         = extendedKeyModifier + 44;
const int KeyPress::playKey         = extendedKeyModifier + 45;
const int KeyPress::stopKey         = extendedKeyModifier + 46;
const int KeyPress::fastForwardKey  = extendedKeyModifier + 47;
const int KeyPress::rewindKey       = extendedKeyModifier + 48;

} // namespace juce
