/*
 * Decompiled with CFR 0.152.
 */
package cc.arduino.utils.network;

import cc.arduino.utils.FileHash;
import cc.arduino.utils.network.CacheControl;
import cc.arduino.utils.network.HttpConnectionManager;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.script.ScriptException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import processing.app.BaseNoGui;
import processing.app.PreferencesData;
import processing.app.helpers.FileUtils;

public class FileDownloaderCache {
    private static final String CACHE_ENABLE_PREFERENCE_KEY = "cache.enable";
    private static final Logger log = LogManager.getLogger(FileDownloaderCache.class);
    private static final Map<String, FileCached> cachedFiles = Collections.synchronizedMap(new HashMap());
    private static final String cacheFolder;
    private static boolean enableCache;

    public static Optional<FileCached> getFileCached(URL remoteURL) throws URISyntaxException, NoSuchMethodException, ScriptException, IOException {
        return FileDownloaderCache.getFileCached(remoteURL, true);
    }

    public static Optional<FileCached> getFileCached(URL remoteURL, boolean enableCache) throws URISyntaxException, NoSuchMethodException, ScriptException, IOException {
        if (!enableCache || !FileDownloaderCache.enableCache) {
            log.info("The cache is not enable.");
            return Optional.empty();
        }
        String[] splitPath = remoteURL.getPath().split("/");
        if (splitPath.length == 0) {
            log.warn("The remote path as no file name {}", (Object)remoteURL);
            return Optional.empty();
        }
        LinkedList<String> addFirstRemoteURL = new LinkedList<String>(Arrays.asList(splitPath));
        addFirstRemoteURL.addFirst(remoteURL.getHost());
        Path cacheFilePath = Paths.get(cacheFolder, addFirstRemoteURL.toArray(new String[0]));
        FileCached fileCached = Optional.ofNullable(cachedFiles.get(remoteURL.toString())).orElseGet(() -> new FileCached(remoteURL.toString(), cacheFilePath.toString()));
        log.info("Get file cached is expire {}, exist {}, info {} ", (Object)fileCached.isExpire(), (Object)fileCached.exists(), (Object)fileCached);
        if (fileCached.isExpire() || !fileCached.exists()) {
            Optional<FileCached> fileCachedInfoUpdated = FileDownloaderCache.updateCacheInfo(remoteURL, (remoteETagClean, cacheControl) -> {
                if (cacheControl.isNoCache() || cacheControl.isMustRevalidate() || cacheControl.isNoStore()) {
                    log.warn("The file {} must not be cache due to cache control header {}", (Object)remoteURL, cacheControl);
                    return Optional.empty();
                }
                log.info("Update cached info of {}, createdAt {}, previous eTag {}, last eTag {}, cache control header {} ", (Object)remoteURL, (Object)fileCached.createdAt, (Object)fileCached.eTag, remoteETagClean, cacheControl);
                FileCached fileCachedUpdateETag = new FileCached(remoteURL.toString(), cacheFilePath.toString(), fileCached.eTag, (String)remoteETagClean, fileCached.md5, (CacheControl)cacheControl);
                cachedFiles.put(remoteURL.toString(), fileCachedUpdateETag);
                return Optional.of(fileCachedUpdateETag);
            });
            FileDownloaderCache.updateCacheFilesInfo();
            return fileCachedInfoUpdated;
        }
        return Optional.of(fileCached);
    }

    private static Optional<FileCached> updateCacheInfo(URL remoteURL, BiFunction<String, CacheControl, Optional<FileCached>> getNewFile) throws URISyntaxException, NoSuchMethodException, ScriptException, IOException {
        HttpURLConnection headRequest = new HttpConnectionManager(remoteURL).makeConnection(connection -> {
            try {
                connection.setRequestMethod("HEAD");
            }
            catch (ProtocolException e) {
                log.error("Invalid protocol", (Throwable)e);
            }
        });
        int responseCode = headRequest.getResponseCode();
        headRequest.disconnect();
        if (responseCode < 200 || responseCode >= 300) {
            log.warn("The head request return a bad response code " + responseCode);
            return Optional.empty();
        }
        String remoteETag = headRequest.getHeaderField("ETag");
        String cacheControlHeader = headRequest.getHeaderField("Cache-Control");
        if (remoteETag != null && cacheControlHeader != null) {
            String remoteETagClean = remoteETag.trim().replace("\"", "");
            CacheControl cacheControl = CacheControl.valueOf(cacheControlHeader);
            return getNewFile.apply(remoteETagClean, cacheControl);
        }
        log.warn("The head request do not return the ETag {} or the Cache-Control {}", (Object)remoteETag, (Object)cacheControlHeader);
        return Optional.empty();
    }

    private static synchronized void updateCacheFilesInfo() throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.enable(SerializationFeature.INDENT_OUTPUT);
        ObjectNode objectNode = mapper.createObjectNode();
        objectNode.putArray("files").addAll((Collection)cachedFiles.values().stream().map(v -> (JsonNode)mapper.convertValue(v, JsonNode.class)).collect(Collectors.toList()));
        Path cachedFileInfo = FileDownloaderCache.getCachedInfoPath();
        if (Files.notExists(cachedFileInfo, new LinkOption[0])) {
            Files.createDirectories(cachedFileInfo.getParent(), new FileAttribute[0]);
        }
        log.info("Update cache file info in {}, number of cached files is {}", (Object)cachedFileInfo.toFile(), (Object)cachedFiles.size());
        mapper.writeValue(cachedFileInfo.toFile(), (Object)objectNode);
    }

    private static Path getCachedInfoPath() {
        return Paths.get(cacheFolder, "cache.json");
    }

    static {
        enableCache = Boolean.valueOf(PreferencesData.get(CACHE_ENABLE_PREFERENCE_KEY, "true"));
        if (!enableCache) {
            log.info("The cache is disable cache.enable=false");
        }
        PreferencesData.set(CACHE_ENABLE_PREFERENCE_KEY, Boolean.toString(enableCache));
        File settingsFolder = BaseNoGui.getSettingsFolder();
        if (settingsFolder != null) {
            cacheFolder = Paths.get(settingsFolder.getPath(), "cache").toString();
        } else {
            enableCache = false;
            cacheFolder = null;
            log.error("The cache will disable because the setting folder is null, cannot generate the cache path");
        }
        Path pathCacheInfo = FileDownloaderCache.getCachedInfoPath();
        log.info("Cache folder {}", (Object)cacheFolder);
        try {
            if (Files.exists(pathCacheInfo, new LinkOption[0])) {
                ObjectMapper mapper = new ObjectMapper();
                JsonNode jsonNode = mapper.readTree(pathCacheInfo.toFile());
                TypeReference<List<FileCached>> typeRef = new TypeReference<List<FileCached>>(){};
                List files = (List)mapper.readValue(mapper.treeAsTokens((TreeNode)jsonNode.get("files")), (TypeReference)typeRef);
                cachedFiles.putAll(Collections.synchronizedMap(files.stream().filter(FileCached::exists).collect(Collectors.toMap(FileCached::getRemoteURL, Function.identity()))));
                log.info("Number of file already in the cache {}", (Object)cachedFiles.size());
            }
        }
        catch (Exception e) {
            log.error("Cannot initialized the cache", (Throwable)e);
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class FileCached {
        private final String remoteURL;
        private final String localPath;
        private final String eTag;
        private final String lastETag;
        private final String md5;
        private final String createdAt;
        private final CacheControl cacheControl;

        FileCached() {
            this.remoteURL = null;
            this.localPath = null;
            this.lastETag = null;
            this.eTag = null;
            this.md5 = null;
            this.createdAt = null;
            this.cacheControl = null;
        }

        FileCached(String remoteURL, String localPath) {
            this.remoteURL = remoteURL;
            this.localPath = localPath;
            this.lastETag = null;
            this.eTag = null;
            this.md5 = null;
            this.createdAt = LocalDateTime.now().format(DateTimeFormatter.ISO_DATE_TIME);
            this.cacheControl = null;
        }

        public FileCached(String remoteURL, String localPath, String eTag, String lastETag, String md5, CacheControl cacheControl) {
            this.remoteURL = remoteURL;
            this.localPath = localPath;
            this.eTag = eTag;
            this.lastETag = lastETag;
            this.md5 = md5;
            this.createdAt = LocalDateTime.now().format(DateTimeFormatter.ISO_DATE_TIME);
            this.cacheControl = cacheControl;
        }

        @JsonIgnore
        public boolean isExpire() {
            LocalDateTime now = LocalDateTime.now();
            return this.getExpiresTime().isBefore(now) || this.getExpiresTime().isEqual(now);
        }

        @JsonIgnore
        public boolean isNotChange() {
            return !this.isChange();
        }

        @JsonIgnore
        public boolean isChange() {
            boolean isChange = false;
            if (this.isExpire()) {
                log.debug("The file \"{}\" is expire. Expire time: {}", (Object)this.localPath, (Object)this.getExpiresTime().format(DateTimeFormatter.ISO_DATE_TIME));
                isChange = true;
            }
            if (this.lastETag != null && !this.lastETag.equals(this.eTag)) {
                log.debug("The file \"{}\" is changed last ETag != now Etag ({}!={})", (Object)this.localPath, (Object)this.lastETag, (Object)this.eTag);
                isChange = true;
            }
            return isChange;
        }

        @JsonIgnore
        public boolean exists() {
            return this.localPath != null && Files.exists(Paths.get(this.localPath, new String[0]), new LinkOption[0]);
        }

        @JsonIgnore
        public Optional<File> getFileFromCache() {
            if (this.md5Check()) {
                return Optional.of(Paths.get(this.localPath, new String[0]).toFile());
            }
            return Optional.empty();
        }

        public synchronized void updateCacheFile(File fileToCache) throws Exception {
            String eTag;
            Path cacheFilePath = Paths.get(this.localPath, new String[0]);
            if (!Files.exists(cacheFilePath.getParent(), new LinkOption[0])) {
                Files.createDirectories(cacheFilePath.getParent(), new FileAttribute[0]);
            }
            FileUtils.copyFile(fileToCache, cacheFilePath.toFile());
            String md5 = this.calculateMD5();
            if (this.lastETag == null) {
                log.warn("The eTag was not calculate this time, is not the right behaviour fileCached={}, md5={}", (Object)this, (Object)md5);
                eTag = this.eTag;
            } else {
                eTag = this.lastETag;
            }
            FileCached newFileCached = new FileCached(this.remoteURL, this.localPath, eTag, eTag, md5, this.cacheControl);
            log.info("Update cache file: {}", (Object)newFileCached);
            cachedFiles.put(this.remoteURL, newFileCached);
            FileDownloaderCache.updateCacheFilesInfo();
        }

        public synchronized void invalidateCache() throws IOException {
            cachedFiles.remove(this.remoteURL);
            Files.deleteIfExists(Paths.get(this.localPath, new String[0]));
        }

        private String calculateMD5() throws IOException, NoSuchAlgorithmException {
            if (this.exists()) {
                return FileHash.hash(Paths.get(this.localPath, new String[0]).toFile(), "MD5");
            }
            return null;
        }

        @JsonIgnore
        public boolean md5Check() {
            try {
                return !Objects.isNull(this.getMD5()) && Objects.equals(this.calculateMD5(), this.getMD5());
            }
            catch (Exception e) {
                log.error("Fail to calculate the MD5. file={}", (Object)this, (Object)e);
                return false;
            }
        }

        @JsonIgnore
        public LocalDateTime getExpiresTime() {
            int maxAge = this.cacheControl != null ? this.cacheControl.getMaxAge() : 0;
            if (this.createdAt != null) {
                return LocalDateTime.parse(this.createdAt, DateTimeFormatter.ISO_DATE_TIME).plusSeconds(maxAge);
            }
            return LocalDateTime.now();
        }

        public String getExpires() {
            return this.getExpiresTime().toString();
        }

        public String getMD5() {
            return this.md5;
        }

        public String geteTag() {
            return this.eTag;
        }

        public String getRemoteURL() {
            return this.remoteURL;
        }

        public String getLocalPath() {
            return this.localPath;
        }

        public String getCreatedAt() {
            return this.createdAt;
        }

        public CacheControl getCacheControl() {
            return this.cacheControl;
        }

        public String toString() {
            return "FileCached{eTag='" + this.eTag + '\'' + ", lastETag='" + this.lastETag + '\'' + ", remoteURL='" + this.remoteURL + '\'' + ", localPath='" + this.localPath + '\'' + ", md5='" + this.md5 + '\'' + ", createdAt='" + this.createdAt + '\'' + ", cacheControl=" + this.cacheControl + '}';
        }
    }
}

