#!/usr/bin/env perl
#
# BEGIN COPYRIGHT BLOCK
# Copyright (C) 2001 Sun Microsystems, Inc. Used by permission.
# Copyright (C) 2013 Red Hat, Inc.
# All rights reserved.
#
# License: GPL (version 3 or any later version).
# See LICENSE for details. 
# END COPYRIGHT BLOCK
#

use lib qw(/usr/lib/dirsrv/perl);
use DSUtil;

# enable the use of our bundled perldap with our bundled ldapsdk libraries
# all of this nonsense can be omitted if the mozldapsdk and perldap are
# installed in the operating system locations (e.g. /usr/lib /usr/lib/perl5)

$ENV{'PATH'} = ":/usr/bin";
$ENV{'SHLIB_PATH'} = "$ENV{'LD_LIBRARY_PATH'}";

DSUtil::libpath_add("");
DSUtil::libpath_add("/usr/lib");
DSUtil::libpath_add("/usr/lib64");

# Add new password policy specific entries

#############################################################################
# enable the use of Perldap functions
require DynaLoader;

use Getopt::Std;
use Mozilla::LDAP::Conn;
use Mozilla::LDAP::Utils qw(:all);
use Mozilla::LDAP::API qw(:api :ssl :apiv3 :constant); # Direct access to C API

#############################################################################
# Default values of the variables

$opt_D = "";
$opt_p = "";
$opt_h = "";
$opt_Z = "";
$opt_v = 0;

#############################################################################

sub usage {
    print (STDERR "ns-newpwpolicy.pl [-Z serverID] [-v] [-D rootdn] { -w password | -j filename } [-P protocol]\n");
    print (STDERR "                  [-p port] [-h host] -U UserDN -S SuffixDN\n\n");
    print (STDERR "Arguments:\n");
    print (STDERR "        -?           - Display usage\n");
    print (STDERR "        -Z serverID  - Server instance identifier\n");
    print (STDERR "        -v           - Verbose output\n");
    print (STDERR "        -D rootdn    - Directory Manager DN\n");
    print (STDERR "        -w rootpw    - password for the Directory Manager DN\n");
    print (STDERR "        -j filename  - Read the Directory Manager's password from file\n");
    print (STDERR "        -p port      - Port\n");
    print (STDERR "        -h host      - Hostname\n");
    print (STDERR "        -U userDN    - User entry DN\n");
    print (STDERR "        -S suffixDN  - Suffix entry DN\n");
    print (STDERR "        -P protocol  - STARTTLS, LDAPS, LDAPI, LDAP (default: uses most secure protocol available)\n");
    exit 1;
}

# Process the command line arguments
{
    usage() if (!getopts('vD:w:j:p:P:h:U:S:Z:'));

    ($opt_Z, $confdir) = DSUtil::get_server_id($opt_Z, "/etc/dirsrv");
    %info = DSUtil::get_info($confdir, $opt_h, $opt_p, $opt_D);
    $info{rootdnpw} = $opt_w;
    $info{protocol} = $opt_P;
    if ($opt_j ne ""){
        die "Error, cannot open password file $opt_j\n" unless (open (RPASS, $opt_j));
        $opt_w = <RPASS>;
        $info{rootdnpw} = chomp($opt_w);
        close(RPASS);
    } 
    
    usage() if( $opt_w eq "" );
    if ($opt_U eq "" && $opt_S eq "") {
        print (STDERR "Please provide at least -S or -U option.\n\n");
    }

    # Now, check if the user/group exists

    if ($opt_S) {
        my $esc_opt_S = $opt_S;
         $esc_opt_S =~ s/,/\\,/g;
         $esc_opt_S =~ s/=/\\=/g;
        print (STDERR "host = $info{host}, port = $info{port}, suffixDN = \"$opt_S\"\n\n") if $opt_v;

        $container="dn: cn=nsPwPolicyContainer,$opt_S\nobjectclass: top\nobjectclass: nsContainer\n\n";
        $pwpolicy="dn: cn=cn\\=nsPwPolicyEntry\\,$esc_opt_S,cn=nsPwPolicyContainer,$opt_S\n" . 
                  "objectclass: top\nobjectclass: ldapsubentry\nobjectclass: passwordpolicy\n\n";
        $template="dn: cn=cn\\=nsPwTemplateEntry\\,$esc_opt_S,cn=nsPwPolicyContainer,$opt_S\n" . 
                  "objectclass: top\nobjectclass: extensibleObject\nobjectclass: costemplate\n" . 
                  "objectclass: ldapsubentry\ncosPriority: 1\n" . 
                  "pwdpolicysubentry: cn=cn\\=nsPwPolicyEntry\\,$esc_opt_S,cn=nsPwPolicyContainer,$opt_S\n\n";
        $cos="dn: cn=nsPwPolicy_cos,$opt_S\nobjectclass: top\nobjectclass: LDAPsubentry\n" . 
             "objectclass: cosSuperDefinition\nobjectclass: cosPointerDefinition\n" . 
             "cosTemplateDn: cn=cn\\=nsPwTemplateEntry\\,$esc_opt_S,cn=nsPwPolicyContainer,$opt_S\n" . 
             "cosAttribute: pwdpolicysubentry default operational-default\n\n";
        $entries = $container . $pwpolicy . $template . $cos;

        $info{args} = "-c -a";
        $retcode = DSUtil::ldapmod($entries, %info);
        if ( $retcode != 0 && $retcode != 68 ) {
            print( STDERR "Error $retcode while adding pwpolicy entries. Exiting.\n" );
            exit ($retcode);
        } else {
            print( STDERR "Successfully added pwpolicy entries\n\n") if $opt_v;
        }
        
        $info{args} = "";
        $modConfig = "dn:cn=config\nchangetype: modify\nreplace:nsslapd-pwpolicy-local\nnsslapd-pwpolicy-local: on\n\n";
        $retcode = DSUtil::ldapmod($modConfig, %info);
        if ( $retcode != 0 ) {
            print( STDERR "Error $retcode while modifing \"cn=config\". Exiting.\n" );
            exit ($retcode);
        } else {
            print( STDERR "Entry \"cn=config\" modified\n\n") if $opt_v;
        }
    } # end of $opt_S

    if ($opt_U) {
        my $norm_opt_U = normalizeDN($opt_U);
        my $esc_opt_U = $norm_opt_U;
        $esc_opt_U =~ s/,/\\,/g;
        $esc_opt_U =~ s/=/\\=/g;
        print (STDERR "host = $info{host}, port = $info{port}, userDN = \"$norm_opt_U\"\n\n") if $opt_v;
        $info{base} = $norm_opt_U;
        $info{filter} = "";
        $info{scope} = "base";
        $info{attrs} = "";
        $retcode = DSUtil::ldapsrch_ext(%info);
        if ($retcode != 0 ) {
            print( STDERR "the user entry $norm_opt_U does not exist. Error $retcode\n");
            exit ($retcode);
        }
        
        print( STDERR "the user entry $norm_opt_U found..\n\n") if $opt_v;
        
        # Now, get the parentDN 
        @rdns = ldap_explode_dn($norm_opt_U, 0);
        shift @rdns;
        $parentDN = join(',', @rdns);

        print (STDERR "parentDN is $parentDN\n\n") if $opt_v;

        $info{args} = "-c -a";
        my $containers="dn: cn=nsPwPolicyContainer,$parentDN\n" . 
                       "objectclass: top\n" . 
                       "objectclass: nsContainer\n\n" .
                       "dn: cn=cn\\=nsPwPolicyEntry\\,$esc_opt_U,cn=nsPwPolicyContainer,$parentDN\n" . 
                       "objectclass: top\n" . 
                       "objectclass: ldapsubentry\nobjectclass: passwordpolicy\n";
        $retcode = DSUtil::ldapmod($containers, %info);
        if ( $retcode != 0 && $retcode != 68 ) {
            print( STDERR "Error $retcode while adding container entries.\n" );
            exit ($retcode);
        } else {
            print (STDERR "Container entries added.\n");
        }
        
        $info{args} = "";
        $target = "cn=cn\\=nsPwPolicyEntry\\,$esc_opt_U,cn=nsPwPolicyContainer,$parentDN";
        $modConfig = "dn: $norm_opt_U\nchangetype: modify\nreplace:pwdpolicysubentry\npwdpolicysubentry: $target\n\n";
        $retcode = DSUtil::ldapmod($modConfig, %info);
        if ( $retcode != 0 ) {
            print( STDERR "Error $retcode while modifing $norm_opt_U. Exiting.\n" );
            exit ($retcode);
        } else {
            print( STDERR "Entry \"$norm_opt_U\" modified\n\n") if $opt_v;
        }

        $modConfig = "dn:cn=config\nchangetype: modify\nreplace:nsslapd-pwpolicy-local\nnsslapd-pwpolicy-local: on\n\n";
        $retcode = DSUtil::ldapmod($modConfig, %info);
        if( $retcode != 0 ) {
            print( STDERR "Error $retcode while modifing \"cn=config\"." );
            exit ($retcode);
        } else {
            print( STDERR "Entry \"cn=config\" modified\n\n") if $opt_v;
        }
    } # end of $opt_U
}
