#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright: (c) 2018, F5 Networks Inc.
# GNU General Public License v3.0 (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function
__metaclass__ = type


ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['stableinterface'],
                    'supported_by': 'certified'}

DOCUMENTATION = r'''
---
module: bigip_dns_nameserver
short_description: Manage LTM DNS nameservers on a BIG-IP
description:
  - Manages LTM DNS nameservers on a BIG-IP. These nameservers form part of what is
    known as DNS Express on a BIG-IP. This module does not configure GTM related
    functionality, nor does it configure system-level name servers that affect the
    base system's ability to resolve DNS names.
version_added: "1.0.0"
options:
  name:
    description:
      - Specifies the name of the nameserver.
    type: str
    required: True
  address:
    description:
      - Specifies the IP address on which the DNS nameserver (client) or back-end DNS
        authoritative server (DNS Express server) listens for DNS messages.
      - When creating a new nameserver, if this value is not specified, the default
        is C(127.0.0.1).
    type: str
  service_port:
    description:
      - Specifies the service port on which the DNS nameserver (client) or back-end DNS
        authoritative server (DNS Express server) listens for DNS messages.
      - When creating a new nameserver, if this value is not specified, the default
        is C(53).
    type: str
  route_domain:
    description:
      - Specifies the local route domain that the DNS nameserver (client) or back-end
        DNS authoritative server (DNS Express server) uses for outbound traffic.
      - When creating a new nameserver, if this value is not specified, the default
        is C(0).
    type: str
  tsig_key:
    description:
      - Specifies the TSIG key the system uses to communicate with this DNS nameserver
        (client) or back-end DNS authoritative server (DNS Express server) for AXFR zone
        transfers.
      - If the nameserver is a client, then the system uses this TSIG key to verify the
        request and sign the response.
      - If this nameserver is a DNS Express server, then this TSIG key must match the
        TSIG key for the zone on the back-end DNS authoritative server.
    type: str
  state:
    description:
      - When C(present), ensures that the resource exists.
      - When C(absent), ensures the resource is removed.
    type: str
    choices:
      - present
      - absent
    default: present
  partition:
    description:
      - Device partition to manage resources on.
    type: str
    default: Common
extends_documentation_fragment: f5networks.f5_modules.f5
author:
  - Tim Rupp (@caphrim007)
'''

EXAMPLES = r'''
- name: Create a nameserver
  bigip_dns_nameserver:
    name: foo
    address: 10.10.10.10
    service_port: 53
    state: present
    provider:
      password: secret
      server: lb.mydomain.com
      user: admin
  delegate_to: localhost
'''

RETURN = r'''
address:
  description: Address which the nameserver listens for DNS messages.
  returned: changed
  type: str
  sample: 127.0.0.1
service_port:
  description: Service port on which the nameserver listens for DNS messages.
  returned: changed
  type: int
  sample: 53
'''

from ansible.module_utils.basic import (
    AnsibleModule, env_fallback
)

from ..module_utils.bigip import F5RestClient
from ..module_utils.common import (
    F5ModuleError, AnsibleF5Parameters, transform_name, f5_argument_spec, fq_name
)


class Parameters(AnsibleF5Parameters):
    api_map = {
        'routeDomain': 'route_domain',
        'port': 'service_port',
        'tsigKey': 'tsig_key'
    }

    api_attributes = [
        'address',
        'routeDomain',
        'port',
        'tsigKey'
    ]

    returnables = [
        'address',
        'service_port',
        'route_domain',
        'tsig_key',
    ]

    updatables = [
        'address',
        'service_port',
        'route_domain',
        'tsig_key',
    ]


class ApiParameters(Parameters):
    pass


class ModuleParameters(Parameters):
    @property
    def tsig_key(self):
        if self._values['tsig_key'] in [None, '']:
            return self._values['tsig_key']
        return fq_name(self.partition, self._values['tsig_key'])

    @property
    def route_domain(self):
        if self._values['route_domain'] is None:
            return None
        return fq_name(self.partition, self._values['route_domain'])

    @property
    def service_port(self):
        if self._values['service_port'] is None:
            return None
        try:
            return int(self._values['service_port'])
        except ValueError:
            # Reserving the right to add well-known ports
            raise F5ModuleError(
                "The 'service_port' must be in numeric form."
            )


class Changes(Parameters):
    def to_return(self):
        result = {}
        try:
            for returnable in self.returnables:
                result[returnable] = getattr(self, returnable)
            result = self._filter_params(result)
        except Exception:
            raise
        return result


class UsableChanges(Changes):
    pass


class ReportableChanges(Changes):
    pass


class Difference(object):
    def __init__(self, want, have=None):
        self.want = want
        self.have = have

    def compare(self, param):
        try:
            result = getattr(self, param)
            return result
        except AttributeError:
            return self.__default(param)

    def __default(self, param):
        attr1 = getattr(self.want, param)
        try:
            attr2 = getattr(self.have, param)
            if attr1 != attr2:
                return attr1
        except AttributeError:
            return attr1

    @property
    def tsig_key(self):
        if self.want.tsig_key is None:
            return None
        if self.have.tsig_key is None and self.want.tsig_key == '':
            return None
        if self.want.tsig_key != self.have.tsig_key:
            return self.want.tsig_key


class ModuleManager(object):
    def __init__(self, *args, **kwargs):
        self.module = kwargs.get('module', None)
        self.client = F5RestClient(**self.module.params)
        self.want = ModuleParameters(params=self.module.params)
        self.have = ApiParameters()
        self.changes = UsableChanges()

    def _set_changed_options(self):
        changed = {}
        for key in Parameters.returnables:
            if getattr(self.want, key) is not None:
                changed[key] = getattr(self.want, key)
        if changed:
            self.changes = UsableChanges(params=changed)

    def _update_changed_options(self):
        diff = Difference(self.want, self.have)
        updatables = Parameters.updatables
        changed = dict()
        for k in updatables:
            change = diff.compare(k)
            if change is None:
                continue
            else:
                if isinstance(change, dict):
                    changed.update(change)
                else:
                    changed[k] = change
        if changed:
            self.changes = UsableChanges(params=changed)
            return True
        return False

    def should_update(self):
        result = self._update_changed_options()
        if result:
            return True
        return False

    def exec_module(self):
        changed = False
        result = dict()
        state = self.want.state

        if state == "present":
            changed = self.present()
        elif state == "absent":
            changed = self.absent()

        reportable = ReportableChanges(params=self.changes.to_return())
        changes = reportable.to_return()
        result.update(**changes)
        result.update(dict(changed=changed))
        self._announce_deprecations(result)
        return result

    def _announce_deprecations(self, result):
        warnings = result.pop('__warnings', [])
        for warning in warnings:
            self.client.module.deprecate(
                msg=warning['msg'],
                version=warning['version']
            )

    def present(self):
        if self.exists():
            return self.update()
        else:
            return self.create()

    def exists(self):
        errors = [401, 403, 409, 500, 501, 502, 503, 504]
        uri = "https://{0}:{1}/mgmt/tm/ltm/dns/nameserver/{2}".format(
            self.client.provider['server'],
            self.client.provider['server_port'],
            transform_name(self.want.partition, self.want.name)
        )
        resp = self.client.api.get(uri)
        try:
            response = resp.json()
        except ValueError as ex:
            raise F5ModuleError(str(ex))

        if resp.status == 404 or 'code' in response and response['code'] == 404:
            return False
        if resp.status in [200, 201] or 'code' in response and response['code'] in [200, 201]:
            return True

        if resp.status in errors or 'code' in response and response['code'] in errors:
            if 'message' in response:
                raise F5ModuleError(response['message'])
            else:
                raise F5ModuleError(resp.content)

    def update(self):
        self.have = self.read_current_from_device()
        if not self.should_update():
            return False
        if self.module.check_mode:
            return True
        self.update_on_device()
        return True

    def remove(self):
        if self.module.check_mode:
            return True
        self.remove_from_device()
        if self.exists():
            raise F5ModuleError("Failed to delete the resource.")
        return True

    def create(self):
        if self.want.address is None:
            self.want.update({'address': '127.0.0.1'})
        if self.want.service_port is None:
            self.want.update({'service_port': '53'})
        if self.want.route_domain is None:
            self.want.update({'route_domain': '/Common/0'})
        self._set_changed_options()
        if self.module.check_mode:
            return True
        self.create_on_device()
        return True

    def create_on_device(self):
        params = self.changes.api_params()
        params['name'] = self.want.name
        params['partition'] = self.want.partition
        uri = "https://{0}:{1}/mgmt/tm/ltm/dns/nameserver/".format(
            self.client.provider['server'],
            self.client.provider['server_port']
        )
        resp = self.client.api.post(uri, json=params)
        try:
            response = resp.json()
        except ValueError as ex:
            raise F5ModuleError(str(ex))

        if resp.status in [200, 201] or 'code' in response and response['code'] in [200, 201]:
            return True
        raise F5ModuleError(resp.content)

    def update_on_device(self):
        params = self.changes.api_params()
        uri = "https://{0}:{1}/mgmt/tm/ltm/dns/nameserver/{2}".format(
            self.client.provider['server'],
            self.client.provider['server_port'],
            transform_name(self.want.partition, self.want.name)
        )
        resp = self.client.api.patch(uri, json=params)
        try:
            response = resp.json()
        except ValueError as ex:
            raise F5ModuleError(str(ex))

        if resp.status in [200, 201] or 'code' in response and response['code'] in [200, 201]:
            return True
        raise F5ModuleError(resp.content)

    def absent(self):
        if self.exists():
            return self.remove()
        return False

    def remove_from_device(self):
        uri = "https://{0}:{1}/mgmt/tm/ltm/dns/nameserver/{2}".format(
            self.client.provider['server'],
            self.client.provider['server_port'],
            transform_name(self.want.partition, self.want.name)
        )
        response = self.client.api.delete(uri)
        if response.status == 200:
            return True
        raise F5ModuleError(response.content)

    def read_current_from_device(self):
        uri = "https://{0}:{1}/mgmt/tm/ltm/dns/nameserver/{2}".format(
            self.client.provider['server'],
            self.client.provider['server_port'],
            transform_name(self.want.partition, self.want.name)
        )
        resp = self.client.api.get(uri)
        try:
            response = resp.json()
        except ValueError as ex:
            raise F5ModuleError(str(ex))

        if resp.status in [200, 201] or 'code' in response and response['code'] in [200, 201]:
            return ApiParameters(params=response)
        raise F5ModuleError(resp.content)


class ArgumentSpec(object):
    def __init__(self):
        self.supports_check_mode = True
        argument_spec = dict(
            name=dict(required=True),
            address=dict(),
            service_port=dict(),
            route_domain=dict(),
            tsig_key=dict(),
            state=dict(
                default='present',
                choices=['present', 'absent']
            ),
            partition=dict(
                default='Common',
                fallback=(env_fallback, ['F5_PARTITION'])
            )
        )
        self.argument_spec = {}
        self.argument_spec.update(f5_argument_spec)
        self.argument_spec.update(argument_spec)


def main():
    spec = ArgumentSpec()

    module = AnsibleModule(
        argument_spec=spec.argument_spec,
        supports_check_mode=spec.supports_check_mode,
    )

    try:
        mm = ModuleManager(module=module)
        results = mm.exec_module()
        module.exit_json(**results)
    except F5ModuleError as ex:
        module.fail_json(msg=str(ex))


if __name__ == '__main__':
    main()
