package Lire::UI::PluginWidget;

use strict;

use base qw/ Curses::UI::Container Lire::UI::Widget /;

use Curses::UI::Common;
use Locale::TextDomain 'lire';

use Lire::Utils qw/ item_index check_param check_object_param /;
use Lire::UI::Utils qw/ button_box_width /;

use Carp;
use vars qw/@CARP_NOT/;

@CARP_NOT = qw/Curses::UI::Container/;

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase(\%userargs);

    check_object_param( $userargs{'value'}, 'value', 'Lire::Config::Plugin' );

    my $self = $class->Curses::UI::Container::new( %userargs,
                                                   '-height' => 1,
                                                   '-releasefocus' => 1 );

    my @options = map { $_->name() } $self->{'value'}->spec()->options();
    my ( $selected, $focusable );
    if ( @options ) {
        my $plugin = $self->{'value'}->get_plugin();
        $selected = ( defined $plugin
                      ? item_index( \@options, $plugin )
                      : undef );
        $focusable = 1;
    } else {
        @options = ( __( '-- empty list --' ) );
        $selected = 0;
        $focusable = 0;
    }

    $self->add( 'list', 'Popupmenu',
                '-onchange' => sub { $self->_on_change_cb_helper() },
                '-selected' => $selected,
                '-values' => \@options,
                '-focusable' => $focusable );

    $self->_update_button();

    $self->layout();

    return $self;
}

sub layout {
    my $self = $_[0];

    $self->{'-height'} = 1;

    return $self->SUPER::layout();
}

sub layout_contained_objects {
    my $self = $_[0];

    my $list = $self->getobj( 'list' );
    return $self unless $list;

    if ( $self->{'value'}->has_properties() ) {
        $list->{'-width'} = $self->canvaswidth() - 6;
        my $btn = $self->getobj( 'button' );
        $btn->{'-x'} = $list->{'-width'} + 1;
    } else {
        $list->{'-width'} = $self->canvaswidth();
    }

    return $self->SUPER::layout_contained_objects();
}

sub _update_button {
    my $self = $_[0];

    my $has_props = $self->{'value'}->has_properties();
    my $button = $self->getobj( 'button' );
    if ( !$button && $has_props) {
        $self->add( 'button', 'Buttonbox',
                    '-buttons' => [ { '-label' => '<...>',
                                      '-onpress' => sub {
                                          $self->_properties_cb_helper()
                                      } } ],
                    '-width' => 5 );
        $self->layout();
        $self->intellidraw();
    } elsif ( $button && !$has_props ) {
        $self->delete( 'button' );
        $self->layout();
        $self->intellidraw();
    }
}

sub _on_change_cb_helper {
    my $self = $_[0];

    my $list = $self->getobj( 'list' );
    $self->{'value'}->set_plugin( $list->get() );
    $self->_update_button();
    $self->run_event( 'onvaluechanged' );

    return;
}

sub _properties_cb_helper {
    my $self = $_[0];

    my $plugin = $self->{'value'}->get_plugin();
    return unless ( defined $plugin
                    && $self->{'value'}->has_properties() );

    my $properties = $self->{'value'}->get_properties();
    my $new_props = $properties->clone();
    my $title = __x( 'Properties for {plugin}', 'plugin' => $plugin );
    if ( $self->root()->userdata()->edit_value_dialog( $new_props, $title ) ) {
        my @comps = map { $_->name() } $new_props->spec()->components();
        foreach my $comp ( @comps ) {
            $properties->set( $new_props->get( $comp ) );
        }
        $self->run_event( 'onvaluechanged' );
    }

    return;
}

1;
