<TeXmacs|1.99.8>

<style|<tuple|tmdoc|british>>

<\body>
  <tmdoc-title|A Tour of Sage>

  This is a tour of Sage adapted from the official
  <hlink|tutorial|http://doc.sagemath.org/html/en/a_tour_of_sage/> for
  <TeXmacs>.

  <paragraph*|Sage as a Calculator>

  The Sage command line has a <code*|sage:> prompt; you do not have to add
  it. If you use the Sage plugin for <TeXmacs>, then put everything after the
  <code*|sage]> prompt in an input cell, and press enter to compute the
  corresponding output.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      3 + 5
    <|unfolded-io>
      <math|8>
    </unfolded-io>
  </session>

  The caret symbol means \Praise to a power\Q.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      57.1 ^ 100
    <|unfolded-io>
      <math|4.60904368661396\<times\>10<rsup|175>>
    </unfolded-io>
  </session>

  We compute the inverse of a <math|2\<times\>2> matrix in Sage.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      matrix([[1,2], [3,4]])^(-1)
    <|unfolded-io>
      <math|<around*|(|<tabular*|<tformat|<cwith|1|-1|1|1|cell-halign|r>|<cwith|1|-1|1|1|cell-lborder|0ln>|<cwith|1|-1|2|2|cell-halign|r>|<cwith|1|-1|2|2|cell-rborder|0ln>|<table|<row|<cell|-2>|<cell|1>>|<row|<cell|<frac|3|2>>|<cell|-<frac|1|2>>>>>>|)>>
    </unfolded-io>
  </session>

  Here we integrate a simple function.

  <\session|sage|default>
    <\input|Sage] >
      x = var('x') \ \ # create a symbolic variable
    </input>

    <\unfolded-io|Sage] >
      integrate(sqrt(x)*sqrt(1+x), x)
    <|unfolded-io>
      <math|<frac|<frac|<around*|(|x+1|)><rsup|<frac|3|2>>|x<rsup|<frac|3|2>>>+<frac|<sqrt|x+1>|<sqrt|x>>|4*<space|0.17em><around*|(|<frac|<around*|(|x+1|)><rsup|2>|x<rsup|2>>-<frac|2*<space|0.17em><around*|(|x+1|)>|x>+1|)>>-<frac|1|8>*<space|0.17em>log
      <around*|(|<frac|<sqrt|x+1>|<sqrt|x>>+1|)>+<frac|1|8>*<space|0.17em>log
      <around*|(|<frac|<sqrt|x+1>|<sqrt|x>>-1|)>>
    </unfolded-io>
  </session>

  This asks Sage to solve a quadratic equation. The symbol <math|==>
  represents equality in Sage.

  <\session|sage|default>
    <\input|Sage] >
      a = var('a')
    </input>

    <\input|Sage] >
      S = solve(x^2 + x == a, x)
    </input>

    <\unfolded-io|Sage] >
      S
    <|unfolded-io>
      <math|<around*|[|x=-<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>,x=<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>|]>>
    </unfolded-io>
  </session>

  The result is a list of equalities.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      S[0].rhs()
    <|unfolded-io>
      <math|-<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>>
    </unfolded-io>
  </session>

  Naturally, Sage can plot various useful functions.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      plot(sin(x) + sin(1.6*x), 0, 40)
    <|unfolded-io>
      <image|<tuple|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|ps>|0.7par|||>
    </unfolded-io>
  </session>

  <paragraph*|Power Computing with Sage>

  First we create a 500\<times\>500 matrix of random numbers.

  <\session|sage|default>
    <\input|Sage] >
      m = random_matrix(RDF,500)
    </input>
  </session>

  It takes Sage a few seconds to compute the eigenvalues of the matrix and
  plot them.

  <\session|sage|default>
    <\input|Sage] >
      e = m.eigenvalues() \ #about 2 seconds
    </input>

    <\input|Sage] >
      w = [(i, abs(e[i])) for i in range(len(e))]
    </input>

    <\unfolded-io|Sage] >
      points(w)
    <|unfolded-io>
      <image|<tuple|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|ps>|0.7par|||>
    </unfolded-io>
  </session>

  Thanks to the GNU Multiprecision Library (GMP), Sage can handle very large
  numbers, even numbers with millions or billions of digits.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      factorial(100)
    <|unfolded-io>
      <math|93326215443944152681699238856266700490715968264381621468592963895217599993229915608941463976156518286253697920827223758251185210916864000000000000000000000000>
    </unfolded-io>

    <\input|Sage] >
      n = factorial(1000000) \ #about 2.5 seconds
    </input>
  </session>

  This computes at least 100 digits of \<pi\>.

  <\session|sage|default>
    <\unfolded-io|Sage] >
      N(pi, digits=100)
    <|unfolded-io>
      <math|3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068>
    </unfolded-io>
  </session>

  This asks Sage to factor a polynomial in two variables.

  <\session|sage|default>
    <\input|Sage] >
      R.\<less\>x,y\<gtr\> = QQ[]
    </input>

    <\input|Sage] >
      F = factor(x^99 + y^99)
    </input>

    <\unfolded-io|Sage] >
      F
    <|unfolded-io>
      <math|<around|(|x+y|)>\<cdot\><around|(|x<rsup|2>-x*y+y<rsup|2>|)>\<cdot\><around|(|x<rsup|6>-x<rsup|3>*y<rsup|3>+y<rsup|6>|)>\<cdot\><around|(|x<rsup|10>-x<rsup|9>*y+x<rsup|8>*y<rsup|2>-x<rsup|7>*y<rsup|3>+x<rsup|6>*y<rsup|4>-x<rsup|5>*y<rsup|5>+x<rsup|4>*y<rsup|6>-x<rsup|3>*y<rsup|7>+x<rsup|2>*y<rsup|8>-x*y<rsup|9>+y<rsup|10>|)>\<cdot\><around|(|x<rsup|20>+x<rsup|19>*y-x<rsup|17>*y<rsup|3>-x<rsup|16>*y<rsup|4>+x<rsup|14>*y<rsup|6>+x<rsup|13>*y<rsup|7>-x<rsup|11>*y<rsup|9>-x<rsup|10>*y<rsup|10>-x<rsup|9>*y<rsup|11>+x<rsup|7>*y<rsup|13>+x<rsup|6>*y<rsup|14>-x<rsup|4>*y<rsup|16>-x<rsup|3>*y<rsup|17>+x*y<rsup|19>+y<rsup|20>|)>\<cdot\><around|(|x<rsup|60>+x<rsup|57>*y<rsup|3>-x<rsup|51>*y<rsup|9>-x<rsup|48>*y<rsup|12>+x<rsup|42>*y<rsup|18>+x<rsup|39>*y<rsup|21>-x<rsup|33>*y<rsup|27>-x<rsup|30>*y<rsup|30>-x<rsup|27>*y<rsup|33>+x<rsup|21>*y<rsup|39>+x<rsup|18>*y<rsup|42>-x<rsup|12>*y<rsup|48>-x<rsup|9>*y<rsup|51>+x<rsup|3>*y<rsup|57>+y<rsup|60>|)>>
    </unfolded-io>

    <\unfolded-io|Sage] >
      F.expand()
    <|unfolded-io>
      <math|x<rsup|99>+y<rsup|99>>
    </unfolded-io>
  </session>

  Sage takes just under 5 seconds to compute the numbers of ways to partition
  one hundred million as a sum of positive integers.

  <\session|sage|default>
    <\input|Sage] >
      z = Partitions(10^8).cardinality() #about 4.5 seconds
    </input>

    <\unfolded-io|Sage] >
      str(z)[:40]
    <|unfolded-io>
      1760517045946249141360373894679135204009
    </unfolded-io>
  </session>

  <tmdoc-copyright|2018|Darcy Shen>

  <tmdoc-license|Permission is granted to copy, distribute and/or modify this
  document under the terms of the GNU Free Documentation License, Version 1.1
  or any later version published by the Free Software Foundation; with no
  Invariant Sections, with no Front-Cover Texts, and with no Back-Cover
  Texts. A copy of the license is included in the section entitled "GNU Free
  Documentation License".>
</body>

<initial|<\collection>
</collection>>