/* Library libcerf:
 *   A self-contained numeric library that provides an efficient and accurate implementation
 *   of complex error functions, along with Dawson, Faddeeva, and Voigt functions.
 *
 * File cerf.h:
 *   Declare exported functions.
 *
 * Copyright:
 *   (C) 2012 Massachusetts Institute of Technology
 *   (C) 2013 Forschungszentrum Jülich GmbH
 *
 * Licence:
 *   Permission is hereby granted, free of charge, to any person obtaining
 *   a copy of this software and associated documentation files (the
 *   "Software"), to deal in the Software without restriction, including
 *   without limitation the rights to use, copy, modify, merge, publish,
 *   distribute, sublicense, and/or sell copies of the Software, and to
 *   permit persons to whom the Software is furnished to do so, subject to
 *   the following conditions:
 *
 *   The above copyright notice and this permission notice shall be
 *   included in all copies or substantial portions of the Software.
 *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *   NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *   LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *   OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *   WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   Steven G. Johnson, Massachusetts Institute of Technology, 2012, core author
 *   Joachim Wuttke, Forschungszentrum Jülich, 2013, package maintainer
 *
 * Website:
 *   http://apps.jcns.fz-juelich.de/libcerf
 *
 * Revision history:
 *   ../CHANGELOG
 *
 * Man pages:
 *   w_of_z(3), dawson(3), voigt(3), cerf(3), erfcx(3), erfi(3)
 */

#ifdef __cplusplus
#error libcerf core code must not be compiled as C++
#endif

#ifndef CERF_CERF_H
#define CERF_CERF_H

#include "cerf_real.h" // defines real functions, and also the EXPORT macro

#include <complex.h>
typedef double _Complex _cerf_cmplx;

//! compute w(z) = exp(-z^2) erfc(-iz), Faddeeva's scaled complex error function
EXPORT _cerf_cmplx w_of_z(_cerf_cmplx z);

//! compute erf(z), the error function of complex arguments
EXPORT _cerf_cmplx cerf(_cerf_cmplx z);

//! compute erfc(z) = 1 - erf(z), the complementary error function
EXPORT _cerf_cmplx cerfc(_cerf_cmplx z);

//! compute erfcx(z) = exp(z^2) erfc(z), an underflow-compensated version of erfc
EXPORT _cerf_cmplx cerfcx(_cerf_cmplx z);

//! compute erfi(z) = -i erf(iz), the imaginary error function
EXPORT _cerf_cmplx cerfi(_cerf_cmplx z);

//! compute dawson(z) = sqrt(pi)/2 * exp(-z^2) * erfi(z), Dawson's integral
EXPORT _cerf_cmplx cdawson(_cerf_cmplx z);

#endif // CERF_CERF_H
