interface RegexObj {
    regex: string;
    exclusive: boolean;
}
export declare class AppServiceRegistration {
    private url;
    /**
     * Generate a random token.
     * @return {string} A randomly generated token.
     */
    static generateToken(): string;
    /**
 * Convert a JSON object to an AppServiceRegistration object.
 * @static
 * @param {Object} obj The registration object
 * @return {?AppServiceRegistration} The registration or null if the object
 * cannot be coerced into a registration.
 */
    static fromObject(obj: any): AppServiceRegistration | null;
    /**
     * Construct a new application service registration.
     * @constructor
     * @param {string} appServiceUrl The base URL the AS can be reached via.
     */
    private id;
    private hsToken;
    private asToken;
    private senderLocalpart;
    private rateLimited;
    /**
     * **Experimental**
     * Signal to the homeserver that this appservice will accept ephemeral events.
     */
    pushEphemeral: boolean;
    private namespaces;
    private protocols;
    private cachedRegex;
    constructor(url: string);
    /**
     * Set the URL which the home server will hit in order to talk to the AS.
     * @param {string} url The application service url
     */
    setAppServiceUrl(url: string): void;
    /**
     * Set the ID of the appservice; must be unique across the homeserver and never change.
     * @param {string} id The ID
     */
    setId(id: string): void;
    /**
     * Get the ID of the appservice.
     * @return {?string} The ID
     */
    getId(): string | null;
    /**
     * Set the list of protocols that this appservice will serve for third party lookups.
     * @param {string[]} protocols The protocols
     */
    setProtocols(protocols: string[]): void;
    /**
     * Get the list of protocols that this appservice will serve for third party lookups.
     * Will return null if no protocols have been set.
     * @return {string[]} The protocols.
     */
    getProtocols(): string[] | null;
    /**
     * Set the token the homeserver will use to communicate with the app service.
     * @param {string} token The token
     */
    setHomeserverToken(token: string): void;
    /**
     * Get the token the homeserver will use to communicate with the app service.
     * @return {?string} The token
     */
    getHomeserverToken(): string | null;
    /**
     * Set the token the app service will use to communicate with the homeserver.
     * @param {string} token The token
     */
    setAppServiceToken(token: string): void;
    /**
     * Get the token the app service will use to communicate with the homeserver.
     * @return {?string} The token
     */
    getAppServiceToken(): string | null;
    /**
     * Set the desired user_id localpart for the app service itself.
     * @param {string} localpart The user_id localpart ("alice" in "@alice:domain")
     */
    setSenderLocalpart(localpart: string): void;
    /**
     * Set whether requests from this AS are rate-limited by the home server.
     * @param {boolean} isRateLimited The flag which is set to true to enable rate
     * rate limiting, false to disable.
     */
    setRateLimited(isRateLimited: boolean): void;
    /**
     * Get the desired user_id localpart for the app service itself.
     * @return {?string} The user_id localpart ("alice" in "@alice:domain")
     */
    getSenderLocalpart(): string | null;
    /**
     * Add a regex pattern to be registered.
     * @param {String} type : The type of regex pattern. Must be 'users', 'rooms', or
     * 'aliases'.
     * @param {String} regex : The regex pattern.
     * @param {Boolean} exclusive : True to reserve the matched namespace.
     * @throws If given an invalid type or regex.
     */
    addRegexPattern(type: "users" | "rooms" | "aliases", regex: string, exclusive?: boolean): void;
    /**
     * Output this registration to the given file name.
     * @param {String} filename The file name to write the yaml to.
     * @throws If required fields hs_token, as_token, url are missing.
     */
    outputAsYaml(filename: string): void;
    /**
     * Get the key-value output which should be written to a YAML file.
     * @return {Object}
     * @throws If required fields hs_token, as-token, url, sender_localpart are missing.
     */
    getOutput(): {
        id: string;
        hs_token: string;
        as_token: string;
        namespaces: {
            users: RegexObj[];
            aliases: RegexObj[];
            rooms: RegexObj[];
        };
        url: string;
        sender_localpart: string;
        rate_limited: boolean;
        protocols: string[] | null;
        "de.sorunome.msc2409.push_ephemeral": boolean;
    };
    /**
     * Check if a user_id meets this registration regex.
     * @param {string} userId The user ID
     * @param {boolean} onlyExclusive True to restrict matching to only exclusive
     * regexes. False to allow exclusive or non-exlusive regexes to match.
     * @return {boolean} True if there is a match.
     */
    isUserMatch(userId: string, onlyExclusive: boolean): boolean;
    /**
     * Check if a room alias meets this registration regex.
     * @param {string} alias The room alias
     * @param {boolean} onlyExclusive True to restrict matching to only exclusive
     * regexes. False to allow exclusive or non-exlusive regexes to match.
     * @return {boolean} True if there is a match.
     */
    isAliasMatch(alias: string, onlyExclusive: boolean): boolean;
    /**
     * Check if a room ID meets this registration regex.
     * @param {string} roomId The room ID
     * @param {boolean} onlyExclusive True to restrict matching to only exclusive
     * regexes. False to allow exclusive or non-exlusive regexes to match.
     * @return {boolean} True if there is a match.
     */
    isRoomMatch(roomId: string, onlyExclusive: boolean): boolean;
    _isMatch(regexList: RegexObj[], sample: string, onlyExclusive: boolean): boolean;
}
export {};
