package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````0!D``#0```"HD0````0`!30`(``)
M`"@`&P`:``$``'``B@```(H```"*```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$````````````````````,B@``#(H```4```````$`
M`0```)B-``"8C0$`F(T!`*@"``#`$@``!@```````0`"````H(T``*"-`0"@
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DF(T`
M`)B-`0"8C0$`:`(``&@"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`UR41Z\4A5^3`Z]C7SU(;>VNIS[,$
M````$`````$```!'3E4```````,````"`````````#T```!/````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PXH!)%B@``"```````````3P``
M``````!0````40```%,`````````5`````````!5````5@```%<```!8````
M`````%D```!;`````````````````````````````````````````%P```!=
M````7P```&$```````````````````!B`````````&,`````````9````&4`
M````````9@```&<`````````:````&D`````````````````````````````
M`&H``````````````&L`````````;```````````````;0``````````````
M``````````"[XY)\:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PTZ/@^TG[
MH6Q@`*\TZ+_:5QO59-1\P^4_+'ZIG)NM2^/`XD?&`<_>1RT-%.*>(?_D#N&G
M`@K;S>.>6>^7>7=4/)2S`8MO#3J7"]EQ6!PS8MOM:PE#UD-%U>QK?YI\````
M``````````````````````!X$P````````,`"P```````)`!```````#`!8`
M;@(`````````````$@````\#`````````````!(```#C`@`````````````B
M````*`,`````````````$0```-`#`````````````!(````!````````````
M```@````U`$`````````````$@```,("`````````````!(``````P``````
M```````2````7P$`````````````$@```$0#`````````````!$```!C````
M```````````2````B0(`````````````$@```,4``````````````!(```!U
M`@`````````````2````!@$`````````````$@```%H!`````````````!(`
M``"0`@`````````````2````4@(`````````````$@```!@"````````````
M`!(````R`P`````````````2````*0(`````````````$@```#8"````````
M`````!(```"X`0`````````````1````%0$`````````````$@```$8`````
M`````````!$```#I```````````````2````R0(`````````````$@```+L"
M`````````````!$```"<```````````````2````.@$`````````````$@``
M`.`!`````````````!(```!D`0`````````````2````J0(`````````````
M$@```-D!`````````````!(```!P`0`````````````2````%@,`````````
M````$@```%,``````````````!(````=```````````````@````2P(`````
M````````$@````("`````````````!(```!@`@`````````````2````=P$`
M````````````$@```-("`````````````!(```"N```````````````2````
MPP$`````````````$@```/("`````````````!(```!%`0`````````````2
M````N@``````````````$@```#`"`````````````!(```!G`@``````````
M```2````EP$`````````````$@```*H!`````````````!(```!_````````
M```````2````W@``````````````$@```'L"`````````````!(```"&`0``
M```````````2````1`(`````````````$@````@#`````````````!(````C
M`0`````````````2````60(`````````````$@```'$``````````````!(`
M``"+```````````````2````*0,`````````````(0```-D"````````````
M`!(```#W```````````````2````U0``````````````$@```"P`````````
M`````"`````J`P`````````````1````L0(`````````````$@````<"````
M`````````!(````^`@`````````````2````_`$`````````````$@```)X"
M`````````````!(````/`@`````````````2````Z`$`````````````$@``
M`'4$``!8H`$``````!``%P"/!```:$H``$`!```2``T`GP,```@D```$````
M$@`-`!X$``!<20``#`$``!(`#0#:!````$D``%P````2``T`!00``-PF```H
M````$@`-`-D#``!T.P``D`$``!(`#0!K!```/)`!``0````1`!8`@@0```0G
M``"8!0``$@`-`+H$``!L+0``7`$``!(`#0#M`P``6*`!```````0`!<`[`,`
M`%B@`0``````$``7`.,$```T-P``0`0``!(`#0"E!```Q"(``%`````2``T`
MO@,``%0M```8````$@`-`#8$```L3```!````!$`#P#X`P``M"8``"@````2
M``T`$00``,@N``!L"```$@`-`(\#```,3```!````!(`#0"M!```#"0``*@"
M```2``T`100``!0C``#T````$@`-`%L$``"H2P``9````!(`#0!Z!```6*`!
M```````0`!<`L`,``$"0`0``````$``7`,P$``"<+```N````!(`#0`K!```
M`)`!```````@`!8`F00``$"0`0``````$``7`"\$``!`&0```````!(`#0`I
M!````)`!```````0`!8`Y0,``$"0`0``````$``6`",#``"8%@``J`(``!(`
M#0``7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%]?9VUO;E]S=&%R=%]?
M`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?
M9W9?9F5T8VAP=@!097)L7V%V7W-H:69T`%!E<FQ?879?9F5T8V@`4&5R;%]M
M9U]S970`4&5R;%]A=&9O<FM?;&]C:P!097)L7W-V7S)P=E]F;&%G<P!097)L
M7V=E=%]S=@!097)L7VYE=UA3`&)O;W1?1'EN84QO861E<@!P97)L7W)U;@!P
M97)L7W!A<G-E`%!E<FQ?<W9?<V5T<'8`4&5R;%]S=E]S971P=FX`<&5R;%]C
M;VYS=')U8W0`<&5R;%]D97-T<G5C=`!097)L7V-R;V%K7VUE;6]R>5]W<F%P
M`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N;V-O;G1E>'0`<&5R;%]F<F5E`%!E
M<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y<U]I;FET,P!097)L7VYE=UA37V9L
M86=S`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%!,7W1H<E]K
M97D`7U]E<G)N;U]L;V-A=&EO;@!R96%D`&]P96XV-`!L<V5E:S8T`'!T:')E
M861?9V5T<W!E8VEF:6,`8VQO<V4`97AI=`!S=')N8VUP`&-L;W-E9&ER`%]?
M<W1A8VM?8VAK7V9A:6P`=6YL:6YK`&UK9&ER`')E86QL;V,`86)O<G0`;65M
M8VAR`&=E='!I9`!S=')D=7``<W1R=&]K`'-T<FQE;@!M96US970`<W1R<W1R
M`')M9&ER`%]?9G!R:6YT9E]C:&L`;65M8W!Y`%]?<W1R;F-P>5]C:&L`7U]L
M>'-T870V-`!O<&5N9&ER`%]?>'-T870V-`!S=&1E<G(`9V5T=6ED`')E861L
M:6YK`'-T<F-H<@!R96%D9&ER-C0`7U]C>&%?9FEN86QI>F4`7U]S<')I;G1F
M7V-H:P!M96UM;W9E`&%C8V5S<P!S=')C;7``7U]L:6)C7W-T87)T7VUA:6X`
M7U]E;G9I<F]N`%]?<F5G:7-T97)?871F;W)K`%]?<W1A8VM?8VAK7V=U87)D
M`&QI8G!E<FPN<V\`;&EB<'1H<F5A9"YS;RXP`&QI8F,N<V\N-@!L9"UL:6YU
M>"UA<FUH9BYS;RXS`%]?;&EB8U]C<W5?9FEN:0!P87)?8W5R<F5N=%]E>&5C
M`%]?8G-S7W-T87)T7U\`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R
M7V-L96%N=7``7V5D871A`%]?8G-S7V5N9%]?`'!A<E]B87-E;F%M90!P87)?
M9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!?7V1A=&%?<W1A<G0`
M7TE/7W-T9&EN7W5S960`<&%R7V-U<G)E;G1?97AE8U]P<F]C`%]?;&EB8U]C
M<W5?:6YI=`!M>5]P87)?<&P`7V5N9`!?7V5N9%]?`'!A<E]I;FET7V5N=@!S
M:&%?9FEN86P`7U]B<W-?<W1A<G0`>'-?:6YI=`!P87)?9FEN9'!R;V<`<&%R
M7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A;@!S:&%?:6YI=`!84U]);G1E
M<FYA;'-?4$%27T)/3U0`1TQ)0D-?,BXT`"]U<W(O;&EB+W!E<FPU+S4N,S(O
M8V]R95]P97)L+T-/4D4``````````@`"``(``@`"`````P`"``(``@`$````
M`@````(```````(``@`"``(``@`"`````````````@`"```````#`````@`#
M``(``@```````@`"``(````"`````P`"```````"``(````````````"````
M`@`"`````@```````@`"``````````(``@`"``(``P`"``(``P`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!`````0`!`'L#```0````(````!1I:0T```0`^00````````!
M``$`80,``!`````@````%&EI#0```P#Y!`````````$``0!Q`P``$```````
M```4:6D-```"`/D$````````F(T!`!<```"<C0$`%P```,2/`0`7````Z(\!
M`!<```#PCP$`%P````20`0`7````#)`!`!<````0D`$`%P```!B0`0`7````
M')`!`!<````@D`$`%P```"20`0`7````*)`!`!<````PD`$`%P```#20`0`7
M````/)`!`!<```#(CP$`%04``,R/`0`5"```T(\!`!4-``#4CP$`%1```-B/
M`0`5&@``W(\!`!4<``#@CP$`%1\``.2/`0`5*0``[(\!`!4V``#TCP$`%4$`
M`/B/`0`51@``_(\!`!5'``#$C@$`%@,``,B.`0`6!```S(X!`!8%``#0C@$`
M%@<``-2.`0`6"0``V(X!`!8*``#<C@$`%@L``.".`0`6#```Y(X!`!8.``#H
MC@$`%@\``.R.`0`6$0``\(X!`!82``#TC@$`%A,``/B.`0`6%```_(X!`!85
M````CP$`%A8```2/`0`6%P``"(\!`!88```,CP$`%AD``!"/`0`6&P``%(\!
M`!8=```8CP$`%AX``!R/`0`6(```((\!`!8A```DCP$`%B(``"B/`0`6(P``
M+(\!`!8D```PCP$`%B4``#2/`0`6)@``.(\!`!8G```\CP$`%B@``$"/`0`6
M*0``1(\!`!8J``!(CP$`%BL``$R/`0`6+```4(\!`!8M``!4CP$`%BX``%B/
M`0`6+P``7(\!`!8P``!@CP$`%C$``&2/`0`6,@``:(\!`!8S``!LCP$`%C0`
M`'"/`0`6-0``=(\!`!8W``!XCP$`%C@``'R/`0`6.0``@(\!`!8Z``"$CP$`
M%CL``(B/`0`6/```C(\!`!8]``"0CP$`%CX``)2/`0`6/P``F(\!`!9```"<
MCP$`%D,``*"/`0`61```I(\!`!9%``"HCP$`%D@``*R/`0`620``L(\!`!9*
M``"TCP$`%DL``+B/`0`63```O(\!`!9-``#`CP$`%DX```A`+>F%`0#K"("]
MZ`3@+>4$X)_E#N"/X`CPON4D>P$``,:/XA?*C.(D^[SE`,:/XA?*C.(<^[SE
M`,:/XA?*C.(4^[SE`,:/XA?*C.(,^[SE`,:/XA?*C.($^[SE`,:/XA?*C.+\
M^KSE`,:/XA?*C.+T^KSE`,:/XA?*C.+L^KSE`,:/XA?*C.+D^KSE`,:/XA?*
MC.+<^KSE`,:/XA?*C.+4^KSE`,:/XA?*C.+,^KSE`,:/XA?*C.+$^KSE`,:/
MXA?*C.*\^KSE`,:/XA?*C.*T^KSE`,:/XA?*C.*L^KSE`,:/XA?*C.*D^KSE
M`,:/XA?*C.*<^KSE`,:/XA?*C.*4^KSE`,:/XA?*C.*,^KSE`,:/XA?*C.*$
M^KSE`,:/XA?*C.)\^KSE`,:/XA?*C.)T^KSE`,:/XA?*C.)L^KSE`,:/XA?*
MC.)D^KSE`,:/XA?*C.)<^KSE`,:/XA?*C.)4^KSE`,:/XA?*C.),^KSE`,:/
MXA?*C.)$^KSE`,:/XA?*C.(\^KSE`,:/XA?*C.(T^KSE`,:/XA?*C.(L^KSE
M`,:/XA?*C.(D^KSE`,:/XA?*C.(<^KSE`,:/XA?*C.(4^KSE`,:/XA?*C.(,
M^KSE`,:/XA?*C.($^KSE`,:/XA?*C.+\^;SE`,:/XA?*C.+T^;SE`,:/XA?*
MC.+L^;SE`,:/XA?*C.+D^;SE`,:/XA?*C.+<^;SE`,:/XA?*C.+4^;SE`,:/
MXA?*C.+,^;SE`,:/XA?*C.+$^;SE`,:/XA?*C.*\^;SE`,:/XA?*C.*T^;SE
M`,:/XA?*C.*L^;SE`,:/XA?*C.*D^;SE`,:/XA?*C.*<^;SE`,:/XA?*C.*4
M^;SE`,:/XA?*C.*,^;SE`,:/XA?*C.*$^;SE`,:/XA?*C.)\^;SE`,:/XA?*
MC.)T^;SE`,:/XA?*C.)L^;SE`,:/XA?*C.)D^;SE`,:/XA?*C.)<^;SE`,:/
MXA?*C.)4^;SE`,:/XA?*C.),^;SE`,:/XA?*C.)$^;SE`,:/XA?*C.(\^;SE
M`,:/XA?*C.(T^;SE`,:/XA?*C.(L^;SE<$`MZ5#"G^4@T$WB3#*?Y0S`C^!(
M0I_E%`"-Y10`C>(0$(WE$!"-X@P@C>4,((WB!$"/X`,PG.<`,)/E'#"-Y0`P
MH..6___K&"*?Y1@RG^4"()3G`P"4YP(0H.%%#0#K"#*?Y0,PE.<`4-/E``!5
MXU<```KX,9_E`T"4YP``E.7;___K`#"@X0``E.6>(]/E`B""XYXCP^75___K
M%#"=Y9XCT.4%,(/B`2""XP<!<^.>(\#E9```B@,!H.%:___KL"&?Y1!`G>4"
M((_@J#&?Y0`0E.4#,(_@`""2Y0@@@.68(9_E`!"`Y0(@C^`,((#E%""=Y8@1
MG^4!`%+C"`"#Y0$0C^`,P(#"!!"`Y000H.,$$(/E!!"$P@)!A,`#``#:!."1
MY`3@K.4!`%3A^___&@T1@N),49_E`#"@XP`PC>4%4(_@`R""X@$Q@.<`,*#A
M-!&?Y1``E>4!$(_@<?__ZP!`4.(G```*(%&?Y1A@C>(%4(_@$`"5Y1G__^L0
M`9_E!A"@X0``C^"E``#K``!0XR````K\,)_E`S"/X!``D^7Z_O_K6?__Z^P@
MG^6L,)_E`B"/X`,PDN<`()/E'#"=Y0(@,^`C```:!`"@X2#0C>)P@+WH#O__
MZ\`@G^4``%#C`B"/X!``@N48```*Y/[_ZWPPG^4#0)3G``"4Y7S__^M)5\#E
MG/__ZA``E>5C___K`$"@X=/__^H,`)7E``!0XP$```JL"`#KV?__ZG``G^4&
M$*#A``"/X'D``.L``%#C#`"%Y=+__PKU___J1/__ZP$`H.,,___KU?[_ZPAX
M`0`8`0``Z'<!`#0!```\`0``)`$``"`!``#4>`$`T'@!`-PW``#`-P``9'@!
M`#0*```T>`$`!#4```QX`0!H=@$`Q'<!`/0S````L*#C`."@XP00G>0-(*#A
M!"`MY00`+>4HH)_E)#"/X@.@BN`@P)_E#,":YP3`+>48,)_E`S":YQ0`G^4`
M`)KGV_[_ZS3__^LP=0$`#`$``#`!```X`0``%#"?Y10@G^4#,(_@`B"3YP``
M4N,>_R\!U?[_ZA!U`0`L`0``+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/
MX![_+P$8,)_E`S"2YP``4^,>_R\!$_\OX71V`0!L=@$`W'0!`!0!```X`)_E
M.!"?Y0``C^`T()_E`1"/X``00>`"((_@H3^@X4$1@^#!$+#A'O\O`1@PG^4#
M,)+G``!3XQ[_+P$3_R_A,'8!`"AV`0"8=`$`0`$``$PPG^5,()_E`S"/X`(@
MC^``,-/E``!3XQ[_+Q$X,)_E$$`MZ0,PDN<``%/C`@``"B@PG^4#`)_G2/[_
MZ\K__^L<,)_E`2"@XP,PC^``(,/E$("]Z.!U`0!4=`$`$`$``'AU`0"@=0$`
MT/__ZLPPG^7P1RWI`)!0X@,PC^`K```*O""?Y0(PD^<`@)/E``!8XR@```H`
M4-GE`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"8Y0``5.,,```*"&"@X00`
MH.$%(*#A"1"@X<O^_^L``%#C`@``&@4PU.<]`%/C"P``"@1`MN4``%3C\___
M&@0`H.'PA[WH"5"@X0$P]>4``%/C/0!3$_O__QH)4$7@YO__Z@A@1N`!4(7B
M!4"$X$9AH.$$`*#A`&"'Y?"'O>@)0*#A[?__Z@A`H.'K___JZ',!`$0!``!D
M(9_E9#&?Y0(@C^#P02WI`%!0XGC03>(#,)+G`#"3Y70PC>4`,*#C`@``"@`P
MU>4``%/C"0``&C0AG^4L,9_E`B"/X`,PDN<`()/E=#"=Y0(@,^!#```:>-"-
MXO"!O>@T_O_K`$!0XO+__PH$<9_E!W"/X`0`H.&"_O_K``!0XRH```H3,-#E
M$V"`XBX`4^,"```:`3#6Y0``4^/T__\*$S#0Y2X`4^,E```*!0"@X3C^_^L`
M@*#A!@"@X37^_^L``(C@`@"`XB#^_^L',*#A`1"@XP1@C>4`(.#C`&"@X0!0
MC>4Z_O_K"""-X@80H.$#`*#C;/[_ZP$`<.,&`*#A`P``"A@PG>4/.@/B`0E3
MXQ(```KL_?_K!@"@X<S]_^L$`*#A5_[_ZP``4./4__\:!`"@X6O^_^L%`*#A
MS?W_Z[S__^H!,-;E+@!3X];__QH",-;E``!3X\7__PK2___JIO__Z^O__^K1
M_?_K$',!`!@!``#8<@$`6#```)3`G^64,)_E#,"/X!!`+>D`0*#A<-!-X@,`
MH.,#,)SG#2"@X000H.$`,)/E;#"-Y0`PH.-)_O_K``!0XP``H!,$```:$#"=
MY0\Z`^(*"E/C`0E3$PD```I$()_E/#"?Y0(@C^`#,)+G`""3Y6PPG>4"(#/@
M!P``&G#0C>(0@+WH`A"@XP0`H.$3_O_K$`]OX:`"H.'O___JIOW_ZY1Q`0`8
M`0``/'$!`!!`+>D`0*#AV?W_ZP$P0.(#,(3@`P!4X0,@H#$#```Z)@``Z@(P
MH.$$`%+A(P``"@(PH.$!$%+D+P!1X_C__PH#`%3A'0``*@$P0^(#`%3A'```
M"@`@T^4O`%+C^?__&@$P0^(#``#J`!#2Y0$P0^(O`%'C`@``&@,@H.$#`%3A
M^/__.@0@0N`",(+B`0I3XP@``(I8`)_E!!"@X0$ZH.,!((+B``"/X`!`H.%Q
M_?_K!`"@X1"`O>@``*#C$("]Z`,`5.'E__\:`##4Y2\`4^,#```*'`"?Y1!`
MO>@``(_@9_W_ZA``G^400+WH``"/X&/]_^KP<0$`T"T``(@M``#P0RWI`$"@
MX1`!G^4!8*#A#!&?Y0``C^`!VDWB#-!-X@"!G^4!.HWB!'"-X@$0D.?_+P#C
M\`"?Y00P@^(`$)'E`!"#Y0`0H.,'$*#A``"/X`B`C^!?_?_K`,!0XB4``-K(
M,)_E"%"-X@,PF.<!@$SB"%"%X`"0D^4$,%7E``!3XQH``!H``%CC'0``"@@@
MH.$O$*#C!P"@X:+]_^L``%#C%P``"@D`H.'(_?_K"#"@X0<@H.$$$*#AK/W_
MZW`0G^4!.HWB6""?Y00P@^(!$(_@`B"1YP`0DN4`()/E`1`RX`T``!H!VHWB
M#-"-XO"#O>@,@*#AY/__ZC`PG^4#,)CG`)"3Y0D`H.&P_?_K!B"@X000H.$L
M_?_KY___ZAO]_^O@;P$`&`$``*1O`0!D+0``(`$``"AO`0`8(I_E&#*?Y0(@
MC^#P1RWI`4"@X0C03>(`4*#A`S"2YP""G^4`,)/E!#"-Y0`PH.,`,-'E"("/
MX#T`4^,!0($"!`"@X3S]_^L-$*#A`&"@X04`H.&;_O_K`)!0XD````HU_?_K
M``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@$P`PH`,%
M```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@#?W_ZP``
M4.,'`(;G%@``"@`PF.4%`*#A`""=Y0(1D^<!,-#D`##!Y0``4^,]`%,3!@``
M"@$@@>(!,-#D`A"@X0$PPN0]`%/C``!3$_G__QH!(*#A/3"@XP`PP>4!,-3D
M`3#BY0``4^/[__\:!"&?Y?0PG^4"((_@`S"2YP`@D^4$,)WE`B`SX#4``!H(
MT(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[___ZL`PG^4#@)CG`'"8Y0`P
ME^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!$:#AD#"?
MY0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^7,_/_K`*!0XM;__PH'$*#A
M"7&@X0<@H.&$_/_K`*"(Y0$PB>(`(*#C`R&*YP"0C>6B___J!P"@X:#\_^L`
MH*#A``!0XP``B.7&__\*"7&@X?+__^H#D*#A"!"@X]___^J-_/_KM&X!`!@!
M``"(;@$`1`$``)!M`0"H;@$`,$`MZ6Q`G^5L,)_E#-!-X@1`C^`#4)3G``"5
MY0_]_^L``)7E#?W_ZU`@G^50,)_E4!"?Y0,PC^`"()3G`1"/X,3\_^L``)7E
M!/W_ZS@PG^4X()_E`,"@XS00G^4#,(_@`B"/X`@0C>@!$(_@SOS_ZPS0C>(P
M@+WH=&P!`"`!```<`0``:&<``!0J``#,+```H!0```0J```00"WI,$"?Y3`P
MG^4$0(_@`S"4YP``D^7J_/_K(""?Y2`PG^4@$)_E`S"/X`(@E.<!$(_@$$"]
MZ*#\_^K@:P$`(`$``!P!``#<9@``B"D``-@0G^4P0"WIU""?Y0$0C^`"VDWB
M)-!-X@(ZC>(40(WB`B"1YQPP@^(`()+E`""#Y0`@H.-R_/_KJ!"?Y0$@`>.D
M,)_E`1"/X/``S>$#,(_@`1"@XP0`H.%^_/_K`1J-XA@0@>($`*#A`2J@XPP0
MC>5"_/_K`$!0X@!0H+,(``"Z`0"$XE+\_^L`4%#B!```"@P0G>4$(*#A%/S_
MZP`PH.,$,,7G1!"?Y0(ZC>(P()_E'#"#X@$0C^`"()'G`!"2Y0`@D^4!$#+@
M`P``&@4`H.$"VHWB)-"-XC"`O>@5_/_KD&L!`!@!``!D*0``4"D``.AJ`0#!
M___J;,*?Y?!/+>D`8*#A9"*?Y0S`C^!"W4WB#-!-XE@"G^5"/8WB&'"-X@(@
MG.<!4*#A``"/X`<0H.$$,(/B`""2Y0`@@^4`(*#CF/W_ZR\0H.,`0*#A!@"@
MX3?\_^L``%#C$0``"A0"G^4&$*#A``"/X-S^_^L&`*#A!!*?Y4(]C>+P(9_E
M!#"#X@$0C^`"()'G`!"2Y0`@D^4!$#+@<P``&D+=C>(,T(WB\(^]Z`8`H.$;
M_/_KS!&?Y0"`H.$%`*#A`1"/X$S\_^L`H%#BY/__"K0QG^4*L*#AL)&?Y0,P
MC^`4,(WEJ#&?Y0F0C^`#,(_@$#"-Y0`PV^4``%/C.P``"@``5.,$```*"Q"@
MX00`H.&@^__K``!0XS\```H+`*#A__O_ZP'`0.(`X*#A##"+X`P@V^<O`%+C
M"@``&@(`0.(#(*#A``"+X`/`3.`#4$#@`*"@XPS@@^`%H,/F`1!RY2\`4>/Z
M__\*`3"(X@[@@^`!"E[CNO__JA@AG^6$4(WB"3"@X0$0H.,"((_@!0"@X40`
MC>D!*J#C`+"-Y>_[_^L'(*#A!1"@X0,`H.,A_/_K``!0XP,``!HH,)WE#SH#
MX@()4^,:```*S!"?Y0``H.,!$(_@"/S_ZP"PH.$``%OCG___"@`PV^4``%/C
MP___&@``5.,:```*`##4Y2X`4^,"```*%+"=Y0'@H./5___J`3#4Y0``4^/Y
M__\:?!"?Y0``H.,!$(_@\_O_ZP"PH.'I___J`1"@XP4`H.'H^__K``!0X]__
M_QI4`)_E!1"@X0``C^!C_O_K!0"@X73[_^N$___J$+"=Y0'@H..]___J<OO_
MZY1J`0`8`0``A"@``%0H```<:@$`$"@``$@G``#L)P``."<``-0F````)P``
MK"8``'`F````,-#E``!3XQ[_+P$!((#B+P!3XP(`H`$!,-+D``!3X_K__QH>
M_R_A`#!0X@,```H`,-/E``!3XP````JQ_?_J!`"?Y0``C^!/^__J."4``!PE
MG^4<Q9_E'#6?Y0(@C^!P0"WI#,"/X`S`C.(PT$WB"&6?Y0,PDN<(X(WB!F"/
MX`1`C>(`,)/E+#"-Y0`PH./L5)_E#P"\Z`\`KN@%4(_@#P"\Z`\`KN@`,)SE
ML##.X000H.$&`*#ATOS_ZP``4.,.```*O#2?Y00`G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JA&2?Y09@
MC^`$$*#A!@"@X;S\_^L``%#C#@``"F0TG^4$`)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZC!DG^4&8(_@
M!!"@X08`H.&F_/_K``!0XPX```H,-)_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^K<8Y_E!F"/X`00
MH.$&`*#AD/S_ZP``4.,.```*M#.?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JB&.?Y09@C^`$$*#A
M!@"@X7K\_^L``%#C#@``"EPSG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><`
M`%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZC1CG^4&8(_@!!"@X08`
MH.%D_/_K``!0XPX```H$,Y_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2
MX_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^K@8I_E!F"/X`00H.$&`*#A
M3OS_ZP``4.,.```*K#*?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q
M__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JC&*?Y09@C^`$$*#A!@"@X3C\
M_^L``%#C#@``"E0RG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__
M"@0P@^($()/E!""#Y```4N/[__\:Z___ZCABG^4&8(_@!!"@X08`H.$B_/_K
M``!0XPX```K\,9_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$
M,(/B!""3Y00@@^0``%+C^___&NO__^KD89_E!F"/X`00H.$&`*#A#/S_ZP``
M4.,.```*I#&?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#
MX@0@D^4$((/D``!2X_O__QKK___JD&&?Y09@C^`$$*#A!@"@X?;[_^L``%#C
M#@``"DPQG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($
M()/E!""#Y```4N/[__\:Z___ZCP!G^4$$*#A``"/X.'[_^L`$%#B`@``"B@!
MG^4``(_@*OW_ZR`!G^4$$*#A``"/X-C[_^L`$%#B`@``"@P!G^4``(_@(?W_
MZP0!G^4$$*#A``"/X,_[_^L`$%#B$0``"O``G^4``(_@&/W_Z^@0G^7H`)_E
M`1"/X```C^`3_?_KW""?Y8`PG^4"((_@`S"2YP`@D^4L,)WE`B`SX!<``!HP
MT(WB<("]Z+@`G^4$$*#A``"/X+?[_^L`$%#B`P``"J0`G^4``(_@`/W_Z^;_
M_^J8$)_E"B"@XQD`C>(!$(_@D?K_ZP``4./?__\:@`"?Y2,0C>(``(_@]/S_
MZ]K__^H(^O_KH&<!`+!B```8`0``N"4``&!G`0!$`0``/"4``.PD``"@)```
M4"0```@D``!0(P``9",``!@C``#,(@``_"$``"0B``#X(0``%"(``-`@```$
M(@``."$``!`B``!T(0``!&,!`+0A``!$(0``G"$``!0A``"8,)_EF!"?Y9@@
MG^4#,(_@`1"/X`3@+>4``)/E#-!-X@(@D><!`'#C`""2Y00@C>4`(*#C"0``
M"FP@G^5D,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`0```:#-"-X@3PG>1(`)_E
M#1"@X0``C^!K^__K``!0XP0```H``-#E,`!0XP``4!,!`*`3``"@`R`PG^4#
M,(_@``"#Y>;__^K#^?_K6&,!``1B`0`8`0``U&$!`"@@``#48@$`$$`MZ9WY
M_^N6^?_K``!0XP``D!40@+WH,"&?Y3`QG^4"((_@\$`MZ2AAG^4<T$WB`%"@
MX0,PDN<&8(_@$!"-X@8`H.$`,)/E%#"-Y0`PH.-$^__K`$!0X@(```H`,-3E
M``!3XPT``!KL()_EX#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@,```&M``G^4%
M$*#A``"/X!S0C>+P0+WH?_S_Z@40H.%F^?_K``!0XPD```JL()_EF#"?Y0(@
MC^`#,)+G`""3Y10PG>4"(#/@'@``&AS0C>+P@+WH!0"@X;[Y_^L`<*#A!`"@
MX;OY_^L``(?@`@"`XJ;Y_^MH,)_E:""?Y0$0H.,#,(_@"$"-Y0(@C^``4(WE
M!""-Y0`@X.,`0*#AO/G_ZT0@G^4D,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^`$
M$*`!!@"@`='__PII^?_K/&$!`!@!```@(```[&`!`,@?``"D8`$`B!X``'P>
M```P8`$`\$\MZ<P7G^6!W$WBR#>?Y0303>+$IY_E`S"/X`%\C>("*8WB=(!'
MXAP`C>40X(/BK`>?Y?P@@N(H4(/B7,!'X@``C^!H0$?BX&!'X@J@C^`!$)#G
M`!"1Y0`0@N4`$*#C!P"3Z0<`B.@/`+[H#P"LZ`,`GN@#`(SH!P"5Z`<`A.A@
M!Y_E!A"@X0``C^#8^O_K`$!0X@(```H`,-3E``!3XR`!`!H8^?_K$?G_ZP"`
M4.+W```*`%"8Y0``5>/S```*!0"@X6?Y_^N``*#A`0"`XE+Y_^L`(-7E`("@
MX0``4N,,```*`+>?Y0A`H.$+L(_@!`"@X0`@C>4+,*#A`"#@XP$0H.,"0(3B
M9?G_ZP$@]>4``%+C]?__&M`&G^584$?B``"/X`(``.H$`)7D``!0XPT```H&
M$*#AK?K_ZP!`4.+X__\*`##4Y0``4^/U__\*/?O_ZP``4./R__\*!`"@X03Y
M_^L`4%#B$P``&@"0H.-TL$?B"2&'X'1`$N4``%3C#```"@%0B>(%48O@`"#4
MY0``4N,'```*!`"@X0&0B>(I^__K``!0XT,``!H$0)7D``!4X_3__QH`4*#C
M!0"@X2CY_^L`0*#A"`"@X27Y_^L``(3@`4N`X@1`A.($`*#A#OG_ZP@VG^4!
M$*#C#("-Y0,PC^`(,(WE^#6?Y0"`H.$`(.#C`%"-Y0,PC^`$,(WEY#6?Y0,P
MC^`A^?_K!QV@XP@`H.$G^?_K`0!PXP,``!H8^?_K`!"0Y1$`4>.L```:!B"@
MX0@0H.$#`*#C5OG_ZP!04.(=```*H!6?Y1PPG>6<)9_E`0":YP"@H.,`,)/E
M`B"/X`"`C>4!$*#C``"0Y1_Y_^M\%9_E`CF-XD0EG^7\,(/B`1"/X`(@D><`
M$)+E`""3Y0$0,N!)`0`:"@"@X8'<C>($T(WB\(^]Z`0`H.&L^/_K`%!0XJG_
M_PJZ___JT#`7Y0\Z`^(!"5/CW?__&LBP%^6(^/_K``!;X=G__QK0,!?E4S#H
MYP<-4^/5__\:!`"@X<7X_^L`H*#A^`2?Y080H.$``(_@-?K_ZP``4..7```*
M'#"=Y0!`D^61^/_K`!"@X00`H.&!_/_K`+!0XH\```H+`*#AQOC_ZP!`H.$#
M`%#C#```FK`4G^4$`$#B``"+X`$0C^!;^/_K``!0XP4``!H$`%3CX0``"@1`
MB^`%,%3E+P!3X]T```J`1)_E!$"/X``PE.4!`'/C>P``"@``4^.)```*:!2?
MY6@$G^4!$(_@9)2?Y0``C^!@=)_E6?O_ZUQDG^6@^/_K6#2?Y0@`C>4!$*#C
M`S"/X`PPC>5(-)_E"@"@X0"`C>4`(.#C`S"/X`0PC>4T-)_E"9"/X`=PC^`&
M8(_@`S"/X*3X_^L/``#JG_C_ZP`PD.41`%/C$0``&HGX_^L!4(7B`,"@X0DP
MH.$$(*#A`1"@XPH`H.$0<(WE#%"-Y4`0C>D`@(WED_C_ZP<=H.,*`*#AF?C_
MZP!`H.$!`'#CZ?__"@@`H.$K^/_KO`.?Y0H0H.$``(_@*?O_ZPH`H.&%_O_K
M?/__Z@6`H.%H0$?B"#&'X&@`$^4``%#C+P``"@&PB.(+483@`P``Z@0`E>0`
M`%#C*0``"@&PB^(&$*#A"X"@X<CY_^L`,%#B]O__"@`PT^4``%/C\___"B/X
M_^L`4%#B\_[_&NC__^H$(Y_E'#"=Y0(`FN<`H*#C`#"3Y2PCG^4$$(WE`1"@
MXP"`C>4"((_@``"0Y7?X_^M6___J7?[_ZPP3G^6@(I_E`CF-X@$0C^#\,(/B
M`B"1YP`0DN4`()/E`1`RX)\``!H$`*#A@=R-X@30C>+P3[WH`_C_ZM12G^4-
M`*#C*OC_ZU,@H.,`@*#A!5"/X-?^_^H<,)WE`+"3Y6S__^JP`I_E!A"@X0``
MC^"4^?_K``!0XP``A`4(```*`##0Y0``4^,P`%,3`3"@$P`PH`,!,*`3`#"$
M!0`PA!5U__\:"P"@X0`0H.,-^/_K`$!0XG#__PH`,.#C$2#@XP)@H.,$4$?B
M`&"-Y?SW_^L&(*#C!1"@X00`H.&\]__K!#!7Y0``4^,"```:`S!7Y4,`4^-'
M```*`#"@XP`@H.,`,(WE!`"@X0`PH./L]__K_00`ZP!@H.$"``#J!1"@X08`
MH.$/!0#K`BF@XP40H.$$`*#AIO?_ZP`@4.+V___*V+&?Y00`H.$PD$?B2/C_
MZT501^(&$*#A"["/X`E`H.$Q8$?B1`!'XD$%`.L!(/7E!`"@X0`@C>4+,*#A
M`1"@XP`@X./[]__K`D"$X@4`5N'U__\:B"&?Y0$0H.,(D(WE"@"@X0(@C^``
M@(WE!""-Y0`@H.-L,9_E""!'Y6@AG^4#,(_@`B"/X`P@C>4`(.#CZ/?_ZUG_
M_^H<,)WE!$"3Y0``5.,=__\*!`"@X=+W_^L#`%#C&?__FC`1G^4$`$#B``"$
MX`$0C^!H]__K``!0XP2PH`$1___J`C!7Y4$`4^.T__\:`3!7Y4,`4^.Q__\:
M`##7Y4@`4^.N__\:`3#7Y44`4^.K__\:`##@XP0`H.$`8(WE.2#@XYGW_^L%
M$*#A*2"@XP0`H.%9]__KP#"?Y<`@G^4!$*#C`S"/X`PPC>6T,)_E`B"/X`H`
MH.$$((WE`S"/X`A0C>4`@(WE`"#@X[3W_^LE___J:O?_ZQ@!```@80$`G%\!
M`*A?`0!P'0``$!X``$P<```@'0``F!L``/`<```H`0``Y!P``%1=`0"4'```
M3!P``*A=`0`P&P``K!H``.0;``",'```G!D``+0<``"P&0``R!L``'P9````
M&@``L%H!`'@9``#,&```>!@``-`6``#4&```G!D``(@8``#H&```\!4```08
M``#8`Y_EV!.?Y0``C^#P3RWIT#.?Y0*++>T4T$WB`1"0YP,PC^#`(Y_E"&"-
MX@`0D>4,$(WE`!"@XQ!J".X"4)/G``"5Y</W_^L`0*#A``"5Y<#W_^L&$*#A
M9##4Y64PP.6(`Y_E``"/X,;X_^L``%#C`@``"@`PT.4Q`%/CNP``"M+[_^L`
M`)7ELO?_ZV`3G^4+,*#C`2"@XP$0C^!*]__K``!0XZ8```I(,Y_E#!"0Y0,P
MC^``L)7E&("1Y00PD^4!`%/C-0``VBRCG^4!FZ#C()!`XR0SG^4`0*#C"J"/
MX`,PC^``,(WE'0``Z@``D^4!8(3B"!":Y0"PE>4&<9'G"!"0Y0D0`>`!"U'C
M#!"0!0D```H+`*#A!#"-Y8WW_^L$,)WE`""@XP`0D^4",*#C#??_ZP"PE>4`
M$*#A!P"@X<KV_^L`,)WE``!0XP0PD^4/```:`D"$X@,`5.$&0*#A"P``J@L`
MH.%Z]__K`#"@XP0@H.$($*#A6/?_ZP`P4.+9__\:>#*?Y0"PE>4#,(_@!#"3
MY0$00^($`%'A60``"@L`H.%K]__K6!*?Y0,PH.,!(*#C`1"/X`/W_^L`0%#B
M"P``"@``E>5B]__K##"4Y30BG^4`$)/E`B"/X-SV_^L,,)3E`#"3Y0@PD^4!
M!1/C:P``&@``E>56]__K#!*?Y0,PH.,!(*#C`1"/X.[V_^L`0%#B"P``"O0!
MG^40&ACN``"/X%?X_^L`8%#B``"5Y5(```I']__K##"4Y08@H.$`$)/EPO;_
MZP``E>5!]__K`$"@X0``E>4^]__KM!&?Y0$@H.,!$(_@[/;_ZZ@AG^4`$*#A
M!`"@X0(@C^"T]O_K``"5Y911G^4R]__K`#"@X8P!G^4%4(_@`""@XQ`:&.X`
M`(_@92##Y3;X_^L`0*#A``!0XPP`A>4C```*!QV@XP0`H.'D]O_K`2"`X@`P
MH.$1`%#C`0!2$S@``(I$(9_E_#"?Y0(@C^`#,)+G`""3Y0PPG>4"(#/@+P``
M&A30C>("B[WL\(^]Z`$`4^.C___:%&&?Y0%`H.,+`*#A!F"/X````.H``)7E
M"O?_ZP@0H.%C]O_K!#"6Y0%`A.($`%/A]___R@"PE>64___J"`"5Y0_]_^L`
M0*#A``!0XPP`A>75__\:Q`"?Y0``C^"T]O_K`3#0Y0``4^-!__\*/___ZO3V
M_^NH,)_E#""4Y0,PC^```)+E"#"3Y0`0D^7Y^/_KJ/__Z@``E>7J]O_K##"4
MY0`0D^6Q]O_KC?__ZE7V_^MP$)_E!""@X6P`G^4!$(_@``"/X`@0D>4`$)'E
MF/;_ZW17`0`8`0``7%<!`"`!``!X%0``\!8``%A8`0`L6`$`*%@!`'Q7`0#<
M%0``O!4``+`5``!X$P``.!4``#P5``!T5@$`[!(``)A4`0#L50$`5!0``'Q5
M`0`\50$`$!0``&@AG^5H,9_E`B"/X/!`+>D`<%#B#-!-X@,PDN<`,)/E!#"-
MY0`PH.,Y```*`##7Y0``4^,V```*@OC_ZP!`H.$`,-3E``!3XP4```H!((3B
M+P!3XP)`H`$!,-+D``!3X_K__QH,89_E`%!7X@%0H!,&8(_@`#"6Y0$`<^,I
M```*``!3XP!0H`,!4`42``!5XPD``!K@()_EU#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@+0``&@S0C>+P@+WH`##7Y0``4^/R__\*L!"?Y00@H.,$`*#A`1"/
MX'_V_^L``%#CZ___&I@@G^6$,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`9```:
M!P"@X0S0C>+P0+WHP_?_ZFP`G^4``(_@ZO7_ZP!`H.'%___J7`"?Y0T0H.$`
M`(_@@_?_ZP``4.,``(8%T___"@`PT.4``%/C,`!3$P`PH`,!,*`3`#"&!<S_
M_PH!,*#C`#"&Y<?__^K9]?_K-%,!`!@!```<5`$`H%(!`)`1``!04@$`I`\`
M`(@0``#P3RWI'""`XN#KG^7@.Y_E4]]-X@[@C^`$0(WB!!"@X5S`@.(#,)[G
M`#"3Y40QC>4`,*#C!#"2Y#,_O^8"`%SA!#"!Y/K__QH-(*#A_$"$X@(0H.$$
MP++E!`!2X3@PD>4DX)'E#!"1Y0XP(^`!,"/@##`CX.,_H.%`,(+E\___&B!`
MD.F9.0?C@CI%XPQ`D.4$P)WE`&"0Y000+N`0<)#E`\",X`40`>!ED:#A!\",
MX`00(>`(()WEYLV,X`'`C.`)$"[@`R""X`80`>`$((+@#A`AX`(0@>`,()WE
M9H&@X>P=@>`#((+@#J""X`@@*>`,(`+@;,&@X0D@(N`*((+@$*"=Y>$M@N`#
MH(K@":"*X`R0*.`!D`G@81&@X0B0*>`*D(G@%*"=Y>*=B>`#H(K@"*"*X`&`
M+.`"@`C@8B&@X0R`*.`"L"'@"H"(X!B@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@
MX0&0*^`,L"+@"I")X!R@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@
M"H"(X""@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X"2@G>4(
ML`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X"B@G>4)L`O@Z8V(X`.@
MBN`!H(K@:1&@X0*0*^`!L"S@"I")X"R@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@
MX0R`*^`"L"'@"H"(X#"@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@X0&0*^`,L"+@
M"I")X#2@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@"H"(X#B@G>4)
ML`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X#R@G>4(L`O@Z)V)X`.@
MBN`,H(K@:,&@X0&`*^`,L"+@"H"(X$"@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@
MX0*0*^`!L"S@"I")X$2@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0R`*^`*@(C@
M2*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@X0&@*N`,H(K@3,"=Y>BMBN`#
MP(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@4,"=Y>H=@>`#,(S@",`IX`K`
M#.`"((/@"<`LX*$[#N/9/D;C`L",X%0@G>5JH:#AX<V,X`,@@N`)D(+@6""=
MY0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X5P0G>7LC8C@`Q"!X`H0@>`)H"K@
M#*`JX&S!H.$*H(+@8""=Y>BMBN`#((+@"2""X`R0*>`(D"G@:(&@X0F0@>!D
M$)WEZIV)X`,0@>`,$('@",`LX`K`+.!JH:#A#,""X&@@G>7IS8S@`R""X`@@
M@N`*@"C@"8`HX&F1H.$(@('@;!"=Y>R-B.`#$('@"A"!X`F@*N`,H"K@;,&@
MX0J@@N!P()WEZ*V*X`,@@N`)((+@#)`IX`B0*>!H@:#A"9"!X'00G>7JG8G@
M`Q"!X`P0@>`(P"S@"L`LX&JAH.$,P(+@>""=Y>G-C.`#((+@""""X`J`*.`)
M@"C@:9&@X0B`@>!\$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"J""X(`@
MG>7HK8K@`R""X`D@@N`,D"G@")`IX&B!H.$)D('@A!"=Y>J=B>!JL:#A`Q"!
MX`P0@>`(P"S@"L`LX`P@@N#IK8+@B""=Y0,@@N`(((+@"X`HX`G`*.!ID:#A
M#!"!X(S`G>7JC8'@"1`KX`/`C.`*$"'@"\",X)"PG>5JH:#A`1""X`.PB^#H
M'8'@";"+X`J0*>`(("G@:(&@X0(@C.#AG8+@E""=Y0,@@N`*((+@"*`JX`'`
M*N"8H)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!@"C@"8`HX&F1H.$(@(+@G""=
MY>R-B."DL)WE`R""X`$@@N`)$"'@#!`AX&S!H.$!H(K@#!`IX.BMBN`($"'@
M`2""X&B!H.'J'8+@H""=Y0,@@N`(,"S@"2""X&J1H.$#H"K@W#P+XQL_2.,*
M((+@`["+X`F@B.$,L(O@J,"=Y>$M@N`!H`K@`\",X&$1H.$(P(S@"8`(X`B@
MBN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&PL)WE#,"(X&(AH.'JC8S@K,"=
MY0.PB^`#P(S@`;"+X`F0C.`"P('A"L`,X`(0`>`!$(SA:J&@X0F0@>`*$(+A
MZ,V)X+20G>4($`'@:(&@X0.0B>`"D(G@"B`"X`(@@>$($(KA"[""X`P0`>`(
M(`K@[+V+X`(0@>&X()WE"1"!X+R0G>5LP:#A`R""X`J@@N`#D(G@#""(X0B0
MB>`+(`+@#(`(X.L=@>`(@(+A:[&@X0J@B.`+((SA88&@X>&MBN`!(`+@P!"=
MY0,0@>`,$('@"\`,X`S`@N$(((OA"<",X`H@`N`(D`O@ZLV,X`D@@N'(D)WE
M`1""X,0@G>5JH:#A`Y")X`,@@N`(D(G@"[""X`H@B.$,(`+@"H`(X.P=@>`(
M@(+A;,&@X0NPB.`,((KA88&@X>&]B^`!(`+@S!"=Y0,0@>`*$('@#*`*X`J@
M@N$(((SA"9"*X`L@`N`(H`S@ZYV)X`H@@N'4H)WE`1""X-`@G>5KL:#A`Z"*
MX`,@@N`(H(K@#,""X`L@B.$)(`+@"X`(X.D=@>`(@(+A:9&@X0S`B.`)((OA
MX8V,X`$@`N!AP:#AV!"=Y0,0@>`+$('@";`+X`NP@N$,((GA"J"+X`@@`N`,
ML`G@Z*V*X`L@@N'@L)WE`1""X-P@G>5H@:#A`["+X`,@@N`,L(O@"9""X`@@
MC.$*(`+@",`,X.H=@>`,P(+A:J&@X0F0C.`*((CA8<&@X>&=B>`!(`+@Y!"=
MY0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@`N`,L`K@Z8V(X`L@@N'LL)WE
M`2""X.@0G>5ID:#A`["+X`,0@>`,L(O@"J"!X`D0C.$($`'@"<`,X.@M@N`,
MP('A"J",X&B!H.'BS8K@\*"=Y0@0B>$#,(K@`A`!X`DP@^!B(:#A")`)X`*@
MB.$)$('A#*`*X`L0@>`"L`C@[!V!X`N@BN%LD:#A`S"*X/3`G>7AK8/@UC$,
MXV(Z3.,#P(S@8;&@X0B`C.#XP)WE`\",X`+`C.`)("+@`2`BX/P0G>4"@(C@
M"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@"I`KX`B0*>`"P(S@"9"!X`01G>4`
M(9WEZ,V,X`,0@>!H@:#A"A"!X`,@@N`(H"K@"R""X`R@*N#LG8G@"J""X`@A
MG>5LP:#AZ:V*X`,@@N`(((+@#(`HX`F`*.!ID:#A"("!X`P1G>7JC8C@`Q"!
MX`P0@>`)P"S@"L`LX&JAH.$,P(+@$"&=Y>C-C.`#((+@"2""X`J0*>`(D"G@
M:(&@X0F0@>`4$9WE[)V)X`,0@>`*$('@"*`JX`R@*N!LP:#A"J""X!@AG>7I
MK8K@`R""X`@@@N`,@"C@"8`HX&F1H.$(@('@'!&=Y>J-B.`#$('@#!"!X`G`
M+.`*P"S@:J&@X0S`@N`@(9WEZ,V,X`,@@N`)((+@"I`IX`B0*>!H@:#A"9"!
MX"01G>7LG8G@`Q"!X`H0@>`(H"K@#*`JX&S!H.$*H(+@*"&=Y>FMBN`#((+@
M""""X`R`*.`)@"C@:9&@X0B`@>`L$9WEZHV(X`,0@>`,$('@"<`LX`K`+.!J
MH:#A#,""X#`AG>7HS8S@`R""X`D@@N`*D"G@")`IX&B!H.$)$('@-)&=Y>P=
M@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N`XH9WEX2V"X`.@BN`(H(K@#(`H
MX`&`*.!A$:#A")")X#R!G>7BG8G@`X"(X`R`B.`!P"S@`L`LX&(AH.$,P(K@
M`G"'X.G-C.`0<(#E;.&.X`C@@.5`X9WE`S".X`8P@^`!,(/@`A`AX`D0(>!I
MD:#A`1"(X`D@(N#L'8'@#"`BX`(P@^!$()_E`5"%X.$=@^`T,)_E`B"/X`10
M@.4)0(3@`!"`Y0Q`@.4#,)+G`""3Y40QG>4"(#/@`0``&E/?C>+PC[WHU?+_
MZYA1`0`8`0``\$4!`!!`+>E@`*#C]O+_ZP`@H.,!PP+C1<=&XXD;"N/-'T[C
M`,"`Y000@.7^S`WCNLA)XW84!>,R$$'C","`Y0P0@.7P$0[CTA-,XQ0@@.40
M$(#E&""`Y5P@@.40@+WH\$<MZ0*0H.&"(:#A%#"0Y0!0H.$!0*#A'("%X@(P
MD^`8()#E%#"`Y5P`D.4!(((BJ2Z"X!@@A>4``%#C(```&C\`6>,N``#:0)!)
MXBECH.$!<(;B!W.$X`00H.$(,*#A0$"$X@#@D>40$('B#,`1Y1`P@^((`!'E
M!"`1Y00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@%`H.$%`*#AP?S_ZP<`5.'K
M__\:!I-)X`D@H.$'$*#A"`"@X7KR_^M<D(7E\(>]Z$!@8.(``(C@!@!9X0E@
MH+$&(*#A<O+_ZUPPE>4#,(;@7#"%Y4``4^/PA[T8!0"@X0:02>`&0(3@J?S_
MZ\[__^H$<*#AY___ZO!!+>D<<('B`4"@X10PD>4`4*#A&&"1Y7\0X.,SC[_F
MTS'EYS9OO^8!((/B.`!2XP(`A^`#$,?G/```VD`@8N(`$*#CNO+_ZP0`H.&2
M_/_K!P"@X3@@H.,`$*#CM/+_ZP0`H.%48(3E6("$Y8K\_^L#,-3E`##%Y00`
MH.&R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%
MY00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A
M"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/
M,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E\$&]Z!3R
M_^HX(&+B`!"@XWWR_^O'___J\$<MZ0!PH.%,8)_E3%"?Y0&`H.$&8(_@`I"@
MX050C^#J\?_K!6!&X$9AL.'PA[T(!%!%X@!`H.,$,+7E"2"@X0@0H.$'`*#A
M`4"$XC/_+^$$`%;A]___&O"'O>C800$`S$$!`![_+^$(,)_E`S"/X``PD^4-
M\O_JZ$,!``A`+>D(@+WH`0`"`"]T;7``````+@```%!!4E]435!$25(``%1%
M35!$25(`5$U0`%5315(`````55-%4DY!344`````)7,O)7,````O````+W!R
M;V,O<V5L9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!);G1E
M<FYA;',Z.E!!4CHZ0D]/5``````O<')O8R\E:2\E<P!E>&4`4$%27U1%35``
M````4$%27U!23T=.04U%`````#H````E<R5S)7,``%!%4DPU3$E"`````%!%
M4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?24Y)5$E!3$E:140`4$%2
M7U-005=.140`4$%27T-,14%.````4$%27T1%0E5'````4$%27T-!0TA%````
M4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,7U1-4$1)4@```%!!4E]'
M3$]"04Q?5$5-4`!005)?1TQ/0D%,7T-,14%.`````%!!4E]#3$5!3CT``#$`
M``!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````"5S)7,E<R5S````
M`'!A<BT`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y
M("5S(&9A:6QE9"`H97)R;F\])6DI"@``)7,Z('!R:79A=&4@<W5B9&ER96-T
M;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@
M>6]U<B!O<&5R871I;VXI"@``4$%42`````!P87)L`````"YP87(`````)7,E
M<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E<R5S=&5M<"TE=2TE=25S
M`````$%21U8`````&````'!E<FP`````4$%2.CI086-K97(Z.E9%4E-)3TX`
M````,2XP-3````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``
M```E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@
M)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``979A;"`G97AE8R`O
M=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U
M;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H
M)%!!4E]-04=)0RP@)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE*3L*
M14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E
M;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E
M8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?
M=&5M<"D["F]U=',H<7%[4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")]
M*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER
M(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@
M*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@
M/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2
M(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@
M/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*<')I;G0@)'1M<"`B(R$O8FEN
M+W-H"G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A
M<E]T96UP)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*
M9FD*<VQE97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R(@+B`D=&UP
M+3YF:6QE;F%M92`N("(G"B(["F-H;6]D(#`W,#`L)'1M<"T^9FEL96YA;64[
M"FUY("1C;60@/2`D=&UP+3YF:6QE;F%M92`N("<@/B]D978O;G5L;"`R/B8Q
M("8G.PIC;&]S92`D=&UP.PIS>7-T96TH)&-M9"D["F]U=',H<7$H4W!A=VYE
M9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A;G5P.B`I"BX@
M)'1M<"T^9FEL96YA;64I.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2
M.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["B10
M05)?34%'24,@/2`B7&Y005(N<&U<;B(["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5#;VYF:6<@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["E]S971?<&%R7W1E;7`H*3L*;7D@*"1S=&%R=%]P;W,L("1D871A7W!O
M<RD["GL*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["F]P96X@7T9(
M+"`G/"<L("1P<F]G;F%M92!O<B!L87-T.PIB:6YM;V1E*%]&2"D["FUY("1B
M=68["FUY("1S:7IE(#T@+7,@)'!R;V=N86UE.PIM>2`D8VAU;FM?<VEZ92`]
M(#8T("H@,3`R-#L*;7D@)&UA9VEC7W!O<SL*:68@*"1S:7IE(#P]("1C:'5N
M:U]S:7IE*2!["B1M86=I8U]P;W,@/2`P.PI](&5L<VEF("@H;7D@)&T@/2`D
M<VEZ92`E("1C:'5N:U]S:7IE*2`^(#`I('L*)&UA9VEC7W!O<R`]("1S:7IE
M("T@)&T["GT@96QS92!["B1M86=I8U]P;W,@/2`D<VEZ92`M("1C:'5N:U]S
M:7IE.PI]"G=H:6QE("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:RA?1D@L("1M
M86=I8U]P;W,L(#`I.PIR96%D*%]&2"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L
M96YG=&@H)%!!4E]-04=)0RDI.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L
M("1005)?34%'24,I*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D["FQA<W0[
M"GT*)&UA9VEC7W!O<R`M/2`D8VAU;FM?<VEZ93L*?0IL87-T(&EF("1M86=I
M8U]P;W,@/"`P.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`["G)E860@
M7T9(+"`D8G5F+"`T.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0@+2!U;G!A
M8VLH(DXB+"`D8G5F*2P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY("5R
M97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@
M("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D
M8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L
M("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F
M.PIO=71S*'%Q*%5N<&%C:VEN9R!F:6QE("(D9G5L;&YA;64B+BXN*2D["FUY
M("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_
M("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@
M;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E
M860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H
M)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY
M("1F:6QE;F%M92`](%]T96UP9FEL92@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U
M*3L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N
M86UE.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L
M;&YA;64["GT*96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N
M9"!D969I;F5D("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?
M=&5M<&9I;&4H(B1B87-E;F%M921E>'0B+"`D8G5F+"`P-S4U*3L*;W5T<R@B
M4TA,24(Z("1F:6QE;F%M95QN(BD["GT*96QS92!["B1R97%U:7)E7VQI<W1[
M)&9U;&QN86UE?2`]"B1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD9G5L;&YA
M;65](#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U
M;&QN86UE+`I].PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@
M/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@
M<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@9&5L971E("1R
M97%U:7)E7VQI<W1[)&UO9'5L97T@;W(@<F5T=7)N.PHD24Y#>R1M;V1U;&5]
M(#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!
M3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/
M.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@)"$["F)I;FUO9&4H)&9H*3L*
M<')I;G0@)&9H("1I;F9O+3Y[8G5F?3L*<V5E:R@D9F@L(#`L(#`I.PIR971U
M<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?=&5M<&9I;&4H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["F]P96X@;7D@)&9H+"`G
M/"<L("1F:6QE;F%M92!O<B!D:64@(F-A;B=T(')E860@)&9I;&5N86UE.B`D
M(2(["F)I;FUO9&4H)&9H*3L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A
M<'!I;F<@9F%I;&5D.B!C86YN;W0@9FEN9"`D;6]D=6QE(5QN(CL*?2P@0$E.
M0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!
M4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL
M93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?
M;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N
M86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O
M:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE
M('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*
M?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FQA<W0@=6YL
M97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PHD<W1A<G1?<&]S(#T@*'1E;&P@
M7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@
M0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?
M;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE
M("A2155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0HD<75I970@/2`P
M('5N;&5S<R`D14Y6>U!!4E]$14)51WT["FEF("@A)'-T87)T7W!O<R!O<B`H
M)$%21U9;,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E
M9&ES=%]C;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A
M<@IU("`@=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P
M87(**3L*:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*
M<VAI9G0@0$%21U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'
M5CL*?0IE;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T
M;U]I;F,["G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"
M3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U
M<V@@0&%D9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A
M;"`B=7-E("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A
M<E]A<F=S+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*
M?0IE;'-I9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS
M:68@*"0Q(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q
M(&5Q("=Q)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*
M;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92`B6%A8.B!#86YN;W0@;W!E
M;B!L;V<Z("0A(CL*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE
M(#T@)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M
M9'LD,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D[
M"G)E<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N
M;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'
M5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II
M9B`H)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV
M93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF
M="A`05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M
M>2`D9F@L("<\)RP@)'!A<B!O<B!D:64@(D-A;FYO="!F:6YD("<D<&%R)SH@
M)"$B.PIB:6YM;V1E*"1F:"D["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z
M:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(
M86YD;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO
M<B!D:64@(E)E860@)R1P87(G(&5R<F]R.B`D(2(["GT*;7D@)65N=B`](&1O
M('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%
M5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^
M(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<[
M"GT*?3L*;&]C86P@)"\@/2!<-#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N
M(%!!4BP@)SPG+"`D<&%R(&]R(&1I92`B)"$Z("1P87(B.PIB:6YM;V1E*%!!
M4BD["F1I92`B)'!A<B!I<R!N;W0@82!005(@9FEL92(@=6YL97-S(#Q005(^
M(&5Q(")02UPP,#-<,#`T(CL*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\
M($E/.CI&:6QE.CI/7U=23TY,62@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*
M,#<W-RP**2!O<B!D:64@)"$["F)I;FUO9&4H)&9H*3L*)"\@/2`H9&5F:6YE
M9"`D9&%T85]P;W,I(#\@7"1D871A7W!O<R`Z('5N9&5F.PIS965K(%]&2"P@
M,"P@,#L*;7D@)&QO861E<B`]('-C86QA<B`\7T9(/CL*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*9F]R96%C:"!M>2`D
M:V5Y("AS;W)T(&ME>7,@)65N=BD@>PIM>2`D=F%L(#T@)&5N=GLD:V5Y?2!O
M<B!N97AT.PHD=F%L(#T@979A;"`D=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*
M;7D@)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B7U\B.PIM
M>2`D<V5T(#T@(E!!4E\B("X@=6,H)&ME>2D@+B`B/21V86PB.PHD;&]A9&5R
M(#U^('-[)&UA9VEC*"`K*7U["B1M86=I8R`N("1S970@+B`H)R`G('@@*&QE
M;F=T:"@D,2D@+2!L96YG=&@H)'-E="DI*0I]96<["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*3L*)"\@/2!U;F1E9CL*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!0
M05(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN
M:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![
M("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D
M7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#
M;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*;7D@)69I;&5S
M.PHD9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J
M96-T<RP@=F%L=65S("5)3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N
M9FEG>VQI8E]E>'1].PIM>2`E=W)I='1E;CL*9F]R96%C:"`H<V]R="!K97ES
M("5F:6QE<RD@>PIM>2`H)&YA;64L("1F:6QE*3L*9F]R96%C:"!M>2`D9&ER
M("A`:6YC*2!["FEF("@D;F%M92`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)%]]*2!["B1F:6QE(#T@)%\["FQA<W0["GT*96QS:68@*"]>*%Q1)&1I<EQ%
M7"\H+BI;7D-C72DI7%HO:2D@>PHH)&9I;&4L("1N86UE*2`]("@D,2P@)#(I
M.PIL87-T.PI]"F5L<VEF("AM(5XO;&]A9&5R+UM>+UTK+R@N*EM>0V-=*5Q:
M(2D@>PII9B`H;7D@)')E9B`]("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD
M,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"F5L
M<VEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O
M)#$B+"`D,2D["FQA<W0["GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA
M;64@86YD(&YO="`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F
M:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"\["F]U=',H(&IO
M:6X@(B(L"G%Q*%!A8VMI;F<@(BDL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M
M97T@.B`D9FEL92P*<7$H(BXN+BD**3L*;7D@)&-O;G1E;G0["FEF("AR968H
M)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIO
M<&5N($9)3$4L("<\)RP@)&9I;&4@;W(@9&EE(")#86XG="!O<&5N("1F:6QE
M.B`D(2(["F)I;FUO9&4H1DE,12D["B1C;VYT96YT(#T@/$9)3$4^.PIC;&]S
M92!&24Q%.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N=&5N
M="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D520D%424U]
M(&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL=&5R
M.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92P@
M)&YA;64I.PI]"F]U=',H<7$H5W)I='1E;B!A<R`B)&YA;64B*2D["B1F:"T^
M<')I;G0H(D9)3$4B*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N9W1H*"1N
M86UE*2`K(#DI*3L*)&9H+3YP<FEN="AS<')I;G1F*`HB)3`X>"\E<R(L($%R
M8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64**2D[
M"B1F:"T^<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI*3L*)&9H
M+3YP<FEN="@D8V]N=&5N="D["GT*?0HD>FEP+3YW<FET951O1FEL94AA;F1L
M92@D9F@I(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N
M86UE+"`P+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@
M*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W
M*#$I.PIO<&5N*&UY("1F:"P@(CPB+"`D;W5T*3L*8FEN;6]D92@D9F@I.PHD
M8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C86-H95]N86UE(#T@
M)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H-#$@
M+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N/2`B0T%#2$4B
M.PHD9F@M/G!R:6YT*"1C86-H95]N86UE*3L*)&9H+3YP<FEN="AP86-K*"=.
M)RP@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I*2D["B1F:"T^<')I;G0H
M)%!!4E]-04=)0RD["B1F:"T^8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X
M:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?
M<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y
M;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C
M:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@
M("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O
M;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I
M(&]R(&1I92`B)"$Z("1`(CL*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHD>FEP+3YR
M96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2!O<B!D:64@(B0A.B`D0"(["D%R8VAI=F4Z.EII<#HZ
M<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB0V%C:&4L
M("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PHD<75I
M970@/2`A)$5.5GM005)?1$5"54=].PIO=71S*'%Q*%PD14Y6>U!!4E]414U0
M?2`]("(D14Y6>U!!4E]414U0?2(I*3L*:68@*&1E9FEN960@)$5.5GM005)?
M5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A
M8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE
M;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M
M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO
M/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q*%-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(BDI.PI](&5L
M<V4@>PIO=71S*'%Q*$5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B*2D["B1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S
M=%]N86UE*3L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H
M<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M
M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!
M;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@
M6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP
M87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H
M:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`
M7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A
M=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@
M*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK
M<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W
M:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB
M.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER
M92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q
M=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO
M.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*
M<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I
M;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR
M97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@
M0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E
M($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y
M.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q
M=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![
M(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL
M('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE
M=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L
M('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*
M979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*
M:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!]
M(#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A
M8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!4
M35!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@
M+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T
M:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L
M('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF
M("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]
M("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@
M>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%
M3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*)'5S
M97)N86UE(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D<&%T:"1#;VYF
M:6=[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I
M<B`D<W1M<&1I<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!3GT@86YD(&UY
M("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P96X@*&UY("1F
M:"P@(CPB+B`D<')O9VYA;64I.PIS965K("1F:"P@+3$X+"`R.PIS>7-R96%D
M("1F:"P@;7D@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`M-3@L(#(["G-Y<W)E860@)&9H+"`D8G5F+"`T,3L*)&)U9B`]
M?B!S+UPP+R]G.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L:6U]8V%C:&4M
M(B`N("1B=68["GT*96QS92!["FUY("1D:6=E<W0@/2!E=F%L"GL*<F5Q=6ER
M92!$:6=E<W0Z.E-(03L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D[
M"F]P96XH;7D@)&9H+"`B/"(L("1P<F]G;F%M92D["F)I;FUO9&4H)&9H*3L*
M)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D9F@I.PHD8W1X+3YH97AD:6=E
M<W0["GT@?'P@)&UT:6UE.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L:6U]
M8V%C:&4M)&1I9V5S="(["GT*8VQO<V4H)&9H*3L*?0IE;'-E('L*)$5.5GM0
M05)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?71E
M;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FUK9&ER("1S
M=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B
M(%]T96UP9FEL92!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIO<&5N(&UY("1F:"P@)SXG+"`D=&5M<&YA;64@;W(@9&EE(")C86XG="!W
M<FET92`D=&5M<&YA;64Z("0A(CL*8FEN;6]D92`D9F@["G!R:6YT("1F:"`D
M8V]N=&5N=',["F-L;W-E("1F:#L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I
M9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I
M(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*
M<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'
M3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R
M;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%2
M7U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^
M/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D
M<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[
M4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E
M;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE
M*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D0V]N
M9FEG>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)$-O;F9I9WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D0V]N9FEG>W!A
M=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6
M>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]
M?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D
M:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG>U]E>&5](BDL(&QA
M<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I
M9WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD
M<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD
M<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G
M;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N
M86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1#
M;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D
M("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97
M:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO
M;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)$-O;F9I9WM?
M9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'
M3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H
M("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E
M;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!3
M4$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*
M9&5L971E("1%3E9[)U!!4E\G+B1??3L*?0IF;W(@*'%W+R!435!$25(@5$5-
M4"!#3$5!3B!$14)51R`O*2!["B1%3E9[)U!!4E\G+B1??2`]("1%3E9[)U!!
M4E]'3$]"04Q?)RXD7WT@:68@97AI<W1S("1%3E9[)U!!4E]'3$]"04Q?)RXD
M7WT["GT*;7D@)'!A<E]C;&5A;B`](")?7T5.5E]005)?0TQ%04Y?7R`@("`@
M("`@("`@("`@("(["FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.
M5GM005)?0TQ%04Y].PI]"F5L<VEF("@A97AI<W1S("1%3E9[4$%27T=,3T)!
M3%]#3$5!3GTI('L*;7D@)'9A;'5E(#T@<W5B<W1R*"1P87)?8VQE86XL(#$R
M("L@;&5N9W1H*")#3$5!3B(I*3L*)$5.5GM005)?0TQ%04Y](#T@)#$@:68@
M)'9A;'5E(#U^("]>4$%27T-,14%./2A<4RLI+SL*?0I]"G-U8B!O=71S('L*
M<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@
M(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I
M;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L
M(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV
M;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V
M96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU
M;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A
M<E]A<F=S*3L*9&EE('%Q*'!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP
M="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN*0IU
M;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H
M)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I
M9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H
M)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*
M9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`````
M<&5R;'AS:2YC````7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@("``
M````0(__?P$`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````"X&@``4!H```$```!6`P```0```&$#
M```!````<0,```$```![`P``#P````,%```,````>!,```T````D3```&0``
M`)B-`0`;````!````!H```"<C0$`'`````0```#U_O]OM`$```4````T"@``
M!@```%0#```*````)@4```L````0````%0`````````#````N(X!``(`````
M`@``%````!$````7````>!$``!$```"8$```$@```.`````3````"````!@`
M````````^___;P$```C^__]O.!```/___V\#````\/__;UH/``#Z__]O$```
M``````````````````````````````````````````````````````"@C0$`
M``````````"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$
M$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03
M``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,`
M`(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``
MA!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$
M$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``#$P`````
M````````````````````````````````````````J$L```````"8%@``````
M``````````````````20`0#_____,$P``#A,````````/$P``$!,``!(3```
MS$P``%!,````````5$P``%Q,````````:$\``$=#0SH@*$=.52D@,3`N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(S+C`N-#$X+60U.3`I(#$P+C(N,`!!
M,@```&%E86)I``$H````!3<M00`&"@=!"`$)`@H$$@03`10!%0$7`Q@!&0$:
M`AP!(@$`+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU+F)U:6QD+6ED`"YN
M;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU+G9E
M<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;"YD>6X`+G)E;"YP;'0`+FEN:70`
M+G1E>'0`+F9I;FD`+G)O9&%T80`N05)-+F5X:61X`"YE:%]F<F%M90`N:6YI
M=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`
M+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````````````````````````
M```````````````````````````````+`````0````(```!4`0``5`$``!D`
M``````````````$`````````$P````<````"````<`$``'`!```D````````
M```````$`````````"8````'`````@```)0!``"4`0``(```````````````
M!``````````T````]O__;P(```"T`0``M`$``*`!```%``````````0````$
M````/@````L````"````5`,``%0#``#@!@``!@````,````$````$````$8`
M```#`````@```#0*```T"@``)@4``````````````0````````!.````____
M;P(```!:#P``6@\``-P````%``````````(````"````6P```/[__V\"````
M.!```#@0``!@````!@````,````$`````````&H````)`````@```)@0``"8
M$```X`````4`````````!`````@```!S````"0```$(```!X$0``>!$````"
M```%````%0````0````(````?`````$````&````>!,``'@3```,````````
M```````$`````````'<````!````!@```(03``"$$P``%`,`````````````
M!`````0```""`````0````8```"8%@``F!8``(PU``````````````0`````
M````B`````$````&````)$P``"1,```(```````````````$`````````(X`
M```!`````@```"Q,```L3```TCT`````````````!`````````"6`````0``
M<((`````B@```(H```@````-``````````0`````````H0````$````"````
M"(H```B*```$```````````````$`````````*L````.`````P```)B-`0"8
MC0``!```````````````!`````0```"W````#P````,```"<C0$`G(T```0`
M``````````````0````$````PP````8````#````H(T!`*"-```8`0``!@``
M```````$````"````,P````!`````P```+B.`0"XC@``2`$`````````````
M!`````0```#1`````0````,`````D`$``)```$````````````````0`````
M````UP````@````#````0)`!`$"0```8$``````````````$`````````-P`
M```!````,`````````!`D```/0```````````````0````$```#E`````P``
M<```````````?9```#,```````````````$``````````0````,`````````
<`````+"0``#U```````````````!````````````
